import gnupg
import tempfile
import struct
import gzip
import io
import os
import hashlib
import shutil
import tarfile
import glob
from lxml import etree
import urllib.request as urlreq
from urllib.error import HTTPError, URLError
from viavi.upgrade import get_user_agent, get_hooks, run_hook, remove_old_files, PrintException, synchronize, sync_password
from viavi.stdhw.bootutils import bootenv
from viavi.stdhw.blockutils import get_root_partition, mount, umount, mounted_point, sync, MountDisk, MountBind
from viavi.upgrade.flashutils import mount_chroot, umount_chroot, chroot_exec, format_device

def open_vui_source(release, proxy=None):
    if release.startswith("http://") or release.startswith("https://"):
        if proxy is not None and proxy.strip() != "DIRECT":
            print("Enable proxy %s" % proxy)
            proxy_handler = urlreq.ProxyHandler({'http': proxy})
        else:
            print("Disable proxy")
            proxy_handler = urlreq.ProxyHandler({})

        opener = urlreq.build_opener(proxy_handler)
        opener.addheaders = [('User-Agent', get_user_agent())]
        urlreq.install_opener(opener)

        # Check mymirror.php
        base_url = release.replace("http://", "").split("/")[0]
        mymirror_url = "http://%s/mymirror.php" % (base_url)
        req = urlreq.Request(mymirror_url)
        try:
            mirror = urlreq.urlopen(req).read().decode('ascii')
            release = release.replace(base_url, mirror)
            url = release.replace(base_url, mirror)
            print("Release with mirror: %s" % release)
        except HTTPError:
            pass

        release_file = "/var/volatile/release.xml"
        req = urlreq.Request(release)
        try:
            vui = urlreq.urlopen(req)
        except HTTPError as e:
            print("HTTP Error:%s %s"%(e.code, release))
            PrintException()
            return None
        except URLError as e:
            print("URL Error: %s %s"%(e.reason, release))
            PrintException()
            return None
        return vui
    elif release.startswith("file://"):
        url = release.replace("file://", "")
    else:
        url = release
    vui = open(url, "rb")
    return vui

def release_verify(releasexml, releasesig):
    sig = io.BytesIO(releasesig)
    with tempfile.NamedTemporaryFile() as xmlf:
        xmlf.write(releasexml)
        xmlf.flush()
        gpg = gnupg.GPG()
        return gpg.verify_file(sig, xmlf.name)

def read_vui_header(vui):
    header = vui.read(12)
    (v,u,i,version,lenxml,lensig) = struct.unpack("<cccBII", header)
    if v != b'V' or u != b'U' or i != b'I':
        print(f"Invalid VUI file")
        return (None,None)
    xmlc = vui.read(lenxml)
    releasexml = gzip.decompress(xmlc)
    sigc = vui.read(lensig)
    releasesig = gzip.decompress(sigc)
    return (releasexml, releasesig)

def create_vui(rxml):
    releasexml = os.path.abspath(rxml)
    file_name, file_extension = os.path.splitext(releasexml)
    path = os.path.dirname(releasexml)
    if file_extension != ".xml":
        print("ERROR: source file must be release.xml")
        return -1
    vuif = None
    with open(f"{path}/release.xml", "rb") as xml, open(f"{path}/release.xml.sig", "rb") as sig:
        releasexml = xml.read()
        releasesig = sig.read()
        verified = release_verify(releasexml, releasesig)
        if not verified:
            print(verified.stderr)
            return -3
        tree = etree.fromstring(releasexml)
        machine = tree.xpath('//release/@device')[0]
        version = tree.xpath('//item[@name = "File System"]/@version')[0]
        archive = tree.xpath('//archive/@file')[0]
        vuif = "%s-%s.vui"%(machine, version)
        with open(vuif, "wb") as vui:
            xmlc = gzip.compress(releasexml)
            sigc = gzip.compress(releasesig)
            vui.write(struct.pack("<cccBII", b'V', b'U', b'I', 1, len(xmlc), len(sigc)))
            vui.write(xmlc)
            vui.write(sigc)
            with open(f"{path}/{archive}", "rb") as tgz:
                key = struct.pack("cccB", b'V', b'U', b'I', 1)
                data = tgz.read(4)
                xdata = struct.pack("<I", struct.unpack('<I', data)[0] ^ struct.unpack('<I', key)[0])
                while xdata:
                    vui.write(xdata)
                    xdata = tgz.read(512)
    return vuif

def extract_vui(vuif, xmlonly=False, proxy=None):
    if not vuif.endswith(".vui"):
        print("ERROR: Source for extract must a .vui file")
        parser.print_help()
        return -1
    vui = open_vui_source(vuif, None)
    (releasexml, releasesig) = read_vui_header(vui)
    if releasexml is None:
        return -2
    verified = release_verify(releasexml, releasesig)
    if not verified:
        print(verified.stderr)
        return -3
    with open("release.xml", "wb") as xmlf:
        xmlf.write(releasexml)
    if xmlonly:
        return 0
    with open("release.xml.sig", "wb") as sigf:
        sigf.write(releasesig)
    tree = etree.fromstring(releasexml)
    machine = tree.xpath('//release/@device')[0]
    archive = tree.xpath('//archive/@file')[0]

    with open(archive, "wb") as tgz:
        key = struct.pack("cccB", b'V', b'U', b'I', 1)
        data = vui.read(4)
        xdata = struct.pack("<I", struct.unpack('<I', data)[0] ^ struct.unpack('<I', key)[0])
        while xdata:
            tgz.write(xdata)
            xdata = vui.read(512)
    return 0


def notify_set_action(text):
    print(text)

def notify_expose(topic, percent, subkey, subtopic, subpercent):
    print("%s %i%% %s %s %i%%" % (topic, percent, subkey, subtopic, subpercent))


def StreamDecorator(original, tarsize, notify):
    setattr(original.__class__, "oldread", original.read)
    setattr(original.__class__, "notify", notify)
    original.lastpercent = 0
    original.tarsize = int(tarsize)
    original.currentsize = 0
    original.dlmd5 = hashlib.md5()

    def myread(self, size):
        percent = int(self.currentsize * 100 / self.tarsize)
        if percent != self.lastpercent:
            self.notify(percent)
            self.lastpercent = percent

        if self.currentsize == 0:
            key = struct.pack("cccB", b'V', b'U', b'I', 1)
            data = self.oldread(4)
            chuck = struct.pack("<I", struct.unpack('<I', data)[0] ^ struct.unpack('<I', key)[0])
            chuck += self.oldread(size - 4)
            self.currentsize += size
        else:
            chuck = self.oldread(size)

        self.dlmd5.update(chuck)
        self.currentsize += size
        return chuck

    myread.__doc__ = "My read"
    myread.__name__ = "read"
    setattr(original.__class__, "read", myread)
    return original


def install_vui(vuif, options=None, mmc_part_new=None):
    result = True
    if options is None:
        options = {
            'proxy': None,
            'flash_block_device': get_root_partition(),
            'parts': [5, 6],
            'synchronize': True,
            'current_rootfs': '',
            'auto_switch': True,
            'test_update_part': False
        }
    vui = open_vui_source(vuif, options['proxy'])
    (releasexml, releasesig) = read_vui_header(vui)
    if releasexml is None:
        return -2
    verified = release_verify(releasexml, releasesig)
    if not verified:
        print(verified.stderr)
        return -3
    tree = etree.fromstring(releasexml)
    machine = tree.xpath('//release/@device')[0]
    archive = tree.xpath('//archive/@file')[0]
    name = tree.xpath("//release/@version")[0]
    sizef = int(tree.xpath("//archive/@size")[0])
    md5f = tree.xpath("//archive/@md5")[0]
    releaseid = tree.xpath("//release/@id")[0]

    model = ""
    with open("/proc/device-tree/model") as ifile:
        model = ifile.read().strip().strip("\x00")
    if machine != model:
        print("Release for %s not for %s" % (machine, model))
        return -4

    hook = get_hooks(tree)
    res = run_hook(hook, 'pre_install_check')

    if os.path.exists('/etc/releasemanager.conf'):
        config = ConfigParser()
        config.read('/etc/releasemanager.conf')
        if config.has_option('GENERAL', 'parts'):
            parts = config.get('GENERAL', 'parts')
            options["parts"] = list(map(int, parts.strip().split(',')))
        print("Imported config is: ")
        print(f"  Parts: {self.__parts}")

    try:
        bootenv.refresh()
        mmc_part = int(bootenv.mmc_part)
        if mmc_part_new is None:
            if mmc_part == options['parts'][0]:
                mmc_part_new = options['parts'][1]
            else:
                mmc_part_new = options['parts'][0]
    except:
        if mmc_part_new is None:
            mmc_part_new = options['parts'][1]
    print("mmc_part is %s" % mmc_part_new)

    # Unmount the partition about to be formatted (all mounts)
    for mount_point in mounted_point(options['flash_block_device'] + "p%s" % mmc_part_new):
        umount(mount_point)

    run_hook(hook, 'pre_vui_install')
    notify_set_action("1: Format partitions")
    if not os.path.exists("/mnt/fs"):
        os.makedirs("/mnt/fs")
    notify_expose("Upgrade", 1, name, "Prepare filesystem %s" % mmc_part_new, 20)
    format_device(mmc_part_new)
    notify_expose("Upgrade", 2, name, "Prepare filesystem", 40)

    with MountDisk(options['flash_block_device'] + "p%s" % mmc_part_new, "/mnt/fs/", "ext4") as fs:
        notify_expose("Upgrade", 3, name, "Prepare filesystem", 60)
        if not os.path.exists("/mnt/fs/user/disk"):
            os.makedirs("/mnt/fs/user/disk")
        notify_expose("Upgrade", 4, name, "Prepare filesystem", 80)

        with MountBind("/user/disk", "/mnt/fs/user/disk") as fsuser:
            notify_expose("Upgrade", 5, name, "Prepare filesystem", 100)
            notify_set_action("2: Install release")

            run_hook(hook, 'pre_tar_untar')

            def print_gui(self, percent):
                notify_expose("Upgrade", 6 + int(percent * 90/100), name, "Installation", percent)

            f = StreamDecorator(vui, sizef, print_gui)
            tar = tarfile.open(fileobj=f, mode='r|gz', errorlevel=1)
            tar.extractall("/mnt/fs")
            sync()
            run_hook(hook, 'post_tar_untar')

            notify_set_action("5: Check release")
            notify_expose("Upgrade", 96, name, "Check integrity", 100)

            if md5f.lower() != f.dlmd5.hexdigest().lower():
                print("BAD SIGNATURE %s %s", md5f.lower(), f.dlmd5.hexdigest().lower())
                notify_set_action("ERROR: BAD SIGNATURE")
                return

            # Prepare CHROOT environement
            try:
                shutil.copyfile("/etc/resolv.conf", "/mnt/fs/etc/resolv.conf")
            except shutil.SameFileError:
                pass

            # Copy all file explicitly listed in /etc/release/backup
            run_hook(hook, 'pre_synchronize')
            if options['synchronize']:
                notify_set_action("6: Synchronize")
                backup = []
                if os.path.exists("{}/etc/release/backup".format(options['current_rootfs'])):
                    for f in os.listdir("{}/etc/release/backup".format(options['current_rootfs'])):
                        with open("{}/etc/release/backup/{}".format(options['current_rootfs'], f)) as fi:
                            backup += [line.strip() for line in fi if line.strip() != '']
                if os.path.exists("/mnt/fs/etc/release/backup"):
                    for f in os.listdir("/mnt/fs/etc/release/backup"):
                        with open("/mnt/fs/etc/release/backup/%s" % f) as fi:
                            backup += [line.strip() for line in fi if line.strip() != '']
                run_hook(hook, 'pre_synchronize_list')

                for bk in set(backup):
                    override = False
                    if bk.startswith("@"):
                        override = True
                        bk = bk[1:].strip()
                    for fn in glob.glob(options['current_rootfs'] + bk):
                        if len(options['current_rootfs']) > 0:
                            print("Sync {} {}".format(fn, fn.replace(options['current_rootfs'], "/mnt/fs")))
                            synchronize(fn, fn.replace(options['current_rootfs'], "/mnt/fs"), override)
                        else:
                            print("Sync {} {}".format(fn, "/mnt/fs/%s" % fn))
                            synchronize(fn, "/mnt/fs/%s" % fn, override)
            run_hook(hook, 'post_synchronize')

            notify_set_action("7: Run post install (don't turn off device)")
            with open("/mnt/fs/etc/release/%s.xml" % releaseid, "wb") as currentxml:
                currentxml.write(releasexml)
            os.system("rm -f /mnt/fs/etc/release/current.xml")
            os.system("ln -s /etc/release/%s.xml /mnt/fs/etc/release/current.xml" % releaseid)

            # Remove old files form /users/disk declared in /etc/release/remove
            remove_old_files("/mnt/fs")

            # Manage password changes
            sync_password("", "/mnt/fs")

            try:
                mount_chroot()
                chroot_exec("/mnt/fs", ["/usr/sbin/run-postinsts"])
            except:
                result = False
                PrintException()
            finally:
                umount_chroot()

            notify_set_action("8: Clean installation")
            notify_expose("Upgrade", 90, name, "Clean installation", 100)

        if os.path.isdir("/mnt/fs/var/lib") and not os.path.isdir("/mnt/fs/var/lib/release-manager"):
            os.mkdir("/mnt/fs/var/lib/release-manager")
        with open("/mnt/fs/var/lib/release-manager/upgrade_status", "w") as f:
            f.write("Success")

    if result:
        run_hook(hook, 'pre_enable')
        notify_set_action("8: Enable new installation")
        notify_expose("Upgrade", 100, name, "Enable new installation", 100)
        if options['auto_switch']:
            print("Setting permanent boot part")
            bootenv.mmc_part = mmc_part_new
        if options['test_update_part']:
            print("Setting temp boot part")
            bootenv.mmc_part_volatile = mmc_part_new
        notify_expose("Upgrade", 100, name, "Finished successful", 100)
        run_hook(hook, 'post_enable')
        notify_set_action("Upgrade Terminate")
    run_hook(hook, 'post_vui_upgrade')
    return result
