import numpy as np
from ctypes import Structure, c_char, c_ulong, c_uint16, c_uint32
from fcntl import ioctl


class FrameBuffer(object):
    '''
    Class to directly write on framebuffer
    '''

    # ioctls
    _FBIOGET_VSCREENINFO = 0x4600
    _FBIOGET_FSCREENINFO = 0x4602
    _FBIOGET_CON2FBMAP = 0x460F

    class _FixedScreenInfo(Structure):
        _fields_ = [
            ('id', c_char * 16),            # identification string eg "TT Builtin"
            ('smem_start', c_ulong),        # Start of frame buffer mem (physical address)
            ('smem_len', c_uint32),		    # Length of frame buffer mem
            ('type', c_uint32),             # see FB_TYPE_*
            ('type_aux', c_uint32),         # Interleave for interleaved Planes
            ('visual', c_uint32),           # see FB_VISUAL_*
            ('xpanstep', c_uint16),		    # zero if no hardware panning
            ('ypanstep', c_uint16),		    # zero if no hardware panning
            ('ywrapstep', c_uint16),        # zero if no hardware ywrap
            ('line_length', c_uint32),      # length of a line in bytes
            ('mmio_start', c_ulong),        # Start of Memory Mapped I/O (physical address)
            ('mmio_len', c_uint32),         # Length of Memory Mapped I/O
            ('accel', c_uint32),            # Indicate to driver which specific chip/card we have
            ('capabilities', c_uint16),     # see FB_CAP_*
            ('reserved', c_uint16 * 2),     # Reserved for future compatibility
        ]

    class _VariableScreenInfo(Structure):

        class _Bitfield(Structure):
            _fields_ = [
                ('offset', c_uint32),       # beginning of bitfield
                ('length', c_uint32),       # length of bitfield
                ('msb_right', c_uint32),    # != 0 : Most significant bit is right
            ]

        _fields_ = [
            ('xres', c_uint32),             # visible resolution
            ('yres', c_uint32),
            ('xres_virtual', c_uint32),     # virtual resolution
            ('yres_virtual', c_uint32),
            ('xoffset', c_uint32),          # offset from virtual to visible
            ('yoffset', c_uint32),          # resolution
            ('bits_per_pixel', c_uint32),   # guess what
            ('grayscale', c_uint32),        # 0 = color, 1 = grayscale, >1 = FOURCC
            ('red', _Bitfield),             # bitfield in fb mem if true color,
            ('green', _Bitfield),           # else only length is significant
            ('blue', _Bitfield),
            ('transp', _Bitfield),          # transparency
            ('nonstd', c_uint32),           # != 0 Non standard pixel format
            ('activate', c_uint32),         # see FB_ACTIVATE_*
            ('height', c_uint32),           # height of picture in mm
            ('width', c_uint32),            # width of picture in mm
            ('accel_flags', c_uint32),      # (OBSOLETE) see fb_info.flags
            # Timing: All values, in pixclocks, except pixclock (of course)
            ('pixclock', c_uint32),         # pixel clock in ps (pico seconds)
            ('left_margin', c_uint32),      # time from sync to picture
            ('right_margin', c_uint32),     # time from picture to sync
            ('upper_margin', c_uint32),     # time from sync to picture
            ('lower_margin', c_uint32),
            ('hsync_len', c_uint32),        # length of horizontal sync
            ('vsync_len', c_uint32),        # length of vertical sync
            ('sync', c_uint32),             # see FB_SYNC_*
            ('vmode', c_uint32),            # see FB_VMODE_*
            ('rotate', c_uint32),           # angle we rotate counter clockwise
            ('colorspace', c_uint32),       # colorspace for FOURCC-based modes
            ('reserved', c_uint32 * 4),     # Reserved for future compatibility
        ]

    def __init__(self, file):
        self.__file = file
        self._fixed_info = self._FixedScreenInfo()
        with open(self.__file, "rb") as fd:
            ioctl(fd, self._FBIOGET_FSCREENINFO, self._fixed_info)
            self._variable_info = self._VariableScreenInfo()
            ioctl(fd, self._FBIOGET_VSCREENINFO, self._variable_info)
            if self._variable_info.bits_per_pixel == 16:
                self.__buffer = np.memmap(self.__file, dtype=np.uint16, mode='r+', shape=(480,800))
            else:
                self.__buffer = np.memmap(self.__file, dtype=np.uint32, mode='r+', shape=(480,800))

    def get_pixel(self,color):
        if type(color) == int:
            r = (color & 0xFF0000) >> 16
            g = (color & 0x00FF00) >> 8
            b = (color & 0x0000FF)
        if type(color) == tuple:
            (r,g,b) = color
        if self._variable_info.bits_per_pixel == 32:
            val = 0xFF000000
        else:
            val = 0x0000
        val |= ((r >> (8 - self._variable_info.red.length))   << self._variable_info.red.offset)
        val |= ((g >> (8 - self._variable_info.green.length)) << self._variable_info.green.offset)
        val |= ((b >> (8 - self._variable_info.blue.length))  << self._variable_info.blue.offset)
        return val

    def setpix(self, x, y, color):
        pixel = self.get_pixel(color)
        if y < len(self.__buffer) and x < len(self.__buffer[0]):
            self.__buffer[y, x] = pixel
        else:
            print("ERROR: (%i, %i) out of screen"%(x,y))

    def rectangle(self, x, y, w, h, color):
        pixel = self.get_pixel(color)
        self.__buffer[y:y+h, x:x+w] = pixel

    def clear(self, color=0):
        pixel = self.get_pixel(color)
        self.__buffer.fill(pixel)
