# -*- coding: utf-8 -*-
import re
import os
import sys
from xml.dom.minidom import parse
import traceback
from . import *
from .dynmodule import get_dynamic_module
import xml.etree.ElementTree as etree

re_py = re.compile("^.*\.py$")
re_items = re.compile("^.*\.items.xml$")
re_events = re.compile("^.*\.events.xml$")


class Project(object):
    '''
    Class of projects,
    This is the class who process event, run command, parse line...
    '''

    def __init__(self, dirname, basedir):
        '''
        Constructor
        '''
        self.__name = dirname
        self.__basedir = basedir

    def reload(self):
        '''
        Reload or first load of all the project
        '''
        if (env("mode.verbose") == "on"):
            if self.__patterns is None:
                print_ok("Load project "+self.__name)
            else:
                print_ok("Reload project "+self.__name)
        self.__patterns = {}
        self.__source = []
        self.__events = []
        self.__listmodules = {}

        toresolve = []
        toresolve.append(self.__name)
        toload = []
        if os.path.isfile(self.__name+"/depends.xml"):
            xml = parse(self.__name+"/depends.xml")
            for elem in xml.getElementsByTagName("package"):
                deps = elem.childNodes[0].nodeValue
                if(deps not in P().List()):
                    if(deps not in toresolve):
                        toload.append(deps)
        for dep in toload:
            if(dep != self.__name):
                P().load(dep)
        toresolve.remove(self.__name)

        pdir = os.listdir(self.__basedir)
        for file in pdir:
            if file == "__init__.py":
                continue
            if re_events.match(file):
                pattern = self.__ImportPatterns(self.__basedir+"/"+file)
                for key in list(pattern.keys()):
                    self.__patterns[key] = pattern[key]
            if re_items.match(file):
                commands = self.__ImportItems(self.__basedir+"/"+file)
            if re_py.match(file):
                self.__AddSource(self.__basedir+"/"+file)
                if (env("mode.verbose") == "on"):
                    O().print_info("Load "+file)
                f = open(self.__basedir+"/"+file)

                modulename = self.__name.replace("/", ".") + "." \
                    + file[0:file.find(".")]
                importobject = __import__(modulename,
                                          fromlist=modulename.split(".")[-1])
                if('initialize' in importobject.__dict__):
                    importobject.__dict__['initialize'](E())
            if os.path.isdir(self.__basedir+"/"+file):
                if file.find(".") == -1:
                    if (env("autoload." + self.__name.replace("/", ".") +
                            "." + file) == "on"):
                        P().load(self.__basedir+"/"+file)
        E().send_event(self.__name.replace("/", "-"), "")

    def __AddSource(self, file):
        '''
        Add a new file to __source with last modification time
        '''
        date = os.path.getmtime(file)
        self.__source.append((date, file))

    def __ImportItems(self, file):
        '''
        Load item from *.items.xml file
        '''
        self.__AddSource(file)
        xml = parse(file)
        if (env("mode.verbose") == "on"):
            print_info("Load "+file)
        for item in xml.getElementsByTagName("item"):
            env(item.getAttribute("name"), item.getAttribute("value"))

    def __ImportPatterns(self, filename):
        '''
        Load pattern from *.events.xml file
        '''
        self.__AddSource(filename)
        if (env("mode.verbose") == "on"):
            E().print_info("Load "+filename)
        linePatterns = {}
        for pattern in etree.parse(filename).findall("pattern"):
            event = pattern.get("event")
            name = pattern.get("name")
            if not name:
                name = event
            linePatterns[name] = {}
            linePatterns[name]["event"] = event
            level = pattern.get("level")
            if level:
                linePatterns[name]["Level"] = level
            counter = pattern.get("counter")
            if counter:
                linePatterns[name]["Counter"] = int(counter)
            else:
                linePatterns[name]["Counter"] = -1
            linePatterns[name]["Pattern"] = pattern.find("exp").text
            linePatterns[name]["Results"] = {}
            groups = pattern.find("groups")
            if(groups is not None):
                for res in groups.findall("res"):
                    linePatterns[name]["Results"][int(res.get("id"))] = \
                        res.get("var")
        return linePatterns

    def AddEvent(self, eventname, modulename, callname):
        '''
        Add a new event in process events list
        '''
        self.__events.append((eventname, modulename, callname))

    def RunEvent(self, event):
        '''
        Execute the event (run all callbacks attached to the event)
        '''
        for ev in self.__events:
            if ev[0] == event.type:
                modulename = ev[1]
                funcname = ev[2]
                modesafe = env("mode.safe")
                good = True
                if modulename not in self.__listmodules:
                    try:
                        self.__listmodules[modulename] = \
                            get_dynamic_module(modulename)
                    except Exception as err:
                        E().print_error("##!!! Error in " + modulename +
                                        ".py: " + str(err))
                        traceback.print_exc()
                        good = False
                        if modesafe == "off":
                            raise err
                if good:
                    try:
                        self.__listmodules[modulename].function(funcname)(
                            E(), event)
                    except Exception as err:
                        E().print_error("##!!! Error in " + modulename + "."
                                        + funcname + "() " + str(err))
                        traceback.print_exc()
                        if modesafe == "off":
                            raise err

    def name(self):
        '''
        Return the directory of the project
        '''
        return self.__name

    def modules(self):
        '''
        Return list of modules used
        '''
        return self.__listmodules

    def PrintEventsOf(self, modulesname):
        '''
        Print all events of this project
        '''
        for ev in self.__events:
            if ev[1] == modulesname:
                E().print_ok("    @ " + ev[0] + " "*(30-len(ev[0])) + " -> "
                             + ev[1] + "~" + ev[2] + "()")

    def GetPatterns(self):
        '''
        Get list of pattern used by this project
        '''
        return self.__patterns

    def GetSources(self):
        '''
        Get list of source files
        '''
        return self.__source

    def GetFunctionsForEvent(self, event):
        '''
        Return list of callbacks call by an event
        '''
        functions = []
        for ev in self.__events:
            if ev[0] == event:
                modulename = ev[1]
                funcname = ev[2]
                modesafe = env("mode.safe")
                good = True
                if modulename not in self.__listmodules:
                    try:
                        self.__listmodules[modulename] = \
                            get_dynamic_module(modulename)
                    except Exception as err:
                        E().print_error("##!!! Error in " + modulename +
                                        ".py: " + str(err))
                        traceback.print_exc()
                        good = False
                        if modesafe == "off":
                            raise err
                if good:
                    try:
                        functions.append(
                            self.__listmodules[modulename].GetFunction(
                                funcname))
                    except Exception as err:
                        E().print_error("##!!! Error in " + modulename + "." +
                                        funcname + "() " + str(err))
                        traceback.print_exc()
                        if modesafe == "off":
                            raise err
        return functions


class ProjectsList(object):
    '''
    Class to manage all Projects
    '''

    def __init__(self):
        '''
        Constructor
        '''
        self.__projects = {}
        self.__todel = []
        __builtins__["projects"] = self

    def __getitem__(self, index):
        k = ".".join(index.split(".")[:-1])
        if k not in self.__projects.keys():
            print_error("ERROR WITH %s" % k)
            print_error("\thave %s" % self.__projects.keys())
            return None
        return self.__projects[k]

    def load(self, dirname):
        '''
        Load a new project in system, this parse all file, attache
        functions to event,
        add new command and add new template to line analyser
        '''
        basedir = None
        if dirname.strip() in self.__projects.keys():
            print_error(dirname+" allready loaded")
        else:
            print_ok("Load "+dirname)
            if os.path.isdir(dirname):
                basedir = dirname
            else:
                for path in E().get_path():
                    if os.path.isdir("%s/%s" % (path, dirname)):
                        basedir = "%s/%s" % (path, dirname)
            if basedir is None:
                print_error("Can't find %s in %s" % (dirname, " ".join(E().get_path())))
            else:
                self.__projects[dirname.strip()] = Project(dirname, basedir)
                self.__projects[dirname.strip()].reload()

    def Unload(self, dirname):
        '''
        Unload a project,
        This revert all things done with load
        '''
        if dirname.strip() in self.__projects.keys():
            print_ok("Unload "+dirname)
            self.__todel.append(dirname.strip())

    def RunEvent(self, event):
        '''
        Execute the event on each projects
        '''
        self.__todel = []

        for k, project in self.__projects.items():
            project.RunEvent(event)

        for d in self.__todel:
            del self.__projects[d]

    def List(self):
        '''
        List all projects name
        (name is the directory path)
        '''
        res = []
        for k, project in self.__projects.items():
            res.append(project.name())
        res.sort()
        return res

    def GetListProjects(self):
        '''
        Return the dict object
        '''
        return self.__projects.values()

    def PrintEventsOf(self, modulesname):
        '''
        Ask to each projects to print event list
        '''
        for k, project in self.__projects.items():
            project.PrintEventsOf(modulesname)

    @classmethod
    def create(cls):
        if "projects" not in __builtins__.keys():
            __builtins__["projects"] = cls()
        return __builtins__["projects"]
