#!/bin/bash
### BEGIN INIT INFO
# Provides:          enhance-security
# Required-Start:
# Required-Stop:
# Default-Start:     S
# Default-Stop:
# Short-Description: Configure other service launch according to security level
# Description:
### END INIT INFO

SERVICE_TARGETED=("rpcbind.socket" "rpcbind")

# Source platform specific code
source /usr/bin/enhance-security-functions.sh

function reset_known_host() {
  # Changing from weak to strong, remove known_host file to avoid server fingerprint mismatch
  rm -f /home/secuser/.ssh/known_hosts
  rm -f /root/.ssh/known_hosts
}

function set_ssh_no_console_mode_login() {

  sshd_cnf="/etc/ssh/sshd_config"

  #  Disable Root login
  prmt=$(cat $sshd_cnf | grep "^PermitRootLogin" | wc -l 2>/dev/null)
  if [ $prmt -ge 1 ]; then
    sed -i 's/^PermitRootLogin.*/PermitRootLogin no/g' $sshd_cnf
  else
    echo "PermitRootLogin no" >>$sshd_cnf
  fi

  # Disable root password
  passwd -l root

  #  Disable password login for all users
  pauth=$(cat $sshd_cnf | grep "^PasswordAuthentication" | wc -l 2>/dev/null)
  if [ $pauth -ge 1 ]; then
    sed -i 's/^PasswordAuthentication.*/PasswordAuthentication no/g' $sshd_cnf
  else
    echo "PasswordAuthentication no" >>$sshd_cnf
  fi

  # ALWAYS disable PortForwarding in this mode
  sed -i 's/^AllowTcpForwarding yes/AllowTcpForwarding no/g' $sshd_cnf
}

function set_ssh_strong_mode_login() {
  sshd_cnf="/etc/ssh/sshd_config"

  #  Enable Root login
  prmt=$(cat $sshd_cnf | grep "^PermitRootLogin" | wc -l 2>/dev/null)
  if [ $prmt -ge 1 ]; then
    sed -i 's/^PermitRootLogin.*/PermitRootLogin yes/g' $sshd_cnf
  else
    echo "PermitRootLogin yes" >>$sshd_cnf
  fi

  # Enable root password
  passwd -u root

  #  Enable password login for all user
  pauth=$(cat $sshd_cnf | grep "^PasswordAuthentication" | wc -l 2>/dev/null)
  if [ $pauth -ge 1 ]; then
    sed -i 's/^PasswordAuthentication.*/PasswordAuthentication yes/g' $sshd_cnf
  else
    echo "PasswordAuthentication yes" >>$sshd_cnf
  fi

  # Enable PortForwarding if ROOT_ACCESS mode is set
  # Otherwise, disable it
  local ROOT_ACCESS=$(fw_printenv -n root_access)
  if [[ "$ROOT_ACCESS" == "true" ]]
  then
    sed -i 's/^AllowTcpForwarding no/AllowTcpForwarding yes/g' $sshd_cnf
  else
    sed -i 's/^AllowTcpForwarding yes/AllowTcpForwarding no/g' $sshd_cnf
  fi
}

function setup_enhance_security_mode() {

  local SSH_NO_CONSOLE=$(fw_printenv -n ssh_strong_no_console)

  TELNET_STATE="no"
  FTP_STATE="no"
  PING_STATE="yes"

  echo "Enforcing enhance security policy "
  # Following service must not start
  for i in "${SERVICE_TARGETED[@]}"; do
    is_service_is_used ${i}
    [ $? -eq 1 ] && deactivate_service ${i}
  done

  # ftp server is conditional
  if [ "$FTP_STATE" == "no" ]; then
    is_service_is_used vsftpd
    [ $? -eq 1 ] && deactivate_service vsftpd
  else
    is_service_is_used vsftpd
    [ $? -eq 1 ] && activate_service vsftpd
  fi

  # telnet is conditional
  if [ -d /etc/xinetd.d ]; then
    if [ "$TELNET_STATE" == "no" ]; then
      sed -i -e 's/disable[ \t]*= no/disable = yes/' /etc/xinetd.d/telnet
      killall -q telnetd
    else
      sed -i -e 's/disable[ \t]*= yes/disable = no/' /etc/xinetd.d/telnet
    fi
  fi

  # response to ping is conditional
  grep icmp_echo_ignore_all /etc/sysctl.conf
  DONT_RESPOND_TO_PING=$?
  # do not respond to ping
  if [[ "$PING_STATE" == "no" && DONT_RESPOND_TO_PING -eq 1 ]]; then
    # Forbid response to ping
    echo "net.ipv4.icmp_echo_ignore_all = 1" >>/etc/sysctl.conf
    sysctl -p -q
  fi
  # respond to ping
  if [[ "$PING_STATE" != "no" && DONT_RESPOND_TO_PING -eq 0 ]]; then
    # activate response to ping
    sed -i -e '/net.ipv4.icmp_echo_ignore_all.*/d' /etc/sysctl.conf
    sysctl -p -q
  fi

  if [[ "$SSH_NO_CONSOLE" == "true" ]]; then
    set_ssh_no_console_mode_login
  else
    set_ssh_strong_mode_login
  fi

  # When using ssl for https use an appropriate certificate :
  #   * if smartotu-certificate.crt and smartotu-private.key exists use them
  #   * if they do not exists use default cetificates : localhost.crt and localhost.key
  # To avoid editing mod_ssl.conf we use symlink from the certificate we want use and smartotu[key|crt]
  if [ -s "/etc/pki/tls/private/smartotu-private.key" ] && [ -s "/etc/pki/tls/certs/smartotu-certificate.crt" ] && [ $( eval /usr/lib/jdsu/scripts/otu_check_certificates.sh 2>/dev/null 1>&2; echo $?; ) -eq 0 ]; then
      # creation des liens vers les certificats ssl utilisateur si on les trouve
      ln -sf /etc/pki/tls/private/smartotu-private.key /etc/pki/tls/private/smartotu.key
      ln -sf /etc/pki/tls/certs/smartotu-certificate.crt /etc/pki/tls/certs/smartotu.crt
      if [ -s "/etc/pki/tls/certs/smartotu-certificate-chain.crt" ]; then
          ln -sf /etc/pki/tls/certs/smartotu-certificate-chain.crt /etc/pki/tls/certs/smartotu-chain.crt
      else
          ln -sf /etc/pki/tls/certs/smartotu-certificate.crt /etc/pki/tls/certs/smartotu-chain.crt
      fi
  else
      # creation des liens vers les certificats autosignes si l'on ne trouve pas les certicats ssl utilisateurs
      ln -sf /etc/pki/tls/private/localhost.key /etc/pki/tls/private/smartotu.key
      ln -sf /etc/pki/tls/certs/localhost.crt /etc/pki/tls/certs/smartotu.crt
      ln -sf /etc/pki/tls/certs/localhost.crt /etc/pki/tls/certs/smartotu-chain.crt
  fi

  # be sure autosigned certificate for ssl are the same in both part
  diff -q /etc/pki/tls/certs/localhost.crt /mnt/fs/unused_part/etc/pki/tls/certs/localhost.crt >/dev/null
  if [ "$?" != "0" ]; then
    mount -o remount,rw /mnt/fs/unused_part/
    cp /etc/pki/tls/certs/localhost.crt /mnt/fs/unused_part/etc/pki/tls/certs/localhost.crt
    mount -o remount,ro /mnt/fs/unused_part/
  fi
  diff -q /etc/pki/tls/private/localhost.key /mnt/fs/unused_part/etc/pki/tls/private/localhost.key
  if [ "$?" != "0" ]; then
    mount -o remount,rw /mnt/fs/unused_part/
    cp /etc/pki/tls/private/localhost.key /mnt/fs/unused_part/etc/pki/tls/private/localhost.key
    mount -o remount,ro /mnt/fs/unused_part/
  fi

  exit 0
}

function setup_basic_security_mode() {
  local SSH_NO_CONSOLE=$(fw_printenv -n ssh_strong_no_console)

  echo "Enforcing basic security policy "
  # Following service must be started
  for i in "${SERVICE_TARGETED[@]}"; do
    is_service_is_used ${i}
    [ $? -eq 0 ] && activate_service ${i}
  done

  is_service_is_used vsftpd
  [ $? -eq 0 ] && activate_service vsftpd


  if [ -d /etc/xinetd.d ] ; then
    # xinetd provides services too, start some of them
    sed -i -e 's/disable[ \t]*= yes/disable = no/' /etc/xinetd.d/telnet
  fi

  # activate response to ping
  sed -i -e '/net.ipv4.icmp_echo_ignore_all.*/d' /etc/sysctl.conf
  sysctl -p -q

  if [[ "$SSH_NO_CONSOLE" == "true" ]]; then
    set_ssh_no_console_mode_login
  else
    set_ssh_strong_mode_login
  fi


  exit 0
}

# Use service configuration file only if in enhance mode
if enhance-security-on ; then
  # Configure environment file for apache2 to run with SSL mode if enhance mode
  echo "ARGS=-D SSL" > /var/tmp/apache-env
  setup_enhance_security_mode
else
  echo "ARGS=" > /var/tmp/apache-env
  setup_basic_security_mode
fi

exit 0
