<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\monitoring\cache;

use app\util\SMTLogger;

use app\sharedmemory\SMTMemoryManager;

/**
 * Handles Operations (measure or test) data kept during the life of the operation
 * 
 * @author Sylvain Desplat
 *
 */
class SMTOperationManager
{              
    /**
     * Whether The operation is registered in the operations cache
     *
     * @return boolean
     */
    public static function hasOperation( $operationId )
    {
    	$operation = self::getMeasureInfo( $operationId );
    	if ( $operation != NULL )
    	{
    		return TRUE;
    	}
    	$operation = self::getTestInfo($operationId);
    	if ( $operation != NULL )
    	{
    		return TRUE;
    	}
    
    	return FALSE;
    }
    
    /**
     * Cache measure info in APC memory
     *
     * @param string $operationId
     * @param int $portId
     * @param string $traceDirectory
     * @param string $traceName
     * @param int $linkId
     */
    public static function addMeasureInfo( $operationId, $portId, $traceDirectory, $traceName, $linkId = NULL )
    {
    	//first, cleanup obsolete data
    	self::purgeMeasureInfo();
    
    	//save measure main settings in memory:
    	$measureCache = new SMTMeasureDataDto();
    	$measureCache->setOperationId($operationId);
    	$measureCache->setPortId($portId);
    	$measureCache->setLinkId($linkId);
    	$measureCache->setTraceDirectory($traceDirectory);
    	$measureCache->setTraceName($traceName);
    	//save dto in cache memory
    	$measureCache->save();
    }
    
    /**
     * Cache test info in APC memory
     *
     * @param string $operationId
     * @param int $testId
     * @param string $traceDirectory
     * @param string $traceName
     */
    public static function addTestInfo( $operationId, $testId, $traceDirectory, $traceName )
    {
    	//first, cleanup obsolete data
    	self::purgeTestInfo();
    
    	//save measure main settings in memory:
    	$testCache = new SMTTestDataDto();
    	$testCache->setOperationId($operationId);
    	$testCache->setTestId($testId);
    	$testCache->setTraceDirectory($traceDirectory);
    	$testCache->setTraceName($traceName);
    	//save dto in cache memory
    	$testCache->save();
    }    
    
    /**
     * Retrieve test info from APC cache
     *
     * @return SMTTestDataDto
     */
    public static function getTestInfo( $operationId )
    {
    	//retrieve measure main settings from memory:
    	return SMTMemoryManager::fetch( SMTTestDataDto::getClass(), $operationId );
    }    
    
    /**
     * Retrieve measure info from APC cache
     *
     * @return SMTMeasureDataDto
     */
    public static function getMeasureInfo( $operationId )
    {
    	//retrieve measure main settings from memory:
    	return SMTMemoryManager::fetch( SMTMeasureDataDto::getClass(), $operationId );
    }
    
    /**
     * Retrieve current measure info from APC cache
     *
     * @return SMTMeasureDataDto
     */
    public static function getCurrentMeasureInfo()
    {
        $currentMeasureInfo = NULL;
    	//retrieve measure main settings from memory:
    	$dtos = SMTMemoryManager::fetchAll( SMTMeasureDataDto::getClass() );
    	if ( is_array( $dtos ) && count( $dtos ) > 0 )
    	{
    	    end($dtos);
    	    $last_key = key($dtos);
    	    $currentMeasureInfo = $dtos[$last_key];
    	}
//     	SMTLogger::getInstance()->trace( "SMTMeasureDataDto: ". var_export( $currentMeasureInfo, true ), SMTLogger::DEBUG );
    	return $currentMeasureInfo;
    }
    
    /**
     * Delete measure info from APC cache
     *
     * @return SMTMeasureDataDto
     */
    public static function deleteMeasureInfo( $operationId )
    {        
    	//retrieve measure main settings from memory:    	        
    	return SMTMemoryManager::delete( SMTMeasureDataDto::getClass(), $operationId );
    }
    
    /**
     * Delete test info from APC cache
     *
     * @return SMTMeasureDataDto
     */
    public static function deleteTestInfo( $operationId )
    {
    	//retrieve measure main settings from memory:
    	return SMTMemoryManager::delete( SMTTestDataDto::getClass(), $operationId );
    }    
    
    /**
     * Cleanup all measures data that are obsolete (not requested since default memory cache timeout: 10 mins).
     * 
     * @param $force boolean Whether data must be deleted whatever they are obsolete or not
     */
    public static function purgeMeasureInfo( $force = FALSE )
    {
    	try
    	{
    		$measuresData = SMTMemoryManager::fetchAll( SMTMeasureDataDto::getClass() );
    		if ( $measuresData != NULL )
    		{
    			foreach ( $measuresData as $measureData )
    			{
    				if ( $measureData != NULL && ( $measureData->isObsolete() || $force ) )
    				{
    					self::deleteMeasureInfo( $measureData->getOperationId() );
    				}
    			}
    		}
    	}
    	catch( \Exception $e )
    	{
    		SMTLogger::getInstance()->traceException($e);
    	}
    }    
    
    /**
     * Cleanup all test data that are obsolete (not requested since default memory cache timeout: 10 mins).
     *
     */
    public static function purgeTestInfo()
    {
    	try
    	{
    		$testsData = SMTMemoryManager::fetchAll( SMTTestDataDto::getClass() );
    		if ( $testsData != NULL )
    		{
    			foreach ( $testsData as $testData )
    			{
    				if ( $testData != NULL && $testData->isObsolete() )
    				{
    					self::deleteTestInfo( $testData->getOperationId() );
    				}
    			}
    		}
    	}
    	catch( \Exception $e )
    	{
    		SMTLogger::getInstance()->traceException($e);
    	}
    }    
    
}

?>