<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI Solutions
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI Solutions
// Copyright JDSU 2017. All rights reserved.
// *********************************************************
namespace app\serviceshelper\monitoring;

/**
 * Monitoring test position buffer encoder/decoder used for switch buffer in OTDR acquisitions, tests...
 * 
 * @author Sylvain Desplat
 */
class SMTPositionBufferEncoderDecoder
{
    const PEAK_POSITION_SIZE = 8; //4 * 2    
    
    /**
     * @var string
     */
    private $peakBufferWithHeader = NULL;
   
    private $peakPositions = array();
    
    /**
     * @param string $peakBufferWithHeader
     */
    function __construct( $peakBufferWithHeader)
    {
    	$this->peakBufferWithHeader= $peakBufferWithHeader;
    } 
    
    /**
     * Encode peaks positions in a buffer string
     *
     * @param array $existingPeakThresholds
     * @return string
     */
    static function encodePeakPositions( array &$existingPeaks,
    		                             $peakCount)
    {
    	$buffer = self::encodeBufferPeakPositionHeader($peakCount);

    	//existing peaks
    	if ( $existingPeaks!= NULL )
    	{
    		foreach($existingPeaks as &$existingPeak)
    		{
    			$buffer .= self::encodeBufferPeakPosition( $existingPeak->getPeakTopDistanceM() );
    		}
    	}
    	
    	return $buffer;
    }
    
    /**
     * @return string
     */
    private static function encodeBufferPeakPositionHeader( $peakCount = 0)
    {
    	$bufferSize = ( $peakCount ) * self::PEAK_POSITION_SIZE;
    	
    	if ($bufferSize < 10)
    		$buffer = sprintf ("#1%d", $bufferSize);
    	else if ($bufferSize <100)
    		$buffer = sprintf ("#2%d", $bufferSize);
    	else if ($bufferSize < 1000)
    		$buffer = sprintf ("#3%d", $bufferSize);
    	else if ($bufferSize < 10000)
    		$buffer = sprintf ("#4%d", $bufferSize);
    				
    	return $buffer;
    }
    
    private static function encodeBufferPeakPosition($position)
    {
    	$positionCm = round( (float) $position *100.0 );
    	return self::int32ToHex($positionCm);
    }
    
    private static function decodePeakPositionBuffer($bufferPosition)
    {
    	$positionCm = self::hexToInt32($bufferPosition);
    	
    	return (floatval($positionCm))/100.0;
    }
    
    /**
     * convert signed 16 bits integer value to 16 bits hexadecimal values 
     * 
     * @param $intVal
     * @return string
     */
    static function convertInt16ToHex( $intVal )
    {
        $hex = sprintf('%04X', $intVal );
        return substr ($hex, strlen($hex) - 4, 4);//16 bits hexa values 0xFFFF
    }
    
    static function int32ToHex($intVal)
    {
    	$firt = ($intVal & 0xFF000000) >> 24;
    	$second = ($intVal & 0x00FF0000) >> 16;
    	$third = ($intVal & 0x0000FF00) >> 8;
    	$forth = ($intVal & 0x000000FF) >> 0;
    	return sprintf('%02X',$firt).sprintf('%02X',$second).sprintf('%02X',$third).sprintf('%02X',$forth);
    }
    
    
    function hexToInt32($hexVal)
    {
    	$firtHex = substr($hexVal, 0, 2);
    	$secondHex = substr($hexVal, 2, 2);
    	$thirdHex = substr($hexVal, 4, 2);
    	$forthHex = substr($hexVal, 6, 2);
    	
    	$first = (hexdec($firtHex) & 0x000000FF) << 24;
    	$second= (hexdec($secondHex) & 0x000000FF) << 16;
    	$third= (hexdec($thirdHex) & 0x000000FF) << 8;
    	$forth= (hexdec($forthHex) & 0x000000FF) << 0;
    	
    	return $first + $second + $third + $forth;
    }
    
    /**
     * Decode the buffer threshold string into arrays
     *
     * @param int $positionCount
     */
    function decodePositionBuffer( $positionCount )
    {
    	//example: #3170 => $sizeOfBufferSize=3  $bufferSize=170
    	$sizeOfBufferSize = intval( substr($this->peakBufferWithHeader, 1, 1) );
    	$bufferSize =  intval( substr($this->peakBufferWithHeader, 2, $sizeOfBufferSize) );
    	
    	$index = strlen( "#" ) + 1 + $sizeOfBufferSize;    	
    	$this->peakThresholds = array();
    	$pos = 0;
    	while ( $pos < $positionCount && $index < $bufferSize)
    	{
    		$position = self::decodePeakPositionBuffer( substr( $this->peakBufferWithHeader, $index, self::PEAK_POSITION_SIZE) );
    		array_push($this->peakPositions, $position);
    		$index += self::PEAK_POSITION_SIZE;
    		$pos++;
    	}
    }
  
    /**
     * @return array of float
     */
    function getPeakPositions()
    {
    	return $this->peakPositions;
    }
    
    /**
     * Convert hexadecimal string value to float.
     *
     * @param $hexStringVal
     * @return float
     */
    static function convertHex16ToFloat( $hexStringVal )
    {
        $decValue = hexdec( $hexStringVal);
        if ( $decValue > hexdec("OxFFFF") / 2 )
        {
            $decValue = $decValue - hexdec("Ox00010000");
        }
        
    	return floatval( $decValue );
    }
    
}


?>