<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\monitoring;

use app\services\monitoring\SMTCurrentPortDto;

use app\services\alarm\SMTOpticalAlarmDto;

use app\util\SMTUtil;

use app\util\SMTInvalidValueException;

use app\util\SMTDto;

class SMTLinkTestListDto extends SMTDto
{   
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\monitoring\SMTLinkTestDto',islist='true')
     * @var array app\services\monitoring\SMTLinkTestDto
     */
    protected $linkTests = array();    
    
    /**
     * The number of output ports available on OTU
     * 
     * @var int
     */
    protected $portCount = 0;
    
    /**
     * @var int
     */
    protected $moduleRunningNumber;
    
    /**
     * Whether sequensor is not stopped
     * @boolean
     */
    protected $sequensorOn = FALSE;    
    
    /**
     * Whether OTU module config is OK
     */
    protected $moduleConfigOK = FALSE;
    
    /**
     * Whether OTU switch config is OK
     */
    protected $switchConfigOK = FALSE;
    
    /**
     * Virtual switch port count used when an external switch not managed by OTU is used
     * 
     * @var int
     */
    protected $virtualSwitchPortCount = SMTCurrentPortDto::INVALID_VIRTUAL_SWITCH_COUNT;
    
    /**
     * Virtual switch current port used when an external switch not managed by OTU is used
     *
     * @var int
     */
    protected $virtualSwitchCurrentPort = SMTCurrentPortDto::NO_PORT_SELECTED;

    /**
     * @param array
     */
    public function setLinkTests( $linkTests )
    {
    	$this->linkTests = $linkTests;
    }
    
    /**
     * @return array
     */
    public function getLinkTests()
    {
    	return $this->linkTests;
    }
    
    /**
     * @param int
     */
    public function setPortCount( $portCount )
    {
        if ( !isset($portCount) || $portCount === NULL || !SMTUtil::isInt( $portCount ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "portCount", $portCount);
        }        
    	$this->portCount = intval($portCount);
    }
    
    /**
     * @return int
     */
    public function getPortCount()
    {
        return $this->portCount !== NULL ? intval($this->portCount) : NULL;
    }        
    
    /**
     * 
     * @param int $moduleRunningNumber
     * @throws SMTInvalidValueException
     */
    public function setModuleRunningNumber( $moduleRunningNumber )
    {
    	if ( !isset($moduleRunningNumber) || $moduleRunningNumber === NULL || !SMTUtil::isInt( $moduleRunningNumber ) || ( $moduleRunningNumber <= 0 || $moduleRunningNumber > 2 ) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "moduleRunningNumber", $moduleRunningNumber );
    	}
    	$this->moduleRunningNumber = intval($moduleRunningNumber);
    }
    
    /**
     * 
     * @return int or NULL
     */
    public function getModuleRunningNumber()
    {
    	return $this->moduleRunningNumber !== NULL ? intval($this->moduleRunningNumber) : NULL;    	
    }    
    
    public function setSequensorOn( $sequensorOn )
    {
    	$this->sequensorOn = $sequensorOn;
    }
    
    public function isSequensorOn()
    {
    	return $this->sequensorOn;
    }    
    
    /**
     * @param boolean $moduleConfigOK
     */
    public function setModuleConfigOK( $moduleConfigOK )
    {
    	$this->moduleConfigOK = $moduleConfigOK;
    }
    
    /**
     * @return boolean
     */
    public function isModuleConfigOK()
    {
    	return $this->moduleConfigOK;
    }
    /**
     * @param boolean $switchConfigOK
     */
    public function setSwitchConfigOK( $switchConfigOK )
    {
    	$this->switchConfigOK = $switchConfigOK;
    }
    
    /**
     * @return boolean
     */
    public function isSwitchConfigOK()
    {
    	return $this->switchConfigOK;
    }    
    
    /**
     * 
     * @param int $virtualSwitchPortCount
     */
    public function setVirtualSwitchPortCount( $virtualSwitchPortCount )
    {
        if ( isset($virtualSwitchPortCount) && $virtualSwitchPortCount !== NULL || SMTUtil::isInt( $virtualSwitchPortCount ) )
        {
            $this->virtualSwitchPortCount = intval($virtualSwitchPortCount);
        }
    }
    
    /**
     *
     * @param int $virtualSwitchCurrentPort
     */
    public function setVirtualSwitchCurrentPort( $virtualSwitchCurrentPort )
    {
        if ( isset($virtualSwitchCurrentPort) && $virtualSwitchCurrentPort !== NULL || SMTUtil::isInt( $virtualSwitchCurrentPort ) )
        {
            $this->virtualSwitchCurrentPort = intval($virtualSwitchCurrentPort);
        }
    }
    
    /**
     * @param SMTLinkTestDto $linkTest
     */
    public function addLinkTest( SMTLinkTestDto $linkTest )
    {
    	array_push( $this->linkTests, $linkTest );
    }       
    
    /**
     * Add the given alarm to its link-test
     * 
     * @param SMTOpticalAlarmDto $opticalAlarm
     */
    public function addOpticalAlarm( SMTOpticalAlarmDto $opticalAlarm )
    {
        if ( $opticalAlarm != NULL )
        {
            foreach ( $this->linkTests as $linkTest )
            {
                if ( $linkTest->getTestId() !== NULL && $linkTest->getTestId() === $opticalAlarm->getTestId() )
                {
                    $linkTest->addOpticalAlarm( $opticalAlarm );
                    break;
                }
            }
        }
    }    
    
    /**
     * Remove link-tests without alarms
     *
     */
    public function removeLinkTestsWithoutAlarm()
    {
        foreach ( $this->getLinkTests() as $key=>$linkTest)
        {
        	$alarmPeakCount = $linkTest->getPeakOpticalAlarms() != NULL? count ($linkTest->getPeakOpticalAlarms()) : 0;
        	if ( ($linkTest->getAttenuationOpticalAlarm() == NULL) && ($alarmPeakCount<= 0) )
        	{
        	    //unset array element in loop is safe
        	    unset( $this->linkTests[$key] );
        	}
        }
        //re-index the array (don't remove!);
        $this->linkTests = array_merge( $this->linkTests );
    }
        
    /**
     * Serialize as Json data the user object.<br> 
     * 
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
        $this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }

    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
        return get_class();
    }      
}

?>