<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of Viavi Solution
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from Viavi Solution
// Copyright Viavi Solution2019. All rights reserved.
// *********************************************************
namespace app\services\api;

use app\services\SMTIService;
use app\services\SMTService;
use app\parser\SMTOtuApi;
use app\http\SMTHttpResponse;
use app\error\SMTRestException;
use app\sharedmemory\SMTMemoryManager;
use app\serviceshelper\system\SMTSystemConfigurationDto;
use app\util\SMTUtil;
use app\admin\SMTUserUtil;
use app\serviceshelper\otdr\SMTMeasurementInstrument;


/**
 * Public API service to execute OTU operations.
 *
 * @author Sylvain Desplat
 */
class SMTApiOtuService extends SMTService implements SMTIService
{
	/**
	 *
	 * @url GET /
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getOtuInfo()
	{
		$this->checkApi();
		
		$otuDto = new SMTApiOtuDto();
		try
		{	
			$otuInfo = $this->sendReceive( SMTOtuApi::getOtuMainInfo() );
//          Problem with regular expressions with PHP 5.6			
//  		$matches = preg_split('/,(?=(?:[^"]*"[^"]*")*[^"]*$)/',$otuInfo);
// 			preg_match_all('/"[^"]*"|[^,]+,/', $otuInfo.',', $matches);
// 			print_r($matches);
			
			list( $serialNumber, $otuType, $moduleType, $portCount, $date, $otuReleaseVersion, $licences ) = explode(',',$otuInfo);
			
			$otuDto->setSerialNumber( trim($serialNumber) );
			$otuDto->setType( trim($otuType));
			$otuDto->setModuleType(trim($moduleType));
			$otuDto->setPortCount(trim($portCount));
			$otuDto->setDate(trim($date));
			$otuDto->setSoftwareVersion( trim($otuReleaseVersion) );
			$licences = trim($licences);
			$otuDto->setLicense( str_replace(' ',',', $licences));
			
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve otu info');
		}

		$this->getContext()->closeSession('api');
		
		return $otuDto->getJsonData();
	}
	
	/**
	 *
	 * @url POST /date
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function updateOtuDate($request_data)
	{
		$this->checkApi();
		
		$dateDto = SMTApiOtuDateDto::getInstance( $request_data );
		try
		{
			$this->send( SMTOtuApi::updateOtuDate($dateDto->getDate()));
			
			
			//update APC cache with new time
			SMTMemoryManager::touchAllDtos();
			
			//retrieve database connection
			$dbConnection = $this->getContext()->getDatabase();
			
			//update timezone in database settings
			$dto = SMTSystemConfigurationDto::fetch($dbConnection);
			$dto->setTimezoneSec( 0 );
			$dto->setTimezoneName( 'UTC' );
			$dto->setTimezoneDst( FALSE );
			$dto->save($dbConnection);
			
			
			$this->getContext()->getLogger()->trace( "clientTimeStamp: ".$dateDto->getDate() );
			$this->getContext()->getLogger()->trace( "clientTimezoneSec: ".$dto->getTimezoneSec() );
			$this->getContext()->getLogger()->trace( "clientTimezoneName: ".$dto->getTimezoneName());
			$this->getContext()->getLogger()->trace( sprintf("clientTimezoneDst: %b",$dto->isTimezoneDst() ));
		}
		catch( \Exception $ex )
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to update OTU date');
		}
		//fetch new date
		try 
		{
			$otuInfo = $this->sendReceive( SMTOtuApi::getOtuMainInfo() );
			list( $serialNumber, $otuType, $moduleType, $portCount, $date, $otuReleaseVersion, $licences ) = explode(",", $otuInfo);
			$dateDto->setDate($date);
		}
		catch (\Exception $e)
		{
			$this->getContext()->getLogger()->traceException($e);
		}
		$this->getContext()->closeSession('api');
		
		return $dateDto->getJsonData();
	}
	
	/**
	 *
	 * @url GET /snmp
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getSnmpConfig()
	{
		$this->checkApi();
		
		$snmpListDto = new SMTApiSnmpListDto();
		try
		{
			for ( $id=1; $id<=5; $id++ )
			{
				try 
				{
					$snmpInfo = $this->sendReceive( SMTOtuApi::getSnmpConfig($id) );
					if ( strlen($snmpInfo) > 0 )
					{
						list( $id, $address, $community, $port ) = explode(",", $snmpInfo);
						if ( strlen($id) > 0 )
						{
							$snmp = new SMTApiSnmpDto();
							$snmp->setId( trim($id) );
							$snmp->setAddress(trim($address) );				
							$snmp->setCommunity(trim($community));
							$snmp->setPort(trim($port));
							if ( $snmp->isConfigValid() )
							{
								$snmpListDto->addSnmpManager($snmp);
							}
						}
					}
				}
				catch(\Exception $ex)
				{
					$this->getContext()->getLogger()->traceException($ex);
				}
			}
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve snmp info');
		}
		
		$this->getContext()->closeSession('api');
		
		return $snmpListDto->getJsonData();
	}
	
	/**
	 *
	 * @url POST /snmp
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function updateSnmpConfig($request_data)
	{
		$this->checkApi();
		
		$snmpListDto = SMTApiSnmpListDto::getInstance( $request_data );
		$snmpConfigUpdated = array();
		try
		{
			foreach ($snmpListDto->getManagers() as $dto )
			{
				array_push( $snmpConfigUpdated, $dto->getId());
				if ( $dto->isConfigValid() )
				{
					$this->send( SMTOtuApi::setSnmpConfig($dto->getId(), $dto->getAddress(), $dto->getCommunity(), $dto->getPort() ));
				}
				else
				{
					$this->send( SMTOtuApi::setSnmpConfig($dto->getId(), '', '', 0 ));
				}
			}
			for ($id=1; $id<=5; $id++)
			{
				if ( !in_array($id, $snmpConfigUpdated) )
				{
					$this->send( SMTOtuApi::setSnmpConfig($id, '', '', 0 ));
				}
			}
		}
		catch( \Exception $ex )
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to update OTU snmp config');
		}
		return $this->getSnmpConfig();
	}
	
	/**
	 *
	 * @url GET /snmp/imalive
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getSnmpImAlive()
	{
		$this->checkApi();
		
		$snmpImAliveDto = new SMTApiSnmpImAliveDto();
		try
		{
			$snmpImAliveInfo = $this->sendReceive( SMTOtuApi::getSnmpImAlive() );
			$snmpImAliveDto->setPeriod(trim($snmpImAliveInfo));
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve snmp ImAlive trap info');
		}
		
		$this->getContext()->closeSession('api');
		
		return $snmpImAliveDto->getJsonData();
	}
	
	/**
	 *
	 * @url POST /snmp/imalive
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function setSnmpImAlive($request_data)
	{
		$this->checkApi();
		
		$snmpImAliveDto = SMTApiSnmpImAliveDto::getInstance( $request_data );
		try
		{
			$this->send( SMTOtuApi::setSnmpImAlive($snmpImAliveDto->getPeriod() ) );
		}
		catch( \Exception $ex )
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to update OTU date');
		}
		return $this->getSnmpImAlive();
	}
	
	/**
	 *
	 * @url POST /snmp/{id}/test
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function testSnmp($id)
	{
		$this->checkApi();
		
		$testStatusDto = new SMTApiStatusDto();
		try
		{
			if($id <0 || $id > 5)
			{
				$testStatusDto->setStatus(SMTApiStatusDto::INVALID_SNMP_MANAGER);
				SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, $testStatusDto->getStatus(), 'No SNMP manager found with id  '.$id);
				
			}
			else
			{
				try
				{
					$snmpInfo = $this->sendReceive( SMTOtuApi::getSnmpConfig($id) );
					if ( strlen($snmpInfo) <= 0 )
					{
						$testStatusDto->setStatus(SMTApiStatusDto::INVALID_SNMP_MANAGER);
						SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, $testStatusDto->getStatus(), 'No SNMP manager found with id  '.$id);
					}
				}
				catch(\Exception $ex)
				{
					$this->getContext()->getLogger()->traceException($ex);
					$testStatusDto->setStatus(SMTApiStatusDto::INVALID_SNMP_MANAGER);
					SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, $testStatusDto->getStatus(), 'No SNMP manager found with id  '.$id);
					
				}
				
				$snmpInfo = $this->sendReceive( SMTOtuApi::getSnmpConfig($id) );
				list( $id, $address, $community, $port ) = explode(",", $snmpInfo);
				
				if (strlen($address) > 0 && strlen($community) > 0 && $port > 0 )
				{
					$this->send( SMTOtuApi::testSnmpConfig(intval($id)) );
					$testStatusDto->setStatus(SMTApiStatusDto::SUCCESS);
				}
				else 
				{
					$testStatusDto->setStatus(SMTApiStatusDto::INVALID_SNMP_MANAGER);
					SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, $testStatusDto->getStatus(), 'No SNMP manager found with id  '.$id);
					
				}
			}
		}
		catch( \Exception $ex )
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to send SNMP test trap');
		}
		$this->getContext()->closeSession('api');
		return $testStatusDto->getJsonData();
	}
	
	/**
	 *
	 * @url GET /module/opt
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getModuleInfo()
	{
		$this->checkApi();
		
		$opticalModuleDto = new SMTApiOpticalModuleDto();
		try
		{
			$otuInfo = $this->sendReceive( SMTOtuApi::getModuleInfo() );
			list( $serialNumber, $position, $type, $wavelengths ) = explode(",", $otuInfo);
			
			$opticalModuleDto->setSerialNumber( trim($serialNumber) );
			$opticalModuleDto->setPosition( trim($position));
			$opticalModuleDto->setType(trim($type));
			$opticalModuleDto->setWavelengths(trim($wavelengths));
			
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve otu info');
		}
		
		$this->getContext()->closeSession('api');
		
		return $opticalModuleDto->getJsonData();
	}
	
	/**
	 *
	 * @url GET /ipv4
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getIpv4()
	{
		$this->checkApi();
		
		$ipv4Dto = new SMTApiIpv4Dto();
		try
		{
			$ipInfo = $this->sendReceive( SMTOtuApi::getIpv4Config('ETH0') );
			list( $dhcp, $ip, $subnet, $dns, $domainName, $gateway ) = explode(",", $ipInfo);
			
			$ipv4Dto->setDhcp( trim($dhcp) );
			$ipv4Dto->setIp( trim($ip) );
			$ipv4Dto->setSubnet( trim($subnet));
			$ipv4Dto->setDns(trim($dns));
			$ipv4Dto->setDomain(trim($domainName));
			$ipv4Dto->setGateway(trim($gateway));
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve otu IPv4 setup');
		}
		
		$this->getContext()->closeSession('api');
		
		return $ipv4Dto->getJsonData();
	}

	/**
	 *
	 * @url GET /ipv4backup
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getIpv4Backup()
	{
	    $this->checkApi();
	    
	    $ipv4Dto = new SMTApiIpv4Dto();
	    try
	    {
	        $ipInfo = $this->sendReceive( SMTOtuApi::getIpv4Config('ETH1') );
	        list( $dhcp, $ip, $subnet, $dns, $domainName, $gateway ) = explode(",", $ipInfo);
	        
	        $ipv4Dto->setDhcp( trim($dhcp) );
	        $ipv4Dto->setIp( trim($ip) );
	        $ipv4Dto->setSubnet( trim($subnet));
	        $ipv4Dto->setDns(trim($dns));
	        $ipv4Dto->setDomain(trim($domainName));
	        $ipv4Dto->setGateway(trim($gateway));
	    }
	    catch ( \Exception $e )
	    {
	        $this->getContext()->getLogger()->traceException($e);
	        SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve otu IPv4 backup setup');
	    }
	    
	    $this->getContext()->closeSession('api');
	    
	    return $ipv4Dto->getJsonData();
	}
	
	/**
	 *
	 * @url GET /ipv6
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getIpv6()
	{
		$this->checkApi();
		
		$ipv6Dto = new SMTApiIpv6Dto();
		try
		{
			$ipInfo = $this->sendReceive( SMTOtuApi::getIpv6Config('ETH0') );
			list( $active, $dhcp, $ip, $gateway, $site, $dns, $link ) = explode(",", $ipInfo);
			
			$ipv6Dto->setActive( trim($active) );
			$ipv6Dto->setDhcp( trim($dhcp) );
			$ipv6Dto->setIp( trim($ip) );
			$ipv6Dto->setGateway(trim($gateway));
			$ipv6Dto->setSite( trim($site));
			$ipv6Dto->setDns(trim($dns));
			$ipv6Dto->setLinkIp(trim($link));
			
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve otu IPv6 setup');
		}
		
		$this->getContext()->closeSession('api');
		
		return $ipv6Dto->getJsonData();
	}
	/**
	 *
	 * @url GET /ipv6backup
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getIpv6Backup()
	{
	    $this->checkApi();
	    
	    $ipv6Dto = new SMTApiIpv6Dto();
	    try
	    {
	        $ipInfo = $this->sendReceive( SMTOtuApi::getIpv6Config('ETH1') );
	        list( $active, $dhcp, $ip, $gateway, $site, $dns, $link ) = explode(",", $ipInfo);
	        
	        $ipv6Dto->setActive( trim($active) );
	        $ipv6Dto->setDhcp( trim($dhcp) );
	        $ipv6Dto->setIp( trim($ip) );
	        $ipv6Dto->setGateway(trim($gateway));
	        $ipv6Dto->setSite( trim($site));
	        $ipv6Dto->setDns(trim($dns));
	        $ipv6Dto->setLinkIp(trim($link));
	        
	    }
	    catch ( \Exception $e )
	    {
	        $this->getContext()->getLogger()->traceException($e);
	        SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve otu IPv6 backup setup');
	    }
	    
	    $this->getContext()->closeSession('api');
	    
	    return $ipv6Dto->getJsonData();
	}
	
	/**
	 *
	 * @url POST /alarm/clear
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function clearAllAlarms()
	{
		$this->checkApi();
		
		$statusDto = new SMTApiStatusDto();
		try
		{
			$this->send( SMTOtuApi::clearAllAlarms() );
			$statusDto->setStatus(SMTApiStatusDto::SUCCESS);
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'failed to clear all OTU alarms  ');
		}
		$this->getContext()->closeSession('api');
		
		return $statusDto->getJsonData();
	}
	
	/**
	 *
	 * @url POST /monitoring/start
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function restartMonitoring()
	{
	    $this->checkApi();
	    
	    $statusDto = new SMTApiStatusDto();
	    try
	    {
	        SMTMeasurementInstrument::restartSequencer($this->getContext(), 0);
	        $statusDto->setStatus(SMTApiStatusDto::SUCCESS);
	    }
	    catch ( \Exception $e )
	    {
	        $this->getContext()->getLogger()->traceException($e);
	        SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'failed to clear all OTU alarms  ');
	    }
	    $this->getContext()->closeSession('api');
	    
	    return $statusDto->getJsonData();
	}
	
	/**
	 *
	 * @url POST /user/smartotu
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function changeSmartOTUCredential($request_data)
	{
		$this->checkApi();
		
		$statusDto = new SMTApiStatusDto();
		
		$dto = SMTApiSmartOtuCredentialDto::getInstance( $request_data );
		try
		{
			$login = SMTUtil::escapeShellArgUtf8WithoutQuotes($dto->getCurrentLogin());
			$password = base64_decode($dto->getCurrentPassword(), TRUE);
			if ($password == false)
			{
				SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid Password');
			}
			
			$password = SMTUtil::escapeShellArgUtf8WithoutQuotes($password);
			$newLogin = SMTUtil::escapeShellArgUtf8WithoutQuotes($dto->getNewLogin());
			$newPassword = base64_decode($dto->getNewPassword(), TRUE);
			if ( strlen($newPassword) == 0 || strlen($newPassword) > SMTApiSmartOtuCredentialDto::CREDENTIAL_MAX_SIZE || ($newPassword == false) || 
				 ( strcmp($newPassword, SMTUtil::escapeShellArgUtf8WithoutQuotes($newPassword)) != 0) )
			{
				SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid new Password : base64 encoded, size > 0 max size <= 255 bytes, \' character not allowed');
			}
			if ( (strlen($newLogin) > SMTApiSmartOtuCredentialDto::CREDENTIAL_MAX_SIZE) || 
					(strcmp( $dto->getNewLogin(), SMTUtil::escapeShellArgUtf8WithoutQuotes($dto->getNewLogin()) ) != 0) )
			{
				SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid new login, max size 255 bytes, \' character not allowed');
			}

			$newPassword = SMTUtil::escapeShellArgUtf8WithoutQuotes($newPassword);
			
			$this->send( SMTOtuApi::changeSmartOTUCredential( $login, $password, $newLogin, $newPassword) );
			$statusDto->setStatus(SMTApiStatusDto::SUCCESS);
		}
		catch( \Exception $ex )
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid credentials');
		}

		$this->getContext()->closeSession('api');
		
		return $statusDto->getJsonData();
	}
	
	/**
	 *
	 * @url POST /user/guest
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function changeSmartOTUGuestCredential($request_data)
	{
		$this->checkApi();
		
		$statusDto = new SMTApiStatusDto();
		
		$dto = SMTApiSmartOtuGuestCredentialDto::getInstance( $request_data );
		try
		{
			$newLogin = SMTUtil::escapeShellArgUtf8WithoutQuotes($dto->getNewLogin());
			$newPassword = base64_decode($dto->getNewPassword(), TRUE);
			if ( strlen($newPassword) == 0 || strlen($newPassword) > SMTApiSmartOtuGuestCredentialDto::CREDENTIAL_MAX_SIZE || ($newPassword == false) ||
					( strcmp($newPassword, SMTUtil::escapeShellArgUtf8WithoutQuotes($newPassword)) != 0) )
			{
				SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid new Password : base64 encoded, size > 0 max size <= 255 bytes, \' character not allowed');
			}
			if ( (strlen($newLogin) > SMTApiSmartOtuGuestCredentialDto::CREDENTIAL_MAX_SIZE) ||
					(strcmp( $dto->getNewLogin(), SMTUtil::escapeShellArgUtf8WithoutQuotes($dto->getNewLogin()) ) != 0) )
			{
				SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid new login, max size 255 bytes, \' character not allowed');
			}
			
			$newPassword = SMTUtil::escapeShellArgUtf8WithoutQuotes($newPassword);
			
			$this->send( SMTOtuApi::changeSmartOTUGuestCredential( $newLogin, $newPassword) );
			$statusDto->setStatus(SMTApiStatusDto::SUCCESS);
		}
		catch( \Exception $ex )
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Invalid credentials');
		}
		
		$this->getContext()->closeSession('api');
		
		return $statusDto->getJsonData();
	}
	
	/**
	 *
	 * @url GET /user/guest
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getSmartOTUGuestCredential()
	{
		$this->checkApi();
		$userToSend = new SMTApiSmartOtuGuestCredentialDto();
		
		try
		{
			$result = $this->sendReceive( SMTOtuApi::hasSmartOTUGuestUser() );
			if ( strcasecmp( $result, 'TRUE' ) == 0 )
			{
				$userToSend->setNewLogin( SMTUserUtil::getSmartOTUGuestLogin() );
				$userToSend->setNewPassword('***');
			}
			else 
			{
				$userToSend->setNewLogin('');
				$userToSend->setNewPassword('');
			}
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Failed to retrieve guest user');
		}
		$this->getContext()->closeSession('api');
		
		return $userToSend->getJsonData();
	}
	
	/**
	 *
	 * @url POST /user/guest/{enable}
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function enableSmartOTUGuestCredential($enable)
	{
		$this->checkApi();
		
		$statusDto = new SMTApiStatusDto();
		
		try
		{
			$enable = filter_var( $enable, FILTER_VALIDATE_BOOLEAN );
			$this->send( SMTOtuApi::enableSmartOTUGuestUser( $enable ) );
			$statusDto->setStatus(SMTApiStatusDto::SUCCESS);
		}
		catch( \Exception $ex )
		{
			$this->getContext()->getLogger()->traceException($ex);
			SMTHttpResponse::handleApiError( $this->getContext(), SMTRestException::BAD_REQUEST, SMTApiDto::FAILURE, 'Failed to enable/disable guest user');
		}
		
		$this->getContext()->closeSession('api');
		
		return $statusDto->getJsonData();
	}
}
