<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\parser;

use app\serviceshelper\ethernet\SMTEthernet;

use app\util\SMTLogger;

use app\http\SMTContext;

class SMTOtuSocketManager
{

    /**
     * The OTU socket: a new socket is used for each http request (1 socket = 1 context = 1 http request)
     *
     * @var SMTOtuSocket
     */
    private $socket = NULL;
    
    /**
     * Application context
     * @var app\http\SMTContext
     */
    private $context = NULL;
    
    const RESCUE_MODE_INDEX = 3;
    const RESCUE_MODE_VALUE = 1;
    
    
    /**
     * Build the application to process an http request.
     *
     */
    function __construct( SMTContext $context )
    {
    	$this->context = $context;
    }
    
    /**
     * Check that the OTU socket is available in the current context (http request context):<br>
     *  1 - check that the OTU socket port has been set in the session<br> 
     *  2 - create the socket and open it.
     *  3 - check whether OTU is in local mode
     * 
     * @param boolean $checkLocalMode TRUE by default: for each query check that OTU is not in local mode
     * 
     * @throws SMTSocketException if :<br>
     *  1 - the port of the OTU application cannot be retrieved.<br>
     *  2 - the socket cannot be opened. 
     *  
     *  @return SMTOtuSocket
     */
    function restoreSocket( $checkLocalMode = TRUE )
    {
        if ( $this->socket === NULL )
        {
            $port = $this->context->getOtuPort();
            
            //open the socket:
            $this->openSocket( $port );
            
            //check OTU local mode for each HTTP request using the parser and update the cache
            if ( $checkLocalMode != FALSE )
            {
                $this->checkLocalMode();
            }
        }
        
        return $this->socket;
    }
    
    /**
     * Create a socket on the given port
     *
     * @param $port The port of the socket to open (By default the otu port 1400 of the OTU external parser)
     * @param $maxRetryCount Max number of retry in case of connection opening failure
     * 
     * @return SMTOtuSocket
     *  
     * @throws SMTSocketException if the socket cannot be opened.
     */ 
    public function openSocket( $port = SMTOtuSocket::OTU_EXTERNAL_PORT, $maxRetryCount = SMTOtuSocket::MAX_RETRY_OPEN_CONNECTION )
    {    
        //open the socket:
        try
        {
        	$this->socket = new SMTOtuSocket( $this->context );
        	$this->socket->open( $port, $maxRetryCount );
        	if ( $port == SMTOtuSocket::OTU_EXTERNAL_PORT)
        	{
        		//Finally, tests whether OTU application is ready.
        		$this->socket->testOTUApplicationReady();
        	}
        }
        catch (\Exception $e)
        {
            self::handleSocketException( $e, $this->context );
        	$this->socket = NULL;
        	throw $e;
        }        
    
    	return $this->socket;
    }
    
    /**
     * Check that the OTU socket is available in the current context (http request context):<br>
     *  1 - check that the OTU socket port has been set in the session<br>
     *  2 - create the socket and open it.
     *  3 - check whether OTU is in local mode
     *
     * @throws SMTSocketException if :<br>
     *  1 - the port of the OTU application cannot be retrieved.<br>
     *  2 - the socket cannot be opened.
     *
     *  @return SMTOtuSocket The current socket opened on OTU parser
     */
    public function checkSocket()
    {
        if ( $this->socket === NULL )
        {
            $port = $this->context->getOtuPort();
            
            //open the socket with no retry in case of failure:
            $this->openSocket( $port, 0 );            
        
        	//check OTU local mode for each HTTP request using the parser and update the cache
      		$this->checkLocalMode();
        }
    
    	return $this->socket;
    }    
        
    /**
     * Return current opened socket on OTU parser
     * @return \app\parser\SMTOtuSocket
     */
    function getCurrentOtuSocket()
    {
        return $this->socket;
    }
    
    /**
     * Check OTU local mode and update the context with the value
     * 
     * 
     * @return boolean Whether the OTU is in local mode and update the context with the value.
     */
    public function checkLocalMode()
    {
        $localMode = SMTEthernet::isLocalMode( $this->socket );
        if ( $localMode )
        {
            $this->context->setLocalMode();
        }
        else
        {
            $this->context->setNormalMode();
        }
        return $localMode;
    }    
    
    /**
     * WARNING, no longer retrieve internal port from ISU: 
     * use external port due to race conditions issues in OTU application between smartOTU and ISU 
     * 
     * (No longer valid: Retrieve the OTU parser port from ISU application.)
     * 
     * @param $context
     * 
     * @throws SMTSocketException if the port of the OTU application cannot be retrieved.
     * @return number the OTU parser port or NULL
     */
    public static function retrieveOtuPort( $context )
    {
        /*$socket = new SMTOtuSocket( $context );
        $portOtu = NULL;
        
        //retrieve OTU port from ISU
        try 
        {
            $socket->open( SMTOtuSocket::ISU_PORT );
            $otu_slic_cmd = sprintf(SMTOtuApi::CMD_otu_port,'BOTH','SLIC2','OTU');
            $portOtu = $socket->sendReceive( $otu_slic_cmd );   
            $socket->close();
        } 
        catch (SMTSocketException $e) 
        {
            $portOtu = NULL;        
            $socket->close();            
        }
        
        if ( $portOtu === NULL )
        {
            try
            {
            	$socket->open( SMTOtuSocket::ISU_PORT );
            	$otu_base_cmd = sprintf(SMTOtuApi::CMD_otu_port,'BOTH','BASE','OTU');
            	$portOtu = $socket->sendReceive( $otu_base_cmd );
            	$socket->close();
            }
            catch (SMTSocketException $e)
            {
                $context->traceException($e);
                $portOtu = NULL;
                $socket->close();                
            }            
        }
                
        // if port otu has not been found, throw an exception.
        if ( $portOtu === NULL )
        {
            $e = new SMTSocketException( SMTSocketException::ERROR_OTU_PORT );
            self::handleSocketException($e, $context );            
        	throw $e;
        }                       
        
        return ($portOtu !== NULL)? intval( $portOtu ) : NULL;*/
        
        return SMTOtuSocket::OTU_EXTERNAL_PORT;
    }    
    
    /**
     * Retrieve FO port from ISU
     *
     * @param $context
     *
     * @throws SMTSocketException if the port of the OTU application cannot be retrieved.
     * @return number the FO parser port or NULL
     */
    public static function retrieveFOPort( $context )
    {
        $socket = new SMTOtuSocket( $context );
        $portFO = NULL;
        
        //retrieve OTU-FO port from ISU
        try
        {
            $socket->open( SMTOtuSocket::ISU_PORT );
            $otu_slic_cmd = sprintf(SMTOtuApi::CMD_otu_port,'BOTH','SLIC2','OTU-FO');
            $portFO = $socket->sendReceive( $otu_slic_cmd );
            $socket->close();
        }
        catch (SMTSocketException $e)
        {
            $portFO = NULL;
            $socket->close();
        }
        
        if ( $portFO === NULL )
        {
            try
            {
                $socket->open( SMTOtuSocket::ISU_PORT );
                $otu_base_cmd = sprintf(SMTOtuApi::CMD_otu_port,'BOTH','BASE','OTU-FO');
                $portFO = $socket->sendReceive( $otu_base_cmd );
                $socket->close();
            }
            catch (SMTSocketException $e)
            {
                $context->traceException($e);
                $portFO = NULL;
                $socket->close();
            }
        }
        
        return ($portFO !== NULL)? intval( $portFO ) : 8002;
    }
    
    /**
     * Check if the OTU application is available only if otu availibility status has not been set or is currently not available.
     * If Otu is not available, update its status in SmartOtu and interrupt the current processing to directly return an error code to the GWT client.
     * 
     * @param $context SMTContext
     * 
     * @throws Exception
     */
    public static function checkOTUAvailable( SMTContext $context )
    {
    	$smartOTUStatus = SMTContext::getSmartOTUStatus();
    	if ( $smartOTUStatus === NULL || !$context->isOTUAvailable() )
    	{
    	    //test rescue mode
    	    if ( self::checkRescueMode() )
    	    {
    	        $context->setRescueMode();
    	    }
    	    else
    	    {
        		//WARNING: force otu status retrieval and throw an exception if a problem occurs:
        		try
        		{
        		    $socketManager = $context->getOtuSocketManager();    		    
        			$socketManager->checkSocket();
        		}
        		catch( \Exception $e )
        		{    			  			
        			SMTLogger::getInstance()->trace( "Couln't connect to OTU parser: ".$e->getMessage(), SMTLogger::ERROR );
                    throw $e;
    		}
    	    }
    	}
    
    	return $context->isOTUAvailable();
    }
    
    /**
     * Retrieve whether the OTU is in rescue mode from disk
     * 
     * @return boolean
     */
    public static function checkRescueMode()
    {
        $rescue = FALSE;
        
        $handle = @fopen("/proc/device-tree/board/rescue-mode", "rb");
        $i = 0;
        if ($handle) 
        {
        	while (!feof($handle)) 
        	{
        		$buffer[$i++]= fread( $handle, 1);
        	}
        	fclose($handle);
        	$rescue = ($buffer[self::RESCUE_MODE_INDEX] == chr(self::RESCUE_MODE_VALUE));
        }
                
        //SMTLogger::getInstance()->trace("rescueString $i:".$buffer[0].$buffer[1].$buffer[2].$buffer[3]); 

        return $rescue;
    }
    
    /**
     * Processes socket exceptions and update OTU application status 
     * 
     * @param \Exception $e
     * @param SMTContext $context
     */
    public static function handleSocketException( \Exception $e, SMTContext $context )
    {
        if ( $e instanceof SMTSocketException )
        {
            switch ( $e->getErrorCode() )
            {
                case SMTSocketException::ERROR_OTU_APPLICATION_STARTING:  
                    $context->setOTUStarting();
                break;
    			case SMTSocketException::ERROR_OTU_APPLICATION_NOT_AVAILABLE:
    			case SMTSocketException::COMMAND_FAILURE:
    			case SMTSocketException::COMMAND_TIMEOUT:
    			case SMTSocketException::ERROR_WRITTING:
    			case SMTSocketException::ERROR_READING:    			    
    			case SMTSocketException::ERROR_CREATING:
    			case SMTSocketException::ERROR_SETTING_OPTIONS:
    			case SMTSocketException::ERROR_OPENING:    	
    			case SMTSocketException::ERROR_OTU_PORT:
                    $context->setOTUNotAvailable();
    			break;
    			default:
    				$context->setOTUNotAvailable();    			        
    			break;               
            }  
        }
        else
        {
            $context->setOTUNotAvailable();
        }
    }
    
    /**
     * Check connection to OTU parser and send back exception directly to GWT client if it fails.
     * It is a lightweigth check without any reading/writing on the socket. cf checkOTUAvailable().
     * If Otu is not available, update its status in SmartOtu.
     *
     * @param $context SMTContext
     *
     * @throws Exception
     */
    public static function checkOtuConnection( SMTContext $context )
    {
    	try
    	{
    		SMTOtuSocket::ping( $context );
    	}
    	catch (SMTSocketException $e)
    	{
    		self::handleSocketException( $e, $context );    
    		$context->getLogger()->trace( "Couln't ping OTU parser: ".$e->getMessage(), SMTLogger::ERROR );
    		throw $e;
    	}
    }
    
    /**
     * Timout of ping function on OTU parser.
     * 
     * @return number
     */
    public static function getPingTimout()
    {
        return SMTOtuSocket::PING_TIMEOUT;
    }
}
?>