from viavi.evm import *
from viavi.diag import *

import viavi.mts.uts_spi as utsspi
import viavi.mts.uts_bdt as utsbdt
import select
import os
import os.path
import subprocess
import struct
import smbus
import hashlib
import sys
import time
from datetime import datetime
import serial
import re


# TACB Specific functions
def Read_FPGA_SynDate(M):
    date = sysfs_read_string("/sys/platform_m%s/read_only/ip_synthesis_date" % (M))
    sd = int(date, 16)
    # timestamp = datetime.datetime.fromtimestamp(sd)
    print("\n##############################################")
    print("FPGA Synthesis date:", end='')
    # print(sd)
    d = (sd >> 27)
    M = ((sd >> 23) & 15)
    Y = ((sd >> 17) & 63)
    h = ((sd >> 12) & 31)
    m = ((sd >> 6) & 63)
    s = ((sd) & 63)
    print("%02d/%02d/20%02d %02d:%02d:%02d" % (d, M, Y, h, m, s))
    print("##############################################\n")


def Read_FPGA_Version(M):
    date = sysfs_read_string("/sys/platform_m%s/read_only/ip_version" % (M))
    sd = int(date, 16)
    print("\n##############################################")
    print("FPGA Version: ", end='')
    # print(sd)
    print(int(sd / 65536), end='')
    print(".", end='')
    print(sd - 65536)
    print("##############################################\n")


def Set_MTS_Power(M, state):
    if state == 1:
        sysfs_write("/sys/platform_m%s/powersupply/all" % (M), 0xFFFFFFFF)
        print("\nSwitch ON MTS Power Supply")
        time.sleep(2)
    else:
        sysfs_write("/sys/platform_m%s/powersupply/all" % (M), 0x0)
        print("\nSwitch OFF MTS Power Supply")


@DIAG("READ FPGA(ZYNQ) VERSION")
def t_tacb_fpga_version(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_fpga_version 1/2")
        return False
    Read_FPGA_Version(Hetnet_Module)
    return True


@DIAG("READ FPGA(ZYNQ) SYNTHESIS DATE")
def t_tacb_fpga_synthesis(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_fpga_version 1/2")
        return False
    Read_FPGA_SynDate(Hetnet_Module)
    return True


def Read_ZYNQ_PS(M):
    print("\nRead Internal ZYNQ PS voltages from Zynq xadc aux. channels:")
    res = 0
    # psint: 1.00V
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/psint" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 3000 / 4095)
    print("   psint = %d mV" % (val_mv))
    res = res + sysfs_check("/sys/platform_m%s/xadc/psint" % (M), "tension psint",
                            vmin=int(CONST("zynq.psint.min", (990 * 4095 * 16) / 3000)),
                            vmax=int(CONST("zynq.psint.max", (1010 * 4095 * 16) / 3000)))

    # psaux: 1.8V
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/psaux" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 3000 / 4095)
    print("   psaux = %d mV" % (val_mv))
    res = res + sysfs_check("/sys/platform_m%s/xadc/psaux" % (M), "tension psaux",
                            vmin=int(CONST("zynq.psaux.min", (1790 * 4095 * 16) / 3000)),
                            vmax=int(CONST("zynq.psaux.max", (1810 * 4095 * 16) / 3000)))

    # psmem: 1.2V
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/psmem" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 3000 / 4095)
    print("   psmem = %d mV" % (val_mv))
    res = res + sysfs_check("/sys/platform_m%s/xadc/psmem" % (M), "tension psmem",
                            vmin=int(CONST("zynq.psmem.min", (1190 * 4095 * 16) / 3000)),
                            vmax=int(CONST("zynq.psmem.max", (1210 * 4095 * 16) / 3000)))

    # plint: 1.0V
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/plint" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 3000 / 4095)
    print("   plint = %d mV" % (val_mv))
    res = res + sysfs_check("/sys/platform_m%s/xadc/plint" % (M), "tension plint",
                            vmin=int(CONST("zynq.plint.min", (990 * 4095 * 16) / 3000)),
                            vmax=int(CONST("zynq.plint.max", (1010 * 4095 * 16) / 3000)))

    # plaux: 1.8V
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/plaux" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 3000 / 4095)
    print("   plaux = %d mV" % (val_mv))
    res = res + sysfs_check("/sys/platform_m%s/xadc/plaux" % (M), "tension plaux",
                            vmin=int(CONST("zynq.plaux.min", (1790 * 4095 * 16) / 3000)),
                            vmax=int(CONST("zynq.plaux.max", (1810 * 4095 * 16) / 3000)))

    # plbram: 1.0V
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/plbram" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 3000 / 4095)
    print("   plbram = %d mV" % (val_mv))
    res = res + sysfs_check("/sys/platform_m%s/xadc/plbram" % (M), "tension plbram",
                            vmin=int(CONST("zynq.plbram.min", (990 * 4095 * 16) / 3000)),
                            vmax=int(CONST("zynq.plbram.max", (1010 * 4095 * 16) / 3000)))

    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/temperature" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 504 / 4095) - 273
    print("   Zynq internal temperature = %d C" % (val_mv))

    return res


@DIAG("Read and Check Internal ZYNQ PS")
def t_tacb_check_zynq_ps(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_check_zynq_ps 1/2")
        return False
    res = Read_ZYNQ_PS(Hetnet_Module)
    return res == 6


def Read_MTS_PS(M):
    print("\nRead MTS PS voltages output to 6K module from Zynq xadc aux. channels:")
    res = 0

    R1 = 100000
    R2 = 8250
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/aux0" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VCC12V8 / AUX0 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform_m%s/xadc/aux0" % (M), "tension VCC12V8",
                            vmin=int(CONST("mtsps.aux0.min", ((((12650 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux0.max", ((((12950 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))

    R1 = 10000
    R2 = 2260
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/aux1" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VCC5V / AUX1 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform_m%s/xadc/aux1" % (M), "tension VCC5V",
                            vmin=int(CONST("mtsps.aux1.min", ((((4950 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux1.max", ((((5050 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))

    R1 = 10000
    R2 = 6810
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/aux2" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VCC1V8 / AUX2 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform_m%s/xadc/aux2" % (M), "tension VCC1V8",
                            vmin=int(CONST("mtsps.aux2.min", ((((1750 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux2.max", ((((1850 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))

    R1 = 10000
    R2 = 1780
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/aux8" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VCC5V8 / AUX8 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform_m%s/xadc/aux8" % (M), "tension VCC5V8",
                            vmin=int(CONST("mtsps.aux8.min", ((((5750 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux8.max", ((((5950 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))

    R1 = 10000
    R2 = 3320
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/aux9" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VCC3V3 / AUX9 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform_m%s/xadc/aux9" % (M), "tension VCC3V3",
                            vmin=int(CONST("mtsps.aux9.min", ((((3250 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux9.max", ((((3380 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))

    R1 = 10000
    R2 = 2260
    R1P = 22600
    R2P = 8250
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/aux3" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    valn_mv = ((R2 + R1) * val_mv) / R2
    print("   VEE12V8 / AUX3 = -%d mV" % ( ((R1P * valn_mv) / R2P)))
    res = res + sysfs_check("/sys/platform_m%s/xadc/aux3" % (M), "tension VEE12V8",
                            vmin=int(CONST("mtsps.aux3.min",
                                           (((((R2P * (12600) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux3.max",
                                           (((((R2P * (12900) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)))

    R1 = 10000
    R2 = 2260
    R1P = 22600
    R2P = 17800
    val_hex = sysfs_read_string("/sys/platform_m%s/xadc/aux11" % (M))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    valn_mv = ((R2 + R1) * val_mv) / R2
    print("   VEE5V8 / AUX11 = -%d mV" % ( (R1P * valn_mv) / R2P))
    res = res + sysfs_check("/sys/platform_m%s/xadc/aux11" % (M), "tension VEE5V8",
                            vmin=int(CONST("mtsps.aux11.min",
                                           (((((R2P * (5600) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux11.max",
                                           (((((R2P * (5900) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)))

    return res


@DIAG("Read and Check MTS PS")
def t_tacb_check_mts_ps(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_check_mts_ps 1/2")
        return False
    res = Read_MTS_PS(Hetnet_Module)
    return res == 7


@DIAG("Verify JTAG ID")
def t_tacb_jtag(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_check_mts_ps 1/2")
        return False
    # print(int(Hetnet_Module))
    res = 0
    f = open("/tmp/diag_jtag", "w")
    f.write('cable tacb interface=%d\n' % (int(Hetnet_Module) - 1))
    f.write('detect\n')
    f.close()
    # res = Read_MTS_PS(Hetnet_Module)
    p = subprocess.Popen("jtag /tmp/diag_jtag", shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                         stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
            # print_info("col %s" % columns[0])
            if ((columns[0] == "Part(0)") and (columns[1] == "LC456ZC-75TN100C (0x1814)")):
                res = 1
    return res == 1


@DIAG("Read Z112 CPLD VERSION REGISTER")
def t_tacb_spi(cmd):
    try:
        (command, Hetnet_Module, version) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_spi 1/2 version(hex)")
        return False

    if (Hetnet_Module == "2"):
        # opside s2
        Utsspi = utsspi.SpiModule(1, 2, "/dev/uts_spi")
    else:
        # pwrside s2
        Utsspi = utsspi.SpiModule(0, 2, "/dev/uts_spi")

    # Read VERSION reg at@0x00 (Should be 0x09)
    ver = Utsspi.read_mod_register8(0x00)
    ver = Utsspi.read_mod_register8(0x01)
    ver = Utsspi.read_mod_register8(0x02)
    ver = Utsspi.read_mod_register8(0x00)
    print("CPLD VERSION 0x%2x" % ver)
    if ver == int(version, 16):
        return True
    else:
        return False


@DIAG("TACB sicutil init")
def t_tacb_sicutil_init(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_sicutil_init 1/2")
        return False
    fab_date = datetime.today().strftime('%Y%m%d')
    err = 0
    p = subprocess.Popen("sicutil -w -sys %s  HardwareSetName TACB" % (Hetnet_Module), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  HardwareSetId 703100B10-001" % (Hetnet_Module), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  Revision 1" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  SerialNumber 00001" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  ManufactureDate %s" % (Hetnet_Module, fab_date), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  CalibrationDate 20000101" % (Hetnet_Module), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  MinimumPower 5" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  CriticalTemp 85" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  PcieEndpoints 5" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  ParentSerialNumber none" % (Hetnet_Module), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -r -sys %s" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", flush=True)
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
    return err == 10


@DIAG("TACB sicutil read")
def t_tacb_sicutil_r(cmd):
    try:
        (command, Hetnet_Module, field) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_sicutil_r 1/2 field/'all'")
        return False
    if (field == "all"):
        p = subprocess.Popen("sicutil -r -sys %s" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    else:
        print(field)
        p = subprocess.Popen("sicutil -r -sys %s %s" % (Hetnet_Module, field), shell=True, stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
    # print(l)
    if l:
        return True
    else:
        return False


@DIAG("TACB sicutil write")
def t_tacb_sicutil_w(cmd):
    try:
        (command, Hetnet_Module, field, field_value) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_sicutil_w 1/2 field field_value")
        print_error(
            "# Possible field are: # HardwareSetName  # HardwareSetId # Revision # SerialNumber # ManufactureDate # CalibrationDate # MinimumPower # CriticalTemp # PcieEndpoints # ParentSerialNumber")
        return False
    print(field)
    p = subprocess.Popen("sicutil -w -sys %s %s %s" % (Hetnet_Module, field, field_value), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    print(err)
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
            return False
    return True


@DIAG("TACB sicutil solution read")
def t_tacb_sicutil_sol_r(cmd):
    try:
        (command, Hetnet_Module, field) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_sicutil_sol_r 1/2 field/'all'")
        return False
    if (field == "all"):
        p = subprocess.Popen("sicutil -r -sol %s" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    else:
        print(field)
        p = subprocess.Popen("sicutil -r -sol %s %s" % (Hetnet_Module, field), shell=True, stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
    # print(l)
    if l:
        return True
    else:
        return False


@DIAG("TACB sicutil solution partition write")
def t_tacb_sicutil_sol_w(cmd):
    try:
        (command, Hetnet_Module, field, field_value) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_sicutil_sol_w 1/2 field field_value")
        print_error(
            "# Possible field are: # ProductName # ManufactureId   # ProductSerialNumber # OTH_Prod_Parameter_[0-9] # OTH_Prod ")
        return False
    print(field)
    p = subprocess.Popen("sicutil -w -sol %s %s %s" % (Hetnet_Module, field, field_value), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    print(err)
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
            return False
    return True


@DIAG("Write TACB QSPI Flash if necessary")
def t_tacb_prog_flash(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_prog_flash 1/2 ")
        return False
    test = os.path.exists("/sys/kernel/config/device-tree/overlays/sol%s" % (Hetnet_Module))
    print(test)
    if test == False:
        print("Hep")
        p = subprocess.Popen("mkdir -p /sys/kernel/config/device-tree/overlays/sol%s" % (Hetnet_Module), shell=True,
                             stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
        p.wait()
        err = p.returncode
        print(err)
        p = subprocess.Popen(
            "cat /usr/share/overlays/703100B10-001_1_%s.dtbo > /sys/kernel/config/device-tree/overlays/sol%s/dtbo" % (
                Hetnet_Module, Hetnet_Module), shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT, close_fds=True)
        p.wait()
        err = p.returncode
        print(err)

    bin_present = os.path.exists("/acterna/storage/fpga/u-boot-tacb.bin")
    print(bin_present)
    if bin_present == True:
        bin_file_path = "/acterna/storage/fpga/u-boot-tacb.bin"
    else:
        bin_file_path = "/tmp/u-boot-tacb.bin"

    print("Binary file path: %s" %(bin_file_path))

    p = subprocess.Popen("fiberprogfpga tacb -sf %s %s" % (Hetnet_Module, bin_file_path), shell=True, stdin=subprocess.PIPE,stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    print(err)

    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]

    if err == 0 or err == 11:
        return True
    else:
        return False


@DIAG("Check if i2c module eeprom is present at 0x52")
def t_tacb_i2c(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_i2c 1/2 ")
    f = open("/sys/platform/uts_scan/i2c_master_s%s" % Hetnet_Module, "r")
    i2c_bus = f.readline()
    f.close()
    i2c_bus = int(i2c_bus)
    print(i2c_bus)
    if i2c_bus <= 0:
        print("Bad i2c_bus detected %d in /sys/platform/uts_scan/i2c_master_s%s" % (i2c_bus, Hetnet_Module))
        return False

    # Check also PCA9535 on TACB is detected
    # This chip is managed by dts overlay and associate a driver, so just check if the file exist
    try:
        f = open("/sys/platform_m%s/boot_from_sdcard/value" % Hetnet_Module, "r")
        print(f.read(), end="")
        print("TACB PCA9525 detected")
    except:
        print("TACB PCA9525 Not detected")
        return False

    eeprom = smbus.SMBus(i2c_bus)
    address = 0x52
    try:
        val = eeprom.read_i2c_block_data(address, 0, 4)
        print("Z112 EEPROM detected")
        return True
    except:
        print("Z112 EEPROM Not detected")
        return False


@DIAG("Ping Ethernet test on TACB ethernet interface'10M HD only)")
def t_tacb_eth(cmd):
    try:
        (command, Hetnet_Module, ip_addr, netmask, hostname_to_ping) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_eth 1/2 ip_addr netmask hostname_to_ping")

    # First setup Eth PHy a 10M HD
    p = subprocess.Popen(
        "mii-tool -A 10baseT-HD eth_M%s" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
        stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    #print(err)
    # Configure eth interface
    p = subprocess.Popen(
        "ifconfig eth_M%s %s netmask %s" % (Hetnet_Module, ip_addr, netmask), shell=True, stdin=subprocess.PIPE,
        stdout=subprocess.PIPE,
        stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    #print(err)
    time.sleep(5)
    # hostname = "10.33.17.16"
    print("ping -I eth_M%s -c 1 %s"%(Hetnet_Module, hostname_to_ping))
    response = os.system("ping -I eth_M%s -c 1 %s"%(Hetnet_Module, hostname_to_ping))
    # and then check the response...
    if response == 0:
        pingstatus = "Network Active"
        print(pingstatus)
        return True
    else:
        pingstatus = "Network Error"
        print(pingstatus)
        return False


@DIAG("PS sync ")
def t_tacb_ps_sync(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_ps_sync 1/2")

    if (Hetnet_Module == "2"):
        # opside s2
        Utsspi = utsspi.SpiModule(1, 2, "/dev/uts_spi")
    else:
        # pwrside s2
        Utsspi = utsspi.SpiModule(0, 2, "/dev/uts_spi")

    meas_ok = 0
    timeout = 10
    # MISC register
    # Write SYNCHRO_MEASURE_START to '1'
    ver = Utsspi.write_mod_register8(0x04, 0x10)
    while meas_ok == 0 and timeout != 0:
        ver = Utsspi.read_mod_register8(0x04)
        meas_ok = (ver & 32) >> 5
        timeout = timeout - 1
    if timeout == 0:
        print("Can't get MEASURE_OK")
        return False
    ver = Utsspi.read_mod_register8(0x05)
    Freq = 33333 / ver
    print("SYNC MEASURE %d KHz (0x%2x)" % (Freq, ver))
    if Freq < 317 or Freq > 323:
        return False
    else:
        return True


@DIAG("Check TACB EMMC")
def t_tacb_emmc(cmd):
    try:
        (command, Hetnet_Module, Id) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_emmc 1/2 13(Manufacturer ID)")

    res = 0

    ser = serial.Serial()
    if (Hetnet_Module == "2"):
        ser.port = "/dev/ttyUSB_M2"
    else:
        ser.port = "/dev/ttyUSB_M1"

    ser.baudrate = 115200
    ser.bytesize = serial.EIGHTBITS  # number of bits per bytes
    ser.parity = serial.PARITY_NONE  # set parity check: no parity
    ser.stopbits = serial.STOPBITS_ONE  # number of stop bits
    # ser.timeout = None          #block read
    ser.timeout = 1  # non-block read
    # ser.timeout = 2              #timeout block read
    ser.xonxoff = False  # disable software flow control
    ser.rtscts = False  # disable hardware (RTS/CTS) flow control
    ser.dsrdtr = False  # disable hardware (DSR/DTR) flow control
    ser.writeTimeout = 2  # timeout for write

    try:
        ser.open()
    except:
        print("error open serial port: ")
        exit()

    if ser.isOpen():

        try:
            ser.flushInput()  # flush input buffer, discarding all its contents
            ser.flushOutput()  # flush output buffer, aborting current output
            # and discard all that is in buffer

            # EMMC is on dev1
            ser.write(b"mmc dev 1\r\n")
            numOfLines = 0
            tmp = ""
            while ((numOfLines < 10) and (tmp[:5] != "TACB>")):
                # print("###"+tmp[:5]+"###")
                # print(len(tmp))
                response = ser.readline()
                tmp = response.decode('Ascii')
                # print(response.decode('Ascii'), end="")
                print(tmp, end="")
                numOfLines = numOfLines + 1
                if (tmp[:30] == "mmc1(part 0) is current device"):
                    res = 1;

            if res == 1:
                ser.flushInput()  # flush input buffer, discarding all its contents
                ser.flushOutput()  # flush output buffer, aborting current output
                # and discard all that is in buffer
                ser.write(b"mmcinfo\r\n")
                numOfLines = 0
                tmp = ""
                while ((numOfLines < 10) and (tmp[:5] != "TACB>")):
                    # print("###"+tmp[:5]+"###")
                    # print(len(tmp))
                    response = ser.readline()
                    tmp = response.decode('Ascii')
                    # print(response.decode('Ascii'), end="")
                    print(tmp, end="")
                    numOfLines = numOfLines + 1
                    if (tmp[:19] == "Manufacturer ID: " + Id):
                        res = 2;

            ser.close()
        except:
            print("error communicating...: ")

    else:
        print("cannot open serial port ")

    print(res)

    if res == 2:
        return True
    else:
        return False


@DIAG("Check TACB LPDDR2 RAM test")
def t_tacb_ram(cmd):
    try:
        (command, Hetnet_Module, test) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_ram 1/2 test(FAST/SIMPLE/MARINESCU/MARCH17N)")

    res = 0

    ser = serial.Serial()
    if (Hetnet_Module == "2"):
        ser.port = "/dev/ttyUSB_M2"
    else:
        ser.port = "/dev/ttyUSB_M1"
    ser.baudrate = 115200
    ser.bytesize = serial.EIGHTBITS  # number of bits per bytes
    ser.parity = serial.PARITY_NONE  # set parity check: no parity
    ser.stopbits = serial.STOPBITS_ONE  # number of stop bits
    ser.timeout = None  # block read
    # ser.timeout = 1  # non-block read
    # ser.timeout = 2              #timeout block read
    ser.xonxoff = False  # disable software flow control
    ser.rtscts = False  # disable hardware (RTS/CTS) flow control
    ser.dsrdtr = False  # disable hardware (DSR/DTR) flow control
    ser.writeTimeout = 2  # timeout for write

    try:
        ser.open()
    except:
        print("error open serial port: ")
        return False

    if ser.isOpen():

        try:
            ser.flushInput()  # flush input buffer, discarding all its contents
            ser.flushOutput()  # flush output buffer, aborting current output
            # and discard all that is in buffer

            # EMMC is on dev1
            # mystring = "t_ram "+test+" 2\r\n"
            if test == "MARINESCU":
                ser.write(b"t_ram MARINESCU 2\r\n")
            elif test == "MARCH17N":
                ser.write(b"t_ram MARCH17N 2\r\n")
            elif test == "SIMPLE":
                ser.write(b"t_ram SIMPLE 2\r\n")
            else:
                ser.write(b"t_ram FAST 2\r\n")

            numOfLines = 0
            tmp = ""
            while ((numOfLines < 10000) and (tmp[:5] != "TACB>")):
                # print("###"+tmp[:5]+"###")
                # print(len(tmp))
                response = ser.readline()
                tmp = response.decode('Ascii')
                # print(response.decode('Ascii'), end="")
                print(tmp, end="")
                numOfLines = numOfLines + 1
                if (tmp[:12] == "t_ram PASSED"):
                    res = 1;
            ser.close()
        except:
            print("error communicating...: ")

    else:
        print("cannot open serial port ")

    print(res)

    if res == 1:
        return True
    else:
        return False


@DIAG("UART loopback ")
def t_tacb_uart(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_uart 1/2")

    if (Hetnet_Module == "2"):
        # opside s2
        Utsspi = utsspi.SpiModule(1, 2, "/dev/uts_spi")
    else:
        # pwrside s2
        Utsspi = utsspi.SpiModule(0, 2, "/dev/uts_spi")

    # MISC register
    # Write ENABLE_LOOP_UART to '1'
    ver = Utsspi.write_mod_register8(0x04, 0x01)

    ser = serial.Serial()
    if (Hetnet_Module == "2"):
        ser.port = "/dev/ttyM2"
    else:
        ser.port = "/dev/ttyM1"

    ser.baudrate = 115200
    ser.bytesize = serial.EIGHTBITS  # number of bits per bytes
    ser.parity = serial.PARITY_NONE  # set parity check: no parity
    ser.stopbits = serial.STOPBITS_ONE  # number of stop bits
    ser.timeout = None  # block read
    # ser.timeout = 1  # non-block read
    # ser.timeout = 2              #timeout block read
    ser.xonxoff = False  # disable software flow control
    ser.rtscts = False  # disable hardware (RTS/CTS) flow control
    ser.dsrdtr = False  # disable hardware (DSR/DTR) flow control
    ser.writeTimeout = 2  # timeout for write

    try:
        ser.open()
    except:
        print("error open serial port: ")
        return False

    if ser.isOpen():

        try:
            ser.flushInput()  # flush input buffer, discarding all its contents
            ser.flushOutput()  # flush output buffer, aborting current output
            # and discard all that is in buffer

            ser.write(b"Hello world\r\n")
            print("Reading serial port ....")

            numOfLines = 0
            tmp = ""
            while ((numOfLines < 10) and (tmp[:11] != "Hello world")):
                # print("###"+tmp[:11]+"###")
                # print(len(tmp))
                response = ser.readline()
                tmp = response.decode('Ascii')
                # print(response.decode('Ascii'), end="")
                print(tmp, end="")
                numOfLines = numOfLines + 1
            ser.close()
        except:
            print("error communicating...: ")
            return False

    else:
        print("cannot open serial port ")
        return False

    return True


@DIAG("IRQ test ")
def t_tacb_irq(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_irq 1/2")

    if (Hetnet_Module == "2"):
        # opside s2
        Utsspi = utsspi.SpiModule(1, 2, "/dev/uts_spi")
    else:
        # pwrside s2
        Utsspi = utsspi.SpiModule(0, 2, "/dev/uts_spi")

    error = 2
    # MISC register
    # Write nIRQ_Sn to '1'
    ver = Utsspi.write_mod_register8(0x04, 0x08)
    misc = sysfs_read_string("/sys/platform_m%s/msel/register_msel" % (Hetnet_Module))
    irq = int(misc, 16) >> 31
    print(irq)
    if irq == 0:
        error = error - 1

    # Write nIRQ_Sn to '0'
    ver = Utsspi.write_mod_register8(0x04, 0x00)
    misc = sysfs_read_string("/sys/platform_m%s/msel/register_msel" % (Hetnet_Module))
    irq = int(misc, 16) >> 31
    print(irq)
    if irq == 1:
        error = error - 1

    if error == 0:
        return True
    else:
        return False

@DIAG("Test LVDS interface")
def t_tacb_lvds(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_lvds 1/2")

    if (Hetnet_Module == "2"):
        # opside s2
        Utsspi = utsspi.SpiModule(1, 2, "/dev/uts_spi")
        bdt = utsbdt.UtsBdt("/dev/bdt_ops2")
    else:
        # pwrside s2
        Utsspi = utsspi.SpiModule(0, 2, "/dev/uts_spi")
        bdt = utsbdt.UtsBdt("/dev/bdt_pws2")

    bdt.open()
    version = bdt.get_version()
    print("TACB BDT Version 0x%x"%(version))

    # REG LVDS registers
    print("Check with pattern 0xAA55")
    ver = Utsspi.write_mod_register8(0x01, 0x55)
    ver = Utsspi.write_mod_register8(0x02, 0xaa)
    # Set bit 16 to 1
    ver = Utsspi.write_mod_register8(0x03, 0x01)

    #BDT OFF
    bdt.rw_register(bdt.WRITE, 0x48, 0xFFFFFFFF, 0x00000000)
    #RESET_PLL
    bdt.rw_register(bdt.WRITE, 0x48, 0x00000004, 0x00000004)
    # BDT_ON + de-assert RESET_PLL
    bdt.rw_register(bdt.WRITE, 0x48, 0x00000044, 0x00000040)
    time.sleep(1)
    version = bdt.rw_register(bdt.READ, 0x48, 0, 0)
    print("Read BDT Misc register 0x%x"%(version))

    #Set OSA mode and reset the IP
    version = bdt.rw_register(bdt.READ, 0x300, 0, 0)
    print("Read AXI bridge version 0x%x"%(version))
    bdt.rw_register(bdt.WRITE, 0x304, 0xFFFFFFFF, 0x02)
    bdt.rw_register(bdt.WRITE, 0x304, 0xFFFFFFFF, 0x03)
    version = bdt.rw_register(bdt.READ, 0x304, 0, 0)
    print("Read AXI bridge status register 0x%x"%(version))

    size = 5
    buffer = bdt.get_buffer(size)
    x = 0
    while x < size:
        if (buffer[x] != buffer[x - 1]):
            print("buffer[%d] = 0x%x" % (x, buffer[x]))
        x = x + 1
    #Bit[12] is unsafe because connected to PLASER on Z112, just ignore it
    if (buffer[0] == 0xaa55aa55 or buffer[0] == 0xba55ba55):
        return True
    else:
        print("Expecting: 0xaa55aa55 or 0xba55ba55")
        return False

@DIAG("Test TACB FANs interface")
def t_tacb_fan(cmd):
    try:
        (command, Hetnet_Module, Fan) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_fan 1/2(Module) 1/2(Fan)")

    if (Fan == "2"):
        fan=2
    else:
        fan=1
    print("Checking fan%d"%(fan))

    res = 0
    ser = serial.Serial()

    if (Hetnet_Module == "2"):
        ser.port = "/dev/ttyUSB_M2"
    else:
        ser.port = "/dev/ttyUSB_M1"
    ser.baudrate = 115200
    ser.bytesize = serial.EIGHTBITS  # number of bits per bytes
    ser.parity = serial.PARITY_NONE  # set parity check: no parity
    ser.stopbits = serial.STOPBITS_ONE  # number of stop bits
    ser.timeout = None  # block read
    # ser.timeout = 1  # non-block read
    # ser.timeout = 2              #timeout block read
    ser.xonxoff = False  # disable software flow control
    ser.rtscts = False  # disable hardware (RTS/CTS) flow control
    ser.dsrdtr = False  # disable hardware (DSR/DTR) flow control
    ser.writeTimeout = 2  # timeout for write

    try:
        ser.open()
    except:
        print("error open serial port: ")
        return False

    if ser.isOpen():

        try:
            ser.flushInput()  # flush input buffer, discarding all its contents
            ser.flushOutput()  # flush output buffer, aborting current output
            # and discard all that is in buffer

            if fan == 1:
                ser.write(b"t_max6639 1 \r\n")
            else:
                ser.write(b"t_max6639 2 \r\n")

            numOfLines = 0
            tmp = ""
            while ((numOfLines < 10000) and (tmp[:5] != "TACB>")):
                # print("###"+tmp[:5]+"###")
                # print(len(tmp))
                response = ser.readline()
                tmp = response.decode('Ascii')
                # print(response.decode('Ascii'), end="")
                print(tmp, end="")
                numOfLines = numOfLines + 1
                if (tmp[:16] == "t_max6639 PASSED"):
                    res = 1;
            ser.close()
        except:
            print("error communicating...: ")

    else:
        print("cannot open serial port ")

    if res == 1:
        return True
    else:
        return False

@DIAG("Test TACB FANs interface on OTH")
def t_tacb_fan_oth(cmd):
    try:
        (command, Hetnet_Module, Fan, Min, Max) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_fan 1/2(Module) 1/2(Fan Min Max)")

    if (Fan == "2"):
        fan=2
    else:
        fan=1

    min =int(Min)
    max = int(Max)

    print("Checking fan%d speed is betwwen %d and %d"%(fan, min, max))

    res = 0
    test = 0
    ser = serial.Serial()

    if (Hetnet_Module == "2"):
        ser.port = "/dev/ttyUSB_M2"
    else:
        ser.port = "/dev/ttyUSB_M1"
    ser.baudrate = 115200
    ser.bytesize = serial.EIGHTBITS  # number of bits per bytes
    ser.parity = serial.PARITY_NONE  # set parity check: no parity
    ser.stopbits = serial.STOPBITS_ONE  # number of stop bits
    ser.timeout = None  # block read
    # ser.timeout = 1  # non-block read
    # ser.timeout = 2              #timeout block read
    ser.xonxoff = False  # disable software flow control
    ser.rtscts = False  # disable hardware (RTS/CTS) flow control
    ser.dsrdtr = False  # disable hardware (DSR/DTR) flow control
    ser.writeTimeout = 2  # timeout for write

    try:
        ser.open()
    except:
        print("error open serial port: ")
        return False

    if ser.isOpen():

        try:
            ser.flushInput()  # flush input buffer, discarding all its contents
            ser.flushOutput()  # flush output buffer, aborting current output
            # and discard all that is in buffer

            if fan == 1:
                ser.write(b"t_max6639 1 \r\n")
            else:
                ser.write(b"t_max6639 2 \r\n")

            numOfLines = 0
            tmp = ""
            while ((numOfLines < 10000) and (tmp[:5] != "TACB>")):
                # print("###"+tmp[:5]+"###")
                # print(len(tmp))
                response = ser.readline()
                tmp = response.decode('Ascii')
                # print(response.decode('Ascii'), end="")
                if (res < 3):
                    print(tmp, end="")
                #print(res)
                numOfLines = numOfLines + 1
                if (tmp[:29] == "max6639: Device ID 0x58 ...OK"):
                    res = res + 1;
                if (tmp[:35] == "max6639: Manufacturer ID 0x4d ...OK"):
                    res = res + 1;
                if (tmp[14:35] == "current duty cycle 50"):
                    res = res + 1;
                if ( (fan == 1 and tmp[:30] == "max6639: FAN1 tachometer count") or
                     (fan == 2 and tmp[:30] == "max6639: FAN2 tachometer count") ):
                        if res == 3:
                            x = tmp.split()
                            speed = int(x[4])
                            print("SPEED: %d"%speed)
                            res = res + 1
                            if speed >= min and speed <= max:
                                print("TEST OK")
                                test=1
            ser.close()
        except:
            print("error communicating...: ")

    else:
        print("cannot open serial port ")

    if test == 1:
        return True
    else:
        return False


@DIAG("Test TACB FANs interface on OTH")
def t_tacb_fan_resume(cmd):
    try:
        (command, Hetnet_Module, Temp_Min) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_fan_resume Treg(Temp regulation min) ")

    res = 0
    test = 0
    ser = serial.Serial()

    if (Hetnet_Module == "2"):
        ser.port = "/dev/ttyUSB_M2"
    else:
        ser.port = "/dev/ttyUSB_M1"
    ser.baudrate = 115200
    ser.bytesize = serial.EIGHTBITS  # number of bits per bytes
    ser.parity = serial.PARITY_NONE  # set parity check: no parity
    ser.stopbits = serial.STOPBITS_ONE  # number of stop bits
    ser.timeout = None  # block read
    # ser.timeout = 1  # non-block read
    # ser.timeout = 2              #timeout block read
    ser.xonxoff = False  # disable software flow control
    ser.rtscts = False  # disable hardware (RTS/CTS) flow control
    ser.dsrdtr = False  # disable hardware (DSR/DTR) flow control
    ser.writeTimeout = 2  # timeout for write

    try:
        ser.open()
    except:
        print("error open serial port: ")
        return False

    if ser.isOpen():

        try:
            ser.flushInput()  # flush input buffer, discarding all its contents
            ser.flushOutput()  # flush output buffer, aborting current output
            # and discard all that is in buffer

            ser.write(b"setup_max6639 " + Temp_Min.encode() + b"\r\n")

            numOfLines = 0
            tmp = ""
            while ((numOfLines < 10000) and (tmp[:5] != "TACB>")):
                # print("###"+tmp[:5]+"###")
                # print(len(tmp))
                response = ser.readline()
                tmp = response.decode('Ascii')
                # print(response.decode('Ascii'), end="")
                print(tmp, end="")
                #print(res)
                numOfLines = numOfLines + 1
            ser.close()
        except:
            print("error communicating...: ")
            return False

    else:
        print("cannot open serial port ")
        return False

    return True


@DIAG("Display TACB uboot version")
def t_tacb_uboot_version(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_fan 1/2(Module) ")


    res = 0
    ser = serial.Serial()

    if (Hetnet_Module == "2"):
        ser.port = "/dev/ttyUSB_M2"
    else:
        ser.port = "/dev/ttyUSB_M1"
    ser.baudrate = 115200
    ser.bytesize = serial.EIGHTBITS  # number of bits per bytes
    ser.parity = serial.PARITY_NONE  # set parity check: no parity
    ser.stopbits = serial.STOPBITS_ONE  # number of stop bits
    ser.timeout = None  # block read
    # ser.timeout = 1  # non-block read
    # ser.timeout = 2              #timeout block read
    ser.xonxoff = False  # disable software flow control
    ser.rtscts = False  # disable hardware (RTS/CTS) flow control
    ser.dsrdtr = False  # disable hardware (DSR/DTR) flow control
    ser.writeTimeout = 2  # timeout for write

    try:
        ser.open()
    except:
        print("error open serial port: ")
        return False

    if ser.isOpen():

        try:
            ser.flushInput()  # flush input buffer, discarding all its contents
            ser.flushOutput()  # flush output buffer, aborting current output
            # and discard all that is in buffer

            ser.write(b"version \r\n")

            numOfLines = 0
            tmp = ""
            while ((numOfLines < 10000) and (tmp[:5] != "TACB>")):
                # print("###"+tmp[:5]+"###")
                # print(len(tmp))
                response = ser.readline()
                tmp = response.decode('Ascii')
                # print(response.decode('Ascii'), end="")
                print(tmp, end="")
                numOfLines = numOfLines + 1
                #if (tmp[:16] == "t_max6639 PASSED"):
                res = 1;
            ser.close()
        except:
            print("error communicating...: ")

    else:
        print("cannot open serial port ")

    if res == 1:
        return True
    else:
        return False



# If any function i selected on the module, we never create (nor update) the tmp file
def read_temp_module():
        try:
                with open("/tmp/OtdrModuleTemp","r") as f:
                        temp = f.readlines()[0].strip()
                        f.close()
                        return float(temp)
        except:
                #print("Can't open /tmp/OtdrModuleTemp file, Check at least one OTDR module function is selected !")
                return (-273.0)


def read_tmp75_module():
    try:
        with open("/sys/class/hwmon/hwmon2/temp1_input", "r") as f:
            temp = f.readlines()[0].strip()
            f.close()
            return (float(temp)/1000)
    except:
        return (-273.0)


@DIAG("Test TACB Temperature monitoring")
def t_tacb_monitor_temp(cmd):
    try:
        (command, Hetnet_Module) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_monitor 1/2(Module)")

    print("Monitoring temperature")

    res = 0
    ser = serial.Serial()

    if (Hetnet_Module == "2"):
        ser.port = "/dev/ttyUSB_M2"
    else:
        ser.port = "/dev/ttyUSB_M1"
    ser.baudrate = 115200
    ser.bytesize = serial.EIGHTBITS  # number of bits per bytes
    ser.parity = serial.PARITY_NONE  # set parity check: no parity
    ser.stopbits = serial.STOPBITS_ONE  # number of stop bits
    ser.timeout = None  # block read
    # ser.timeout = 1  # non-block read
    # ser.timeout = 2              #timeout block read
    ser.xonxoff = False  # disable software flow control
    ser.rtscts = False  # disable hardware (RTS/CTS) flow control
    ser.dsrdtr = False  # disable hardware (DSR/DTR) flow control
    ser.writeTimeout = 2  # timeout for write

    print("time", end="\t")
    print("date", end="\t")
    print("zynq", end="\t")
    print("max6639", end="\t")
    print("fan1_tacho", end="\t")
    print("fan2_tacho", end="\t")
    print("fan1_dutyCycle", end="\t")
    print("fan2_dutuCycle", end="\t")
    print("tmp75", end="\t")
    print("module", end="\t")
    print("", end="\n")
    loop_forever = True
    while (loop_forever):

        print("%s"%(datetime.today().strftime('%H:%M:%S\t%d/%m/%Y')), end="\t")

        try:
            ser.open()
        except:
            print("error open serial port: ")
            return False

        if ser.isOpen():

            try:
                ser.flushInput()  # flush input buffer, discarding all its contents
                ser.flushOutput()  # flush output buffer, aborting current output
                # and discard all that is in buffer

                ser.write(b"monitor_max6639\r\n")

                numOfLines = 0
                tmp = ""
                while ((numOfLines < 10000) and (tmp[:5] != "TACB>")):
                    # print("###"+tmp[:5]+"###")
                    # print(len(tmp))
                    response = ser.readline()
                    tmp = response.decode('Ascii')
                    # print(response.decode('Ascii'), end="")
                    #print(tmp, end="")
                    tmp1 = tmp.split()
                    if (tmp1[0] == "max6639:"):
                        if (tmp1[1] == "Zynq"):
                            zynq_temp = float(tmp1[3])/1000
                            print("%s"%(zynq_temp), end="\t")
                        if (tmp1[1] == "Internal"):
                            max6639_temp = float(tmp1[3])/1000
                            print("%s" % (max6639_temp), end="\t")
                        if (tmp1[1] == "FAN1" and tmp1[2] == "tachometer"):
                            fan1_tacho = tmp1[4]
                            print("%s" % (fan1_tacho), end="\t")
                        if (tmp1[1] == "FAN2" and tmp1[2] == "tachometer"):
                            fan2_tacho = tmp1[4]
                            print("%s" % (fan2_tacho), end="\t")
                        if (tmp1[1] == "FAN1" and tmp1[2] == "current"):
                            fan1_dc = tmp1[5]
                            print("%s" % (fan1_dc), end="\t")
                        if (tmp1[1] == "FAN2" and tmp1[2] == "current"):
                            fan2_dc = tmp1[5]
                            print("%s" % (fan2_dc), end="\t")

                    numOfLines = numOfLines + 1
                ser.close()
            except:
                print("error communicating...: ")

        else:
            print("cannot open serial port ")

        print("%f"%(read_tmp75_module()), end="\t")
        print("%f"%(read_temp_module()))

        #print("", end="\n")
        #loop_forever = False
        try:
            time.sleep(60)
        except KeyboardInterrupt:
            loop_forever = False

    return True


@DIAG("Check if OSB board is ready (preliminary)")
def t_tacb_is_ready(cmd):
    try:
        (command) = cmd.split()
    except:
        print_error("# Command must be: t_tacb_is_ready")

    p = subprocess.Popen(
        "systemctl is-active init-otdr-modules", shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
        stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    print(err)

    if err == 0:
        return True
    else:
        return False

