#! /bin/sh

function init_tacb_init {
	isTACBUpdated[1]=0
	isTACBUpdated[2]=0

	# TACB management
	#
	# Search for HETNET modules in position 1 or 2 and look for a TACB
	# For each TACB found, flash it's FPGA bitstream and take care of the module under it - if any
	for i in $(seq 1 2);
	do
		echo -n "TACB #$i"

		# Check if a TACB is detected - If not, try another position
		qdbus --literal --system com.viavisolutions.platform.SystemInventory /com/viavisolutions/platform/SystemInventory/$i com.viavisolutions.platform.SystemInventory.ModuleResources.SystemRecords | grep -q "703100B10-"
		[ $? -ne 0 ] && echo " - Not found, continuing" && continue || echo

		# Program the TACB fpga if needed
		/usr/bin/fiberprogfpga tacb -sf ${i} "/acterna/storage/fpga/u-boot-tacb.bin"
		case $? in
			0) # 0  = Flash success - We need to reboot
				echo "FPGA bitstream updated"
				isTACBUpdated[$i]=1
				;;

			11) # 11 = Already up to date - There is nothing to be done
				echo "FPGA bitstream already up to date."
				isTACBUpdated[$i]=0
				;;

			*)
				echo "ERROR: FPGA flash failed, continuing anyway"
				isTACBUpdated[$i]=0
				;;
		esac

		# If the TACB has been updated, we can't handle the module under it
		[ ${isTACBUpdated[$i]} -eq 1 ] && continue

		SYS_PLATFORM_DIR="/sys/platform_m${i}"

		# Switch on all TACB power-supplies
		echo "0xFFFFFFFF" > "${SYS_PLATFORM_DIR}/powersupply/all"
		sleep 0.1

		# Flash a "generic" fpga for the module under the TACB
		/usr/bin/load_tacb_submodule_fpga.sh ${i}
		sleep 0.1

		# Select the module, modules can be selected all the time in a TACB
		echo "0x00000081" > "${SYS_PLATFORM_DIR}/msel/register_msel"
		echo "0x000000A1" > "${SYS_PLATFORM_DIR}/msel/register_msel"
		echo "0x00000000" > "${SYS_PLATFORM_DIR}/msel/register_msel"
	done

	if [ ${isTACBUpdated[1]} -eq 1 -o ${isTACBUpdated[2]} -eq 1 ]
	then
		echo "A TACB has been updated - Reboot in 5 seconds"

		# Simulate Otu launch to avoid rebooting in rescue mode - if script exists
		[ -f /sbin/otu_confirm_start.sh ] && /sbin/otu_confirm_start.sh

		sleep 5

		/sbin/reboot
		exit -1
	fi
}

function init_tacb_init_post {
	# Apply post op on fpga (programming the specific bitstream for the module under the TACB)
	for i in $(seq 1 2);
	do
		# Check if a TACB is detected - If not, try another position
		qdbus --literal --system com.viavisolutions.platform.SystemInventory /com/viavisolutions/platform/SystemInventory/$i com.viavisolutions.platform.SystemInventory.ModuleResources.SystemRecords | grep -q "703100B10-"
		[ $? -ne 0 ] && continue

		# Check if a compatible module is detected under the TACB - If not, ignore
		CFG_FILE=$(grep -E "^2[[:space:]]$((i - 1))" /sys/platform/uts_scan/modules | tr -s "\t" | cut -f6 | tr -d " ")
		[ -z "$CFG_FILE" ] && echo "TACB #${i} - No compatible submodule found" && continue

		# Check if a corresponding bitstream exists - If not, ignore
		SVF_FILE=/acterna/storage/fpga/"${CFG_FILE%.cfg}.svf"
		[ ! -e "${SVF_FILE}" ] && echo "TACB #${i} - Specific bitstream '${SVF_FILE}' not found for the submodule" && continue

		# Update the module's FPGA bitstream
		/usr/bin/load_tacb_submodule_fpga.sh ${i} "${SVF_FILE}"
		sleep 0.1

		# Select the module, module can be selected all the time in a TACB
		SYS_PLATFORM_DIR="/sys/platform_m${i}"
		echo "0x00000081" > "${SYS_PLATFORM_DIR}/msel/register_msel"
		echo "0x000000A1" > "${SYS_PLATFORM_DIR}/msel/register_msel"
		echo "0x00000000" > "${SYS_PLATFORM_DIR}/msel/register_msel"
	done
}

function init_tacb_deinit {
	# Search for ALE module (position 1 or 2) if there is a tacb
	# For each TACB detected - de-init what is needed
	for i in `seq 1 2`;
	do
		# Check if a TACB is detected - If not, try another position
		qdbus --literal --system com.viavisolutions.platform.SystemInventory /com/viavisolutions/platform/SystemInventory/$i com.viavisolutions.platform.SystemInventory.ModuleResources.SystemRecords | grep -q "703100B10-"
		[ $? -ne 0 ] && continue

		echo "TACB #$i"

		# Stop the network interface
		/sbin/ifconfig eth_M$i down

		# Reset all power
		# The sequence we follow is there to ensure the different voltages are shutdown in the correct order
		# The goal of this sequence is to avoid the bug with the U90 component on the 6k/8k modules
		# 1 - Cut off CLK80M + Shutdown VDD_3V3_MTS and its derivated VDD_1V8_MTS
		echo "0xFFFeFFe7" > "/sys/platform_m$i/powersupply/all"
		sleep 1

		# 2 - Shutdown VDD_5V_MTS and its negative derivated VEE_12V8_MTS, VEE_5V8_MTS
		echo "0xFFFeFF80" > "/sys/platform_m$i/powersupply/all"
		sleep 1

		# 3 - Shutdown everything else
		echo "0x00000000" > "/sys/platform_m$i/powersupply/all"

		# Shutdown the Zynq - Waiting 4 seconds for the Zynq to take into account the reset signal
		echo "0" > "/sys/platform_m$i/zynq_hw_reset_n/value"
		sleep 4

		# Switch off the TACB through Power-Budget
		qdbus --system com.viavisolutions.platform.PowerBudget /com/viavisolutions/platform/PowerBudget com.viavisolutions.platform.PowerBudget.ModulePowerOff $i
	done
}
