<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\util;

use app\database\SMTSmartOtuDB;


/**
 * Base Class for dto objects built from database resultset with a primary key with auto index. 
 * Those DTOs shouldn't be updated.
 *
 * @author Sylvain Desplat
 */
abstract class SMTResultSetWithAutoIndexDto extends SMTResultSetDto
{    
    /**
     * Insert new record in database with new Id generation
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param string $tableName
     */
    protected function insertNewRecord( SMTSmartOtuDB $dbConnection, $tableName )
    {
    	SMTLogger::getInstance()->trace( sprintf("Create new record for table %s.", $tableName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	//search properties bounded to columns
    	$boundColumns= $this->bindTableParametersForInsert($dbConnection, $tableName );
    	//create insert query
    	$query = $this->formatInsertQuery( $tableName, array_keys( $boundColumns ) );
    	SMTLogger::getInstance()->trace( sprintf("Insert query: %s", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	 
    	$bindParameters = "";
    	$statement = $dbConnection->prepare($query);
    
    	self::handleStatementError( $dbConnection, $statement, "Prepare Insert", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	foreach ( $boundColumns as $column=>$propertyValue )
    	{
    		//for logging
    		$bindParameters.= " :".$column."=>".$propertyValue;
    
    		$statement->bindValue(":".$column, $propertyValue );
    	}
    
    	self::handleStatementError( $dbConnection, $statement->execute(), "Insert", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );    	
    }

    /**
     * Search properties of the current dto instance bound to table (tableName) except for the primary key wich is auto-indexed
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param string $tableName The database table to map
     *
     * @return array Database columns bound with current dto instance properties
     */
    protected function bindTableParametersForInsert( SMTSmartOtuDB $dbConnection, $tableName )
    {
    	$boundColumns= array();
    	$dtoClassName = $this->getDtoClassName();
    	SMTLogger::getInstance()->trace( sprintf("Search properties of dto instance: %s, bound to table: %s.", $dtoClassName, $tableName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	$tablecolumns = $this->getTableColumns( $dbConnection, $tableName );
    	//SMTLogger::getInstance()->trace( sprintf("Table columns for query: %s", print_r( $tablecolumns, TRUE) ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	 
    	foreach ( $tablecolumns as $column )
    	{
    		$propertyName = $this->checkDtoPropertyName( $column );
    		//don't bind the primary key which is auto-indexed
    		if ( $propertyName != NULL && $propertyName != SMTResultSetDto::PRIMARY_KEY_ID )
    		{
    			$bindParameterValue = SMTResultSetDto::getBindParameterValue( $propertyName, $column, $this );
    			if ( $bindParameterValue !== NULL )
    			{
    				$boundColumns[$column] = $bindParameterValue;
    				//SMTLogger::getInstance()->trace( sprintf("Found bound column %s in class %s for property %s with value %s", $column, $dtoClassName, $propertyName, $bindParameterValue ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    			}
    			else
    			{
    				SMTLogger::getInstance()->trace( sprintf("Database column %s not bound in class %s", $column, $dtoClassName), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
    			}
    		}
    		//don't log if column is not found when we search to bind the primary key.
    		else if ( $propertyName != SMTResultSetDto::PRIMARY_KEY_ID )
    		{
    			SMTLogger::getInstance()->trace( sprintf("Database column %s not found in class %s", $column, $dtoClassName), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		}
    	}
    
    	return $boundColumns;
    }
    
    /**
     * Returns NULL as DTO has an autoindex
     *
     * @param SMTSmartOtuDB $dbConnection
     *
     * @return NULL as DTO has an autoindex
     */
    protected function generateId( SMTSmartOtuDB $dbConnection ) 
    {
        return NULL;
    }
    
    /**
     * Build the insert statement to insert the new dto but not its auto generated primary key and not its attached dto properties
     *
     * @param string $tableName the database table to update
     * @param string $tablecolumns the columns of the database table to fill (except the primary key)
     *
     * @return insert sql statement with bindings
     */
    protected function formatInsertQuery( $tableName, $tablecolumns )
    {
    	$insertKeys = "";
    	$insertBindKeys = "";
    
    	foreach ( $tablecolumns as $columnName )
    	{
    	    if ( $columnName != SMTResultSetDto::PRIMARY_KEY_ID )
    	    {
        		$insertKeys = $insertKeys.$columnName."," ;
        		$insertBindKeys = $insertBindKeys.":".$columnName."," ;
    	    }
    	}
    	//remove trailing comma
    	$insertKeys = trim($insertKeys,",");
    	$insertBindKeys = trim($insertBindKeys,",");
    
    	return sprintf( SMTResultSetDto::INSERT_STATEMENT, $tableName, $insertKeys, $insertBindKeys);
    }
}
?>