<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otdr;

use app\services\otdr\SMTAcquisitionParametersDto;

use app\events\SMTEventFactory;

use app\serviceshelper\monitoring\SMTMeasureOnDemandException;

use app\serviceshelper\monitoring\SMTLinkTest;

use app\services\otdr\SMTOtdrMeasureOnDemandParametersDto;

use app\services\otdr\SMTStartOtdrMeasureOnDemandParametersDto;

use app\serviceshelper\monitoring\SMTMeasureOnDemandEventDto;

use app\serviceshelper\module\SMTModule;

use app\serviceshelper\otau\SMTSwitchBufferEncoderDecoder;

use app\util\SMTLogger;

use app\events\operations\SMTOperationEventDto;

use app\parser\SMTOtuApi;
use app\parser\SMTOtuSocket;
use app\services\setup\SMTModuleDto;

/**
 * Handles measure operations: 
 * - Fetch measure parameters
 * - Start a measurement on a port.
 * - utility functions for a measure
 * 
 * @author Sylvain Desplat
 */
class SMTMeasurement extends SMTCommonMeasurement
{         
    /**
     * Start measurement on undeclared link
     *
     * @param SMTStartOtdrMeasureOnDemandParametersDto $acquisitionParameters
     *
     * @return SMTMeasureOnDemandEventDto;
     *
     * @throws SMTMeasureOnDemandException if measurement couldn't be launched
     */
    function startMeasure( SMTStartOtdrMeasureOnDemandParametersDto $acquisitionParameters )
    {        
        $measurementStatus = SMTEventFactory::createMeasureOnDemandEvent( NULL, $acquisitionParameters->getOperationId(), SMTOperationEventDto::NO_PROGRAMMED );
    
        if ( $this->acquireMeasureOnDemandLock() )
        {
            if ( $acquisitionParameters->getOperationId() == NULL )
            {
                $operationId = SMTMeasureOnDemandEventDto::generateMeasureOnDemandOperationId();
            }
            else
            {
                $operationId = $acquisitionParameters->getOperationId();
            }

            SMTLogger::getInstance()->trace( sprintf("Start measurement with id: ",  $operationId ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
            
            //check that no measurement is running
            //if a measurement is already running, abort start measurement operation
            try
            {
                $measurementStatus = $this->checkNoMeasurementRunning( NULL , $operationId );
            }
            catch ( \Exception $e )
            {
                $this->releaseMeasureOnDemandLock();
                $this->getContext()->getLogger()->traceException($e);
                throw $e;
            }            
    
            try
            {
                SMTModule::checkModuleRunningNumber( $acquisitionParameters->getModuleRunningNumber() );
                                
                //check that a switch is available
                $linkHelper = new SMTLinkTest();
                $linkHelper->setContext( $this->getContext() );
                $hasSwitch = $linkHelper->hasSwitches();
                
                //case of no switch available                                
                if ( $acquisitionParameters->getPortNumber() === NULL || !$hasSwitch )
                {
                    $switchCount = 0;
                    $switchBuffer = SMTOtuApi::EMPTY_BUFFER;
                }
                else
                {
                    $switchCount = SMTSwitchBufferEncoderDecoder::DEFAULT_SWITCH_COUNT;
                    $switchBuffer = SMTSwitchBufferEncoderDecoder::encodeSwitch( SMTSwitchBufferEncoderDecoder::DEFAULT_SWITCH_INPUT, $acquisitionParameters->getPortNumber() );
                }
                
                SMTLogger::getInstance()->trace( sprintf("Start measurement with module %s, switch buffer %s, acquisition mode %s, pulse %s, range %s, duration %s, wavelength %s, resolution %s, fiber index %s  ", $acquisitionParameters->getModulePosition(), $switchBuffer,
                        $acquisitionParameters->getAcquisitionMode(), $acquisitionParameters->getPulseNs(), $acquisitionParameters->getRangeKm(), $acquisitionParameters->getAcquisitionTimeSec(),$acquisitionParameters->getWavelengthNm(),$acquisitionParameters->getResolutionCm(), $acquisitionParameters->getFiberIndex() ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);                

                $isDwdm = $acquisitionParameters->getDwdmFrequencyGhz() != NULL;
                
                if ( $isDwdm )
                {
                	$tracePathInfos = $this->getContext()->getOtuSocket()->sendReceive( SMTOtuApi::getExecuteDwdmMeasurementCommand(
	                        $acquisitionParameters->getModulePosition(), 
	                        $switchCount,
	                        $switchBuffer,
	                        $acquisitionParameters->getAcquisitionMode(),
	                        $acquisitionParameters->getPulseNs(),
	                        $acquisitionParameters->getRangeKm(),
	                        $acquisitionParameters->getAcquisitionTimeSec(),
	                        $acquisitionParameters->getFiberIndex(),
	                        0,
	                        $acquisitionParameters->getResolutionCm(),
	                        $acquisitionParameters->getFunctionName(),
	                		$acquisitionParameters->getShortAcq(),
	                		$acquisitionParameters->getDwdmFrequencyGhz()), 
	                		SMTOtuSocket::DEFAULT_SLEEP, SMTOtuSocket::LONGER_TIME_OUT_READING, TRUE);
                }
                else 
                {
                	$tracePathInfos = $this->getContext()->getOtuSocket()->sendReceive( SMTOtuApi::getExecuteMeasurementCommand(
                			$acquisitionParameters->getModulePosition(),
                			$switchCount,
                			$switchBuffer,
                			$acquisitionParameters->getAcquisitionMode(),
                			$acquisitionParameters->getPulseNs(),
                			$acquisitionParameters->getRangeKm(),
                			$acquisitionParameters->getAcquisitionTimeSec(),
                			$acquisitionParameters->getFiberIndex(),
                			$acquisitionParameters->getWavelengthNm(),
                			$acquisitionParameters->getResolutionCm(),
                			$acquisitionParameters->getFunctionName(),
                			$acquisitionParameters->getShortAcq() ), SMTOtuSocket::DEFAULT_SLEEP, SMTOtuSocket::LONGER_TIME_OUT_READING, TRUE);
                }
                
                $tracePathDirNameArray = preg_split( "/[;]/", $tracePathInfos );            
            }
            catch ( \Exception $e )
            {
                //nothing to do, try to get info with getMeasurementStatus
                $this->getContext()->getLogger()->traceException($e);
            }
    
            try
            {
                $measurementStatus = $this->getMeasurementAddStatus( NULL, $operationId );
                $this->releaseMeasureOnDemandLock();
    
                //1 -measurement was started: save some measure features
                $traceDirectory = array_shift( $tracePathDirNameArray );
                $traceName = array_shift( $tracePathDirNameArray );
                $portId = $acquisitionParameters->getPortNumber();
                $this->cacheMeasureInfo($operationId, $portId, $traceDirectory, SMTBaseOtdrTraceFileUtil::removeExtension( $traceName ) );
                
                //2 - measurement was started: start the polling on measure in an asynchronous process
                $measureOnDemandClass = SMTMeasureOnDemandEventDto::getClass();
                $measureOnDemandClass = str_replace("\\", "\\\\", $measureOnDemandClass);
                
                self::executeAsynchronousMeasureProcess( __DIR__.'/measurementProcess.php', $measureOnDemandClass, $operationId );
            }
            catch( \Exception $e )
            {
                $this->releaseMeasureOnDemandLock();
                throw  $e;
            }
        }
        else
        {
            $this->getContext()->getLogger()->trace("Couldn't acquire lock for SMTMeasurement to start measure with operation Id:".$acquisitionParameters->getOperationId(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            throw new SMTMeasureOnDemandException( SMTOperationEventDto::FAILED, MSG_ERROR_START_MEASURE ) ;
        }
    
        return $measurementStatus;
    }               
    
    /**
     * 
     * @param int $linkId
     * @param int $portNumber
     * @param app\services\otdr\SMTStartOtdrMeasureOnDemandParametersDto $acquisitionParametersDto
     */
    function fetchMeasureOnDemandParameters( $linkId, $portNumber, $acquisitionParametersDto = NULL )
    {
        $measureOnDemandParameters = new SMTOtdrMeasureOnDemandParametersDto();
        //default acquisition parameters sent by client:
        if ( $acquisitionParametersDto == NULL )
        {
            $acquisitionParametersDto = new SMTStartOtdrMeasureOnDemandParametersDto();
        }
        
        //retrieve OTDR module functions, wavelengths...
        $module = new SMTModule();
        $module->setContext( $this->getContext() );
        $otdrModule = $module->fetchOtdrModuleFunctions();
        
        if ( $portNumber !== NULL )
        {
            $acquisitionParametersDto->setPortNumber($portNumber);
        }
                
        //force module and function in acquisition parameters
        $acquisitionParametersDto->setModulePosition( $otdrModule->getPosition() );
        $functions = $otdrModule->getFunctions();
        $acquisitionParametersDto->setFunctionName( $functions[0]->getName() );

        //update measure on demand parameters
        $measureOnDemandParameters->setOtdrModule( $otdrModule );
        $measureOnDemandParameters->setOtdrMeasureParameters( $acquisitionParametersDto );
        $measureOnDemandParameters->setForceUseOfParameters( FALSE );

        //acquisition on Link
        if ( $linkId !== NULL )
        {
            $acquisitionParametersDto->setLinkId($linkId);
            //if acquisition parameter retrieval fails, launch the measurement without test parameters
            try 
            {
                $this->processLinkDefaultParameters( $linkId, $measureOnDemandParameters );
            }
            catch(\Exception $e)
            {
                $this->getContext()->getLogger()->traceException($e);
            }
        }
        
        return $measureOnDemandParameters;
    }
    
    /**
     * Fetch test parameters if a test exist.
     * 
     * @param String $linkId
     * @param SMTOtdrMeasureOnDemandParametersDto $measureOnDemandParameters
     */
    //TODO WARNING, impossible to implement because test parameters are not SI. 
    private function processLinkDefaultParameters( $linkId, SMTOtdrMeasureOnDemandParametersDto $measureOnDemandParameters )
    {
        $acquisitionParametersDto = $measureOnDemandParameters->getOtdrMeasureParameters();
        $forceUseOfParameters = FALSE;
        
        //update link Identifier
        $acquisitionParametersDto->setLinkId($linkId);
        
        //if the link has a test, retrieve test parameters
        //retrieve test identifier:
        $linkTestIds = $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId ) );
        $testId = SMTLinkTest::parseLinkTestId( $linkTestIds );
        if ( $testId !== NULL && $testId >= 0 )
        {
        	try 
        	{
	            //force use of monitoring test parameters
	            $forceUseOfParameters = TRUE;
	            $measureOnDemandParameters->setForceUseOfParameters( $forceUseOfParameters );
	            $acquisitionParametersDto->setTestId( $testId );
	            
	            //force manual acquisition
	            $acquisitionParametersDto->setAcquisitionModeAuto(FALSE);
	            
	            //fetch monitoring test
	            $linkHelper = new SMTLinkTest();
	            $linkHelper->setContext( $this->getContext() );
	            
	            //WARNING, test parameters are not SI.            
	       		$testDetailedDto = $linkHelper->fetchTestAcquisitionParameters( $testId, $linkId );
	       		$acquisitionParametersDto->setAcquisitionTimeSec( $testDetailedDto->getAcquisitionTimeSec() );
	       		$acquisitionParametersDto->setPulseNs( $testDetailedDto->getPulseNs() );
	       		$acquisitionParametersDto->setRangeKm( $testDetailedDto->getRangeKm() );
	       		$acquisitionParametersDto->setResolutionCm( $testDetailedDto->getResolutionCm() );
	       		$acquisitionParametersDto->setDwdmFrequencyGhz( $testDetailedDto->getFrequencyGhz() );
	       		$acquisitionParametersDto->setWavelengthNm( $testDetailedDto->getWavelengthNm() );
	       		$acquisitionParametersDto->setFiberIndex( SMTCommonMeasurement::DEFAULT_FIBER_INDEX );
	            //WARNING, impossible to implement because test parameters are not SI.
        	}
        	catch( \Exception $e)
        	{
        		$this->getContext()->getLogger()->trace('Invalid test: '. $testId .' using AUTO parameters ', SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        		// if the test is invalid DTSS? ignore it
        		$forceUseOfParameters = FALSE;
        		$measureOnDemandParameters->setForceUseOfParameters( $forceUseOfParameters );
        		$acquisitionParametersDto->setTestId( NULL );
        		$acquisitionParametersDto->setAcquisitionModeAuto(AUTO);
        	}
        }
    }   
    
    /**
     * Create measurement data for test purpose
     *
     * @param $portNumber WARNING: start from 1
     * 
     * @return SMTStartOtdrMeasureOnDemandParametersDto
     */
    static function createMeasurementTestData( $portNumber )
    {
        $measurementDto = new SMTStartOtdrMeasureOnDemandParametersDto();
        $measurementDto->setModulePosition(SMTModuleDto::MOD2);
        $measurementDto->setPortNumber( intval( $portNumber ) );
        $measurementDto->setAcquisitionModeAuto(TRUE);
        $measurementDto->setAcquisitionTimeSec( 10 );
        $measurementDto->setFunctionName("SM-OTDR");
        $measurementDto->setPulseNs( SMTAcquisitionParametersDto::DEFAULT_PULSE_NS );//3ns
        $measurementDto->setRangeKm(0);//auto
        $measurementDto->setResolutionCm(0);//auto
        $measurementDto->setFiberIndex( SMTCommonMeasurement::DEFAULT_FIBER_INDEX );
        $measurementDto->setWavelengthNm( 1625 );
    
        return $measurementDto;
    }    
}
?>