<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI Solutions
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI Solutions
// Copyright VIAVI Solutions 2022. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otau;

use app\parser\SMTOtuApi;
use app\services\setup\SMTConfDmuxDto;
use app\services\setup\SMTDmuxDto;
use app\serviceshelper\SMTServiceHelper;
use app\util\SMTInvalidValueException;
use app\util\SMTLogger;
use app\util\SMTUtil;
use app\services\setup\SMTModuleDto;
use app\admin\SMTOtuType;
use app\http\SMTContext;

/**
 * Retrieve/set Dmux config
 *
 * @author Sylvain Desplat
 */
class SMTDmuxManager extends SMTServiceHelper
{		
    const DMUX_NAME_INDEX = 0;
    const DMUX_OUTPUT_INDEX = 1;
    const DMUX_GRID_INDEX = 2;
	/**
	 * 
	 * @param SMTDmuxDto $dmux
	 * @throws \InvalidArgumentException
	 * @throws SMTInvalidValueException
	 */
	function add( SMTDmuxDto $dmux)
	{			
	    if( $dmux->isValid() )
		{
			try 
			{
				//Ajout du DMUX dans l'OTU
			    $cmd_switch_config_Add_Dmux_Formatted = sprintf(SMTOtuApi::CMD_add_dmux, trim($dmux->getName(),'\'"'), $dmux->getOutput(),$dmux->getGrid());
				$this->sendReceive($cmd_switch_config_Add_Dmux_Formatted);
			}
			catch(\Exception $e)
			{
			    throw new \InvalidArgumentException( sprintf( MSG_BAD_DMUX, trim($dmux->getName(),'\'"'), $dmux->getGrid(), $dmux->getOutput() ) );
			}
		}
		else
		{
		    throw new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "Dmux grid", $dmux->getGrid());
		}
		
	}
	
	/**
	 * 
	 * @param int $dmuxhId
	 * 
	 * @throws SMTInvalidValueException
	 */
	function remove( $dmuxId )
	{
	    if( intval($dmuxId) > 0)
		{						
		    $this->sendReceive( sprintf(SMTOtuApi::CMD_remove_dmux, $dmuxId) );
		}
		else
		{
		    throw new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "DmuxId", $dmuxId);
		}
	}
	
	/**
	 *
	 * @param SMTConfDmuxDto $dmux
	 * @throws \InvalidArgumentException
	 * @throws SMTInvalidValueException
	 */
	function update( SMTConfDmuxDto $dmuxConf)
	{
	    $dmuxConfCurrent = new SMTConfDmuxDto();
	    $updateDmux = FALSE;
	    
	    try
	    {
            $this->retrieve($dmuxConfCurrent);
        }
        catch ( \Exception $e )
        {
            SMTLogger::getInstance()->traceException($e);
        }
        
        try 
        {
            if ($dmuxConfCurrent->getMaxFrequency() != $dmuxConf->getMaxFrequency() )
            {
                // Update DMUX initial frequency in calibration
                $cmd_update_dmux_initial_frequency_Formatted = sprintf(SMTOtuApi::CMD_set_initial_frequency, $dmuxConf->getMaxFrequency());
                $this->send($cmd_update_dmux_initial_frequency_Formatted);
            }
        } 
        catch (Exception $e) 
        {
            SMTLogger::getInstance()->traceException($e);
        }
	    
        foreach($dmuxConf->getConfiguredDmux() as $dmux)
        {
            $updateDmux = FALSE;
            if ( $dmux->isValid() )
            {
                try
                {
                    
                    
                    $updateDmux = $dmuxConfCurrent->getDmux($dmux->getIndex()) != NULL;
                    
                    if ( $updateDmux )
                    {
                        //Update du DMUX dans l'OTU
                        $cmd_switch_config_Update_Dmux_Formatted = sprintf(SMTOtuApi::CMD_update_dmux, $dmux->getIndex(), trim($dmux->getName(),'\'"'), $dmux->getOutput(),$dmux->getGrid());
                        $this->send($cmd_switch_config_Update_Dmux_Formatted);
                    }
                    else
                    {
                        //Ajout du DMUX dans l'OTU
                        $cmd_switch_config_Add_Dmux_Formatted = sprintf(SMTOtuApi::CMD_add_dmux, trim($dmux->getName(),'\'"'), $dmux->getOutput(),$dmux->getGrid());
                        $this->sendReceive($cmd_switch_config_Add_Dmux_Formatted);
                    }
                }
                catch(\Exception $e)
                {
                    throw new \InvalidArgumentException( sprintf( MSG_BAD_DMUX, $dmux->getName(), $dmux->getGrid(), $dmux->getOutput() ) );
                }
            }
	        else
	        {
	            throw new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "Dmux grid", $dmux->getGrid());
	        }	        
        }
        // deleted DMUX
        foreach($dmuxConfCurrent->getConfiguredDmux() as $dmux)
        {
            $dmuxFound = TRUE;
            if ( $dmux->isValid() )
            {
                try
                {
                    $dmuxFound = $dmuxConf->getDmux($dmux->getIndex()) != NULL;
                    
                    if (!$dmuxFound)
                    {                        
                        //Suppression du DMUX dans l'OTU
                        $cmd_switch_config_Remove_Dmux_Formatted = sprintf(SMTOtuApi::CMD_remove_dmux, $dmux->getIndex());
                        $this->send($cmd_switch_config_Remove_Dmux_Formatted);
                    }
                }
                catch(\Exception $e)
                {
                    throw new \InvalidArgumentException( sprintf( MSG_BAD_DMUX, $dmux->getName(), $dmux->getGrid(), $dmux->getOutput() ) );
                }
            }
        }
	}	
	
	/**
	 * Whether a Dmux optical switch can be used
	 * @param SMTContext $context
	 * @return boolean
	 */
	static function isDmuxSwitchSupported(SMTContext $context)
	{
	    if ( SMTOtuType::isOTH() || SMTOtuType::isFTH7K() )
	    {
	        $moduleName = trim( $context->getOtuSocket()->sendReceive( SMTOtuApi::CMD_get_DWDM_MOD )," \t\n\r\0\"" );
	       return (strpos($moduleName, SMTOtuApi::RES_YES) !== false);
	    }
	    else
	    {
	        return FALSE;
	    }
	}
	
	/**
	 *
	 * @param SMTConfDmuxDto $dmuxConf
	 * @throws \InvalidArgumentException
	 * @throws SMTInvalidValueException
	 */
	function retrieve( SMTConfDmuxDto $dmuxConf)
	{
	    if ( SMTOtuType::isOTH() || SMTOtuType::isFTH7K() )
        {
            $dmuxList = $this->sendReceive(SMTOtuApi::CMD_get_dmux_list);
            $dmuxIdsArray = explode (",", $dmuxList );
            foreach ($dmuxIdsArray as $dmuxId)
            {
                if( intval($dmuxId) > 0)
                {
                    $dmuxData = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_dmux, $dmuxId) );
                    $dmuxInfoArray = SMTUtil::splitStringByComma( $dmuxData );
                    $name = trim( $dmuxInfoArray[self::DMUX_NAME_INDEX] );
                    $name = trim($name,'\'"');
                    $output = trim( $dmuxInfoArray[self::DMUX_OUTPUT_INDEX] );
                    $grid = trim( $dmuxInfoArray[self::DMUX_GRID_INDEX] );
                    $dto = SMTDmuxDto::buildDmux($dmuxId, $name, $output, $grid);
                    if ($dto->isValid())
                    {
                        $dmuxConf->addConfigured($dto);
                    }
                }
            }
            if ( self::isDmuxSwitchSupported($this->getContext()) )
            {
                $dmuxConf->setActive( TRUE );
            }
            try
            {
                //Get DMUX initial frequency from calibration
                $initialFrequency = trim($this->sendReceive(SMTOtuApi::CMD_get_initial_frequency));
                $dmuxConf->setMaxFrequency($initialFrequency);
            }
            catch (Exception $e)
            {
                SMTLogger::getInstance()->traceException($e);
            }
        }
	}
}
?>