<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2014. All rights reserved.
// *********************************************************
namespace app\services\setup;

use app\settings\SMTSmartOtuSettings;

use app\util\SMTIOException;

use app\view\SMTFileDownloadPage;

use app\serviceshelper\media\SMTSnmp;

use app\serviceshelper\media\SMTMediaException;
use app\view\SMTTestRestApiPage;
use app\services\SMTIService;
use app\services\SMTService;
use app\parser\SMTOtuApi;

/**
 * Service to setup the OTU SNMP features and display its configuration.
 * 
 * @author Sylvain Desplat
 */
class SMTSnmpService extends SMTService implements SMTIService
{    
    /**
     * Create empty Dto Email to be sent to the client
     *
     * @url GET /new
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function newSnmp()
    {
    	try
    	{
    		$snmpDto = new SMTSnmpDto();
    		$snmpDto->setPort( SMTSnmpDto::SNMP_DEFAULT_PORT );
    		$snmpDto->setCommunity( SMTSnmpDto::SNMP_DEFAULT_COMMUNITY );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $snmpDto->getJsonData();
    }
    
    /**
     * Create empty Dto SNMPV3 to be sent to the client
     *
     * @url GET /newv3
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function newSnmpV3()
    {
        try
        {
            $snmpDto = new SMTSnmpV3Dto();
            $snmpDto->setPort( SMTSnmpDto::SNMP_DEFAULT_PORT );
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }
        return $snmpDto->getJsonData();
    }
    
    /**
     * Update SNMP config
     *
     * @url POST
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function updateSnmp( $request_data )
    {
        $snmp = new SMTSnmp();
        $snmp->setContext( $this->getContext() );
    	try
    	{
    		$snmpConfDto = SMTConfSnmpDto::getInstance( $request_data );
    		
    		$snmp->updateSnmp( $snmpConfDto );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}

    	//return SNMP configuration
    	return $this->fetchSnmp();
    }        
    
    /**
     * Fetch SNMP configuration
     *
     * @url GET
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchSnmp()    
    {        
        $snmpConfDto = new SMTConfSnmpDto();
        $snmp = new SMTSnmp();
        $snmp->setContext( $this->getContext() );
        
        try
        {
            $snmpConfDto = $snmp->retrieveSnmpConf();      	        
    	}
    	catch ( \Exception $e )
    	{
    	    $this->handleException($e);
    	}
    
    	return $snmpConfDto->getJsonData();
    }
    
    /**
     * Download SNMP MIBs
     *
     * @url GET /download
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function downloadSnmpMibs()
    {
        try 
        {
            $fileName = SMTSnmp::retrieveMibs();
            
            if ( $fileName != NULL )
            {
                $page = new SMTFileDownloadPage($fileName);            
                $this->getContext()->getHttpResponse()->send( $page );            
            }
            else
            {
                $e = new SMTIOException( SMTIOException::FILE_NOT_FOUND );            
                $this->handleException($e);
            }    	
        }
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }
    
    /**
     * Perform a Test of SNMP media: send a TRAP
     *
     * @url POST /test/
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function testSnmp( $request_data )
    {
        try 
        {
            $snmp = new SMTSnmp();
            $snmp->setContext( $this->getContext() );
            $snmpDto = SMTSnmpDto::getInstance( $request_data );
    		
   		    $testMessage = $snmp->buildTestSnmpMessage();
			$cmd_test_media = SMTOtuApi::testSnmpCommand( $snmpDto->getManager(), $snmpDto->getCommunity(), $snmpDto->getPort(), $testMessage );
			$this->send( $cmd_test_media );
    	}
    	catch ( \Exception $e )
    	{
    	    $this->getContext()->traceException($e);
    		if ( !( $e instanceof SMTMediaException ) )
    	    {
    	        $e = new SMTMediaException( SMTMediaException::ERROR_INVALID_CONFIG );
    	    }    		   
    		$this->handleException($e);
    	}    			
    	return self::RESULT_OK;
    }
    
    /**
     * Perform a Test of SNMP media: send a TRAP
     *
     * @url POST /testv3/
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function testSnmpV3( $request_data )
    {
        try
        {
            $snmp = new SMTSnmp();
            $snmp->setContext( $this->getContext() );
            $snmpDto = SMTSnmpV3Dto::getInstance( $request_data );
            
            $testMessage = $snmp->buildTestSnmpMessage();
            $cmd_test_media = SMTOtuApi::testSnmpV3Command( $snmpDto->getManager(), $snmpDto->getUser(), $snmpDto->getPort(), $snmpDto->getAuthType(), $snmpDto->getAuthPassword(), $snmpDto->getEncryptionType(), $snmpDto->getEncryptionPassword(), $testMessage );
            $this->send( $cmd_test_media );
        }
        catch ( \Exception $e )
        {
            $this->getContext()->traceException($e);
            if ( !( $e instanceof SMTMediaException ) )
            {
                $e = new SMTMediaException( SMTMediaException::ERROR_INVALID_CONFIG );
            }
            $this->handleException($e);
        }
        return self::RESULT_OK;
    }
    
    /**
     * @url GET /test/updatesnmp
     */
    function testUpdateSnmp()
    {        
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$snmpConfDto = SMTConfSnmpDto::buildSMTConfSnmpDto( TRUE, FALSE, SMTSnmp::DEFAULT_SNMP_HEART_BEAT_PERIOD, FALSE );
        	
        	$snmpDto = new SMTSnmpDto();
        	$snmpDto->setCommunity(SMTSnmpDto::SNMP_DEFAULT_COMMUNITY);
        	$snmpDto->setIndex(1);
        	$snmpDto->setManager("none");
        	$snmpDto->setPort(SMTSnmpDto::SNMP_DEFAULT_PORT);
        	$snmpConfDto->addSnmp($snmpDto);
        	
        	$url = '/snmp';
        	$json_string = json_encode( $snmpConfDto->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }
    
    /**
     * @url GET /test/snmptest
     */
    function testSnmpTest()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
            $snmpConfDto = SMTConfSnmpDto::buildSMTConfSnmpDto( TRUE, FALSE, SMTSnmp::DEFAULT_SNMP_HEART_BEAT_PERIOD, FALSE );
            
            $snmpDto = new SMTSnmpDto();
            $snmpDto->setCommunity(SMTSnmpDto::SNMP_DEFAULT_COMMUNITY);
            $snmpDto->setIndex(1);
            $snmpDto->setManager("none");
            $snmpDto->setPort(SMTSnmpDto::SNMP_DEFAULT_PORT);
            $snmpConfDto->addSnmp($snmpDto);
            
        	$url = '/snmp/test';
        	$json_string = json_encode( $snmpConfDto->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }

    /**
     * @url GET /test/snmpv3test
     */
    function testSnmpV3Test()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
            $snmpConfDto = SMTConfSnmpDto::buildSMTConfSnmpDto( TRUE, FALSE, SMTSnmp::DEFAULT_SNMP_HEART_BEAT_PERIOD, TRUE );
            
            $snmpDto = new SMTSnmpV3Dto();
            $snmpDto->setUser(SMTSnmpDto::SNMP_DEFAULT_COMMUNITY);
            $snmpDto->setAuthType('MD5');
            $snmpDto->setAuthPassword(SMTSnmpDto::SNMP_DEFAULT_COMMUNITY);
            $snmpDto->setIndex(1);
            $snmpDto->setManager('stelx-desplat');
            $snmpDto->setPort(SMTSnmpDto::SNMP_DEFAULT_PORT);
            $snmpConfDto->addSnmpV3($snmpDto);
            
            $url = '/snmp/test';
            $json_string = json_encode( $snmpConfDto->getJsonData() );
            $page = new SMTTestRestApiPage($url, $json_string, 'POST');
            $this->getContext()->getHttpResponse()->send( $page );
        }
    }
}

?>