<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\services\maintenance;

use app\sharedmemory\SMTIMemorySupport;

use app\sharedmemory\SMTMemoryManager;

use app\util\SMTDto;

/**
 * DTO tracking the Upgrade status
 * 
 * @author Sylvain Desplat
 */
class SMTUpgradeDto extends SMTDto implements SMTIMemorySupport
{
    const UNIQUE_UPGRADE_ID="UNIQUE_UPGRADE_ID"; 
    
    const UPGRADE_NO_ERROR = "UPGRADE_NO_ERROR";
    
    const UPGRADE_REBOOT = "UPGRADE_REBOOT";
    
    const UPGRADE_SUCCESSFULL = "UPGRADE_SUCCESSFULL";

    const UPGRADE_FAILED = "UPGRADE_FAILED";
    
    const UPGRADE_ALLREADY_RUNNING_ERROR = "UPGRADE_ALLREADY_RUNNING_ERROR";
    
    const DUPLICATION_ALLREADY_RUNNING_ERROR = "DUPLICATION_ALLREADY_RUNNING_ERROR";
    
    const NO_UPGRADE_ARCHIVE_ERROR = "NO_UPGRADE_ARCHIVE_ERROR";
    
    /**
     * Last access to query or update the upgrade status cache
     *
     * @var float
     */
    private $lastAccess = 0;
    
    /**
     * The name of the current upgrade
     * 
     * @var string
     */
	protected $currentUpgradeName = NULL;
		
	/**
	 * Whether an upgrade is in progress
	 * 
	 * @var boolean
	 */
	protected $isUpgradeInProgressOnOtu = FALSE;
	
	/**
	 * Whether a duplication is in progress
	 *
	 * @var boolean
	 */
	protected $isDuplicationInProgressOnOtu = FALSE;
	
	/**
	 * Current upgrade error status among error possible values: 
	 * UPGRADE_NO_ERROR, UPGRADE_FAILED, UPGRADE_SUCCESSFULL, UPGRADE_ALLREADY_RUNNING_ERROR, NO_UPGRADE_ARCHIVE_ERROR, UPGRADE_REBOOT
	 *
	 * @var string
	 */
	protected $upgradeError = self::UPGRADE_NO_ERROR;	
	
	/**
	 * Upgrade progress log embedded in an array of string; last operation logged is at the end
	 * 
	 * @var array of string
	 */
	protected $upgradeProgressLogArray = array();
	
	/**
	 * Last upgrade start date or NULL if no upgrade log file is available
	 * 
	 * @var string
	 */
	protected $lastUpgradeStartDate = NULL;

	/**
	 * Current OTU version
	 *
	 * @var string
	 */
	protected $currentOtuVersion = NULL;
	
	/**
	 * Current OTU rescue version
	 *
	 * @var string
	 */
	protected $currentOtuRescueVersion = NULL;
	
	/**
	 * Current upgrade version
	 *
	 * @var string
	 */
	protected $currentUpgradeVersion = NULL;
	
	/**
	 * Whether an upgrade is in progress
	 *
	 * @param boolean $inProgress
	 */
	public function setIsUpgradeInProgressOnOtu( $inProgress )
	{
		$this->isUpgradeInProgressOnOtu = $inProgress;
	}
	
	/**
	 * Whether a duplication is in progress
	 *
	 * @param boolean $inProgress
	 */
	public function setIsDuplicationInProgressOnOtu( $inProgress )
	{
		$this->isDuplicationInProgressOnOtu = $inProgress;
	}
	
	/**
	 * 
	 * @return boolean
	 */
	public function isUpgradeInProgressOnOtu()
	{
	    return $this->isUpgradeInProgressOnOtu;
	}
	
	/**
	 *
	 * @return boolean
	 */
	public function isDuplicationInProgressOnOtu()
	{
		return $this->isDuplicationInProgressOnOtu;
	}
	
	/**
	 * The name of the current upgrade
	 * 
	 * @param string $archiveName
	 */
	public function setCurrentUpgradeName( $archiveName )
	{
	    $this->currentUpgradeName = $archiveName;
	}
	
	/**
	 * The version of the current upgrade
	 *
	 * @param string $version
	 */
	public function setCurrentUpgradeVersion( $version )
	{
		$this->currentUpgradeVersion= $version;
	}
	
	/**
	 * The name of the current upgrade
	 *
	 * @param string $archiveName
	 */
	public function getCurrentUpgradeName()
	{
		return $this->currentUpgradeName;
	}
	
	/**
	 * The name of the current upgrade
	 *
	 * @param string $archiveName
	 */
	public function getCurrentUpgradeVersion()
	{
		return $this->currentUpgradeVersion;
	}
	
	/**
	 * The current upgrade error status
	 *
	 * @param string current upgrade error status among error possible values: UPGRADE_NO_ERROR, UPGRADE_ALLREADY_RUNNING_ERROR, NO_UPGRADE_ARCHIVE_ERROR
	 */
	public function setUpgradeError( $upgradeError )
	{
		$this->upgradeError = $upgradeError;
	}		

	/**
	 * Update the current upgrade progress log array by appending the new log to the end of the array
	 *
	 * @param string current upgrade progress log
	 */
	public function addUpgradeProgressLog( $upgradeProgressLog )
	{
		if ( $upgradeProgressLog != "" )
		{
			array_push( $this->upgradeProgressLogArray, $upgradeProgressLog );
		}
	}
	
	/**
	 * Upgrade progress log embedded in an array of string; 
	 * last operation logged is at the end
	 * 
	 * @return array of string 
	 */
	public function getUpgradeProgressLogArray()
	{
	    return $this->upgradeProgressLogArray;
	}
	
	/**
	 * Set last upgrade start date
	 * 
	 * @param string $lastUpgradeStartDate last upgrade start date
	 */
	public function setLastUpgradeStartDate( $lastUpgradeStartDate)
	{
	    $this->lastUpgradeStartDate = $lastUpgradeStartDate;
	}
	
	/**
	 * Set current OTU Rescue Version
	 *
	 * @param string $currentOtuRescueVersion current Rescue Version
	 */
	public function setCurrentOtuRescueVersion( $currentOtuRescueVersion)
	{
		$this->currentOtuRescueVersion = $currentOtuRescueVersion;
	}
	
	/**
	 * Set current OTU Version
	 *
	 * @param string $currentOtuVersion current Otu Version
	 */
	public function setCurrentOtuVersion( $currentOtuVersion)
	{
		$this->currentOtuVersion = $currentOtuVersion;
	}
	
	/**
	 * Serialize as Json data (array) the user object.<br>
	 *
	 * @return $objectProperties the object properties serialized as a Json array.
	 */
	function getJsonData()
	{
		$objectProperties = get_object_vars($this);
		$this->serializeObjectAsJsonData( $objectProperties );
		return $objectProperties;
	}

	/**
	 * Returns Dto class name.
	 *
	 * @return string the dtop class name
	 */
	function getDtoClassName()
	{
		return get_class();
	}
	
	/**
	 * Returns the dto class name.
	 *
	 * @return the dto class name.
	 */
	public static function getClass()
	{
		return __CLASS__;
	}
	
	/**
	 * Save Dto in shared memory and updates its last access timestamp
	 * WARNING: updates last access time stamp
	 *
	 * @throws SMTAPCMemoryException
	 */
	public function save()
	{
	    $this->touchAccess();
		SMTMemoryManager::saveDto( $this );
	}
	
	/**
	 * Delete Dto from shared memory
	 *
	 * @throws SMTAPCMemoryException
	 */
	public function delete()
	{
		SMTMemoryManager::deleteDto( $this );
	}
	
	/**
	 * The DTO identifier
	 */
	public function getId()
	{
		return self::UNIQUE_UPGRADE_ID;
	}
	
	/**
	 * No obsolescence
	 *
	 * @return FALSE
	 */
	public function isObsolete()
	{
	    return FALSE;
	}
	
	/**
	 * Returns the timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
	 * @see isObsolete()
	 * @return float Timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
	*/
    public function getLastAccess()
    {
    	return $this->lastAccess;
    }    
	
	/**
	 * Update Timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
	 * @see isObsolete()
	 *
	 */       
    public function touchAccess()
    {
    	$this->lastAccess = microtime(true);
    }   
}


?>