<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\alarm;

use app\serviceshelper\monitoring\SMTGpsRoute;

use app\util\SMTResultSetDto;

use app\util\SMTLogger;

use app\database\SMTDatabaseException;

use app\util\SMTInvalidValueException;

/**
 * Attenuation optical alarm event Dto
 * 
 * @author Sylvain Desplat
 * @SMTClassTableNameAnnotation(tablename='attenuation_optical_alarm_event', tablekey='id') 
 */
class SMTAttenuationOpticalAlarmEventDto extends SMTOpticalAlarmEventDto
{
    /**
     * @var float
     */
    protected $distanceM;
    
    /**
     * Level in dB
     * 
     * @var float
     */
    protected $levelDb;
    
    /**
     * Whether level attenuation is valid
     * 
     * @var integer
     */
    protected $levelValid = SMTResultSetDto::BOOLEAN_VALUE_FALSE;
    
    /**
     * Whether distance is valid
     *
     * @var integer
     */
    protected $distanceValid = SMTResultSetDto::BOOLEAN_VALUE_FALSE;
    
    /**
     * alarm longitude (decimal degree)
     *
     * @var float
     */
    protected $gpsX = 0;
    /**
     * alarm latitude (decimal degree)
     *
     * @var float
     */
    protected $gpsY = 0;
    
    /**
     * GPS google maps url
     * http://maps.google.com/maps?q=%f+%f
     * @var string
     */
    protected $gpsUrl;
    
    /**
     * Whether gps coordinate is valid
     *
     * @var integer
     */
    protected  $gpsValid = FALSE;
    

    const FETCH_CURRENT_ATTENUATION_OPTICAL_ALARM_EVENT_OF_ALARM =
    "SELECT attenuation_optical_alarm_event.*, optical_alarm_event.* FROM attenuation_optical_alarm_event, optical_alarm_event, optical_alarm WHERE optical_alarm.id=:alarmId AND optical_alarm_event.id=optical_alarm.current_alarm_event_id and attenuation_optical_alarm_event.id = optical_alarm_event.id";

    const FETCH_ATTENUATION_OPTICAL_ALARM_EVENTS_OF_ALARM =
    "SELECT * FROM attenuation_optical_alarm_event, optical_alarm_event WHERE optical_alarm_event.alarm_id=:alarmId AND attenuation_optical_alarm_event.id = optical_alarm_event.id";
    	
    /**
     * Fetch the current attenuation optical alarm event of the given alarm 
     * 
     * @param \SQLite3 $dbConnection
     * @param string $alarmId
     * 
     * @return SMTAttenuationOpticalAlarmEventDto
     */
    public static function fetch( \SQLite3 $dbConnection, $alarmId )
    {
        $query = self::FETCH_CURRENT_ATTENUATION_OPTICAL_ALARM_EVENT_OF_ALARM;
    	$statement = $dbConnection->prepare( $query );
    	
    	if ( $statement === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement failed. Query: %s.", $query), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement Successfull. Query: %s.", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	}
    	    	
    	$bindParameters = ':alarmId=>'.$alarmId;
    	$statement->bindValue(':alarmId', $alarmId, SQLITE3_TEXT );
    	$resultSet = $statement->execute();
    	
    	if ( $resultSet === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch failed. Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch successfull for Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::DEBUG);
    	}    	
    	
    	$alarmEvent= parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    	if ( $alarmEvent != NULL )
    	{
    	    $alarmEvent->buildGpsUrl();
    	}
    	
    	return $alarmEvent;
    }
    
    /**
     * Fetch the list of attenuation optical alarm events of an alarm
     *
     * @param \SQLite3 $dbConnection
     * @param string $alarmId
     *
     * @return array SMTAttenuationOpticalAlarmEventDto
     */
    public static function fetchAlarmEvents( \SQLite3 $dbConnection, $alarmId )
    {
        $query = self::FETCH_ATTENUATION_OPTICAL_ALARM_EVENTS_OF_ALARM;
    	$statement = $dbConnection->prepare( $query );
    	
    	if ( $statement === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement failed. Query: %s.", $query), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement Successfull. Query: %s.", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	}
    	
    	$bindParameters = ':alarmId=>'.$alarmId;
    	$statement->bindValue(':alarmId', $alarmId, SQLITE3_TEXT );
    	$result = $statement->execute();
    	
    	if ( $result === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch failed. Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch successfull for Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::DEBUG);
    	}    	
    	
    	$opticalAlarmEvents = parent::getListFromResultSet($resultSet, get_class() );
    	 
    	foreach ( $opticalAlarmEvents as &$alarmEvent )
    	{
    		//translates alarm specific problem
    		if ( $alarmEvent != NULL )
    		{
    			$alarmEvent->buildGpsUrl();
    		}
    	}
    	
        return $opticalAlarmEvents;
    }       
    
    /**
     * Whether level attenuation is valid
     * 
     * @var boolean
     */
    public function isLevelValid()
    {
    	return SMTResultSetDto::ToBoolean( $this->levelValid );
    }    
    
    /**
     * Set whether level attenuation is valid
     * 
     * @var boolean $valid
     */
    public function setLevelValid( $valid )
    {
    	$this->levelValid = SMTResultSetDto::FromBoolean( $valid );
    }
    
    /**
     * Whether distance for attenuation level is valid
     * 
     * @var boolean
     */
    public function isDistanceValid()
    {
    	return SMTResultSetDto::ToBoolean( $this->distanceValid );
    }
    
    /**
     * Set whether distance for level attenuation is valid
     * @var boolean $valid
     */
    public function setDistanceValid( $valid )
    {
    	$this->distanceValid = SMTResultSetDto::FromBoolean( $valid );
    }    
    
    /**
     * Whether gps coordinate is valid
     *
     * @var boolean
     */
    public function isGpsValid()
    {
    	return SMTResultSetDto::ToBoolean( $this->gpsValid );
    }
    
    /**
     * Set whether level attenuation is valid
     *
     * @var boolean $valid
     */
    public function setGpsValid( $valid )
    {
    	$this->gpsValid = SMTResultSetDto::FromBoolean( $valid );
    }
    
    /**
     *
     * @param float $level
     */
    public function setLevelDb( $levelDb )
    {
    	if ( $levelDb !== NULL && !is_numeric( $levelDb ) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "levelDb", $levelDb );
    	}
    	$this->levelDb = $levelDb;
    }
    
    /**
     *
     * @return float
     */
    public function getLevelDb()
    {
    	return $this->levelDb;
    }
    
    /**
     *
     * @param float $distance
     */
    public function setDistanceM( $distanceM)
    {
    	if ( $distanceM!== NULL && !is_numeric( $distanceM) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "distanceM", $distanceM);
    	}
    	$this->distanceM= $distanceM;
    }
    
    /**
     *
     * @return float
     */
    public function getDistanceM()
    {
    	return $this->distanceM;
    }
    
    /**
     *
     * @param float $gpsX
     */
    public function setGpsX( $gpsX )
    {
    	if ( $gpsX !== NULL && !is_numeric( $gpsX ) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "gspX", $gpsX );
    	}
    	$this->gpsX = $gpsX;
    	$this->buildGpsUrl();
    }
    /**
     *
     * @param float $gpsY
     */
    public function setGpsY( $gpsY )
    {
    	if ( $gpsY !== NULL && !is_numeric( $gpsY ) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "gpsY", $gpsY );
    	}
    	$this->gpsY = $gpsY;
    	$this->buildGpsUrl();
    }
    /**
     *
     * @return float
     */
    public function getGpsX()
    {
    	return $this->gpsX;
    }
    /**
     *
     * @return float
     */
    public function getGpsY()
    {
    	return $this->gpsY;
    }
    
    /**
     *
     * @return string
     */
    public function getGpsUrl()
    {
    	return $this->gpsUrl;
    }
    
    /**
     *
     */
    public function buildGpsUrl()
    {
        $this->gpsUrl = SMTGpsRoute::createGoogleMapsUrl($this->gpsX, $this->gpsY);
    }    
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    public function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    public function getDtoClassName()
    {
    	return get_class();
    }
}



?>