<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\parser;

use app\util\SMTUtil;

use app\util\SMTLogger;
use app\http\SMTContext;

/**
 * Wrapper for php socket:
 * - perform querries on OTU parser and process the errors. 
 * 
 * @author Sylvain Desplat
 */
class SMTOtuSocket
{    
    const RES_TAG = "<RES>";
    const RES_TAG_END = "</RES>";
    const PORT_TAG = "<PORT>";
    const PORT_TAG_END = "</PORT>";
    const QUERY_TAG = "<QUERY>";
    const QUERY_TAG_END = "</QUERY>";
    const ESR_TAG = "<ESR>";
    const ESR_TAG_END = "</ESR>";
    const COMMAND_ERROR_TAG = "<COMMAND_ERROR>";
    const COMMAND_ERROR_TAG_END = "</COMMAND_ERROR>";
    const OPEN_SOCKET_TAG = "<OPEN_SOCKET>";
    const OPEN_SOCKET_TAG_END = "</OPEN_SOCKET>";
    const CLOSE_SOCKET_TAG = "<CLOSE_SOCKET>";
    const CLOSE_SOCKET_TAG_END = "</CLOSE_SOCKET>";
    const OPEN_SOCKET_ERROR_TAG = "<OPEN_SOCKET_ERROR>";    
    const OPEN_SOCKET_ERROR_TAG_END = "</OPEN_SOCKET_ERROR>";
    const SOCKET_ID_TAG = "<SOCKET_ID>";
    const SOCKET_ID_TAG_END = "</SOCKET_ID>";    
    const MESSAGE_TAG = "<MESSAGE>";
    const MESSAGE_TAG_END = "</MESSAGE>";
    
    /**
     * OTU application default dynamic port of the internal parser
     * 
     * @var int
     */
    const OTU_DEFAULT_PORT = 8003;
    
    /**
     * OTU application port of the external parser
     *
     * @var int
     */
    const OTU_EXTERNAL_PORT = 1400;
    
    /**
     * ISU application port
     *  
     * @var int
     */
    const ISU_PORT = 8000;


    /**
     * ISU application port
     *
     * @var int
     */
    const FO_PORT = 8002;
    
    /**
     * Force remote mode on OTU: bypass authentification with automatic logon
     * @var string
     */    
    const CMD_REM = "*rem";

    /**
     * Remote authentication for OTU external parser: authentification with manual logon
     * 
     * @var string
     */    
    const CMD_MANUAL_AUTHENTIFICATION = '*PASS "RTU" "PASSWORD"';
    
    /**
     * Force local mode on OTU: for authentification
     * @var string
     */
    const CMD_LOC = "*loc";
    /**
     * registry success/failure
     * @var string
     */
    const CMD_ESR = "*esr?";
    
    /**
     * Otu parser ESR error code status
     * 
     * @var string
     */
    const ESR_0  = "0"; //NO ERROR
    const ESR_16 = "16";
    const ESR_32 = "32";
    
    /**
     * sleep 20ms between 2 readings on parser when the first read fails.
     * 
     * @var int
     */
    const SLEEP_TIME_BETWEEN_READS = 20000;
    
    /**
     * sleep 10ms before reading on parser.
     *
     * @var int
     */
    const SLEEP_TIME_BEFORE_READS = 10000;
    
    /**
     * Default sleep
     * 
     * @var integer
     */
    const DEFAULT_SLEEP = 0;
    
    /**
     * Default sleep for isu queries, wait for the query to be processed before reading. ISU can send data not in a single flush
     *
     * @var integer
     */
    const DEFAULT_SLEEP_FOR_ISU = 150000;
    
    /**
     * 5s short timeout for OTU ready command to 
     * limit the flow of new connection allowed on OTU parser.
     *
     * @var int
     */
    const TIME_OUT_READING_OTU_READY = 5;

    /**
     * 10s TIME OUT FOR READING ESR IN SOCKET : 10s
     *
     *
     * @var number
     */
    const TIME_OUT_READING_ESR = 10;
    
    /**
     * 60s TIME OUT FOR READING ESR IN SOCKET : 60s
     *  
     * must be large because some command (configuration, tests are long)
     * 
     * @var number
     */
    const LONG_TIME_OUT_READING_ESR = 60;
    /**
     * 30s TIME OUT FOR READING RESPONSE IN SOCKET : 30s; readings should be shorter than commands
     * At least 30s are required to wait for a response.
     * 
     * @var number
     */
    const TIME_OUT_READING = 30;
    
    /**
     * 300s LONG TIME OUT FOR READING RESPONSE IN SOCKET : 300s
     *
     * @var int LONG TIME OUT FOR READING RESPONSE IN SOCKET : 300s
     */
    const LONG_TIME_OUT_READING = 300;
    
    const LONGER_TIME_OUT_READING = 90;
    
    /**
     * bloc size of data read in socket stream
     * 
     * @var int bloc size of data read in socket stream 
     */
    const CHUNK_SIZE = 256;
    
    /**
     * Typically returned when trying to read on socket and data are not already available.
     */
    const SOCKET_RESOURCE_TEMPORARILY_UNAVAILABLE = 11;
    
    const SOCKET_CREATE_LOCK_FILE = "/../../tmp/socket.lock";
    
    /**
     * Max number of retry in case of connection opening failure
     * 
     * @var number
     */
    const MAX_RETRY_OPEN_CONNECTION = 3;
    
    /**
     * Ping on OTU parser timeout: 10s
     * 
     * @var integer
     */
    const PING_TIMEOUT = 10;
    
    private static $socketLockFile = NULL;
    
    /**
     * Application context
     * 
     * @var SMTContext
     */
    private $context = NULL;
    
    /**
     * The php socket
     * 
     * @var resource socket
     */
    private $socket = NULL;
    
    /**
     * The current port used by the socket
     * @var string
     */
    private $port = NULL;
    
    /**
     * Whether the socket connection is opened
     * 
     * @var boolean
     */
    private $opened = false;    
    /**
     * Identifier of the OtuSocket object used internally for logging
     * @var integer
     */
    private $id = NULL;
    
    /**
     * Current query or command ESR code value to be used for debug in TestIp function
     * @var string
     */
    private $lastEsrCode = self::ESR_0;
    
    function __construct( SMTContext $context)
    {
        $this->context = $context;
        //generate random identifier
        mt_srand();
        $this->id = mt_rand();
    }
    
    /**
     * Try to release socket when exiting
     */
    function __destruct()
    {
        $this->close();
   		$this->socket = NULL;
    }    
    
    
    /**
     * Whether the otu socket is opened. If the socket is not opened, try to open it
     * @return true if the otu soket is opened.
     * 
     * @throws SMTSocketException in case of error.
     */
    private function checkOpened()
    {
         if ( !$this->opened )
         {
             $this->open();    
         }
         return $this->opened ;
    }
    
    /**
     * Open socket (retry 3 times in case of failure by default)
     * 
     * @param $port (optional) used to force to open the socket on the given port.
     * @param $maxRetryCount Max number of retry in case of connection opening failure
     * 
     * @throws SMTSocketException in case of error.
     */
    function open( $port = NULL, $maxRetryCount = self::MAX_RETRY_OPEN_CONNECTION )
    {
        $retryCount = 0;        
        $success = FALSE;        
        $this->port = ( $port === NULL ) ?  $this->context->getOtuPort(): $port;   
        
        do
        {   
            try 
            {
                $this->performOpenSocket();
                $success = TRUE;
            } 
            catch (\Exception $e) 
            {
                $success = FALSE;                
                if ( $retryCount >= $maxRetryCount )
                {
                    $this->traceOtuQuery( self::OPEN_SOCKET_ERROR_TAG."Failed to open connection : ".$e->getMessage().self::OPEN_SOCKET_ERROR_TAG_END, SMTLogger::ERROR );
                    // Force the retrieval of OTU port when we cannot connect to OTU
                    $this->context->setOtuPort( NULL );
                    throw $e;
                }
                else
                {
                    $this->traceOtuQuery( self::OPEN_SOCKET_ERROR_TAG."Retry to open connection: ".$e->getMessage().self::OPEN_SOCKET_ERROR_TAG_END, SMTLogger::ERROR );
                }
            }    
        } while ( !$success && ( $retryCount++ < $maxRetryCount) );       
    }    
    
    /**
     * Perform socket opening
     * 
     * @throws Exception if the socket couldn't be opened successfully 
     */
    private function performOpenSocket()
    {
        //synchronize the creation and the opening of the socket on OTU
        if ( self::acquireLock() )
        {
            $this->socket = socket_create(AF_INET, SOCK_STREAM, SOL_TCP);
            
            if ( $this->socket === FALSE )
            {
                self::releaseLock();
                
                $this->traceOtuQuery( self::OPEN_SOCKET_ERROR_TAG.SMTSocketException::ERROR_CREATING.": ".$this->socket.self::OPEN_SOCKET_ERROR_TAG_END, SMTLogger::ERROR );
                
                //throws exception
                $this->processSocketError( SMTSocketException::ERROR_CREATING );
            }

            try 
            {
            	$this->traceOtuQuery(self::OPEN_SOCKET_TAG.$this->socket.self::OPEN_SOCKET_TAG_END, SMTLogger::DEBUG);
                
                //assign timeout to socket:
                // socket_set_option parameters: socket resource/ socket level / (send/receive)timeout option / {timeout in seconds,timeout in microseconds}
                @socket_set_option($this->socket, SOL_SOCKET, SO_SNDTIMEO, array("sec"=>1, "usec"=>0)) || $this->processSocketError( SMTSocketException::ERROR_SETTING_OPTIONS );
                @socket_set_option($this->socket, SOL_SOCKET, SO_RCVTIMEO, array("sec"=>1, "usec"=>0)) || $this->processSocketError( SMTSocketException::ERROR_SETTING_OPTIONS );
                
                if ( socket_connect( $this->socket,$this->context->getOtuAddress(), $this->port ) || $this->processSocketError( SMTSocketException::ERROR_OPENING ) )
                {
                	$this->opened = TRUE;
                	
                    try 
                    {
                        $this->authentify();
                    }
                    catch(\Exception $e)
                    {
                        //if authentification fails, generate a socket opening error exception to close current socket and try to open a new one
                        $this->processSocketError( SMTSocketException::ERROR_OPENING );
                    }                    
                }

                self::releaseLock();
            }
            catch( \Exception $e)
            {
                self::releaseLock();
                throw $e;
            }
        }
        else
        {
            $this->traceOtuQuery( self::OPEN_SOCKET_ERROR_TAG." Failed to acquire lock to create socket: ".self::OPEN_SOCKET_ERROR_TAG_END, SMTLogger::ERROR );
            $this->processSocketError( SMTSocketException::ERROR_CREATING );
        }
    }
    
    /**
     * Authentification on OTU socket
     * 
     * @throws Exception if authentification fails
     */
    private function authentify()
    {
        $query = self::CMD_REM;
        try
        {               
        	//try authentification on OTU
        	if ( $this->port == self::OTU_EXTERNAL_PORT )
        	{
        	    $query = self::CMD_MANUAL_AUTHENTIFICATION;
        	}
        	$this->socketWrite( $query );
            
        	//sleep to let time for authentification
        	usleep (5000);
        	
        	//hide authentification
        	$query = "*****";
        	//Don't reduce ESR default timeout for authentification with OTU
        	$esrCode = $this->getInfoEsr( $query, 1000 );
        	$this->processEsr( $query, $esrCode );
        } 
        catch (\Exception $e) 
        {           
            $socketErrorTrace = self::COMMAND_ERROR_TAG.SMTSocketException::ERROR_OTU_AUTHENTIFICATION_FAILURE.": ".$e->getMessage().self::COMMAND_ERROR_TAG_END;                    
            $socketException = new SMTSocketException( SMTSocketException::ERROR_OTU_AUTHENTIFICATION_FAILURE, $socketErrorTrace );
            $this->traceOtuQuery( $socketErrorTrace.self::MESSAGE_TAG.$socketException->getMessage().self::MESSAGE_TAG_END, SMTLogger::ERROR);
                                
            throw $socketException;
        }                 
    }
    
    /**
     * Test if OTU application is ready and throw an exception if OTU application is not available
     * 
     * 
     * @throws SMTSocketException if OTU application is not available.
     */
    function testOTUApplicationReady()
    {
        $otuReady = FALSE;
        //if we don't dialog with ISU application, check whether OTU application is available or not
        if ( ( $this->port != SMTOtuSocket::ISU_PORT ) && 
             ( $this->port != SMTOtuSocket::FO_PORT ) )
        {
            //short timeout to limit the flow of new connection allowed on OTU parser, but not too short:
            if ( $this->sendReceive(SMTOtuApi::isOTUReady(), 0, self::TIME_OUT_READING_OTU_READY ) != SMTOtuApi::RES_READY )
            {
                $this->processSocketError( SMTSocketException::ERROR_OTU_APPLICATION_STARTING );
            }
            else
            {
                $otuReady = TRUE;
            }
        }
        return $otuReady;
    }
    
    /**
     * close the socket.
     * 
     */
    function close()
    {
    	if($this->opened)
    	{
    		$this->opened = FALSE;
    		if ( $this->socket != NULL )
    		{
        		//force authentification mode on OTU when closing the socket opened with *REM 
        		@socket_write($this->socket, self::CMD_LOC."\n");
        		$this->traceOtuQuery(self::CLOSE_SOCKET_TAG.$this->socket.self::CLOSE_SOCKET_TAG_END, SMTLogger::DEBUG);
        		//stop writing and reading
        		socket_shutdown($this->socket,2);
        		socket_close($this->socket);    		
        		$this->socket = NULL;
    		}
    	}
    }    
    
    /**
     * Send a command (without response) to the OTU parser
     * 
     * @param string $query query string 
     * @param number $sleep (in microseconds) by default no sleep after having sent the command and before querying the esr status
     * @param number $readTimeOut (in s) wait until $readTimeOut for the ESR status by default 60s
     * 
     * @throws SMTSocketException in case of error.
     */
    function send( $query, $sleep = 0, $readTimeOut = self::LONG_TIME_OUT_READING_ESR  )
    {
        //check that the socket is opened and if it is not opened, try to re-open it
    	if( $this->checkOpened() )
    	{            		    	    
    	    //synchronize the creation and the opening of the socket on OTU
    	    if ( self::acquireLock() )
    	    {    	    
    	        try 
    	        {
            	    $this->socketWrite( $query);
            	    $esrCode = $this->getInfoEsr( $query, $sleep, $readTimeOut);			
            		$this->processEsr( $query, $esrCode );
            		self::releaseLock();
        		}
        		catch( \Exception $e)
        		{
        			self::releaseLock();
        			throw $e;
        		}
    		}
    		else
    		{
    			$this->traceOtuQuery( self::OPEN_SOCKET_ERROR_TAG." Failed to acquire lock for update".self::OPEN_SOCKET_ERROR_TAG_END, SMTLogger::ERROR );
    			$this->processSocketError( SMTSocketException::ERROR_CREATING, $query );
    		}        		
    	}
    }    
    
    /** 
     * Send a query to the Otu parser and wait for the result.
     * 
     * @param string $query query string
     * @param number $sleep (in microseconds) sleep before starting to retrieve the answer
     * @param number $readTimeOut (in s) wait until $readTimeOut for the ESR status
     * @param string $removeQuotes TRUE by default: remove double quotes from returned string
     *
     * @return The result of the query.
     * @throws SMTSocketException in case of error.
     */
    function sendReceive($query, $sleep = self::DEFAULT_SLEEP, $readTimeOut = self::TIME_OUT_READING, $removeQuotes = TRUE )
    {
        $data = NULL;
        //check that the socket is opened and if it is not opened, try to re-open it
    	if( $this->checkOpened() )
    	{
		    $this->socketWrite( $query );
		    $queryDate = SMTLogger::getDate();
		    //$this->traceOtuQuery( self::QUERY_TAG.trim($query).self::QUERY_TAG_END.self::PORT_TAG.$this->port.self::PORT_TAG_END, SMTLogger::DEBUG);
		    if ( $this->port == SMTOtuSocket::ISU_PORT )
		        {
		        $sleep = self::DEFAULT_SLEEP_FOR_ISU;
		        }
		    usleep ( $sleep ); //sleep 0ms before attempting to read by default
		    
			$data = $this->receive( $query, $readTimeOut );												
			
		    //need to wait for ESR of the command
		    if ( $data != NULL && $data != '' )
		    {
		    	$esrCode = $this->getInfoEsr( $query, 1000, self::TIME_OUT_READING_ESR, FALSE);
		    }
		    // need only to wait shortly for ESR ( $readTimeOut = 0.1 ): something got wrong
		    else
		    {
		    	$esrCode = $this->getInfoEsr($query, 1000, 0.1, FALSE);
		    }
		    $this->traceOtuQuery( self::QUERY_TAG.trim($query).self::QUERY_TAG_END.self::PORT_TAG.$this->port.self::PORT_TAG_END.self::ESR_TAG.$esrCode.self::ESR_TAG_END, SMTLogger::INFO, $queryDate, self::RES_TAG.( $data !== NULL? trim( $data ) : "" ).self::RES_TAG_END);
		    
			$this->processEsr( $query, $esrCode );				
    	}
    	
    	
    	return ($data !== NULL)? ( $removeQuotes? rtrim( trim( str_replace("\"","", $data ) ), "\n" ): rtrim( trim( $data ), "\n" ) ) : NULL;
    }    
    
    /**
     * Read the buffer of the socket. Must be launched after a write socket command.
     *  
     * @param string $query query string
     * @param number $readTimeOut (in s) wait until $readTimeOut for the ESR status
     * 
     * @return int The number of bytes read.
     * @throws SMTSocketException in case of error.
     */
    private function receive( $query, $readTimeOut = self::TIME_OUT_READING )
    {
    	$start_time = microtime(true);
    	$byteRead = -1;
    	$data = '';
    	$buffer = '';    	
       
    	usleep ( self::SLEEP_TIME_BEFORE_READS ); //sleep 10ms before attempting to read
    
    	//wait for data
    	//socket_recv returns FALSE if there is no data and 0 if the socket is widowed (disconnected by remote side)    	
    	while (
    	        ( ( ( $byteRead = $this->socketRead( $query, $buffer) ) < 0 ) || $byteRead === FALSE || $byteRead == '') 
                && ( (microtime(true) - $start_time) < $readTimeOut) )
    	{
//     	    $this->traceOtuQuery("sleep: ".$byteRead." ".$query);
    		usleep ( self::SLEEP_TIME_BETWEEN_READS ); //sleep 20ms before attempting to read again
    	}    	     	
    	
    	$data.= $buffer;
    	$buffer='';
    	 
    	while ($byteRead == self::CHUNK_SIZE)
    	{
    	    $byteRead = $this->socketRead( $query, $buffer);
    		$data .= $buffer;
    		$buffer='';
    	}
    	
    	//Even in case of error, the parser should return a line feed.
    	if( $byteRead <= 0 || $byteRead === FALSE )
    	{    	    
    	    //Even if the command seems to have timed-out (nothing read in socket), check the ESR status to track command errors 
    	    //(That test shouldn't be usefull if line feed was always set in case of error).
    	    //Wait shortly for ESR ( $readTimeOut = 0.1 ): the command has already been read or is in timeout.
    	    $esrCode = $this->getInfoEsr($query, 1000, 0.1);
    	    $this->processEsr( $query, $esrCode );
    	    
    	    //If ESR OK, process timeout
    		$this->processSocketError( SMTSocketException::COMMAND_TIMEOUT, $query );
    	}    	
    
    	return $data;
    }    
    
    /**
     * Returns socket ESR of the last executed query or command
     * @return string
     */
    public function getLastEsrCode()
    {
        return $this->lastEsrCode;
    }
    
    /**
     * Returns the esr code of the command.
     * Retrieves the esr code of the command and wait until $readTimeOut
     * 
     * @param string $query query string
     * @param number $sleep sleep (in microseconds) before querying the esr status
     * @param number $readTimeOut (in s) wait until $readTimeOut for the ESR status
     * @param boolean $traceESR TRUE by default to log the query with the ESR
     * 
     * @return string esr code of the query
     * @throws SMTSocketException in case of error.
     */
    private function getInfoEsr($query, $sleep = 0, $readTimeOut = self::TIME_OUT_READING_ESR, $traceESR = TRUE)
    {
    	$start_time = microtime( true );
    	$byteRead = -1;
    	$data = "";
    	$esrCode = "";
    
    	if($this->opened)
    	{
            $this->socketWrite( self::CMD_ESR  );
            
    		/* sleep before attempting to read */
    		usleep ($sleep);
    		
    		//wait for data
    		while ( ( ( ( $byteRead = @socket_recv($this->socket, $data, 8, 0) ) < 0) || $byteRead === FALSE || $byteRead =='')
    				&& ( (microtime( true ) - $start_time) < $readTimeOut) )
    		{
    		    //$this->traceOtuQuery("sleep esr: ".$byteRead." ".$query);
    			usleep ( self::SLEEP_TIME_BETWEEN_READS ); //sleep 20ms before attempting to read again
    		}
    		
    		$esrCode = trim($data);
    	}
    	
    	if ($traceESR)
    	{
    		$this->traceOtuQuery( self::QUERY_TAG.trim($query).self::QUERY_TAG_END.self::PORT_TAG.$this->port.self::PORT_TAG_END.self::ESR_TAG.$esrCode.self::ESR_TAG_END);
    	}
    	
    	return $esrCode;
    }       
    
    /**
     * Write the query on the current socket and process the error.
     * 
     * @param string $query query string
     * 
     * @throws SMTSocketException in case of error. 
     */
    private function socketWrite( $query )
    {
        if ( SMTUtil::endsWith( $query, "\n" ) === false ) 
        { 
            $query = $query."\n"; 
        }
        ( socket_write($this->socket, $query, strlen($query) ) !== false ) || $this->processSocketError( SMTSocketException::ERROR_WRITTING, $query );
    }

    /**
     * Read on the current socket and process the error.
     * 
     * @param string $query query string
     * @param string $buffer The buffer string with the the read bytes.
     * 
     * @return int the number of bytes read.
     * @throws SMTSocketException in case of error.
     */
    private function socketRead( $query, &$buffer )
    {
        if ( ( $byteRead = @socket_recv($this->socket, $buffer, self::CHUNK_SIZE, 0)) == FALSE )
        {
            if ( socket_last_error($this->socket) != self::SOCKET_RESOURCE_TEMPORARILY_UNAVAILABLE )
            {
                $this->processSocketError( SMTSocketException::ERROR_READING, $query );
            }
        } 
        return $byteRead;
    }    
    
    /**
     * Process the esr error code and call the processing of the error if needed
     *
     * @param string $query query string
     * @param string $esrCode esr query code
     * @throws SMTSocketException in case of error.
     */
    private function processEsr( $query, $esrCode )
    {
        $this->lastEsrCode = $esrCode;
    	//if the command didn't return anything and if we couldn't read the ESR, we close the socket
    	if ( $esrCode == "" )
    	{
    		$this->processSocketError( SMTSocketException::COMMAND_TIMEOUT, $query, $esrCode );
    	}
    	else if ( $esrCode == self::ESR_16 || $esrCode == self::ESR_32 || strlen( $esrCode ) > 2 ) //esrCode can include query answer if we don't wait enough time before sending the *esr command
    	{
    		$this->processSocketError( SMTSocketException::COMMAND_FAILURE, $query, $esrCode );
    	}
    	
    	//no error: $esrCode == 0
    	//set OTU application available if we don't dialog with ISU application
        if ( ( $this->port != SMTOtuSocket::ISU_PORT ) && 
             ( $this->port != SMTOtuSocket::FO_PORT ) )
    	{
    	    $this->context->setOTUAvailable();
    	}
    }
    
    /**
     * Process SMTSocketException error code: 
     * - trace errors 
     * - release the socket connection if needed.
     * - update SmartOTU status
     * 
     * Throws a SMTSocketException in any case.
     * 
     * @param string $error socket exception error code (SMTSocketException).
     * @param string $query (optional)
     * @param string $esrCode (optional)
     * 
     * @throws SMTSocketException in any case.      
     */
    private function processSocketError( $error, $query = "", $esrCode = "" )
    {                
        $socketErrorTrace = ( $query != "") ? self::COMMAND_ERROR_TAG.': '.trim($query).self::COMMAND_ERROR_TAG_END: "";
        
        switch ( $error )
        {
            //socket was created but connection openning failed
            case SMTSocketException::ERROR_SETTING_OPTIONS:
            case SMTSocketException::ERROR_OPENING:
                $socketErrorTrace = self::OPEN_SOCKET_ERROR_TAG.SMTSocketException::translateErrorLabel($error).": ".$this->context->getOtuAddress().':'.$this->port.self::OPEN_SOCKET_ERROR_TAG_END;
                $socketErrorTrace .= socket_strerror(socket_last_error());                
                //socket_create was successful but socket couldn't be opened: close it                
                if ( $this->socket !== NULL )
                {
                    socket_close($this->socket);
                }
                $this->opened = FALSE;
                $this->socket = NULL;
                break;                
            //socket wasn't created
            case SMTSocketException::ERROR_CREATING:
                $socketErrorTrace = self::OPEN_SOCKET_ERROR_TAG.SMTSocketException::translateErrorLabel($error).": ".$this->context->getOtuAddress().':'.$this->port.self::OPEN_SOCKET_ERROR_TAG_END;
                $socketErrorTrace .= socket_strerror(socket_last_error());
                $this->opened = FALSE;
                $this->socket = NULL;
                break;
            case SMTSocketException::ERROR_OTU_APPLICATION_NOT_AVAILABLE:
            case SMTSocketException::ERROR_OTU_APPLICATION_STARTING:
                //close socket to force reinitialization
                if ( $this->socket !== NULL )
                {
                    socket_close($this->socket);
                }
                $this->opened = FALSE;
                $socketErrorTrace .=self::OPEN_SOCKET_ERROR_TAG.SMTSocketException::translateErrorLabel($error).": ".self::OPEN_SOCKET_ERROR_TAG_END;
                $this->traceOtuQuery(self::CLOSE_SOCKET_TAG.$this->socket.self::CLOSE_SOCKET_TAG_END, SMTLogger::INFO);
                $this->socket = NULL;
                break;
                
        	case SMTSocketException::ERROR_WRITTING:
        	case SMTSocketException::ERROR_READING:         	    
        	    $socketErrorTrace .= socket_strerror(socket_last_error());     
        	    socket_clear_error($this->socket);
        	    socket_shutdown($this->socket,2);
        	    socket_close($this->socket);        	    
        	    $this->opened = FALSE;
        	    $socketErrorTrace .=self::COMMAND_ERROR_TAG.SMTSocketException::translateErrorLabel($error).": ".trim($query).self::COMMAND_ERROR_TAG_END;
        	    $this->traceOtuQuery(self::CLOSE_SOCKET_TAG.$this->socket.self::CLOSE_SOCKET_TAG_END, SMTLogger::INFO);
        	    $this->socket = NULL;
        	    break;        	    
        	case SMTSocketException::COMMAND_TIMEOUT:        	    
        	    if ( $this->socket !== NULL )
        	    {
            	    //stop OTU writing 
            	    socket_shutdown($this->socket,1);
            	    //stop reading 
            	    socket_shutdown($this->socket,0);
            		socket_close($this->socket);
        	    }
        		$this->opened = FALSE;
        		$socketErrorTrace .=self::COMMAND_ERROR_TAG.SMTSocketException::translateErrorLabel($error).": ".trim($query).self::COMMAND_ERROR_TAG_END;
        		$this->traceOtuQuery(self::CLOSE_SOCKET_TAG.$this->socket.self::CLOSE_SOCKET_TAG_END, SMTLogger::INFO);
        		$this->socket = NULL;
        		//socket timeout can be due to OTU application no longer on same PORT: force the retrieval of the port for next connexion:
        		$this->context->setOtuPort( NULL );        		
        		break;            	
        		
        	case SMTSocketException::COMMAND_FAILURE:
        	    $socketErrorTrace = self::COMMAND_ERROR_TAG.SMTSocketException::translateErrorLabel($error).": ".trim($query).self::COMMAND_ERROR_TAG_END.self::ESR_TAG.$esrCode.self::ESR_TAG_END;
        	    break;
        }        
        
        $socketException = new SMTSocketException( $error, $socketErrorTrace );
    	$this->traceOtuQuery( $socketErrorTrace.self::MESSAGE_TAG.$socketException->getMessage().self::MESSAGE_TAG_END, SMTLogger::ERROR);    	    	
    	
        throw $socketException;
    }    
    
    /**
     * Returns internal key composed of Otu socket instance identifier and socket object identifier
     * @return string
     */
    private function getCompositeKey()
    {
        return ($this->socket != NUll )? $this->socket."_".$this->id : "Resource id #_".$this->id;
    }
    
    /**
     * Centralized message trace method for Otu Socket to add Socket key identifier
     * 
     * @param $message 
     * @param $level (optional) default SMTLogger::INFO 
     * @param $queryDate string The date of the query execution (used for logs) or NULL        
     * @param $resultMessage string the result of the query (NULL for a command)
     */
    private function traceOtuQuery( $message, $level = SMTLogger::INFO, $queryDate= NULL, $resultMessage = NULL  )
    {
    	$this->context->traceOtuQuery( SMTLogger::isDebugLogLevel()? $message.self::SOCKET_ID_TAG.$this->getCompositeKey().self::SOCKET_ID_TAG_END : $message, $level, $queryDate, $resultMessage);
    }
     
    /**
     * All creation of sockets are synchronized
     *  
     * @return resource Lock file
     */
    private static function getSocketLockFile()
    {
    	if ( self::$socketLockFile == NULL )
    	{
    		$handle = fopen( __DIR__.self::SOCKET_CREATE_LOCK_FILE, "w+");
    		 
    		self::$socketLockFile = ( $handle != FALSE)? $handle : NULL;
    	}
    	return self::$socketLockFile;
    }    
    
    /**
     * Open and close a connection on OTU parser.
     * Ping timeout after 10s
     * 
     * @param $context SMTContext
     * 
     * @throws SMTSocketException if connection cannot be performed after 3 trials
     */
    public static function ping( SMTContext $context )
    {
    	$success = FALSE;
    	$retryCount = 0;
    	$maxRetryCount = 3;
    	$port = $context->getOtuPort();
    	$otuIpAddress = $context->getOtuAddress();    	
    	
    	//retry 3 times the connection to OTU parser
    	while ( !$success && $retryCount++ < $maxRetryCount)
    	{    	    
    		try
    		{
    			//synchronize the creation and the opening of the socket on OTU
    			if ( self::acquireLock() )
    			{
    			    // AF_INET = access to OTU application on local loop: use only IPv4 Internet based protocols
    				$socket = socket_create(AF_INET, SOCK_STREAM, SOL_TCP);
    	
    				if ( $socket === FALSE )
    				{
   				    	self::releaseLock();
    					$socketErrorTrace = socket_strerror(socket_last_error());
    					throw new SMTSocketException( SMTSocketException::ERROR_CREATING, $socketErrorTrace );
    				}
    				else
    				{
    				    //timeout 1s for reading/writing
    					@socket_set_option($socket, SOL_SOCKET, SO_SNDTIMEO, array('sec'=>1, 'usec'=>0));
    					@socket_set_option($socket, SOL_SOCKET, SO_RCVTIMEO, array('sec'=>1, 'usec'=>0));
    	
    					if ( socket_connect( $socket, $otuIpAddress, $port ) )
    					{
    						$success = TRUE;
    						//stop writing and reading 
    						socket_shutdown($socket,2);
    						socket_close($socket);
    					}
    					else
    					{
    						$socketErrorTrace = socket_strerror(socket_last_error());
    						//socket_creation was successful but socket couldn't be opened: close it
    						socket_close($socket);
    						throw new SMTSocketException( SMTSocketException::ERROR_OPENING, $socketErrorTrace );
    					}
    				}
    				self::releaseLock();
    			}
    		}
    		catch (\Exception $e)
    		{
    			$success = FALSE;
    			self::releaseLock();
    			if ( $retryCount >= $maxRetryCount )
    			{    			    
    				throw $e;
    			}
    			else
    			{
    			    SMTLogger::getInstance()->trace( "Ping failed, retry: ".$e->getMessage() );
    			    //retry after sleeping 2s
    			    usleep(2000000); // ( floor( self::PING_TIMEOUT / $maxRetryCount ) -1 ) * 1000000
    			}
    		}
    	}
    }    
    
    /**
     * Creation of sockets must be synchronized 
     * @return TRUE if lock could be acquired
     */
    static function acquireLock()
    {
        $retries = 0;
        $max_retries = 60;
        
        // keep trying to get a lock as long as possible (max 37s)
        do 
        {
        	if ($retries > 0)
        	{
        		usleep( 500 * ( $retries*$retries ) );
        	}
        	$retries += 1;
        } while ( (!flock(self::getSocketLockFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
        
        return ($retries < $max_retries);        
    }
    
    /**
     * Release socket creation lock file
     * @return TRUE if lock could be released
     */
    static function releaseLock()
    {
        return (isset(self::$socketLockFile) && self::$socketLockFile != NULL)? flock( self::$socketLockFile, LOCK_UN ) : TRUE;
    }
}

?>