<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\admin;

use app\http\SMTISerializable;
use app\util\SMTDto;

/**
 * User roles
 * 
 * @author sdesplat
 */
class SMTRoleDto extends SMTDto implements SMTISerializable
{
    const USER = 'user';
    const INSTALL = 'install';
    const INSTALL_IP = 'install_ip';
    const DEBUG = 'debug';
    const SUPER_USER = 'super_user';
    
    /**
     * The role name 
     * 
     * @var string
     */
    protected $roleName = self::USER;
    
    /**
     * Instanciate a role; construct a user role if the role name is valid.
     * 
     * @param string $roleName
     */
    public function __construct( $roleName )
    {
        if ( self::isValid($roleName))
        {
    	    $this->roleName = $roleName;
        }
    }
    
    /**
     * Whether the string value is an existing role
     * 
     * @param string $roleName
     * @return boolean
     */
    public static function isValid( $roleName )
    {
        if ( strcasecmp ($roleName, self::USER ) == 0  || strcasecmp( $roleName, self::INSTALL ) == 0  || strcasecmp( $roleName, self::INSTALL_IP) == 0  || strcasecmp( $roleName, self::DEBUG) == 0 || strcasecmp( $roleName, self::SUPER_USER) == 0)
        {
            return TRUE;
        }
        return FALSE;
    }
    
    /**
     * Test whether the rolename is equivalent to the current role.
     * 
     * @param string $roleName
     * 
     * @return whether the rolename is equivalent to the given one;
     */
    public function isEqual( $roleName )
    {
        return strcasecmp ( $this->roleName, $roleName ) == 0;
    }
    
    /**
     * Test whether the role is read-only
     *
     *
     * @return whether the role is read-only
     */
    public function isReadOnly()
    {
    	return strcasecmp ($roleName, self::USER ) == 0 ;
    }
    
    /**
     * Whether the given role name is included in the current role
     * 
     * @param string $roleName
     * 
     * @return boolean
     */
    public function isIncludingRole( $roleName )
    {
        $includeRole = false;
        
        switch ( $roleName )
        {
            case self::USER:
                $includeRole = true;
                break;
            case self::INSTALL:
                if ( $this->roleName != self::USER)
                {
            	    $includeRole = true;
                }
            	break;
        	case self::INSTALL_IP:
        		if ( $this->roleName == self::INSTALL_IP || $this->roleName == self::DEBUG || $this->roleName == self::SUPER_USER)
        		{
        			$includeRole = true;
        		}
    		case self::DEBUG:
    			if ( $this->roleName == self::DEBUG || $this->roleName == self::SUPER_USER )
    			{
    				$includeRole = true;
    			}        		
        		break;            
    		case self::SUPER_USER:
    			if ( $this->roleName == self::SUPER_USER )
    			{
    				$includeRole = true;
    			}
                            
        }
        return $includeRole;        
    }
    
    /**
     * Serialize as Json data the role object.<br> 
     * 
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars( $this );
        $this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }    
    
    /**
     * Serialize object
     *
     * @return multitype:string
     */
    public function __sleep()
    {
    	return $this->getSerializableProperties();
    }
    
    /**
     * Returns the serializable properties of the object.
     *
     * @see \app\http\SMTISerializable::getSerializableProperties()
     * 
     * @return array with the object properties to serialize.
     */
    public function getSerializableProperties()
    {
    	$objectProperties = get_object_vars($this);
    	return array_keys( $objectProperties );
    }    
    
    /**
     * Deserialize object.
     */
    public function __wakeup()
    {
    }
}

?>