# -*- coding: utf-8 -*-


import sys
from os import getcwd, walk, path
from importlib.util import spec_from_file_location, module_from_spec
from typing import List, Dict, Any
from types import ModuleType

from .json_proxy import JsonProxy

import syslog
from syslog import syslog as log


def dynamic_import(path_py: str,
                   module_name: str) -> ModuleType:
    """TODO

    :param path_py:
    :param module_name:
    :return    
    """
    module_spec = spec_from_file_location(module_name, path_py)
    module = module_from_spec(module_spec)
    module_spec.loader.exec_module(module)
    
    return module


def get_modules(src: str) -> List[ModuleType]:
    """TODO

    :param src:
    :return
    """
    cwd = getcwd()
    modules = [] 
    for root, _, files in walk(src):
        for file in files:
            if file.endswith(".py"):
                try:
                    module = dynamic_import(path.join(cwd, root, file), file[:-3])
                    modules.append(module)
                except Exception as err:
                    log(syslog.LOG_ERR, "%s" % str(err))

    return modules


RulesType = List[Dict[str, Any]]


def is_valide_rules(rules: RulesType) -> bool:
    """TODO

    :param rules:
    :return
    """
    if not isinstance(rules, list):
        return False

    for rule in rules:
        if not isinstance(rule, dict):
            return False

        cdm_path = rule.get("cdmPath", False)

        if not isinstance(cdm_path, str):
            return False

        meta_key = rule.get("metadataKey", False)

        if not isinstance(meta_key, str):
            return False

        fmt = rule.get("formatter", lambda entries: entries[0])

        if not callable(fmt):
            return False
        
    return True


def process(cdm: JsonProxy,
            rules: RulesType) -> str:
    """TODO

    :param cdm:
    :param rules:
    :return
    """
    meta_data = list()

    for rule in rules:
        cdm_path = rule.get("cdmPath", False)
        meta_key = rule.get("metadataKey", False)
        fmt = rule.get("formatter", lambda entries: entries[0])
        
        if all((cdm_path, meta_key)):
            value = cdm.get(cdm_path, "")

            value = fmt(value)

            if value:
                meta_data.append("{}={}".format(meta_key, value))
            
    return ",".join(meta_data)


class CdmMetadataConverter:
    def __init__(self, rules_dir: str):
        """Default constructor
        """
        sys.path.append(rules_dir)

        self._rules: RulesType = dict()

        modules = get_modules(rules_dir)
        
        for module in modules:
            ttype = getattr(module, "__test_type__", False)      
            cdm_ver = getattr(module, "__cdm_version__", False)
            schema_ver = getattr(module, "__schema_version__", False)
            rules = getattr(module, "__rules__", False)   

            if all((ttype, cdm_ver, schema_ver, rules)):
                if is_valide_rules(rules):
                    if ttype not in self._rules:
                        self._rules[ttype] = {}

                    if cdm_ver not in self._rules[ttype]:
                        self._rules[ttype][cdm_ver] = {}
                        
                    self._rules[ttype][cdm_ver][schema_ver] = rules
                else:
                    log(syslog.LOG_ERR, "Invalide rules definition: %s" % module.__name__)

        log(syslog.LOG_INFO, "CdmMetadataConverter is ready!")

    def __call__(self, cdm_str: str) -> str:
        """TODO
        
        :param cdm_str: raw cdm data
        :return: stratasync meta data
        """
        # Deserialize cdm
        cdm = JsonProxy.from_str(cdm_str)
        
        # Search for specific rules
        rules = self.search_rules(cdm)
        
        # Making meta-data
        meta_data = process(cdm, rules)

        return meta_data

    def search_rules(self, cdm: JsonProxy) -> RulesType:
        """TODO

        :param cdm:
        :return
        """
        ttype = cdm.get("$.tests[0].type", "")[0]
        cdm_ver = cdm.get("$.cdmVersion", "")[0]
        schema_ver = cdm.get("$.tests[0].schemaVersion", "")[0]

        return self._rules.get(ttype, {}).get(cdm_ver, {}).get(schema_ver, [])

