#!/usr/bin/env bash
# OSB is a 2k/4k module with PCIe feature on SLOTA/PWRS
# It is seen as module 4 because of hardware configuration

function init_osb_init() {
	echo -n "OSB #4"

	# Check if the OSB is detected - If not, do nothing
	# There is two versions of the OSB : 703100B01 (legacy one) and 70310EB01 (newer one, compatible with the Cobra/DTSS module)
	qdbus --literal --system com.viavisolutions.platform.SystemInventory /com/viavisolutions/platform/SystemInventory/4 com.viavisolutions.platform.SystemInventory.ModuleResources.SystemRecords | grep -q "703100B01-\|70310EB01-"
	[ $? -ne 0 ] && echo " - Not found, continuing" && return 1 || echo

	# Ensure it is powered on
	dbus-send --system --type=method_call --dest=com.viavisolutions.platform.PowerBudget /com/viavisolutions/platform/PowerBudget com.viavisolutions.platform.PowerBudget.RequestModulePowerOn uint32:4
	sleep 0.1

	# Do not programm in pmulti
	if grep -q /dev/mmcblk0 /proc/cmdline
	then
		# Check the FPGA bitstream and upgrade it if needed
		/usr/bin/advisordff ste1 -sf 4 /usr/share/firmware/osb/osb.bin
		case $? in
			0) # 0  = Flash success - We need to reboot
				echo "FPGA bitstream updated"
				isOSBUpdated=1
				;;

			11) # 11 = Already up to date - There is nothing to be done
				echo "FPGA bitstream already up to date."
				isOSBUpdated=0
				;;

			*)
				echo "ERROR: FPGA flash failed, continuing anyway"
				isOSBUpdated=0
				;;
		esac

		# If the OSB has been updated, we can't handle the module under it - just reboot
		if [ ${isOSBUpdated} -eq 1 ]
		then
			echo "The OSB has been updated - Reboot in 5 seconds"

			# Simulate Otu launch to avoid rebooting in rescue mode - if script exists
			[ -f /sbin/otu_confirm_start.sh ] && /sbin/otu_confirm_start.sh

			sleep 5

			/sbin/reboot
			exit -1
		else
			# Set the 2K/4K Module in OTDR mode (required for interrupt)
			/usr/bin/hamma action.write port.moduleSelector.MODULE_SEL value.0x02

			# Flash a "generic" fpga for the module under the OSB
			/usr/bin/load_osb_submodule_fpga.sh
			sleep 0.1

			# Select the Cobra Module
			echo "0x00000081" > /sys/platform/msel_osb/register_msel
			echo "0x000000A1" > /sys/platform/msel_osb/register_msel
			echo "0x00000000" > /sys/platform/msel_osb/register_msel
		fi
	else
		echo "Nothing to do in pmulti mode"
	fi
}

#### TODO: Add a post-init function to flash the 'specific' FPGA bitsream of the submodules ?
#			If so, we need to modify loag_osb_submodules_fpga to handle the peculiar case
#			of the Cobra module (generic == specific) - don't flash twice !

function init_osb_deinit() {
	# Check if the OSB is detected - If not, do nothing
	qdbus --literal --system com.viavisolutions.platform.SystemInventory /com/viavisolutions/platform/SystemInventory/4 com.viavisolutions.platform.SystemInventory.ModuleResources.SystemRecords | grep -q "703100B01-"
	[ $? -ne 0 ] && return 0

	echo "OSB #4"

	# De-select the Cobra Module
	echo "0x00000080" > /sys/platform/msel_osb/register_msel
	echo "0x000000A0" > /sys/platform/msel_osb/register_msel
	echo "0x00000000" > /sys/platform/msel_osb/register_msel
}
