import QtQuick 2.5
import QtQuick.Controls 2.0
import elf_plugin 1.0
import graph_plugin 1.0
import formatters_plugin 1.0

import "ViaviStyle"

FiberContainer
    {
        id: rootFiberContainer

        ////////////////////////////////////////////////////////////////////////////////
        //////////////        CONFIGS FOR THE COMPONENT                /////////////////
        ////////////////////////////////////////////////////////////////////////////////

        property string graphBackendId

        //Visible
        property ElfData isVisible: platform.getElfData(':config:Fiber_Optic:trace:isGraphVisible' + graphBackendId + ':')
        visible: isVisible.value

        //Index
        property ElfData selectedIndex: platform.getElfData(':config:Fiber_Optic:trace:selectedIndex' + graphBackendId + ':')

        //Zoom
        property ElfData zoomRectMinWidth:  platform.getElfData(':config:Fiber_Optic:trace:zoomRectMinWidth' + graphBackendId + ':')
        property ElfData zoomRectMinHeight:  platform.getElfData(':config:Fiber_Optic:trace:zoomRectMinHeight' + graphBackendId + ':')
        property ElfData zoomAreaToUI:  platform.getElfData(':config:Fiber_Optic:trace:zoomAreaToUI' + graphBackendId + ':')
        property ElfData zoomLeftPosFromUI:  platform.getElfData(':config:Fiber_Optic:trace:zoomLeftPosFromUI' + graphBackendId + ':')
        property ElfData zoomTopPosFromUI:  platform.getElfData(':config:Fiber_Optic:trace:zoomTopPosFromUI' + graphBackendId + ':')
        property ElfData zoomWidthFromUI:  platform.getElfData(':config:Fiber_Optic:trace:zoomWidthFromUI' + graphBackendId + ':')
        property ElfData zoomHeightFromUI:  platform.getElfData(':config:Fiber_Optic:trace:zoomHeightFromUI' + graphBackendId + ':')

        //Cursor position
        property ElfData cursorAPosition:  platform.getElfData(':config:Fiber_Optic:trace:cursorAPosition' + graphBackendId + ':')
        property ElfData cursorBPosition:  platform.getElfData(':config:Fiber_Optic:trace:cursorBPosition' + graphBackendId + ':')
        property ElfData cursorAyPosition:  platform.getElfData(':config:Fiber_Optic:trace:cursorAyPosition' + graphBackendId + ':')
        property ElfData cursorByPosition:  platform.getElfData(':config:Fiber_Optic:trace:cursorByPosition' + graphBackendId + ':')
        property ElfData cursorAdditionalAPosition:  platform.getElfData(':config:Fiber_Optic:trace:cursorAdditionalAPosition' + graphBackendId + ':')
        property ElfData cursorAdditionalBPosition:  platform.getElfData(':config:Fiber_Optic:trace:cursorAdditionalBPosition' + graphBackendId + ':')
        property ElfData cursorAdditionalCPosition:  platform.getElfData(':config:Fiber_Optic:trace:cursorAdditionalCPosition' + graphBackendId + ':')

        //Cursor function
        property ElfData elfUpdateZoomFromUI:  platform.getElfData(':action:Fiber_Optic:trace:updateZoomFromUI' + graphBackendId + ':')
        property ElfData elfUpdateCursorAFromUI:  platform.getElfData(':action:Fiber_Optic:trace:updateCursorAFromUI' + graphBackendId + ':')
        property ElfData elfUpdateCursorAyFromUI:  platform.getElfData(':action:Fiber_Optic:trace:updateCursorAyFromUI' + graphBackendId + ':')
        property ElfData elfUpdateCursorBFromUI:  platform.getElfData(':action:Fiber_Optic:trace:updateCursorBFromUI' + graphBackendId + ':')
        property ElfData elfUpdateCursorByFromUI:  platform.getElfData(':action:Fiber_Optic:trace:updateCursorByFromUI' + graphBackendId + ':')
        property ElfData elfUpdateCursorAdditionalAFromUI:  platform.getElfData(':action:Fiber_Optic:trace:updateCursorAdditionalAFromUI' + graphBackendId + ':')
        property ElfData elfUpdateCursorAdditionalBFromUI:  platform.getElfData(':action:Fiber_Optic:trace:updateCursorAdditionalBFromUI' + graphBackendId + ':')
        property ElfData elfUpdateCursorAdditionalCFromUI:  platform.getElfData(':action:Fiber_Optic:trace:updateCursorAdditionalCFromUI' + graphBackendId + ':')
        property ElfData elfUpdateCursorsPosition:  platform.getElfData(':action:Fiber_Optic:trace:updateCursorsPosition' + graphBackendId + ':')

        //Axis
        property ElfData isXAxisDecreasing: platform.getElfData(':result:Fiber_Optic:trace:isXAxisDecreasing' + graphBackendId + ':')
        property ElfData pointsXRangeEnd: platform.getElfData(':result:Fiber_Optic:trace:points' + graphBackendId + '0_xRangeEnd' + ':')
        property ElfData pointsXRangeBegin: platform.getElfData(':result:Fiber_Optic:trace:points' + graphBackendId + '0_xRangeBegin' + ':')
        property ElfData pointsYRangeEnd: platform.getElfData(':result:Fiber_Optic:trace:points' + graphBackendId + '0_yRangeEnd' + ':')
        property ElfData pointsYRangeBegin: platform.getElfData(':result:Fiber_Optic:trace:points' + graphBackendId + '0_yRangeBegin' + ':')
        property ElfData unitDisplayedX: platform.getElfData(':result:Fiber_Optic:trace:unitDisplayedX' + graphBackendId + ':')
        property ElfData unitDisplayedY: platform.getElfData(':result:Fiber_Optic:trace:unitDisplayedY' + graphBackendId + ':')
        property ElfData conversionUnit: platform.getElfData(':result:Fiber_Optic:trace:conversionUnit' + graphBackendId + ':')

        property ElfData cursorABDistance: platform.getElfData(':result:Fiber_Optic:trace:cursorABDistance' + graphBackendId + ':')

        //Object name
        property string graphObjectName: "graph" + graphBackendId
        property string xAxisObjectName: "xAxis" + graphBackendId
        property string yAxisObjectName: "yAxis" + graphBackendId
        property string xAxisOsaObjectName: "xAxisOsa" + graphBackendId
        property string yAxisOsaObjectName: "yAxisOsa" + graphBackendId
        property string wrapperEventsObjectName: "wrapperEvents" + graphBackendId
        property string eventsObjectName: "events" + graphBackendId
        property string graphDebugTailingObjectName: "graphDebugTailing" + graphBackendId

        //Models
        property ElfData crossesModel: platform.getElfData(':result:Fiber_Optic:trace:crosses' + graphBackendId + ':')
        property ElfData segmentsModel: platform.getElfData(':result:Fiber_Optic:trace:segments' + graphBackendId + ':')
        property ElfData integralsModel: platform.getElfData(':result:Fiber_Optic:trace:integrals' + graphBackendId + ':')
        property ElfData textsModel: platform.getElfData(':result:Fiber_Optic:trace:texts' + graphBackendId + ':')
        property ElfData iconsModel: platform.getElfData(':result:Fiber_Optic:trace:icons' + graphBackendId + ':')
        property ElfData customPainterModel:  platform.getElfData(':result:Fiber_Optic:trace:customPainter' + graphBackendId + ':')
        property ElfData rectanglesModel:  platform.getElfData(':result:Fiber_Optic:trace:rectangles' + graphBackendId + ':')
        property ElfData activeMarkerFirstLabelsStringARRAY:  platform.getElfData(':result:Fiber_Optic:trace:activeMarkerFirstLabels' + graphBackendId + ':stringARRAY:')
        property ElfData activeMarkerSecondLabelsStringARRAY:  platform.getElfData(':result:Fiber_Optic:trace:activeMarkerSecondLabels' + graphBackendId + ':stringARRAY:')
        property ElfData curveSelector: platform.getElfData(':result:Fiber_Optic:trace:curveSelector' + graphBackendId + ':')

        //Elf
        property string elfDataPoints: ':result:Fiber_Optic:trace:points' + graphBackendId
        property string elfDataPointsAsSegments: ':result:Fiber_Optic:trace:pointsAsSegments' + graphBackendId + ':'
        property string elfDataTraceGroupIndex: ':result:Fiber_Optic:trace:traceGroupIndex' + graphBackendId + ':'
        property string elfDataEvents: ':result:Fiber_Optic:trace:events' + graphBackendId
        property string elfDataWrapperPoints: ':result:Fiber_Optic:trace:wrapperPoints' + graphBackendId


        //dbussAddress
        property string dbussAddressPoints: 'gem.fiber.Fiber_Optic,' + elfDataPoints
        property string dbussAddressEvents: 'gem.fiber.Fiber_Optic,' + elfDataEvents
        property string dbussAddressWrapperPoints: 'gem.fiber.Fiber_Optic,' + elfDataWrapperPoints


        ////////////////////////////////////////////////////////////////////////////////
        //////////////        REST OF THE VARIABLES                    /////////////////
        ////////////////////////////////////////////////////////////////////////////////


        property bool zoomUpdateInProgress: false
        property real zoomXValue:1
        property real zoomYValue:1

        readonly property int axisLabelsZ: 150
        readonly property int axisLinesZ: 50
        readonly property int graphZ: axisLabelsZ - 10
        readonly property int integralZ: 140
        readonly property int segmentZ: 160


        readonly property int nrOfVerticalTicks : 10
        readonly property real tickLabelsToSpaceRatio: 0.33

        readonly property var verticalAxisDashPattern : [3.0, 3 * ViaviStyle.layouts.grapLineWidth]
        readonly property var horizontalAxisDashPattern: [5.0, 2 * ViaviStyle.layouts.grapLineWidth]

        readonly property string cursorALabel: "A"
        readonly property string cursorAdditionalALabel: "a"
        readonly property string cursorBLabel: "B"
        readonly property string cursorAdditionalBLabel: "b"
        readonly property string cursorAdditionalCLabel: "C"

        property bool backgroundVisible : false

        property bool osaAxisVisible: status_res_osaGridLinesEnabled.value && osaGridLines_res_specific.value
        property bool xTicksVisible: !status_res_osaGridLinesEnabled.value
                                     || ( status_res_osaGridLinesEnabled.value && osaGridLines_res_visible.value
                                         && !osaGridLines_res_specific.value )
        property bool yTicksVisible: !status_res_osaGridLinesEnabled.value
                                     || ( status_res_osaGridLinesEnabled.value && osaGridLines_res_visible.value )

        property bool additionalCursors: status_res_analysisEnabled.value
                                         && analysis_cfg_measType.enums.emtLoss === analysis_cfg_measType.value
                                         && analysis_cfg_selectedMethod.enums.esm5PtLoss === analysis_cfg_selectedMethod.value

        readonly property int additionalCursorsInitialOffset: ViaviStyle.layouts.buttonHeight

        property bool additionalCursorALock: false
        property bool additionalCursorBLock: false

        property alias cursorDragA: cursorDragA
        property alias cursorDragB: cursorDragB
        property alias cursorDragAy: cursorDragAy
        property alias cursorDragBy: cursorDragBy
        property alias cursorDragAdditionalA: cursorDragAdditionalA
        property alias cursorDragAdditionalB: cursorDragAdditionalB
        property alias cursorDragAdditionalC: cursorDragAdditionalC

        property alias oXAxisVisibleLeft: xAxis.minVisible
        property alias oXAxisVisibleRight: xAxis.maxVisible

        property alias graphCount: graphRepeaterId.model
        property alias tailingGraphCount: graphTailingRepeaterId.model

        property alias zoomButtonsLayout: zoomButtonsLayout

        property CursorDragFiber lastUsedCursor

        property Item traceComponent


        ////////////////////////////////////////////////////////////////////////////////
        //////////////        COMPONENT METHODS                        /////////////////
        ////////////////////////////////////////////////////////////////////////////////

       function transformX(value) {
            var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
            return (value - graphItem.width * ( graphItem.axisX.minVisible - graphItem.axisX.minValue ) / graphItem.axisX.visibleRange )
        }

       function transformY(value) {
           var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
           return value - (graphItem.axisY.maxValue - graphItem.axisY.maxVisible) * graphItem.height / graphItem.axisY.visibleRange 
        }

       function workingValueToScreenValue(value) {
            var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
            if(graphItem){
                if( isXAxisDecreasing.value )
                {
                    value = graphItem.axisX.minValue + ( graphItem.axisX.maxValue - value )
                }
                return (( value - graphItem.axisX.minVisible ) * graphItem.width / graphItem.axisX.visibleRange )
            }

            return value
        }

       function workingValueToScreenValueOY(value) {
               var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
               if(graphItem){
                   return (graphItem.axisY.maxVisible - value) / Math.abs(graphItem.axisY.minVisible - graphItem.axisY.maxVisible) * graphItem.height;
               }

               return value
           }

		function screenDifToWorkingDif(value){
			var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
            if(graphItem){
                return value * graphItem.axisX.visibleRange / graphItem.width
            }
            return 0			
		}

        function screenValueToWorkingValue(value) {
            var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
            if(graphItem){
                var result = (value * graphItem.axisX.visibleRange / graphItem.width + graphItem.axisX.minVisible)
                if( isXAxisDecreasing.value )
                {
                    result = graphItem.axisX.minValue + ( graphItem.axisX.maxValue - result )
                }
                return result
            }
            return 0
        }

        function workingPointToZoomRect(point) {
            var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
            var result = Qt.point(
                        (point.x - graphItem.axisX.minValue) / (graphItem.axisX.maxValue - graphItem.axisX.minValue) ,
                        (point.y - graphItem.axisY.minValue) / (graphItem.axisY.maxValue - graphItem.axisY.minValue) )
            return result
        }

        function zoomRectPointToWorking(point) {
            var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
            var x = point.x * (graphItem.axisX.maxValue - graphItem.axisX.minValue) + graphItem.axisX.minValue
            var y = point.y * (graphItem.axisY.maxValue - graphItem.axisY.minValue) + graphItem.axisY.minValue
            var result = Qt.point(x, y)
            return result
        }

        function updateZoomToUI(){
            if(!zoomUpdateInProgress && !flick.moving  && !flick.pinching){
                var xCenter = (zoomAreaToUI.value.left + zoomAreaToUI.value.width / 2 - xAxis.minValue) / xAxis.valueRange
                var yCenter = (zoomAreaToUI.value.top - zoomAreaToUI.value.height / 2 - yAxis.minValue) / yAxis.valueRange
                var xScale = zoomAreaToUI.value.width / xAxis.valueRange
                var yScale = zoomAreaToUI.value.height / yAxis.valueRange

                graphId.zoomRect.setCenterAndZoom(Qt.vector2d(xScale, yScale)
                                                  , Qt.point(xCenter, yCenter))

                zoomXValue = graphId.zoomRect.width
                zoomYValue = graphId.zoomRect.height
            }
        }

        // @param {double} extendRatio - extend the range within which x is considered to be contained. Value is a proportion of the are width.
        function workingValueInVisibleRangeX(x, extendRatio) {
            var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
            var extendValue = 0
            if (extendRatio != 0) {
                extendValue = Math.abs(graphItem.axisX.maxVisible - graphItem.axisX.minVisible) * extendRatio
            }
            var result = (x >= graphItem.axisX.minVisible - extendValue) && (x <= graphItem.axisX.maxVisible + extendValue)
            return result
        }

        function workingValueInVisibleRangeY(y, extendRatio) {
            var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
            var extendValue = 0
            if (extendRatio != 0) {
                extendValue = Math.abs(graphItem.axisY.maxVisible - graphItem.axisY.minVisible) * extendRatio
            }
            var result = (y >= graphItem.axisY.minVisible - extendValue) && (y <= graphItem.axisY.maxVisible + extendValue)
            return result
        }

        // NOTE: function assumes trace samples are distributed uniformly, no sparse arrays.
        // Not currently used because of difficulties with .csor curves.
        function graphPointAtWorkingValue(value) {
            var graphItem = graphRepeaterId.itemAt(selectedIndex.value)
            if(graphItem) {
                var xAddress = elfDataPoints + selectedIndex.value +'_xData:doubleARRAY:'
                var yAddress = elfDataPoints + selectedIndex.value +'_yData:doubleARRAY:'
                var pointsX = platform.getElfData(xAddress)
                var pointsY = platform.getElfData(yAddress)

                var index = (value - graphItem.axisX.minValue) / (graphItem.axisX.maxValue  - graphItem.axisX.minValue) * pointsX.valueCount
                if (index < 0 || index > pointsX.valueCount) {
                    console.log('Warning! graphPointAtWorkingValue could not find graph point at x ' + value)
                }
                else {
                    var res = Qt.point(pointsX.at(index), pointsY.at(index))
                    return res
                }
            }
            console.log('Warning! graphPointAtWorkingValue called on invalid graph')
            return Qt.point(0, 0)
        }



        function findInvariantPoint() {
            // Target points will be considered within the visible area if they are outside with less than 1%.
            // This will prevent cases where the target point is right at the edge and will jump outside on zoom due to floating point precision.
            var extraMarginRatio = 0.01
            var x =  graphId.zoomRect.x + graphId.zoomRect.width / 2
            switch (trace_cfg_cursorSelected.value) {
            case trace_cfg_cursorSelected.enums.ecCursor_A:
                if (workingValueInVisibleRangeX(cursorA.posValue, extraMarginRatio)) {
                    x = workingPointToZoomRect(Qt.point(cursorA.posValue, 0)).x
                }
                break;
            case trace_cfg_cursorSelected.enums.ecCursor_B:
                if (workingValueInVisibleRangeX(cursorB.posValue, extraMarginRatio)) {
                    x = workingPointToZoomRect(Qt.point(cursorB.posValue, 0)).x
                }
                break;
            case trace_cfg_cursorSelected.enums.ecCursor_AB:
                if (workingValueInVisibleRangeX(cursorA.posValue, extraMarginRatio) &&
                    workingValueInVisibleRangeX(cursorB.posValue, extraMarginRatio)) {
                    x = workingPointToZoomRect(
                                Qt.point(Math.min(cursorA.posValue, cursorB.posValue) + (Math.max(cursorA.posValue, cursorB.posValue) - Math.min(cursorA.posValue, cursorB.posValue)) / 2, 0)).x
                }
                break;
            default:
                break;
            }

            var y = graphId.zoomRect.y + graphId.zoomRect.height / 2
            switch (trace_cfg_cursorSelected.value) {
            case trace_cfg_cursorSelected.enums.ecCursor_A:
                if (workingValueInVisibleRangeX(cursorA.posValue, extraMarginRatio)) {
                    var workingIntersection = cursorA.yValues[cursorA.primaryPlotIndex]
                    if (workingValueInVisibleRangeY(workingIntersection, extraMarginRatio)) {
                        y = workingPointToZoomRect(Qt.point(0, workingIntersection)).y
                    }
                }
                break;
            case trace_cfg_cursorSelected.enums.ecCursor_B:
                if (workingValueInVisibleRangeX(cursorB.posValue, extraMarginRatio)) {
                    var workingIntersection = cursorB.yValues[cursorB.primaryPlotIndex]
                    if (workingValueInVisibleRangeY(workingIntersection, extraMarginRatio)) {
                        y = workingPointToZoomRect(Qt.point(0, workingIntersection)).y
                    }
                }
                break;
            case trace_cfg_cursorSelected.enums.ecCursor_AB:
                if (workingValueInVisibleRangeX(cursorA.posValue, extraMarginRatio) &&
                    workingValueInVisibleRangeX(cursorB.posValue, extraMarginRatio)) {
                    var workingIntersectionA = cursorA.yValues[cursorA.primaryPlotIndex]
                    var workingIntersectionB = cursorB.yValues[cursorB.primaryPlotIndex]
                    y = workingPointToZoomRect(
                                Qt.point(0, Math.min(workingIntersectionA, workingIntersectionB) + (Math.max(workingIntersectionA, workingIntersectionB) - Math.min(workingIntersectionA, workingIntersectionB)) / 2)).y
                }
                break;
            default:
                break;
            }

            var result = Qt.point(x, y)
            return result
        }


        function autoZoom(){
            zoomXValue = 1.0
            zoomYValue = 1.0
            var center = Qt.point(graphId.zoomRect.x + graphId.zoomRect.width / 2,
                                  graphId.zoomRect.y + graphId.zoomRect.height /2)
            graphId.zoomRect.zoom(Qt.vector2d(zoomXValue, zoomYValue),
                                  center)
        }


        function setZoomLevelChanged() {
            traceComponent.zoomLevelChanged(graphId.zoomRect.x,
                             graphId.zoomRect.y,
                             graphId.zoomRect.width,
                             graphId.zoomRect.height)
        }

        function setZoomFromUI(){
            zoomLeftPosFromUI.value = xAxis.minVisible
            zoomTopPosFromUI.value = yAxis.maxVisible
            zoomWidthFromUI.value = xAxis.visibleRange
            zoomHeightFromUI.value = yAxis.visibleRange
        }

        function updateTraceZoomValuesFromUI() {

            if(!isVisible.value)
                return;

            zoomUpdateInProgress = true
            setZoomFromUI();
            zoomUpdateInProgress = false

            elfUpdateZoomFromUI.invoke()
        }

        function updateCursorsPosition(x) {
            var currentCursor = rootFiberContainer.lastUsedCursor
            if (currentCursor) {
                traceComponent.pressAndHoldTimer.start()
                elfUpdateCursorsPosition.invokeWithStringArg(x)
            }
        }

        function onCursorFocusChanged(cursorDrag){
            if(rootFiberContainer.lastUsedCursor !== cursorDrag){
                rootFiberContainer.lastUsedCursor = cursorDrag;
            }

            onGraphFocusChanged();
            traceComponent.onCursorFocusChanged(cursorDrag);
        }

        function onGraphFocusChanged(){
           trace_cfg_currentFocusedGraphName.value = graphBackendId;
           traceComponent.onGraphFocusedChanged(graphBackendId, rootFiberContainer);
        }

        Connections{
            target:  zoomAreaToUI
            function onValueChanged() {
                updateZoomToUI();
            }
        }

        ////////////////////////////////////////////////////////////////////////////////
        //////////////        GRAPH CONTAINER                          /////////////////
        ////////////////////////////////////////////////////////////////////////////////


        CurveCursorInfoBar {
            id: curveCursorInfoBar
            width: traceComponent.width
            trace: traceComponent
            fiberViewContainer: rootFiberContainer

            cursorDragA: cursorDragA
            cursorDragB: cursorDragB
            cursorDragAy: cursorDragAy
            cursorDragBy: cursorDragBy

            cursorDragAdditionalA: cursorDragAdditionalA
            cursorDragAdditionalB: cursorDragAdditionalB

            selectedIndex: rootFiberContainer.selectedIndex
            unitDisplayedX: rootFiberContainer.unitDisplayedX
            unitDisplayedY: rootFiberContainer.unitDisplayedY
            conversionUnit: rootFiberContainer.conversionUnit
            curveSelector: rootFiberContainer.curveSelector
            cursorABDistance: rootFiberContainer.cursorABDistance
        }


        GraphView {

            id : graphId

            anchors.top: curveCursorInfoBar.bottom
            anchors.left: parent.left
            anchors.right: parent.right
            anchors.bottom: parent.bottom

            anchors.topMargin: ViaviStyle.layouts.smallMargin
            anchors.leftMargin: ViaviStyle.layouts.smallMargin
            anchors.rightMargin: ViaviStyle.layouts.smallMargin
            anchors.bottomMargin: ViaviStyle.layouts.smallMargin

            zoomRect.zoomAxis: Qt.Horizontal | Qt.Vertical
            zoomRect.minWidth: zoomRectMinWidth.value
            zoomRect.minHeight: zoomRectMinHeight.value

            onWidthChanged: {
                cursorDragA.setPosValue(cursorAPosition.value)
                cursorDragB.setPosValue(cursorBPosition.value)
                updateZoomToUI()
            }

            onHeightChanged: {
                updateZoomToUI()
            }

            zoomRect.onRectChanged: { setZoomLevelChanged() }

            GraphGestureArea {
                id: flick
                flickableDirection: Flickable.HorizontalAndVerticalFlick
                multipleDirectionPinch: true
                zoomRect: graphId.zoomRect

                onContentWidthChanged: {
                    if(pinching){
                        updateTraceZoomValuesFromUI()
                    }
                }

                onContentHeightChanged: {
                    if(pinching){
                        updateTraceZoomValuesFromUI()
                    }
                }

                onContentXChanged: {
                    if(moving){
                        updateTraceZoomValuesFromUI()
                    }
                }

                onContentYChanged: {
                    if(moving){
                        updateTraceZoomValuesFromUI()
                    }
                }

                MouseArea {
                    anchors.fill: parent
                    hoverEnabled: false
                    preventStealing: false
                    onClicked: {

                        onGraphFocusChanged();

                        var currentCursor = rootFiberContainer.lastUsedCursor
                        var mouseTransformed = 0;

                        if(currentCursor.isVerticalCursor){
                            mouseTransformed = transformX(mouseX)
                            if(currentCursor.connectedCursor){
                                var connectedCursorX = currentCursor.connectedCursor

                                var currentDistX = Math.abs(currentCursor.x - mouseTransformed)
                                var connectedDistX = Math.abs(connectedCursorX.x - mouseTransformed)

                                if(connectedDistX < currentDistX){
                                    currentCursor = connectedCursorX
                                }
                            }
                        }
                        else{
                            mouseTransformed = transformY(mouseY)
                            if(currentCursor.connectedCursor){
                                var connectedCursorY = currentCursor.connectedCursor

                                var currentDistY = Math.abs(currentCursor.y - mouseTransformed)
                                var connectedDistY = Math.abs(connectedCursorY.y - mouseTransformed)

                                if(connectedDistY < currentDistY){
                                    currentCursor = connectedCursorY
                                }
                            }
                        }

                        currentCursor.updateConnectedCursorOffset()
                        currentCursor.updatePosition(mouseTransformed, true, true)
                        rootFiberContainer.forceActiveFocus()

                    }
                    onPressed: {
                        rootFiberContainer.forceActiveFocus()
                        onGraphFocusChanged();
                    }
                }
            }

            XYScale {
                id: xyLabels
                z: axisLabelsZ
            }
            XYScale {
                id: xyLines
                z: axisLinesZ
            }

            axes: [
                ValueAxis {
                    id: xAxisOsa
                    visible: osaAxisVisible
                    objectName: xAxisOsaObjectName
                    alignment: Qt.AlignBottom
                    linePen: ViaviStyle.colors.borderColor
                    labelPen: ViaviStyle.colors.secondaryInfoColor
                    lineJoinStyle: Qt.MiterJoin
                    lineDashPattern: horizontalAxisDashPattern
                    lineWidth: ViaviStyle.layouts.graphTickWidth
                    labelFont: ViaviStyle.layouts.traceUnitFont
                    maxValue: pointsXRangeEnd.value
                    minValue: pointsXRangeBegin.value
                    unit: unitDisplayedX.value
                    tickInterval: ValueAxis.FixedTicks
                    ticks: {
                        var ticksArray = []
                        var arrayLength = osaGridLines_res_lines_doubleARRAY.valueCount
                        for (var i = 0; i < arrayLength ; i++) {
                            ticksArray.push ( osaGridLines_res_lines_doubleARRAY.at(i) )
                        }
                        return ticksArray;
                    }
                    axisStyle: ValueAxis.TicksOnly
                    rangeOrder: isXAxisDecreasing.value ? ValueAxis.Decreasing : ValueAxis.Increasing

                    labelsPlane: xyLabels
                    linesPlane: xyLines
                },

                ValueAxis {
                    id: xAxis
                    objectName: xAxisObjectName
                    alignment: Qt.AlignBottom
                    linePen: ViaviStyle.colors.borderColor
                    labelPen: ViaviStyle.colors.secondaryInfoColor
                    lineJoinStyle: Qt.MiterJoin
                    lineDashPattern: horizontalAxisDashPattern
                    lineWidth: ViaviStyle.layouts.graphTickWidth
                    labelFont: ViaviStyle.layouts.traceUnitFont
                    maxValue: pointsXRangeEnd.value
                    minValue: pointsXRangeBegin.value
                    unit: unitDisplayedX.value
                    axisStyle: xTicksVisible ? ValueAxis.TicksAndLabels : ValueAxis.LabelsOnly
                    tickInterval: ValueAxis.DisplayDynamic
                    dynamicTicksRatio: tickLabelsToSpaceRatio
                    rangeOrder: isXAxisDecreasing.value ? ValueAxis.Decreasing : ValueAxis.Increasing

                    labelsPlane: xyLabels
                    linesPlane: xyLines

                    formatter: DecimalFormatter{
                        totalDigits: 8
                    }

                    onMaxValueChanged:{
                        cursorDragA.setPosValue(cursorAPosition.value)
                        cursorDragB.setPosValue(cursorBPosition.value)
                        updateZoomToUI()
                    }

                    onMinValueChanged:{
                        cursorDragA.setPosValue(cursorAPosition.value)
                        cursorDragB.setPosValue(cursorBPosition.value)
                        updateZoomToUI()
                    }
                },

                ValueAxis {
                    id: yAxis
                    objectName: yAxisObjectName
                    alignment: Qt.AlignLeft
                    linePen: ViaviStyle.colors.borderColor
                    labelPen: ViaviStyle.colors.secondaryInfoColor
                    lineJoinStyle: Qt.MiterJoin
                    lineDashPattern: verticalAxisDashPattern
                    lineWidth: ViaviStyle.layouts.graphTickWidth
                    labelFont: ViaviStyle.layouts.traceUnitFont
                    maxValue: pointsYRangeEnd.value
                    minValue: pointsYRangeBegin.value
                    numTicks: nrOfVerticalTicks
                    unit: unitDisplayedY.value
                    axisStyle:  yTicksVisible ? ValueAxis.TicksAndLabels : ValueAxis.LabelsOnly
                    tickInterval: ValueAxis.DisplayDynamic
                    dynamicTicksRatio: tickLabelsToSpaceRatio

                    labelsPlane: xyLabels
                    linesPlane: xyLines

                    formatter: DecimalFormatter{
                        totalDigits: 5
                    }

                    onMaxValueChanged:{
                        updateZoomToUI()
                    }
                    onMinValueChanged:{
                        updateZoomToUI()
                    }
                }
            ]

            Repeater {
                id: graphRepeaterId

                GraphPlot {
                    id: reflections
                    objectName: graphObjectName + index

                    property bool controlledVisibility: !misc_res_hideAllTraceVisible.value ? true : emphasize
                    visible: platform.getElfData(elfDataPoints + index + '_xData:doubleARRAY:').valueCount > 0
                             && controlledVisibility

                    property ElfData pointAsSegment: platform.getElfData(elfDataPointsAsSegments + index + ":")
                    property ElfData groupIndex: platform.getElfData(elfDataTraceGroupIndex + index + ":")

                    property bool isPmdScreen: status_res_osaMode.value === status_res_osaMode.enums.ecFunction_Pmd
                    property bool emphasize: (isPmdScreen && multigraphsStates_res_isPmdFftActive.value) ? true : (selectedIndex.value === colorIndex)

                    plotData: PlotPoints {
                        dbusAddress: dbussAddressPoints + index
                        axisY: yAxis
                        axisX: osaAxisVisible ? xAxisOsa: xAxis
                        usePointsAsSegments: pointAsSegment.value

                        active: reflections.colorIndex === selectedIndex.value
                    }
                    antialiasing : false

                    property int colorIndex : (groupIndex.value !== -1) ? groupIndex.value : index

                    penColor: emphasize ? ViaviStyle.colors.activeColor : ViaviStyle.colors.traceColors[colorIndex]
                    plotStyle: GraphPlot.LinePlot
                    maxLineWidth: emphasize ?
                                      ViaviStyle.layouts.isSmallScreen ? 3 * ViaviStyle.layouts.grapLineWidth : 2 * ViaviStyle.layouts.grapLineWidth
                                        : ViaviStyle.layouts.grapLineWidth
                    minLineWidth: emphasize ?
                                      ViaviStyle.layouts.isSmallScreen ? 2 * ViaviStyle.layouts.grapLineWidth : 1 * ViaviStyle.layouts.grapLineWidth
                                        : ViaviStyle.layouts.grapLineWidth
                    useIntermittentPlot: false

                    //the axises have a z of 100
                    //in order to have the trace above the dashes from the vertical axis we need to set the z to something heigher
                    z: rootFiberContainer.graphZ + (emphasize ? 1 : 0)
                }
            }


            GraphPlot {
                id: wrapperEventsId
                objectName: wrapperEventsObjectName

                plotData: PlotPoints {
                    dbusAddress: dbussAddressWrapperPoints
                    axisY: yAxis
                    axisX: xAxis
                }

                antialiasing : false

                penColor: ViaviStyle.colors.extraTraceEventColor
                plotStyle: GraphPlot.ScatterPlot

                maxLineWidth: 3 * ViaviStyle.layouts.grapLineWidth
                minLineWidth: 3 * ViaviStyle.layouts.grapLineWidth

                //the axises have a z of 100
                //in order to have the trace above the dashes from the vertical axis we need to set the z to something heigher
                z: rootFiberContainer.graphZ + 2
            }



            GraphPlot {
                id: eventsId
                objectName: eventsObjectName

                plotData: PlotPoints {
                    dbusAddress: dbussAddressEvents + selectedIndex.value
                    axisY: yAxis
                    axisX: osaAxisVisible ? xAxisOsa: xAxis
                }
                graphPoint: GraphPointActiveMarker{
                    //opacity: parent.opacity
                    firstLabels: activeMarkerFirstLabelsStringARRAY
                    secondLabels: activeMarkerSecondLabelsStringARRAY
                    pointSize: ViaviStyle.layouts.activeMarkerPointSize
                    labelColor: ViaviStyle.colors.mainInfoColor
                    labelFontSize: ViaviStyle.layouts.minuteFontSize

                    secondLabelColor: ViaviStyle.colors.activeColor
                    secondLabelFontSize: ViaviStyle.layouts.minuteFontSize
                }

                antialiasing : false

                penColor: ViaviStyle.colors.mainInfoColor
                plotStyle: GraphPlot.ScatterPlot

                maxLineWidth: ViaviStyle.layouts.traceEventLineWidth
                minLineWidth: ViaviStyle.layouts.traceEventLineWidth

                //the axises have a z of 100
                //in order to have the trace above the dashes from the vertical axis we need to set the z to something heigher
                z: rootFiberContainer.graphZ
            }
            //<Tailing>
            // this is where we plot the tailing in debug Mode
            Repeater {
                id: graphTailingRepeaterId

                GraphPlot {
                    id: tailingId
                    objectName: graphDebugTailingObjectName + index

                    plotData: PlotPoints {
                        dbusAddress: "gem.fiber.Fiber_Optic,:result:Fiber_Optic:debugInfoTailing:points"
                        axisY: yAxis
                        axisX: xAxis
                    }
                    antialiasing : false

                    penColor: ViaviStyle.colors.extraTraceEventColor
                    plotStyle: GraphPlot.LinePlot

                    maxLineWidth: 3 * ViaviStyle.layouts.grapLineWidth//ViaviStyle.layouts.traceEventLineWidth
                    minLineWidth: 3 * ViaviStyle.layouts.grapLineWidth//ViaviStyle.layouts.traceEventLineWidth

                    //the axises have a z of 100
                    //in order to have the trace above the dashes from the vertical axis we need to set the z to something heigher
                    z: rootFiberContainer.graphZ+10
                }
            }
            //</Tailing>
            ContainerPlot{
                id: crosses
                axisY: yAxis
                axisX: xAxis

                Repeater{
                    model: crossesModel.value
                    CrossPainter{
                        point1X: model.point1X
                        point1Y: model.point1Y
                        crossColor: model.crossColor

                        lineWidth: ViaviStyle.layouts.separatorSize
                        crossSize: ViaviStyle.layouts.crossEventSize
                    }
                }
            }

            ContainerPlot{
                id: segments
                axisY: yAxis
                axisX: xAxis


                Repeater{
                    model: segmentsModel.value
                    SegmentPainter {
                        plotParent: segments
                        point1: Qt.point(model.point1X, model.point1Y)
                        point2: Qt.point(model.point2X, model.point2Y)

                        lineColor: model.lineColor
                        label: model.label
                        labelColor: model.labelColor
                        labelFontSize: ViaviStyle.layouts.minuteFontSize

                        drawLimits: model.drawLimits
                        drawSlope: model.drawSlope

                        labelOffset: ViaviStyle.layouts.mediumMargin
                    }
                }

                z: rootFiberContainer.segmentZ
            }

            ContainerPlot{
                id: integrals
                axisY: yAxis
                axisX: xAxis

                Repeater{
                    model: integralsModel.value
                    IntegralPainter{
                        x1: model.x1
                        x2: model.x2

                        fillColor: model.fillColor
                        label: model.label
                        labelColor: model.labelColor
                        labelFontSize: ViaviStyle.layouts.minuteFontSize
                        labelOffset: 3 * ViaviStyle.layouts.mediumMargin

                        plotData: graphRepeaterId.count > 0 ? graphRepeaterId.itemAt(selectedIndex.value).plotData : null

                    }
                }

                z: rootFiberContainer.integralZ
            }

            ContainerPlot{
                id: texts
                axisY: yAxis
                axisX: xAxis

                Repeater{
                    model: textsModel.value
                    TextPainter {
                        plotParent: texts
                        origin: Qt.point(model.point1X, model.point1Y)

                        label: model.label
                        labelColor: model.labelColor
                        font: ViaviStyle.layouts.largeFont
                        hTextAlignment: model.hTextAlignment
                        vTextAlignment: model.vTextAlignment
                    }
                }

                z: rootFiberContainer.integralZ
            }

            Item {
                x: eventsId.x
                clip: true
                //Note: icons has an Item as parent because GraphView re-anchors with 'fill' all its direct children, thus causing icons to lose their size binding
                id: iconsHolder
                Repeater {
                    id: icons
                    model: iconsModel.value
                    FiberColorImage {
                        source: model.iconPath
                        color: ViaviStyle.colors.mainInfoColor

                        function getAllignmentOffset()
                        {
                            return (model.hIconAlignment === Text.AlignLeft) ? (- width)
                                                                             : (model.hIconAlignment === Text.AlignHCenter) ? (- width / 2) : 0
                        }

                        x: (model.point1X - xAxis.minVisible) * (eventsId.width - 1.0 ) / xAxis.visibleRange + getAllignmentOffset()
                        y: zoomButtonsLayout.height

                        fillMode: Image.PreserveAspectFit

                        width: ViaviStyle.layouts.traceIconEventSize
                        height: ViaviStyle.layouts.traceIconEventSize

//                        visible: model.point1X >= xAxis.minVisible && model.point1X <= xAxis.maxVisible - paintedWidth * xAxis.visibleRange / (eventsId.width - 1.0 )

                        z: rootFiberContainer.integralZ
                    }
                }
            }


            Item {
                id: rectanglesHolder
                clip: true
                anchors.bottomMargin: 2 * ViaviStyle.layouts.largeMargin

                Repeater{
                    id: evRecranglesRepeater
                    model: rectanglesModel.value
                    Rectangle {
                        x: workingValueToScreenValue(model.point1X)
                        y: workingValueToScreenValueOY(model.point2Y)

                        width: model.point2X * parent.width / xAxis.visibleRange
                        height: (model.point2Y - model.point1Y) * parent.height / yAxis.visibleRange
                        color: model.boolDescIsFull ?  model.fillColor : "transparent"
                        border.color: model.fillColor
                        border.width: 2

                        z: rootFiberContainer.graphZ + 1
                    }
                }
                z: rootFiberContainer.graphZ + 1
            }

            Item {
                id: numLabelRepeaterParent
                clip: true
                anchors.bottomMargin: 2 * ViaviStyle.layouts.largeMargin

                Repeater{
                    id: evNumLabelRepeater
                    model: customPainterModel.value
                    QMLPainter {
                        x: workingValueToScreenValue(model.point1X) - width / 2
                        y: parent.height - height

                        textField.text: model.label
                        textField.color: ViaviStyle.colors.mainInfoColor
                        textField.font: ViaviStyle.layouts.largeFont
                        radius: 5
                        border.color: ViaviStyle.colors.borderHighContrastColor
                        z: rootFiberContainer.graphZ + 1
                    }
                }
                z: rootFiberContainer.graphZ + 1
            }


            GraphCursor {
                id: cursorA
                visible: status_res_cursorsEnabled.value &&
                         (trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_X || trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_XY)
                enabled: status_res_cursorsEnabled.value &&
                         (trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_X || trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_XY)

                opacity: cursorA.enabled ? 1.0 : 0.0
                color: ViaviStyle.colors.redCursorColor
                staysOnScreen: false

                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                //This CursorDrag is getting in the way of pinch-to-zoom!
                CursorXDragFiber {
                    id: cursorDragA
                    connectedCursor: additionalCursors
                                     ? (additionalCursorALock ? cursorDragAdditionalA : null)
                                     : trace_res_selectedCursors.value === "Cursors_AB"
                                       ? cursorDragB : null

                    parentGraph: rootFiberContainer
                    parentTrace: traceComponent
                    updateElf: elfUpdateCursorAFromUI
                    backendItem: cursorAPosition
                    xAxis: xAxis
                    backendId: trace_cfg_cursorSelected.enums.ecCursor_A
                    
                    refrenceValue: cursorAPosition.value

                    text: cursorALabel
                    color: ViaviStyle.colors.redCursorColor
                    
                }
            }

            GraphCursorY {
                id: cursorAy
                visible: status_res_cursorsEnabled.value &&
                         (trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_Y || trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_XY)
                enabled: status_res_cursorsEnabled.value &&
                         (trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_Y || trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_XY)

                opacity: enabled ? 1.0 : 0.0
                color: ViaviStyle.colors.redCursorColor
                staysOnScreen: false

                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                //This CursorDrag is getting in the way of pinch-to-zoom!
                CursorYDragFiber {
                    id: cursorDragAy
                    connectedCursor: additionalCursors
                                     ? (additionalCursorALock ? cursorDragAdditionalA : null)
                                     : trace_res_selectedCursors.value === "Cursors_AB"
                                       ? cursorDragBy : null

                    parentGraph: rootFiberContainer
                    parentTrace: traceComponent
                    updateElf: elfUpdateCursorAyFromUI
                    backendItem: cursorAyPosition
                    xAxis: yAxis
                    backendId: trace_cfg_cursorSelected.enums.ecCursor_A
                    refrenceValue: cursorAyPosition.value

                    text: cursorALabel + "y"
                    color: ViaviStyle.colors.redCursorColor
                }
            }

            GraphCursor {
                id: cursorAdditionalA
                visible: status_res_cursorsEnabled.value && additionalCursors
                enabled: status_res_cursorsEnabled.value && additionalCursors

                //must use opacity instead of visible because GraphCursor redefines a final property. but for backwords compatability it must kept as it is
                opacity: enabled ? 1.0 : 0.0
                staysOnScreen: false
                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                CursorXDragFiber {
                    id: cursorDragAdditionalA
                    connectedCursor: (additionalCursors && additionalCursorALock)
                                     ? cursorDragA : null
                    parentGraph: rootFiberContainer
                    parentTrace: traceComponent
                    updateElf: elfUpdateCursorAdditionalAFromUI
                    backendItem: cursorAdditionalAPosition
                    xAxis: xAxis

                    additionalCursor: true                    
                    refrenceValue: cursorDragA.posValue + screenDifToWorkingDif(additionalCursorsInitialOffset)

                    text: cursorAdditionalALabel
                    color: ViaviStyle.colors.redCursorColor
                    
                }
            }

            GraphCursor {
                id: cursorB
                visible: status_res_cursorsEnabled.value &&
                         (trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_X || trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_XY)
                enabled: status_res_cursorsEnabled.value &&
                         (trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_X || trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_XY)

                opacity: enabled ? 1.0 : 0.0
                color: ViaviStyle.colors.blueCursorColor
                staysOnScreen: false

                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                CursorXDragFiber {
                    id: cursorDragB
                    connectedCursor: additionalCursors
                                     ? (additionalCursorBLock ? cursorDragAdditionalB : null)
                                     : trace_res_selectedCursors.value === "Cursors_AB"
                                       ? cursorDragA : null
                    parentGraph: rootFiberContainer
                    parentTrace: traceComponent
                    updateElf: elfUpdateCursorBFromUI
                    backendItem: cursorBPosition
                    
                    xAxis: xAxis
                    backendId: trace_cfg_cursorSelected.enums.ecCursor_B
                    refrenceValue: cursorBPosition.value

                    text: cursorBLabel
                    color: ViaviStyle.colors.blueCursorColor
                    
                }
            }

            GraphCursorY {
                id: cursorBy
                visible: status_res_cursorsEnabled.value &&
                         (trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_Y || trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_XY)

                enabled: status_res_cursorsEnabled.value &&
                         (trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_Y || trace_cfg_cursorDraw.value === trace_cfg_cursorDraw.enums.ecDraw_Cursor_XY)

                opacity: enabled ? 1.0 : 0.0
                color: ViaviStyle.colors.blueCursorColor
                staysOnScreen: false

                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                //This CursorDrag is getting in the way of pinch-to-zoom!
                CursorYDragFiber {
                    id: cursorDragBy
                    connectedCursor: additionalCursors
                                     ? (additionalCursorBLock ? cursorDragAdditionalB : null)
                                     : trace_res_selectedCursors.value === "Cursors_AB"
                                       ? cursorDragAy : null

                    parentGraph: rootFiberContainer
                    parentTrace: traceComponent
                    updateElf: elfUpdateCursorByFromUI
                    backendItem: cursorByPosition
                    xAxis: yAxis
                    backendId: trace_cfg_cursorSelected.enums.ecCursor_B
                    refrenceValue: cursorByPosition.value

                    text: cursorBLabel + "y"
                    color: ViaviStyle.colors.blueCursorColor
                }
            }

            GraphCursor {
                id: cursorAdditionalB
                visible: status_res_cursorsEnabled.value && additionalCursors
                enabled: status_res_cursorsEnabled.value && additionalCursors

                //must use opacity instead of visible because GraphCursor redefines a final property. but for backwords compatability it must kept as it is
                opacity: enabled ? 1.0 : 0.0

                staysOnScreen: false

                color: ViaviStyle.colors.blueCursorColor

                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                CursorXDragFiber {
                    id: cursorDragAdditionalB
                    connectedCursor: (additionalCursors && additionalCursorBLock)
                                     ? cursorDragB : null
                    parentGraph: rootFiberContainer
                    parentTrace: traceComponent
                    updateElf: elfUpdateCursorAdditionalBFromUI
                    backendItem: cursorAdditionalBPosition
                    xAxis: xAxis
                    
                    additionalCursor: true                    
                    refrenceValue: cursorDragB.posValue + screenDifToWorkingDif(additionalCursorsInitialOffset)

                    text: cursorAdditionalBLabel
                    color: ViaviStyle.colors.blueCursorColor
                    
                }
            }

            GraphCursor {
                id: cursorAdditionalC
                visible: status_res_cursorsEnabled.value && additionalCursors
                enabled: status_res_cursorsEnabled.value && additionalCursors

                //must use opacity instead of visible because GraphCursor redefines a final property. but for backwords compatability it must kept as it is
                opacity: enabled ? 1.0 : 0.0

                staysOnScreen: false

                color: ViaviStyle.colors.secondaryInfoColor

                autoPrimaryPlot: true

                CursorXDragFiber {
                    id: cursorDragAdditionalC
                    parentGraph: rootFiberContainer
                    parentTrace: traceComponent
                    updateElf: elfUpdateCursorAdditionalCFromUI
                    backendItem: cursorAdditionalCPosition
                    xAxis: xAxis
                    
                    additionalCursor: true                    
                    refrenceValue: (cursorDragA.posValue + cursorDragB.posValue) / 2

                    text: cursorAdditionalCLabel
                    color: ViaviStyle.colors.secondaryInfoColor
                    
                }
            }
        }



        Row {
            id: zoomButtonsLayout
            visible: trace_res_hasZoomButtons.value ? true : false

            anchors.top: curveCursorInfoBar.bottom
            anchors.right: parent.right

            clip: true

            property real zoomInFactor: 2/3
            function zoomBy(zoomFactorX, zoomFactorY) {
                var zoomXValue = Math.min(1.0, rootFiberContainer.zoomXValue * zoomFactorX)
                var zoomYValue = Math.min(1.0, rootFiberContainer.zoomYValue * zoomFactorY)
                var invariantPoint = findInvariantPoint()
                var center = Qt.point(graphId.zoomRect.x + graphId.zoomRect.width / 2,
                                      graphId.zoomRect.y + graphId.zoomRect.height / 2)

                center.x += (invariantPoint.x - center.x) * (1 - zoomFactorX)
                center.y += (invariantPoint.y - center.y) * (1 - zoomFactorY)

                graphId.zoomRect.setCenterAndZoom(Qt.vector2d(zoomXValue, zoomYValue), center)
                updateTraceZoomValuesFromUI()
                zoomXValue = graphId.zoomRect.width
                zoomYValue = graphId.zoomRect.height
            }

            property int buttonApparentSize: ViaviStyle.layouts.imageButtonHeight
            property int buttonTopPadding: ViaviStyle.layouts.mediumMargin
            property int buttonHPadding: ViaviStyle.layouts.largeMargin

            FiberImageButton  {
                id: zoomInBtn
                smooth: true
                source: ViaviStyle.images.zoomInBtnImg
                color: 'transparent'
                width: parent.buttonApparentSize + leftPadding + rightPadding
                height: parent.buttonApparentSize + topPadding
                leftPadding: parent.buttonHPadding
                rightPadding: parent.buttonHPadding
                topPadding: parent.buttonTopPadding
                onClicked: {
                    zoomButtonsLayout.zoomBy(zoomButtonsLayout.zoomInFactor, zoomButtonsLayout.zoomInFactor)
                }
            }

            FiberImageButton  {
                id: zoomOutBtn
                smooth: true
                source: ViaviStyle.images.zoomOutBtnImg
                color: 'transparent'
                width: parent.buttonApparentSize + leftPadding + rightPadding
                height: parent.buttonApparentSize + topPadding
                leftPadding: parent.buttonHPadding
                rightPadding: parent.buttonHPadding
                topPadding: parent.buttonTopPadding
                onClicked: {
                    zoomButtonsLayout.zoomBy(1/zoomButtonsLayout.zoomInFactor, 1/zoomButtonsLayout.zoomInFactor)
                }
            }

            FiberImageButton  {
                id: zoomAutoBtn
                smooth: true
                source:ViaviStyle.images.zoomAutoBtnImg
                color: 'transparent'
                width: parent.buttonApparentSize + leftPadding + rightPadding
                height: parent.buttonApparentSize + topPadding
                leftPadding: parent.buttonHPadding
                rightPadding: parent.buttonHPadding
                topPadding: parent.buttonTopPadding
                onClicked: {
                    trace_act_doAutoZoom.invoke()
                }
            }
        }
    }

