<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\settings;

use app\admin\SMTOtuType;

use app\parser\SMTOtuApi;

use app\http\SMTContext;

use app\services\monitoring\SMTOtuMonitoringTestThresholdDto;

use app\sharedmemory\SMTMemoryManager;
use app\util\SMTLogger;
use app\admin\SMTOtuMode;
use app\serviceshelper\maintenance\SMTLicence;

/**
 * SmartOTU settings manager:
 * - retrieve and edit application settings from .ini file
 * - retrieve and edit application settings from cache memory 
 * 
 * @author Sylvain Desplat
 */
class SMTSmartOtuSettings
{
    const SMART_OTU_VERSION_FILE = "Otu_Smart_Otu.ver";
    const SMART_OTU_VERSION_HEADER = "SOFTWARE VERSION";
    const INI_FILE = "data/settings.ini";
    
    const OTU_MODE = "mode";
    const OTU_OPTION = "option";
    const MAP_URL  = "map";
    const SESSION_TIME_OUT = "session_timeout";
    const MESURE_UNIT = "measure_unit";
    const OTU_ADDRESS = "otu_address";
    const LOG_LEVEL = "log_level";
    const DEBUG_MODE = "debug_mode";
    const DATE_TIME_MAX_DRIFT = "date_time_max_drift";    
    
    const DEFAULT_SESSION_TIME_OUT = 20;
    const DEFAULT_MESURE_UNIT = "KM";
    const DEFAULT_OTU_ADDRESS = "127.0.0.1";    
    const DEFAULT_HTTP_PORT = 80;
    const DEFAULT_DEBUG_MODE = FALSE;
    const DEFAULT_DATE_TIME_MAX_DRIFT = 900;
    const DEFAULT_LOG_LEVEL = SMTLogger::ERROR;
    const DEFAULT_MAP_URL = "http://maps.google.com/maps/place/%f,%f";
    
    private $sessionTimeOut = self::DEFAULT_SESSION_TIME_OUT;
    private $measureUnit = self::DEFAULT_MESURE_UNIT;    
    private $otuAddress = self::DEFAULT_OTU_ADDRESS;
    // attention plus de mode par defaut!!!
    private $otuMode = NULL;
    private $otuOption= NULL;
    //attention plus de type d'OTU par defaut!!!
    private $otuType = NULL;
    private $mapUrl = self::DEFAULT_MAP_URL;
    private $restApi = NULL;
    
    /**
     * The OTU log level
     *
     * @var string SMTLogger::ERROR or SMTLogger::INFO or SMTLogger::DEBUG
     */
    private $logLevel = self::DEFAULT_LOG_LEVEL;
    private $debugMode = self::DEFAULT_DEBUG_MODE;
    private $dateTimeMaxDrift = self::DEFAULT_DATE_TIME_MAX_DRIFT;
    
    /**
     * Single instance of otu setting DTO
     * 
     * @var SMTSmartOtuSettings
     */
    private static $smartOtuSettings = NULL;
    
    /**
     * Single instance of otu setting cache DTO
     *
     * @var SMTSmartOtuSettingsCache
     */
    private static $otuSettingsCache = NULL;
    
    /**
     * Returns the otu settings instance: retrieve it from APC memory if needed
     *
     * @param $createIfNotExistInAPC boolean Whether to force the creation of the instance if it doesn't exist in APC memory
     *
     * @return \app\settings\SMTSmartOtuSettingsCache
     */
    private static function getOtuSettingsDto( $createIfNotExistInAPC = FALSE )
    {
    	if ( self::$otuSettingsCache == NULL )
    	{
    		if ( ( ( $settings = SMTMemoryManager::fetch( SMTSmartOtuSettingsCache::getClass(), SMTSmartOtuSettingsCache::ID ) ) == NULL ) && $createIfNotExistInAPC )
    		{
    			$settings = new SMTSmartOtuSettingsCache();
    		}
    		self::$otuSettingsCache = $settings;
    	}
    	else
    	{
    		$settings = self::$otuSettingsCache;
    	}
    	return $settings;
    }
    
    /**
     *
     */
    private function __construct( )
    {
    	$this->retrieveSettings();
    }
    
    /**
     * @return \app\settings\SMTSmartOtuSettings
     */
    public static function getInstance()
    {
        if ( self::$smartOtuSettings == NULL )
        {
            self::$smartOtuSettings = new SMTSmartOtuSettings();
        }
        return self::$smartOtuSettings;
    }
    
    
    /**
     * Retrieve application settings from data/settings.ini file
     * 
     */
    function retrieveSettings()
    {
        $conf_array = parse_ini_file(self::INI_FILE);
        
        //ini file is not mandatory
        if ( is_array( $conf_array ) )
        {
            if( array_key_exists( SMTSmartOtuSettings::OTU_MODE, $conf_array) )
            {
            	$this->otuMode = $conf_array[SMTSmartOtuSettings::OTU_MODE];
            }
            if( array_key_exists( SMTSmartOtuSettings::OTU_OPTION, $conf_array) )
            {
            	$this->otuOption = $conf_array[SMTSmartOtuSettings::OTU_OPTION];
            }
            else 
            {   
            	//by default no option
            	$this->otuOption = "";
            }
            if( (array_key_exists( SMTSmartOtuSettings::SESSION_TIME_OUT, $conf_array)) && $conf_array[SMTSmartOtuSettings::SESSION_TIME_OUT] > 1)
            {
                $this->sessionTimeOut = intval($conf_array[SMTSmartOtuSettings::SESSION_TIME_OUT]);
            }
            if( array_key_exists( SMTSmartOtuSettings::MESURE_UNIT, $conf_array) )
            {
            	$this->measureUnit = $conf_array[SMTSmartOtuSettings::MESURE_UNIT];
            }
            if( array_key_exists( SMTSmartOtuSettings::OTU_ADDRESS, $conf_array) )
            {
            	$this->otuAddress = $conf_array[SMTSmartOtuSettings::OTU_ADDRESS];
            }   
            if( array_key_exists( SMTSmartOtuSettings::LOG_LEVEL, $conf_array) )
            {
            	$this->logLevel = $conf_array[SMTSmartOtuSettings::LOG_LEVEL];
            }       
            if( array_key_exists( SMTSmartOtuSettings::DEBUG_MODE, $conf_array) )
            {
                $debugMode = $conf_array[SMTSmartOtuSettings::DEBUG_MODE];
            	$this->debugMode = ($debugMode !== NULL)? filter_var($debugMode, FILTER_VALIDATE_BOOLEAN) : SMTSmartOtuSettings::DEFAULT_DEBUG_MODE;
            }
            if( array_key_exists( SMTSmartOtuSettings::DATE_TIME_MAX_DRIFT, $conf_array) )
            {
            	$this->dateTimeMaxDrift = $conf_array[SMTSmartOtuSettings::DATE_TIME_MAX_DRIFT];
            }
        }
        if ( SMTOtuMode::queryIsRescueMode() )
        {
        	$this->otuMode = SMTOtuMode::RESCUE;
        }
        $this->otuType = SMTOtuType::queryOtuType();       
    }
    
    /**
     * Modify application settings in .ini file
     * 
     * @param array $settings: $settingKey=>$settingValue. WARNING $settings array is passed by value to avoid to have to declare explicitly the array ( writeSettings( array(...) ) fails)
     * @param string $settingsFile file where the settings are saved. self::INI_FILE by default.
     */
    static function writeSettings( array $settings, $settingsFile = self::INI_FILE )
    {
    	$iniFile = fopen( $settingsFile,"r");    	     	
    	
    	$replaced = FALSE;
    	$file_contents = "";
    	
    	if ( $iniFile != FALSE)
    	{
	    	while( ( $content = fgets( $iniFile ) ) !== FALSE)
	    	{
	    	    $replaced = FALSE;
	    	    
	    	    foreach ($settings as $settingKey => $settingValue )
	    	    {
	        		if(preg_match("/".$settingKey."/", $content)) 
	        		{
	        		    $file_contents = $file_contents.$settingKey."=".$settingValue."\r\n";        		    
	        			$replaced = TRUE;
	        			unset( $settings[$settingKey] );        			
	        			break;
	        		} 
	    	    }
	    	    if ( !$replaced )
	    	    {
	    	    	$file_contents = $file_contents.$content;
	    	    }
	    	}
	    	fclose($iniFile);
    	}
    	    	
    	// If, the setting wasn't replaced.
    	if( count( $settings ) > 0 ) 
    	{
    	    foreach ($settings as $settingKey => $settingValue )
    	    {
    	        $file_contents = $file_contents.$settingKey."=".$settingValue."\r\n";
    	    }
    	}
    	
    	file_put_contents( $settingsFile, $file_contents, LOCK_EX );
    }
        
    /**
     * Get session timeout from from cache or from .ini file
     * @return number
     */
    public static function getSessionTimeOut()
    {
        $sessionTimeOut = NULL;
    	$settings = self::getOtuSettingsDto();
    	if ( $settings != NULL )
    	{
    		$sessionTimeOut = $settings->getSessionTimeOut();
    	}
    
    	if ( $sessionTimeOut == NULL )
    	{
    		//retrieve logLevel from settings file or use default value (INFO)
    		$sessionTimeOut =  self::getInstance()->sessionTimeOut;
    		//update cache
    		$settings = self::getOtuSettingsDto(TRUE);
    		$settings->setSessionTimeOut( $sessionTimeOut );
    		$settings->save();
    	}
    
    	return $sessionTimeOut;
    }

    /**
     * 
     * @return string
     */
    public static function getMeasureUnit()
    {
    	return self::getInstance()->measureUnit;
    }          

    /**
     * Set log level in cache and save it in .ini file
     *
     * @param string SMTLogger::ERROR or SMTLogger::INFO or SMTLogger::DEBUG 
     * @param SMTContext $context Can be null: if null, don't change OTU log level 
     * 
     * @throws \InvalidArgumentException
     */
    public static function setLogLevel( $logLevel, SMTContext $context = NULL )
    {    	
        $settings = self::getOtuSettingsDto(TRUE);
        $settings->setLogLevel($logLevel);
        $settings->save();
                
    	switch( $logLevel )
    	{
    	    case SMTLogger::ERROR:
    	        self::writeSettings( array( self::LOG_LEVEL=>SMTLogger::ERROR) );
    	        self::getInstance()->logLevel = SMTLogger::ERROR;
    	        break;    	        
	        case SMTLogger::INFO:
	        	self::writeSettings( array( self::LOG_LEVEL=>SMTLogger::INFO) );
	        	self::getInstance()->logLevel = SMTLogger::INFO;
	        	break;
        	case SMTLogger::DEBUG:
        		self::writeSettings( array( self::LOG_LEVEL=>SMTLogger::DEBUG ) );
        		self::getInstance()->logLevel = SMTLogger::DEBUG;
        		break;	        	
	        default:
	            throw new \InvalidArgumentException( sprintf( MSG_INVALID_LOG_LEVEL_PARAMETER, SMTLogger::ERROR.", ".SMTLogger::INFO .", ".SMTLogger::DEBUG) );
	            break;     	    	            	            
    	}    	
    	
    	if ( $context != NULL )
    	{
    		try
    		{
    			self::updateOtuLogLevel( $logLevel, $context );
    		}
    		//don't throw exception to continue processing: OTU can be unavailable and user wants to change the log level.
    		catch(\Exception $e)
    		{
    			SMTLogger::getInstance()->traceException($e);
    		}
    	}
    }    
    
    /**
     * Set log level in OTU
     *
     * @param string SMTLogger::ERROR or SMTLogger::INFO or SMTLogger::DEBUG
     * @param SMTContext $context

     */
    private static function updateOtuLogLevel( $logLevel, SMTContext $context = NULL )
    {
        switch( $logLevel )
        {
        	case SMTLogger::ERROR:
        	    $context->getOtuSocket()->send( SMTOtuApi::changeLogLevel( SMTOtuApi::LOG_LEVEL_ERROR ) );
        		break;
        	case SMTLogger::INFO:
        	    $context->getOtuSocket()->send( SMTOtuApi::changeLogLevel( SMTOtuApi::LOG_LEVEL_INFO ) );
        		break;
        	case SMTLogger::DEBUG:
        	    $context->getOtuSocket()->send( SMTOtuApi::changeLogLevel( SMTOtuApi::LOG_LEVEL_DEBUG ) );
        		break;
        	default:
        		break;
        }
    }
    
    
    /**
     * Get loglevel from cache or from .ini file
     * @return string
     */
    public static function getLogLevel($save = TRUE )
    {
        $logLevel = NULL;
        $settings = self::getOtuSettingsDto();
        if ( $settings != NULL )
        {
        	$logLevel = $settings->getLogLevel();
        }
        
        if ( $logLevel == NULL )
        {
        	//retrieve logLevel from settings file or use default value (ERROR)
        	$logLevel =  self::getInstance()->logLevel;
        	if ($save)
        	{
            	//update cache
            	$settings = self::getOtuSettingsDto(TRUE);
            	$settings->setLogLevel($logLevel);
            	$settings->save();
        	}
        }
        
        return $logLevel;
    }           
    
    /**
     * Get debug mode from cache or from .ini file
     * @return boolean 
     */
    public static function isDebugMode()
    {
        $debugMode = NULL;
        $settings = self::getOtuSettingsDto();
        if ( $settings != NULL )
        {
        	$debugMode = $settings->isDebugMode();
        }
        
        if ( $debugMode === NULL || !isset($debugMode) )
        {
        	//retrieve debugMode from settings file or use default value (FALSE)
        	$debugMode =  self::getInstance()->debugMode;
        	//update cache
        	$settings = self::getOtuSettingsDto(TRUE);
        	$settings->setDebugMode($debugMode);
        	$settings->save();
        }
        
        return $debugMode;
    }
    
    /**
     * Retrieve smartOTU version from Otu_Smart_Otu.ver
     * 
     * @return string
     */
    public static function getSmartOtuVersion()
    {
        $smartOtuVersion = "";
        $handle = fopen( self::SMART_OTU_VERSION_FILE , "r");
        if ( $handle != FALSE )
        {
            $data='';
            while( !feof( $handle ) )
            {
            	$data = fgets($handle);
            	if ( strpos($data, self::SMART_OTU_VERSION_HEADER) !== FALSE )
            	{
            	    //jump one line
            	    fgets($handle);
            	    //retrieve the next one
            	    $smartOtuVersion = trim( fgets($handle) );
            	    break;
            	}
            }
            fclose($handle);
        }
        	
        return $smartOtuVersion;        
    }
    
    /**
     * Retrieve OTU version from getinfoversion Otu
     * 
     * @return string:
     */
    public static function getOtuVersion()
    {
        $otuInfo = exec('sudo /sbin/getinfoversion Otu');
        list( $otuType, $application, $executable, $otuReleaseVersion ) = explode(";", $otuInfo);
        return $otuReleaseVersion;
    }
    
    /**
     * Retrieve OTU rescue version from getinfoversion Otu
     *
     * @return string:
     */
    public static function getOtuRescueVersion()
    {
    	$otuInfo = exec('sudo /sbin/getinfoversion -r Otu');
    	list( $otuType, $application, $executable, $otuReleaseVersion ) = explode(";", $otuInfo);
    	return $otuReleaseVersion;
    }
    
    /**
     * Get Max date-time drift of the OTU in seconds (900 by default) from cache or from .ini file
     * @return integer
     */
    public static function getDateTimeMaxDrift()
    {
        $dateTimeMaxDrift = NULL;
        $settings = self::getOtuSettingsDto();
        if ( $settings != NULL )
        {
        	$dateTimeMaxDrift = $settings->getDateTimeMaxDrift();
        }
        
        if ( $dateTimeMaxDrift == NULL )
        {
        	//retrieve logLevel from settings file or use default value (INFO)
        	$dateTimeMaxDrift =  self::getInstance()->dateTimeMaxDrift;
        	//update cache
        	$settings = self::getOtuSettingsDto(TRUE);
        	$settings->setDateTimeMaxDrift( $dateTimeMaxDrift );
        	$settings->save();
        }
    	return intval($dateTimeMaxDrift);
    }
    
    /**
     * Register the otu port in CACHE MEMORY
     *
     * @param number $otuPort The port for the OTU parser.    
     */
    public static function setOtuPort( $otuPort )
    {
        $settings = self::getOtuSettingsDto(TRUE);
		$settings->setOtuPort( $otuPort );
		$settings->save();
    }    
        
    /**
     * Set whether attenuation monitoring critical thresholds are set in OTU
     *
     * @param boolean $isCriticalThresholdsSet
     */
    public static function setIsCriticalThresholdsSet($isCriticalThresholdsSet)
    {
    	$settings = self::getOtuSettingsDto(TRUE);
    	$settings->setIsCriticalThresholdsSet($isCriticalThresholdsSet);
    	$settings->save();
    }
    
    /**
     * Whether attenuation monitoring critical thresholds are set in OTU
     *
     * @return boolean
     */
    public static function IsCriticalThresholdsSet()
    {
        $settings = self::getOtuSettingsDto(TRUE);
    	return $settings->IsCriticalThresholdsSet();
    }
    
    
    /**
     * Get the otu parser address from CACHE MEMORY: by default the parser is on the same machine => local loop.
     * In debug mode SmartOTU can be located on a remote computer.
     *
     * @return string
     */
    public static function getOtuAddress()
    {
        $otuIpAddress = NULL;
        $settings = self::getOtuSettingsDto();
        if ( $settings != NULL )
        {
            $otuIpAddress = $settings->getOtuIpAddress();
        }
        
        if ( $otuIpAddress == NULL )
        {
            //retrieve otu address from settings file or use default value (local loop)
            $otuIpAddress =  self::getInstance()->otuAddress;
            self::setOtuAddress($otuIpAddress);
        }            
        
        return $otuIpAddress;
    }      
    
    /**
     * Register the otu parser ip address in CACHE MEMORY
     *
     * @param string $otuParserIpAddress The ip address of the OTU parser.
     */
    public static function setOtuAddress( $otuParserIpAddress )
    {
    	$settings = self::getOtuSettingsDto(TRUE);
    	$settings->setOtuIpAddress( $otuParserIpAddress );
    	$settings->save();
    }
    
    /**
     * Get the port of the otu parser from CACHE MEMORY if it has been set, either return NULL.
     *
     * @return number
     */
    public static function getOtuPort()
    {
    	$otuPort = NULL;
    	$settings = self::getOtuSettingsDto();
    	if ( $settings != NULL )
    	{
    		$otuPort = $settings->getOtuPort();
    	}
    	return ($otuPort !== NULL)? intval( $otuPort ) : NULL;
    }    
    
    /**
     * Get the otu mode from CACHE MEMORY.
     * If mode is not found, try to parse again the settings file (link on settings file is created by OTU and could not exist when WebI looks for it)
     *
     * @return string The OTU mode or NULL (settings file link doesn't exist)
     */
    public static function getOtuMode()
    {
    	$otuMode = NULL;
    	$settings = self::getOtuSettingsDto();
    	if ( $settings != NULL )
    	{
    		$otuMode = $settings->getOtuMode();
    	}
    
    	if ( $otuMode == NULL )
    	{
    		//retrieve otu mode from settings file or use default value
    		$otuMode =  self::getInstance()->otuMode;
    		if ($otuMode != NULL)
    		{
    			self::setOtuMode($otuMode);
    		}
    		else 
    		{
    			//force retrieval of settings
    			self::getInstance()->retrieveSettings();
    			$otuMode =  self::getInstance()->otuMode;
    			if ($otuMode != NULL)
    			{
    				self::setOtuMode($otuMode);
    			}
    		}
    	}
    
    	return $otuMode;
    }
    
    /**
     * Register the otu mode in CACHE MEMORY
     *
     * @param string $otuMode The OTU mode.
     */
    public static function setOtuMode( $otuMode )
    {
    	$settings = self::getOtuSettingsDto(TRUE);
    	$settings->setOtuMode( $otuMode );
    	$settings->save();
    }

    /**
     * Get the otu option from CACHE MEMORY.
     * If mode is not found, try to parse again the settings file (link on settings file is created by OTU and could not exist when WebI looks for it)
     *
     * @return string The OTU mode or NULL (settings file link doesn't exist)
     */
    public static function getOtuOption()
    {
    	$otuOption = NULL;
    	$settings = self::getOtuSettingsDto();
    	if ( $settings != NULL )
    	{
    		$otuOption = $settings->getOtuOption();
    	}
    	
    	if ( $otuOption == NULL )
    	{
    		//retrieve otu mode from settings file or use default value
    		$otuOption =  self::getInstance()->otuOption;
    		if ($otuOption != NULL)
    		{
    			self::setOtuOption($otuOption);
    		}
    		else
    		{
    			//force retrieval of settings
    			self::getInstance()->retrieveSettings();
    			$otuOption =  self::getInstance()->otuOption;
    			if ($otuOption != NULL)
    			{
    				self::setOtuOption($otuOption);
    			}
    		}
    	}
    	
    	return $otuOption;
    }
    
    /**
     * Register the otu option in CACHE MEMORY
     *
     * @param string $otuOption The OTU mode.
     */
    public static function setOtuOption( $otuOption)
    {
    	if ( $otuOption != NULL )
    	{
    		$settings = self::getOtuSettingsDto(TRUE);
    	}
    	else
    	{
    		$settings = self::getOtuSettingsDto(FALSE);
    	}
    	if ( $settings != NULL )
    	{
    		$settings->setOtuOption( $otuOption);
    		$settings->save();
    	}
    }

    /**
     * Get the otu mode from CACHE MEMORY.
     *
     * @return string
     */
    public static function getOtuType()
    {
    	$otuType = NULL;
    	$settings = self::getOtuSettingsDto();
    	if ( $settings != NULL )
    	{
    		$otuType = $settings->getOtuType();
    	}
    
    	if ( $otuType == NULL )
    	{
    		//retrieve otu type from settings file or use default value
    		$otuType =  self::getInstance()->otuType;
    		self::setOtuType($otuType);
    	}
    
    	return $otuType;
    }
    
    /**
     * Register the otu mode in CACHE MEMORY
     *
     * @param string $otuType The OTU type.
     */
    public static function setOtuType( $otuType )
    {
    	$settings = self::getOtuSettingsDto(TRUE);
    	$settings->setOtuType( $otuType );
    	$settings->save();
    }    
    
    /**
     * Get the google map url from CACHE MEMORY.
     *
     * @return string google map url
     */
    public static function getMapUrl()
    {
    	$mapUrl = NULL;
    	$settings = self::getOtuSettingsDto();
    	if ( $settings != NULL )
    	{
    		$mapUrl = $settings->getMapUrl();
    	}
    
    	if ( $mapUrl == NULL )
    	{
    		//use default value
    		$mapUrl =  self::getInstance()->mapUrl;
    		self::setMapUrl($mapUrl);
    	}
    
    	return $mapUrl;
    }
    
    /**
     * Register the google map url in CACHE MEMORY.
     *
     * @param string $mapUrl The google map url.
     */
    public static function setMapUrl( $mapUrl )
    {
    	$settings = self::getOtuSettingsDto(TRUE);
    	$settings->setMapUrl( $mapUrl );
    	$settings->save();
    }
    
    /**
     * @param boolean $valid
     */
    public static function setSmsOptionValid ( $smsOptionValid )
    {
        $settings = self::getOtuSettingsDto(TRUE);
		$settings->setSmsOptionValid( $smsOptionValid );
		$settings->save();
    }
    
    /**
     * Whether the SMS option is available on the OTU or NULL if the data wasn't yet retrieved
     * 
     * @return boolean or NULL if not set
     */
    public static function isSmsOptionValid()
    {
        $smsOptionValid = FALSE;
        $settings = self::getOtuSettingsDto();
        if ( $settings != NULL )
        {
            $smsOptionValid = $settings->isSmsOptionValid();
        }
        return ($smsOptionValid !== NULL)? filter_var($smsOptionValid, FILTER_VALIDATE_BOOLEAN) : NULL;
    }
    
    /**
     * @param boolean $valid
     */
    public static function setRelayOptionValid ( $relayOptionValid )
    {
    	$settings = self::getOtuSettingsDto(TRUE);
    	$settings->setRelayOptionValid( $relayOptionValid);
    	$settings->save();
    }
    
    /**
     * Whether the Relay option is available on the OTU or NULL if the data wasn't yet retrieved
     *
     * @return boolean or NULL if not set
     */
    public static function isRelayOptionValid()
    {
    	$relayOptionValid= FALSE;
    	$settings = self::getOtuSettingsDto();
    	if ( $settings != NULL )
    	{
    		$relayOptionValid= $settings->isRelayOptionValid();
    	}
    	return ($relayOptionValid!== NULL)? filter_var($relayOptionValid, FILTER_VALIDATE_BOOLEAN) : NULL;
    }
    
    /**
     * Register the default peaks thresholds
     *
     * @param SMTOtuMonitoringTestThresholdDto $peaksThresholdsDb The otu peaks thresholds.
     *
     */
    public static function setDefaultPeakThresholdsDb( SMTOtuMonitoringTestThresholdDto $peaksThresholdsDb)
    {
    	SMTSmartOtuThresholds::setDefaultPeakThresholds($peaksThresholdsDb);
    }
    
    /**
     * Get default peaks thresholds
     *
     * @return SMTOtuMonitoringTestThresholdDto The otu peaks default thresholds.
     *
     */
    public static function getDefaultPeakThresholdsDb()
    {
    	return SMTSmartOtuThresholds::getDefaultPeakThresholds();
    }
    
    /**
     * Register the otu First Marker attenuation thresholds
     *
     * @param SMTOtuMonitoringTestThresholdDto $firstMarkerThresholds The otu First Marker attenuation thresholds.
     *
     */
    public static function setDefaultFirstMarkerAttenuationThresholds( SMTOtuMonitoringTestThresholdDto $firstMarkerThresholds )
    {
    	SMTSmartOtuThresholds::setDefaultFirstMarkerAttenuationThresholds($firstMarkerThresholds);
    }
    
    /**
     * Get the otu First Marker attenuation thresholds
     *
     * @return SMTOtuMonitoringTestThresholdDto The otu First Marker attenuation thresholds.
     *
     */
    public static function getDefaultFirstMarkerAttenuationThresholds()
    {
    	return SMTSmartOtuThresholds::getDefaultFirstMarkerAttenuationThresholds();
    }
        
    
    /**
     * Register the otu budget (LastMarker - First Marker) attenuation thresholds
     *
     * @param SMTOtuMonitoringTestThresholdDto $budgetVariationMarkerThresholds The otu budget attenuation thresholds.
     *
     */
    public static function setDefaultBudgetVariationThresholds( SMTOtuMonitoringTestThresholdDto $budgetVariationThresholds )
    {
        SMTSmartOtuThresholds::setDefaultBudgetVariationThresholds($budgetVariationThresholds);
    }    
    
    /**
     * Get the otu budget (LastMarker - First Marker) attenuation thresholds
     *
     * @return SMTOtuMonitoringTestThresholdDto The otu budget attenuation thresholds.
     *
     */
    public static function getDefaultBudgetVariationThresholds()
    {
    	return SMTSmartOtuThresholds::getDefaultBudgetVariationThresholds();
    }
    
    /**
     * Get the otu tapping attenuation thresholds
     *
     * @return SMTOtuMonitoringTestThresholdDto The otu tapping attenuation thresholds.
     *
     */
    public static function getDefaultTappingThresholds()
    {
    	return SMTSmartOtuThresholds::getDefaultTappingThresholds();
    }
    
    /**
     * Get the Monitoring Acceptable Noise Margin (in Db) from cache or from ini file.
     * Used to display that min noise level in trace viewer and to force user to set markers in traceviewer outside noise level.
     *
     * @return int Monitoring Acceptable Noise Margin (in Db)
     */
    public static function getMonitoringAcceptableNoiseMarginDbDefault()
    {
        return SMTSmartOtuThresholds::getMonitoringAcceptableNoiseMarginDbDefault();
    }
    
    /**
     * Get the Tapping Acceptable Noise Margin (in Db) from cache or from ini file.
     * Used to display that min noise level in trace viewer and to force user to set markers in traceviewer outside noise level.
     *
     * @return int Monitoring Acceptable Noise Margin (in Db)
     */
    public static function getTappingAcceptableNoiseMarginDbDefault()
    {
    	return SMTSmartOtuThresholds::getTappingAcceptableNoiseMarginDbDefault();
    }
    
    /**
     * Get the delta between TO and BACK TO thresholds for attenuation (in Db) from cache or from ini file.
     *
     * @return The delta between TO and BACK TO thresholds for attenuation (in Db)
     */
    public static function getAttenuationToBtoDeltaDbDefault()
    {
    	return SMTSmartOtuThresholds::getAttenuationToBtoDeltaDbDefault();
    }
    
    /**
     * Get the delta between TO and BACK TO thresholds for tappingattenuation (in Db) from cache or from ini file.
     *
     * @return The delta between TO and BACK TO thresholds for attenuation (in Db)
     */
    public static function getTappingToBtoDeltaDbDefault()
    {
    	return SMTSmartOtuThresholds::getTappingToBtoDeltaDbDefault();
    }
    
    /**
     * Get the attenuation critical threshold (in Db).
     *
     * @return float attenuation critical threshold (in Db)
     */
    public static function getAttenuationCriticalThresholdDbDefault()
    {
    	return SMTSmartOtuThresholds::getAttenuationCriticalThresholdDbDefault();
    }
    
    /**
     * Get the delta between TO and BACK TO thresholds for peak (in Db)
     *
     * @return float delta between TO and BACK TO thresholds for peak (in Db)
     */
    public static function getPeakToBtoDeltaDbDefault()
    {
    	return SMTSmartOtuThresholds::getPeakToBtoDeltaDbDefault();
    }
    
    /**
     * Get the peaks shift limit  (in points).
     *
     * @return int peaks shift limit in points
     */
    public static function getMissingPeakThresholdPtDefault()
    {
    	return SMTSmartOtuThresholds::getMissingPeakThresholdPtDefault();
    }
    
    /**
     * Get the threshold to detect peaks (in Db).
     *
     * @return float threshold to detect peaks (in Db)
     */
    public static function getMonitorablePeakThresholdDbDefault()
    {
    	return SMTSmartOtuThresholds::getMonitorablePeakThresholdDbDefault();
    }

    /**
     * Whether REST API can be used.
     *
     * @return boolean or NULL if not found
     */
    public static function isRestApi()
    {
    	$restApi = NULL;
    	$settings = self::getOtuSettingsDto();
    	if ( $settings != NULL )
    	{
    		$restApi= $settings->isRestApi();
    	}
    	
    	return $restApi;
    }
    
    /**
     * Set whether REST API can be used.
     *
     * @param boolean $restApi
     */
    public static function setRestApi( $restApi)
    {
    	$settings = self::getOtuSettingsDto(TRUE);
    	$settings->setRestApi( $restApi);
    	$settings->save();
    }
    
    /**
     * Whether vpncan be used.
     *
     * @return boolean or NULL if not found
     */
    public static function isVpn()
    {
        $vpn = NULL;
        $settings = self::getOtuSettingsDto();
        if ( $vpn != NULL )
        {
            $vpn= $settings->isVpn();
        }
        
        return $vpn;
    }
    
    /**
     * Set whether vpn can be used.
     *
     * @param boolean $restApi
     */
    public static function setVpn( $vpn)
    {
        $settings = self::getOtuSettingsDto(TRUE);
        $settings->setVpn( $vpn);
        $settings->save();
    }
} 

?>