<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\maintenance;

use app\util\SMTIOException;

use app\util\SMTLogger;
use app\serviceshelper\SMTServiceHelper;
use app\util\SMTUtil;

/**
 *
 * @author Sylvain Desplat
 */
class SMTCertificate extends SMTServiceHelper
{
    const SSH_AUTHORIZED_PUBLIC_KEYS = "/root/.ssh/authorized_keys";
    const UPDATE_SSH_AUTHORIZED_PUBLIC_KEYS_CMD = "sudo /usr/lib/jdsu/scripts/otu_update_ssh_authorized_keys.sh %s";
    const READ_SSH_AUTHORIZED_PUBLIC_KEYS_CMD = "sudo /usr/lib/jdsu/scripts/otu_read_ssh_authorized_keys.sh";
    const COPY_CERTFICATE_FILE_CMD = "sudo /usr/lib/jdsu/scripts/smartotu_copy_files.sh %s %s";
    const CHECK_CERTIFICATES_CMD = "sudo /usr/lib/jdsu/scripts/otu_check_certificates.sh %s %s";
    const SSL_KEY  = 'ssl_key';
    const SSL_CERT = 'ssl_cert';
    const SSL_CHAIN_CERT = 'ssl_chain-cert';
    const SSH_TEMP_PUBLIC_KEY='/tmp/temp_public_keys';
    const SANITIZED_SSL_CHAIN_FILE='/tmp/sanitized_chain_certificate_file.crt';
    const SANITIZED_SSL_CERT_FILE='/tmp/sanitized_certificate_file.crt';
    const SANITIZED_SSL_KEY_FILE='/tmp/sanitized_ssl_key_file';
    
    /**
     * Read ssh public keys file
     * 
     * @return The ssh authorized file as an array of string
     */
    function readSshPublicKeys()
    {
        $lines = array();

        $command = self::READ_SSH_AUTHORIZED_PUBLIC_KEYS_CMD;
        $handle = popen($command, 'r');
        SMTLogger::getInstance()->trace( sprintf("command: %s.", $command ), SMTLogger::DEBUG);
        
        while (!feof($handle)) 
        {
        	$data = fgets($handle);
        	$data = trim( str_replace('\n',"",$data) );
        	if ( $data !== "" )
        	{
        	    array_push($lines, $data);
        	    SMTLogger::getInstance()->trace( sprintf("data: %s.", $data ), SMTLogger::DEBUG);
        	}
        }

        return $lines;
    }
    
    /**
     * Save the ssh public keys from an array of string
     * 
     * @param array
     */
    function updateSshPublicKeys( $lines )
    {
        $success = -1;

        // append \n on each line
		$lines = array_map(
		function($line) { return $line."\n"; },
		$lines
		); 

		file_put_contents(self::SSH_TEMP_PUBLIC_KEY, $lines);
        
		$command = sprintf(self::UPDATE_SSH_AUTHORIZED_PUBLIC_KEYS_CMD, self::SSH_TEMP_PUBLIC_KEY);
        SMTLogger::getInstance()->trace( sprintf("command: %s.", $command ), SMTLogger::DEBUG);
        
        // update ssh keys
        $value = exec ($command, $output, $success);
        
        SMTLogger::getInstance()->trace(sprintf("update ssh keys output: %s success %s", $value, $success) );
        
        //unlink(self::SSH_TEMP_PUBLIC_KEY);
        // 0 -> update ssh keys OK, otherwise KO
        if ($success !== 0) 
        {
            throw new SMTIOException(SMTIOException::INVALID_FILE, "update ssh keys");
        }         
    }
    
    /**
     * copy ssl certficates to their target directory
     *
     * @param $filename The name of the certificate file
     *
     */
    static function copySslCertificate($fileName, $certificateType)
    {
    	$fullPath = SMTUtil::SMARTOTU_TEMP_DIR.$fileName;
    	$success = -1;
    	
    	//delete uploaded file
    	if ( file_exists( $fullPath) )
    	{
    		unlink ($fullPath);
    	}
    	
    	switch ($certificateType)
    	{
    		case SMTCertificate::SSL_KEY:
    			$fullPath = self::SANITIZED_SSL_KEY_FILE;
    			break;
    		case SMTCertificate::SSL_CERT:
    			$fullPath = self::SANITIZED_SSL_CERT_FILE;
    			break;
    		case SMTCertificate::SSL_CHAIN_CERT:
    			$fullPath = self::SANITIZED_SSL_CHAIN_FILE;
    			break;
    		default:    			
    			break;
    	}
    	
    	try
    	{
    		if ( file_exists( $fullPath) )
    		{
    			$command = sprintf(self::COPY_CERTFICATE_FILE_CMD, SMTUtil::escapeShellArgUtf8($fullPath), SMTUtil::escapeShellArgUtf8($certificateType) );
    			SMTLogger::getInstance()->trace( sprintf("command: %s.", $command ), SMTLogger::INFO);
    			 
    			// update ssh keys
    			$value = exec ($command, $output, $success);
    			 
    			SMTLogger::getInstance()->trace(sprintf("copy uploaded certificate file output: %s success %s", $value, $success), SMTLogger::DEBUG);
    			
    			// 0 -> update license OK, otherwise KO
    			if ($success !== 0)
    			{
    				throw new SMTIOException(SMTIOException::INVALID_FILE, "failed to copy uploaded certificate file");
    			}
    		}
    		else
    		{
    			throw new SMTIOException(SMTIOException::FILE_NOT_FOUND, sprintf("uploaded certificate file %s not found", $fullPath) );
    		}
    	}
    	catch ( \Exception $e )
    	{
    		SMTLogger::getInstance()->traceException( $e );
    		throw $e;
    	}
    }
    
    /**
     * check ssl certficates from their temp dir
     *
     * @param $filename The name of the certificate file
     * @param $fullPath the temporary file full path
     * @param $certificateType
     *
     */
    static function checkSslCertificate($fileName, $fullPath, $certificateType)
    {
    	$success = -1;
    	
    	try
    	{
    		if ( file_exists( $fullPath) )
    		{
    			$command = sprintf(self::CHECK_CERTIFICATES_CMD, SMTUtil::escapeShellArgUtf8($fullPath), SMTUtil::escapeShellArgUtf8($certificateType) );
    			SMTLogger::getInstance()->trace( sprintf("command: %s.", $command ), SMTLogger::INFO);
    			
    			// check certificate
    			$value = exec ($command, $output, $success);
    			
    			SMTLogger::getInstance()->trace(sprintf("check certificate file output: %s success %s", $value, $success), SMTLogger::DEBUG);
    			
    			// 0 -> update license OK, otherwise KO
    			if ($success !== 0)
    			{
    				unlink($fullPath);
    				throw new SMTIOException(SMTIOException::INVALID_FILE, "failed to check certificate file ".$fileName);
    			}
    		}
    		else
    		{
    			throw new SMTIOException(SMTIOException::FILE_NOT_FOUND, sprintf("uploaded certificate file %s not found", $fileName) );
    		}
    	}
    	catch ( \Exception $e )
    	{
    		SMTLogger::getInstance()->traceException( $e );
    		throw $e;
    	}
    }
}
?>
