<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\events;

use app\util\SMTRwDto;

use app\sharedmemory\SMTMemoryManager;
use app\sharedmemory\SMTIMemorySupport;

/**
 * Dto storing data to filter filter messages:
 * -$conversationId all messages for a given destinator client are tunnelled in a single "conversation" connection
 * -$consumerId the destinator of the message
 * -$messageClassName the type of message to send to the destinator 
 * 
 * @author Sylvain Desplat
 */
class SMTConversationMessageFilterDto extends SMTRwDto implements SMTIMemorySupport
{
    /**
     * All messages for a given client are tunnelled in a single "conversation" connection
     * 
     * @var string
     */
    protected $conversationId = NULL;
    /**
     * he destinator of the message
     * @var string
     */
    protected $consumerId = NULL;
    /**
     * SMTMessageDto class name to filter : the type of message to send to the destinator
     * @var string
     */
    protected $messageType = NULL;
    
    /**
     * Last access to query the event consumer
     *
     * @var float
     */
     private $lastAccess;

     public function getLastAccess()
     {
     	return $this->lastAccess;
     }
     
     public function touchAccess()
     {
     	$this->lastAccess = microtime(true);
     }         
    
    public function setConversationId( String $conversationId )
    {
    	$this->conversationId = $conversationId;
    }
    
    public function getConversationId()
    {
    	return $this->conversationId;
    }
    
    public function setConsumerId( String $consumerId )
    {
    	$this->consumerId = $consumerId;
    }
    
    public function getConsumerId()
    {
    	return $this->consumerId;
    }

    public function setMessageType( String $messageType )
    {
    	$this->messageType = $messageType;
    }
    
    public function getMessageType()
    {
    	return $this->messageType;
    }    
    
	/**
	 * Serialize as Json data the user object.<br>
	 *
	 * @return $objectProperties the object properties serialized as a Json string.
	 */
	public function getJsonData()
	{
		$objectProperties = get_object_vars($this);
		$this->serializeObjectAsJsonData( $objectProperties );
		return $objectProperties;
	}

	/**
	 * Returns Dto class name.
	 *
	 * @return string the dtop class name
	 */
	public function getDtoClassName()
	{
		return get_class();
	}
	
	/**
	 * Returns the dto class name.
	 *
	 * @return the dto class name.
	 */
	public static function getClass()
	{
		return __CLASS__;
	}	
	
	/**
	 * Create the Dto from a Json Dto in an array
	 *
	 * @param array $json_array json data in an array
	 *
	 * @return \app\events\SMTConversationMessageFilterDto
	 */
	static function getInstance( array &$json_array )
	{
		return self::forge( get_class(), $json_array );
	}	
	
	/**
	 * Identifier for memory manager
	 * 
	 * @return string
	 */
	public function getId()
	{
		return $this->getConsumerId();
	}
		
	/**
	 * Save Dto in shared memory
	 *
	 * @throws SMTAPCMemoryException
	 */
	public function save()
	{
		SMTMemoryManager::saveDto( $this );
	}
		
	/**
	 * Delete Dto from shared memory
	 *
	 * @throws SMTAPCMemoryException
	 */
	public function delete()
	{
		SMTMemoryManager::deleteDto( $this );
	}			
	
	/**
	 * Check whether the conversation was not requested since a time > 5 times the period between 2 message retrievals.
	 *
	 *
	 * @return boolean Whether the event consumer was not requested since a time > 5 times the period between 2 message retrievals.
	 */
	public function isObsolete()
	{
	    if ( !isset( $this->lastAccess ) )
	    {
	        $this->touchAccess();
	    }
		//if 5 times the period between 2 message retrieval has been spent, the message is obsolete		
// 	    SMTLogger::getInstance()->trace("isObsolete: ".(microtime(TRUE) - $this->getLastAccess()), SMTLogger::DEBUG);
		return ( microtime(TRUE) - $this->getLastAccess() ) > 5 * SMTEventMessageManager::SLEEP_TIME_SEC;
	}	
}

?>