from viavi.evm import *
from viavi.diag import *
import viavi.mts.uts_spi as utsspi
import time, os
import smbus
import serial

import subprocess
from datetime import datetime

DIAG_VERSION        = 0.4

#
#   reg_name             reg_addr         reg_access    value @ reset
#
#
# OSB SPI MASTER
#
# REG_OSB_SPI_BASE_ADDR   = 0x0C00
#
REG_OSB_SPI_SLV0        = 0x0C00
REG_OSB_SPI_SLV1        = 0x0C04
REG_OSB_SPI_CTRL        = 0x0C08
REG_OSB_SPI_TRANSMIT    = 0x0C0C
#
# GLU_IP_BASE_ADDR      = 0x1400
REG_FPGA_VERSION        = 0x1400            # RO        <current_version>
REG_FPGA_SYNTH_DATE     = 0x1404            # RO        <current_synth_date>
REG_PSU_CTRL            = 0x1408            # R/W       0x0FF
REG_SHIFTER_CTRL        = 0x140C            # R/W       0x0 ?
REG_MUX_CTRL            = 0x1410            # RO        ?
REG_SYS_FAN1_CTRL       = 0x1418            # R/W       0x0009003A
REG_SYS_FAN2_CTRL       = 0x141C            # R/W       0x00090019
REG_SYS_FAN3_CTRL       = 0x1420            # R/W       0x00090014
REG_OSB_FAN1_CTRL       = 0x1424            # R/W       0x0013001E
REG_OSB_FAN2_CTRL       = 0x1428            # R/W       0x00130028
REG_DBG_CTRL            = 0x1438            # R/W
REG_MSEL                = 0x143C            # R/W       0x0
REG_SYS_FAN1_TACHI_VAL  = 0x1440            # RO        -
REG_SYS_FAN2_TACHI_VAL  = 0x1444            # RO        -
REG_SYS_FAN3_TACHI_VAL  = 0x1448            # RO        -
REG_OSB_FAN1_TACHI_VAL  = 0x144C            # RO        -
REG_OSB_FAN2_TACHI_VAL  = 0x1450            # RO        -
REG_OSB_PUSHB_VAL       = 0x1454            # RO        -
REG_SYS_PSYNC_700K_MEAS = 0x1458            # RO
REG_MOD_FAN_RDBCK       = 0x1474            # RO        -
REG_MOD_PRESENCE_RDBCK  = 0x1478            # RO        -
REG_RST_CTRL            = 0x147C            # R/W       0xFFFFFFFF

URJTAG_BASE_ADDR        = 0x1800
REG_URJTAG_FREQ         = URJTAG_BASE_ADDR + 0x0
REG_URJTAG_IO           = URJTAG_BASE_ADDR + 0x4
REG_URJTAG_STATUS       = URJTAG_BASE_ADDR + 0x8
REG_URJTAG_DUMMY        = URJTAG_BASE_ADDR + 0xC

# UART devices

UART_MOD_BASE_ADDR      = 0x10000
UART_RS485_BASE_ADDR    = 0x19000

UART_RBR_DLL            = 0x01000
UART_DLM                = 0x01004
UART_FCR                = 0x01008
UART_LCR                = 0x0100C
UART_MCR                = 0x01010
UART_MSR                = 0x01018

# XADC

XADC_BASE_ADDR          = 0x2000

#
#""""""""""""""""""""""""""""""""
# Status Registers (Read Only) :
#""""""""""""""""""""""""""""""""
#
XADC_TEMP_REG           = XADC_BASE_ADDR + 0x200    # RO
#
XADC_VAUX_0_REG         = XADC_BASE_ADDR + 0x240    # RO
XADC_VAUX_1_REG         = XADC_BASE_ADDR + 0x244    # RO
XADC_VAUX_2_REG         = XADC_BASE_ADDR + 0x248    # RO
XADC_VAUX_3_REG         = XADC_BASE_ADDR + 0x24C    # RO
#XADC_VAUX_4_REG         = XADC_BASE_ADDR + 0x250    # RO : not available in Artix7
#XADC_VAUX_5_REG         = XADC_BASE_ADDR + 0x254    # RO : not available in Artix7
#XADC_VAUX_6_REG         = XADC_BASE_ADDR + 0x258    # RO : not available in Artix7
#XADC_VAUX_7_REG         = XADC_BASE_ADDR + 0x25C    # RO : not available in Artix7
XADC_VAUX_8_REG         = XADC_BASE_ADDR + 0x260    # RO
XADC_VAUX_9_REG         = XADC_BASE_ADDR + 0x264    # RO
XADC_VAUX_10_REG        = XADC_BASE_ADDR + 0x268     # RO
XADC_VAUX_11_REG        = XADC_BASE_ADDR + 0x26C     # RO
#XADC_VAUX_12_REG        = XADC_BASE_ADDR + 0x270    # RO : not available in Artix7
#XADC_VAUX_13_REG        = XADC_BASE_ADDR + 0x274    # RO : not available in Artix7
#XADC_VAUX_14_REG        = XADC_BASE_ADDR + 0x278    # RO : not available in Artix7
#XADC_VAUX_15_REG        = XADC_BASE_ADDR + 0x27C    # RO : not available in Artix7
#
# SPI B200
#
SPI_ADDR_VERSION        = 0x0   # RO
SPI_ADDR_REGLVDS_7_0    = 0x1   # R/W
SPI_ADDR_REGLVDS_15_8   = 0x2   # R/W
SPI_ADDR_REGLVDS_20_16  = 0x3   # R/W
SPI_ADDR_MISC           = 0x4   # R/W
SPI_ADDR_SYNC_MEAS1_LSB = 0x5   # RO
SPI_ADDR_SYNC_MEAS1_MSB = 0x6   # RO
SPI_ADDR_SYNC_MEAS2_LSB = 0x7   # RO
SPI_ADDR_SYNC_MEAS2_MSB = 0x8   # RO
SPI_ADDR_SYNC_MEAS3_LSB = 0x9   # RO
SPI_ADDR_SYNC_MEAS3_MSB = 0xa   # RO
SPI_ADDR_SYNC_MEAS4_LSB = 0xb   # RO
SPI_ADDR_SYNC_MEAS4_MSB = 0xc   # RO
SPI_ADDR_SYNC_MEAS5_LSB = 0xd   # RO
SPI_ADDR_SYNC_MEAS5_MSB = 0xe   # RO
SPI_ADDR_SYNC_MEAS6_LSB = 0xf   # RO
SPI_ADDR_SYNC_MEAS6_MSB = 0x10  # RO

SPI_ADDR_BOARD_PRES     = 0x11 # R/W
SPI_ADDR_UART           = 0x12 # R/W
SPI_ADDR_DTSS_REG       = 0x13 # R/W
SPI_ADDR_MSEL_REG       = 0x14 # RO
SPI_ADDR_EXPCB          = 0x15 # RO
SPI_ADDR_CPLD33_REG     = 0x16 # RO

SPI_ADDR_SYNC_MEAS7_LSB = 0x17  # RO
SPI_ADDR_SYNC_MEAS7_MSB = 0x18  # RO

SPI_ADDR_CCAN_PLAS      = 0x19  # RO
#
# SPI DTSS
#
DTSS_SPI_XVERSION       = 0x7F
DTSS_SPI_TNUMBER        = 0x20
DTSS_SPI_NDURATION      = 0x23
DTSS_SPI_ANUMBER        = 0x24
DTSS_SPI_PULSE          = 0x2A
DTSS_SPI_MISC           = 0x30
DTSS_SPI_STATUS         = 0x31
DTSS_SPI_ON_ALIM        = 0x41
DTSS_SPI_FAN_PWM_DUTY   = 0x4F
DTSS_SPI_FREQUENCE_READ = 0x50
DTSS_SPI_FAN_PWM_DIV    = 0x51
#

REG_BDT_SEQ_SLV1        = 0x4   # slv1
REG_BDT_SEQ_MISC        = 0x48  # slv18
REG_BDT_SEQ_SLV26       = 0x68  # slv26 (RW)
REG_BDT_SEQ_SLV27       = 0x6C  # slv27 (RO)
REG_BDT_SEQ_SLV28       = 0x70  # slv28 (RO)
REG_BDT_SEQ_SLV29       = 0x74  # slv29
REG_BDT_SEQ_SLV30       = 0x78  # slv30
REG_BDT_SEQ_VERSION     = 0x7C  # slv31

REG_BDT_IP_VERSION      = 0x300
REG_BDT_CTRL            = 0x304 # RW
REG_BDT_FIFO_RD_CNT     = 0x308 # RO
REG_BDT_FIFO_OUT        = 0x30C # RO
#
#**************************************
def i2c_table_parser(FileName, debug):
#**************************************

    i2c_detected_list = []

    with open(FileName,'r') as tmpfile:
        for line in tmpfile.readlines():
            col = 0
            if ':' in line:
                (addr_list, data_list) = line.split(': ')
                if debug:
                    print("* addr = ", addr_list, " ; data_list = ", data_list)

                var = data_list.split(' ')
                if debug:
                    print("     - var = ", var, " ; length = %d" %len(var))

                if len(var) > 17:
                    j = 0
                    var1 = []
                    while j < len(var):
                        if var[j] == '':
                            var1.append('  ')
                            j += 2
                        else:
                            var1.append(var[j])
                        j += 1

                    var = var1

                    if debug:
                        print("         > var1 = ", var1, " ; length = %d" %len(var1))
                        print("         > var  = ", var, " ; length = %d" %len(var))

                for elt in enumerate (var):
                    if elt[1] != '  ' and elt[1] != '--' and  elt[1] != '\n':
                        if debug:
                            print("     - elt[%d][%d]" %(int(addr_list), col), "= %s" %elt[1])
                            print("     - elt[%X][%x]" %(int(addr_list, base=16), col), "= %s" %elt[1])

                        i2c_addr = int(addr_list, base=16) + col
                        i2c_addr_h = "%x" %i2c_addr

                        i2c_detected_list.append(i2c_addr_h)
                    col += 1

    tmpfile.close()

    return i2c_detected_list

#************************
def get_glu_base_addr():
#************************

    debug = False

    FileName = "/usr/share/pydiag/oth/glu_base_addr_tmp.txt"

    os_base_cmd = "lspci -s 06:00.0 -v | grep size=128K"
    os_cmde = os_base_cmd+" > "+FileName

    if debug == True:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))

    with open(FileName,'r') as tmpfile:
        line = tmpfile.read()
    tmpfile.close()

    if debug:
        print("File %s: line is %s" %(FileName, line))

    tmp_list = line.split()

    if debug == True:
        print("tmp_list is" , tmp_list)

    base_addr = tmp_list[2]

    base_address = int(base_addr, base=16)

    if debug == True:
        print("base_address is 0x%x" %base_address)
        print("base_address is d'%d\n" %base_address)

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return base_address

#****************************
def get_pci_bdt_base_addr():
#****************************

    debug = False

    FileName = "/usr/share/pydiag/oth/bdt_base_addr_tmp.txt"

    os_base_cmd = "lspci -s 06:00.1 -v | grep size=64K"
    os_cmde = os_base_cmd+" > "+FileName

    if debug == True:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))

    with open(FileName,'r') as tmpfile:
        line = tmpfile.read()
    tmpfile.close()

    if debug:
        print("File %s: line is %s" %(FileName, line))

    tmp_list = line.split()

    if debug == True:
        print("tmp_list is" , tmp_list)

    base_addr = tmp_list[2]

    base_address = int(base_addr, base=16)

    if debug == True:
        print("base_address is 0x%x" %base_address)
        print("base_address is d'%d\n" %base_address)

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return base_address

#*********************************************
def get_reg_abs_addr(base_addr, addr_offset):
#*********************************************

    debug = False

    if debug:
        print("addr_offset is 0x%x" %addr_offset)

    reg_address = base_addr + addr_offset

    if debug:
        print("reg_address is 0x%x" %reg_address)

    return reg_address

#****************************
def get_reg_value(reg_addr):
#****************************

    debug = False

    var = str(reg_addr)
    FileName = "/usr/share/pydiag/oth/"+var+"_tmp.txt"

    os_base_cmd = "mem 0x%x" %reg_addr+" w"
    os_cmde = os_base_cmd+" > "+FileName

    if debug == True:
        print("os_base_cmd is", os_base_cmd)
        print("os_cmde is ", os_cmde)

    (os.system(os_cmde))

    with open(FileName,'r') as tmpfile:
        line = tmpfile.read()
    tmpfile.close()

    tmp_list = line.split()

    if debug == True:
        print("File %s: line is %s" %(FileName, line))
        print("tmp_list is" , tmp_list)

    reg_val = tmp_list[1]

    register_value = int(reg_val, base=16)

    if debug == True:
        print("register_value is 0x%x %d" %register_value)
        print("register_value is d'%d\n" %register_value)

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return register_value

#*************************************
def reg_write(reg_addr, data_value):
#*************************************

    debug = False

    var = "%x" %reg_addr

    FileName = "/usr/share/pydiag/oth/"+var+"_tmp.txt"

    if debug == True:
        print("reg_addr is 0x%x ; %d" %(reg_addr, reg_addr))
        print("data_value is 0x%x ; %d" %(data_value, data_value))

    os_base_cmd = "mem 0x%x" %reg_addr+" w "+"0x%x" %data_value
    os_cmde = os_base_cmd+" > "+FileName

    if debug == True:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))
    os_cmde = "rm -f "+FileName

    if debug == True:
        print("os_cmde is", os_cmde)

    (os.system(os_cmde))

    return 0

#*****************************************
def periph_reset_gen(flag, verbose_flag):
#*****************************************

    glu_reg_base_addr = get_glu_base_addr()

    if verbose_flag:
        print("glu_reg_base_addr = 0x%x" % glu_reg_base_addr)

    reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_RST_CTRL)
    if verbose_flag:
        print("reset_ctrl_value = 0x%x" % reset_ctrl_value)

#   if reset_ctrl_value == 0:
#       print("Periph reset allready set")
#   else:

    data_val =  0x0
    reg_write(glu_reg_base_addr + REG_RST_CTRL, data_val)

    if verbose_flag:
        print ("     * Enable reset")
    reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_RST_CTRL)

    if verbose_flag:
        print("         => reset_ctrl_value = 0x%x" % reset_ctrl_value)

    if flag == "release":
        if verbose_flag:
            print("     * Wait for 0.1s")
        time.sleep(0.1)

        if verbose_flag:
            print("     * Disable reset")
        data_val =  0x1
        reg_write(glu_reg_base_addr + REG_RST_CTRL, data_val)

        reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_RST_CTRL)

        if verbose_flag:
            print("         => reset_ctrl_value = 0x%x" % reset_ctrl_value)

    return True

#**************************************************************
def module_presence_info(OSB_flag, B200_flag, verbose_flag):
#**************************************************************

    glu_reg_base_addr = get_glu_base_addr()
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    if verbose_flag == 1 and OSB_flag == 1:
        module_presence_rdback_value = get_reg_value(glu_reg_base_addr + REG_MOD_PRESENCE_RDBCK)

        switch_bit        = module_presence_rdback_value & 0x1
        dtss_bit          = (module_presence_rdback_value & 0x2)/2
        dtss_fpga_rdy_bit = (module_presence_rdback_value & 0x4)/4

        print("* OSB module_presence_rdback register details :")
        print("     * module_presence_rdback_value read = 0x%x" % module_presence_rdback_value)
        print("         - module_presence_rdback(2) = DTSS_FPGA_RDY = %d ('0' = ABSENT ; '1' = present)" %dtss_fpga_rdy_bit)
        print("         - module_presence_rdback(1) = DTSS_PRES     = %d ('0' = ABSENT ; '1' = present)" %dtss_bit)
        print("         - module_presence_rdback(0) = SWITCH_PRES   = %d ('0' = ABSENT ; '1' = present)" %switch_bit, "\n")

    if verbose_flag == 1 and B200_flag == 1:
        value = Utsspi.read_mod_register8(SPI_ADDR_BOARD_PRES)

        switch_pres_b_bit = value & 0x1
        dtss_pres_b_bit   = (value & 0x2)/2
        toggle_bit        = (value & 0x4)/4
        board_id          = (value & 0xc0)/64

        print("* B200 module_presence_rdback register details :")
        print("     * module_presence_rdback read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_BOARD_PRES, value))
        print("         - BOARD_ID           = %d (2 = SWITCH ; 1 = DTSS)" %board_id)
        print("         - toggle_board_state = %d " %toggle_bit)
        print("         - DTSS_PRES_B        = %d ('0' = Selected ; '1' = DeSelected)" %dtss_pres_b_bit)
        print("         - SWITCH_PRES_B      = %d ('0' = Selected ; '1' = DeSelected)\n" %switch_pres_b_bit)

    return 0

#******************************************************
def b200_module_type_set(module_to_set, verbose_flag):
#******************************************************

    glu_reg_base_addr = get_glu_base_addr()

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    if verbose_flag:
        print("* B200 CPLD version is 0x%2x" % cpld_version)

    # Read spi from spi_dtss module

    module_presence_info(OSB_flag = 1, B200_flag = 1, verbose_flag = verbose_flag)

    if verbose_flag:
        mux_ctrl_value = get_reg_value(glu_reg_base_addr + REG_MUX_CTRL)

        print("* OSB mux_ctrl_value = 0x%x" % mux_ctrl_value)
        print("     - MUX_CTRL(0) = mux_sel = %d (0 : switch selected ; 1 : DTSS selected)" %(mux_ctrl_value & 0x1))
        print("     - MUX_CTRL(1) = mux_oe  = %d\n" %((mux_ctrl_value & 0x2)/2))

    value = Utsspi.read_mod_register8(SPI_ADDR_BOARD_PRES)

    switch_pres_b_bit = value & 0x1
    dtss_pres_b_bit   = (value & 0x2)/2
    board_id          = int(value & 0xc0)/64

    if module_to_set == "DTSS":
        if dtss_pres_b_bit == 0:
            if verbose_flag:
                print(" %s module (%d) allready selected : no action" %(module_to_set, board_id))
            return True
        else:
            if verbose_flag:
                print(" Select %s module (Apply a RESET) : " %module_to_set)
    else:
        if switch_pres_b_bit == 0:
            if verbose_flag:
                print("%s module (%d) allready selected : no action" %(module_to_set, board_id))
            return True
        else:
            if verbose_flag:
                print("Select %s module (Apply a RESET) : " %module_to_set)

    if verbose_flag:
        print ("\n NOW WRITE TO TOGGLE\n")

    wr_data = 0x0

    if verbose_flag:
        print("     * Write 0x%x into B200 REGISTER_MOD (@ addr 0x%1x)\n" %(wr_data, SPI_ADDR_BOARD_PRES))

    ver = Utsspi.write_mod_register8(SPI_ADDR_BOARD_PRES, wr_data)

    module_presence_info(OSB_flag = 1, B200_flag = 0, verbose_flag = verbose_flag)

    if verbose_flag:
        mux_ctrl_value = get_reg_value(glu_reg_base_addr + REG_MUX_CTRL)

        print("* mux_ctrl_value = 0x%x" % mux_ctrl_value)
        print("     - MUX_CTRL(0) = mux_sel = %d (0 : switch selected ; 1 : DTSS selected)" %(mux_ctrl_value & 0x1))
        print("     - MUX_CTRL(1) = mux_oe  = %d" %((mux_ctrl_value & 0x2)/2))

        # At this point, module has been switched

        print("\n******************************************")
        print("* At this point, module has been switched")
        print("******************************************\n")

    my_verbose = False
    periph_reset_gen("release", my_verbose)

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    if verbose_flag:
         print("* B200 CPLD version is 0x%2x\n" % cpld_version)

    module_presence_info(OSB_flag = 1, B200_flag = 1, verbose_flag = verbose_flag)

    return 0
#
#=============================================================================================
#
# find the correspondance of hwmon<x> drivers and fan positions, knowing that :
#
# lateral_fan1 is @ REAR position and is controlled by REG_SYS_FAN1_CTRL register
# lateral_fan0 is @ MIDDLE position and is controlled by REG_SYS_FAN2_CTRL register
# lateral_fan2 is @ FRONT position and is controlled by REG_SYS_FAN3_CTRL register
#
#=============================================================================================
#
#***********************
def OSB_fan_topology():
#***********************

    debug = False

    # os_cmde : ls -ltr /sys/class/hwmon | grep -i "bar1:fan"

    FileName = "/usr/share/pydiag/oth/fan_topology.txt"

    os_base_cmd = "ls -ltr /sys/class/hwmon | grep -i \"bar1:fan\""
    os_cmde = os_base_cmd+" > "+FileName

    if debug:
        print("\n * os_cmde is :", os_cmde, "\n")

    (os.system(os_cmde))

    tmp_list = []
    drv_vs_pos_list = []

    line_num = 1
    with open(FileName,'r') as tmpfile:
        for line in tmpfile.readlines():
            #print("****** line is ", line)

            tmp_list = line.split()

            if debug == True:
                print(" * line_num = %d :" %line_num)
                print(" ==================\n")
                print("   => tmp_list is", tmp_list)

            hwmon_flag = False
            fan_nb_flag = False

            for elt in enumerate(tmp_list):
                if 'hwmon' in elt[1]:
                    if '/' not in elt[1]:
                        if debug:
                            print("\n   * elt is ", elt, ": hwmon string found")
                        hmon_str = elt[1]
                        hwmon_flag = True
                    else:
                        if 'bar1:fan' in elt[1]:
                            my_list1 = elt[1].split('/')
                            if (debug):
                                print("\n   * my_list1 is", my_list1)
                            for el in enumerate(my_list1):
                                if 'bar1:fan' in el[1]:
                                    my_list2 = el[1].split(':')
                                    if (debug):
                                        print("   * my_list2 = ", my_list2)
                                    for e in enumerate(my_list2):
                                        if 'fan' in e[1]:
                                            fan_nb = e[1]
                                            if (debug):
                                                print("   * fan_nb = ", fan_nb)
                                            fan_nb_flag = True

                    if hwmon_flag == True and fan_nb_flag == True:
                        drv_vs_pos_list.append(hmon_str)
                        drv_vs_pos_list.append(fan_nb)

                        hwmon_flag = False
                        fan_nb_flag = False
            if debug:
                print("\n   * line_num = %d :" %line_num,"drv_vs_pos_list is", drv_vs_pos_list, "\n")

            line_num = line_num + 1

    tmpfile.close()

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return drv_vs_pos_list

#***********************
def OSB_fan_func():
#***********************

    debug = False

    # Test OSB_fan functionality
    #============================

    glu_reg_base_addr = get_glu_base_addr()
    #print("glu_reg_base_addr = 0x%x" % glu_reg_base_addr)

    psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)

    if debug == True:
        print("psu_ctrl_value = 0x%x\n" % psu_ctrl_value)

    #var = psu_ctrl_value and 0x200
    #print("var = %x" %var)

    if psu_ctrl_value and 0x200 == 0x0200:
        print("\n1. OSB_FAN power supply DISABLE")
    else:
        print("\n1. OSB_FAN power supply ENABLE : DISABLE it")

        data_val = psu_ctrl_value and 0x1FF
        reg_write(glu_reg_base_addr + REG_PSU_CTRL, data_val)

        psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)

        if debug == True:
            print("psu_ctrl_value = 0x%x" % psu_ctrl_value)

        time.sleep(3)   # wait for fan stabilization

    print(" * OSB fans tachi values @ STANDBY mode")

    osb_fan1_value1 = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_TACHI_VAL)
    print("     > osb_fan1_tachi_value = 0x%x" % osb_fan1_value1)

    osb_fan2_value1 = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_TACHI_VAL)
    print("     > osb_fan2_tachi_value = 0x%x\n" % osb_fan2_value1)

    print("2. Turn \"ON\" OSB_FAN power supply\n")

    psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)

    if debug == True:
        print("psu_ctrl_value = 0x%x" % psu_ctrl_value)

    data_val = psu_ctrl_value | 0x200
    reg_write(glu_reg_base_addr + REG_PSU_CTRL, data_val)

    time.sleep(3)

    print("3. Turn \"OFF\" OSB_FAN power supply")

    psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)
    data_val = psu_ctrl_value and 0x1ff
    reg_write(glu_reg_base_addr + REG_PSU_CTRL, data_val)

    print(" * OSB fans tachi values @ RUNNING mode")

    osb_fan1_value2 = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_TACHI_VAL)
    print("     > osb_fan1_tachi_value = 0x%x" % osb_fan1_value2)

    osb_fan2_value2 = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_TACHI_VAL)
    print("     > osb_fan2_tachi_value = 0x%x\n" % osb_fan2_value2)

    if osb_fan1_value1 == osb_fan1_value2:
        if osb_fan1_value2 == 0:
            print("OSB fan1 functionality : KO : fan absent")
        else:
            print("OSB fan1 functionality : KO")
    else:
        print("OSB fan1 functionality : OK")

    if osb_fan2_value1 == osb_fan2_value2:
        if osb_fan2_value2 == 0:
            print("OSB fan2 functionality : KO : fan absent")
        else:
            print("OSB fan2 functionality : KO")
    else:
        print("OSB fan2 functionality : OK")

    return None

#========================================================================================
#
#========================================================================================

#******************************************************
def spi_master_transfer(readNwrite, address, wr_data):
#******************************************************

    debug_flag = True

    if readNwrite == 1:
        print("readNwrite = Read")
        header = 0xf0
    else:
        print("readNwrite = Write")
        header = 0xe0

    print("address = 0x%x" %address)
    print("wr_data = 0x%x" %wr_data)

    # header trfer :
    #================

    glu_reg_base_addr = get_glu_base_addr()

    print("\n** Header : set OSB SPI Master SPI_TRANSMIT value = 0x%x" % header)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, (header * 65536))

    if debug_flag:
        header_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
        print("            => OSB SPI Master SPI_TRANSMIT reg. read value = 0x%x" % header_value)

    time.sleep(0.5)

    mod_spi_sel16 = 0x0
    mod_spi_cpol  = 0x0
    mod_spi_cpha  = 0x0
    mod_spi_freq  = 0x12

    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    if debug_flag:
        print("            Launch SPI transfer with OSB SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    # reg_addr trfer :
    #==================

    if debug_flag:
        input("\nAddress transfer : Enter")
        print("")

    reg_addr_value = address * 2

    print("** Register Address : Set OSB SPI Master SPI_TRANSMIT value = 0x%x" % reg_addr_value)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, ( reg_addr_value * 65536))

    if debug_flag:
        addr_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
        print("                   => OSB SPI Master SPI_TRANSMIT reg. read value = 0x%x" % addr_value)

    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    if debug_flag:
        print("                   Launch SPI transfer with OSB SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    # reg_addr_value = SPI_ADDR_VERSION

    if debug_flag:
        input("\nData transfer : Enter")
        print("")

    # data trfer :
    #==============

    print("** Write_data : OSB SPI Master SPI_TRANSMIT value = 0x%x" % wr_data)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, (wr_data * 65536))

    if debug_flag:
        data_wr_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
        print("             => OSB SPI Master SPI_TRANSMIT reg. read value = 0x%x" % data_wr_value)

    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    if debug_flag:
        print("             Launch SPI transfer with OSB SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    time.sleep(0.5)

    if readNwrite == 1:
        if debug_flag:
            print("\nOSB SPI Master SPI_TRANSMIT Receive value = 0x%x" % wr_data)

        reg_data_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
        return_value = reg_data_value & 0xff
        if debug_flag:
            print("             OSB SPI Master SPI_TRANSMIT read value = 0x%x" % return_value)

    if readNwrite == 1:
        return return_value
    else:
        return None

#========================================================================================
#
#========================================================================================

def oth_usb_topology():

    debug = False

    FileName = "/usr/share/pydiag/oth/usb_topology.txt"

    os_base_cmd = "ls /sys/bus/usb/devices/"

    os_cmde = os_base_cmd+" > "+FileName

    if debug:
        print("\n * os_cmde is :", os_cmde, "\n")

    (os.system(os_cmde))

    tmp_list = []
    usb_topology = []
    line_num = 1

    with open(FileName,'r') as tmpfile:
        for line in tmpfile.readlines():
            if debug:
                print("\n****** line is ", line)

            tmp_list = line.split()

            if debug:
                print(" * line_num = %d :" %line_num)
                print(" ==================\n")
                print("   => tmp_list is", tmp_list)

            for elt in tmp_list:
                fileidVendor = "/sys/bus/usb/devices/"+elt+"/idVendor"
                if(os.path.isfile(fileidVendor)):
                    msg = "idVendor file exist"
                    mfg = sysfs_read_string(fileidVendor)
                else:
                    msg = "idVendor file don't exist"
                    mfg = "False"

                fileidProduct = "/sys/bus/usb/devices/"+elt+"/idProduct"
                if(os.path.isfile(fileidProduct)):
                    msg = "idProduct file exist"
                    prod = sysfs_read_string(fileidProduct)
                else:
                    msg = "idProduct file don't exist"
                    prod = "False"

                if debug:
                    print("   elt is ", elt, ":", msg)
                    print("           mfg  = ", mfg)
                    print("           prod = ", prod)

                usb_topology.append(elt)
                usb_topology.append(mfg)
                usb_topology.append(prod)

                if debug:
                    print("\n***** usb_topology is ", usb_topology)

            line_num = line_num + 1

    tmpfile.close()

    os_cmde = "rm -f "+FileName

    if debug == True:
        print("\nos_cmde is", os_cmde)

    (os.system(os_cmde))

    return usb_topology

#
#********************************
def osb_uart_mod_init (verbose):
#********************************

    glu_reg_base_addr = get_glu_base_addr()

    if verbose:
        print("glu_reg_base_addr = 0x%x" %glu_reg_base_addr, "\n")

    # Verify Internal Optical module present : MOD_PRESENCE_RDBK reg

    # MOD_PRESENCE_RDBK(0) = SWITCH_PRESENCE ( value = 1 => Switch_module present ; value = 0 => Switch_module absent)
    # MOD_PRESENCE_RDBK(1) = DTSS_PRESENCE ( value = 1 => DTSS_module present   ; value = 0 => DTSS_module absent)
    # MOD_PRESENCE_RDBK(2) = DTSS_FPGA_READY  ( value = 1 => DTSS_FPGA present ?   ; value = 0 => DTSS_FPGA absent ?)

    UART_BASE_ADDR = UART_MOD_BASE_ADDR

    if verbose:
        print("UART_BASE_ADDR = 0x%x" % UART_BASE_ADDR, "\n")

    module_presence_rdback_value = get_reg_value(glu_reg_base_addr + REG_MOD_PRESENCE_RDBCK)
    switch_bit = module_presence_rdback_value & 0x1
    dtss_bit = (module_presence_rdback_value & 0x2)/2
    dtss_fpga_rdy_bit = (module_presence_rdback_value & 0x4)/4

    if verbose:
        print("* OSB module_presence_rdback register details :")
        print("  module_presence_rdback_value read = 0x%x" % module_presence_rdback_value)
        print("     * module_presence_rdback(2) = DTSS_FPGA_RDY = %d ('0' = ABSENT ; '1' = present)" %dtss_fpga_rdy_bit)
        print("     * module_presence_rdback(1) = DTSS_PRES     = %d ('0' = ABSENT ; '1' = present)" %dtss_bit)
        print("     * module_presence_rdback(0) = SWITCH_PRES   = %d ('0' = ABSENT ; '1' = present)\n" %switch_bit)

    # Read LCR register (0x100C) : verify LCR(7) = 0
    LCR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR))
    if verbose:
        print("Read LCR_value = 0x%x" % LCR_value, "@ addr = 0x%x (LCR reg.)" %UART_LCR)

    # LCR(7) = 0 : read @ 0x01000 addr i.e RBR register
    RBR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL))
    if verbose:
        print("Read RBR_value = 0x%x" % RBR_value, "@ addr = 0x%x (RBR_DLL reg.)" %UART_RBR_DLL)

    # LCR register (0x100C) : set LCR(7) = 1
    data_val =  0x80
    if verbose:
        print("Write = 0x%x" % data_val, " @ addr = 0x%x (LCR reg.)" %UART_LCR)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR), data_val)

    LCR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR))
    if verbose:
        print("Read LCR_value = 0x%x" % LCR_value, "@ addr = 0x%x (LCR reg.)" %UART_LCR)

    # LCR(7) = 1 : read @ 0x01000 addr i.e DLL register
    DLL_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL))
    if verbose:
        print("Read DLL_value = 0x%x" % DLL_value, "@ addr = 0x%x (RBR_DLL reg.)" %UART_RBR_DLL)

    # Program uart baudrate :

    data_val =  0x8b
    if verbose:
        print("Write = 0x%x" % data_val, "@ addr = 0x%x (LCR reg.)" %UART_RBR_DLL)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL), data_val)

    DLL_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL))
    if verbose:
        print("Read DLL_value = 0x%x" % DLL_value, "@ addr = 0x%x (RBR_DLL reg.)" %UART_RBR_DLL)

    # LCR(7) = 1 : read @ 0x01004 addr i.e DLM register
    DLM_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_DLM))
    if verbose:
        print("Read DLM_value = 0x%x" % DLM_value, "@ addr = 0x%x" %UART_DLM)

    data_val =  0x2
    if verbose:
        print("Write = 0x%x" % data_val, "@ addr = 0x%x (LCR reg.)" %UART_DLM)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_DLM), data_val)

    DLM_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_DLM))
    if verbose:
        print("Read DLM_value = 0x%x" % DLM_value, "@ addr = 0x%x" %UART_DLM)

    # Write 0x3 into LCR register (0x100C) : clear LCR(7)
    #
    # LCR(1:0) = WLS (Word Length Select) = b'11 => 8 bits/character
    # LCR(2) = STB (STop Bit) => 0 = 1 Stop bit

    data_val =  0x3
    if verbose:
        print("Write = 0x%x" % data_val, "@ addr = 0x%x (LCR reg.)" %UART_LCR)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR), data_val)

    LCR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR))
    if verbose:
        print("Read LCR_value = 0x%x" % LCR_value, "@ addr = 0x%x (LCR reg.)" %UART_LCR)

    # LCR(7) = 0 : write 0x1 @ 0x01008 addr i.e FCR register (Fifo Control register)
    data_val =  0x1
    if verbose:
        print("Write = 0x%x" % data_val, "@ addr = 0x%x (FCR reg.)" %UART_FCR)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_FCR), data_val)

    if verbose:
        print("**** Ready to launch Tx values" )

    return 0
#
#=============================================================================================
#
# Current version of OSB python diag
#
#=============================================================================================
#
@DIAG("OSB DIAG VERSION")
def t_osb_diag_ver(cmd):
    print("OSB DIAG version is", DIAG_VERSION)
    return True
#
#=============================================================================================
#
#
#=============================================================================================

@DIAG("Test PCIe")
def t_osb_pcie(cmd):

    npcied = 0
    if(os.path.isdir("/sys/bus/pci/devices/0000:06:00.0")):
        vendor = sysfs_read_string("/sys/bus/pci/devices/0000:06:00.0/vendor")
        device = sysfs_read_string("/sys/bus/pci/devices/0000:06:00.0/device")
        print_info("# Solution board FPGA Func 0 detected: %s %s"%(vendor,device))
        npcied = npcied + 1
    else:
        print_error("# Solution board FPGA Func 0 not detected")

    if(os.path.isdir("/sys/bus/pci/devices/0000:06:00.1")):
        vendor = sysfs_read_string("/sys/bus/pci/devices/0000:06:00.1/vendor")
        device = sysfs_read_string("/sys/bus/pci/devices/0000:06:00.1/device")
        print_info("# Solution board FPGA Func 1 detected: %s %s"%(vendor,device))
        npcied = npcied + 1
    else:
        print_error("# Solution board FPGA Func 1 not detected")

    if(npcied==2):
        print_ok("# PCIe: 2 devices detected")
        return True
    else:
        print_error("# PCIe only %d/2 devices detected"%(npcied))
        return False
#
#=============================================================================================
#
#
#=============================================================================================
#
@DIAG("Test USB")
def t_osb_usb(cmd):

    debug = False

    usb_topology_list = oth_usb_topology()

    i = 0
    driver_list = []
    mfg_list = []
    prod_list = []

    for elt in enumerate(usb_topology_list):
        if debug == True:
            print("\nelt_%d = " %i, elt)

        if i == 0 or i%3 == 0:
            driver_list.append(elt[1])
        elif i%3 == 1:
            mfg_list.append(elt[1])
        else:
            prod_list.append(elt[1])

        if debug:
            print("\ndriver_list is", driver_list)
            print("mfg_list    is", mfg_list)
            print("prod_list   is", prod_list)

        i += 1

    print("\n OTH USB topology :")

    i = 0
    while i < len(driver_list):
        driver_str = driver_list[i]
        mfg_str    = mfg_list[i]
        prod_str   = prod_list[i]
        print("     * i = %d :" %i, driver_str, mfg_str, prod_str)
        i += 1
    print("")

#======================================================================================

    nbusb = 0
    if(os.path.isdir("/sys/bus/usb/devices/1-1.1")):
        mfg = sysfs_read_string("/sys/bus/usb/devices/1-1.1/idVendor")
        prod = sysfs_read_string("/sys/bus/usb/devices/1-1.1/idProduct")
        print_info("# CPB USB HUB 2514: %s %s"%(mfg,prod))
        nbusb = nbusb + 1
    else:
        print_error("# CPB USB HUB 2514 not detected")

    if(os.path.isdir("/sys/bus/usb/devices/1-1.3")):
        mfg = sysfs_read_string("/sys/bus/usb/devices/1-1.3/idVendor")
        prod = sysfs_read_string("/sys/bus/usb/devices/1-1.3/idProduct")
        print_info("# USB A CPB FRONT PANEL: %s %s"%(mfg,prod))
        nbusb = nbusb + 1
    else:
        print_error("# USB A CPB FRONT PANEL not detected")

    # if(os.path.isdir("/sys/bus/usb/devices/1-1.3")):
    #     mfg = sysfs_read_string("/sys/bus/usb/devices/1-1.3/manufacturer")
    #     prod = sysfs_read_string("/sys/bus/usb/devices/1-1.3/product")
    #     print_info("# USB J6: %s %s"%(mfg,prod))
    #     nbusb = nbusb + 1
    # else:
    #     print_error("# USB J6 not detected")
    #
    # if(os.path.isdir("/sys/bus/usb/devices/1-1.4")):
    #     mfg = sysfs_read_string("/sys/bus/usb/devices/1-1.4/manufacturer")
    #     prod = sysfs_read_string("/sys/bus/usb/devices/1-1.4/product")
    #     print_info("# USB J5: %s %s"%(mfg,prod))
    #     nbusb = nbusb + 1
    # else:
    #     print_error("# USB J5 not detected")

    if(nbusb==2):
        print_ok("# USB: 2 devices detected")
        return True
    else:
        print_error("# USB only %d/2 devices detected"%(nbusb))
        return False
#
#=============================================================================================
#
@DIAG("OSB Ethernet test")
def t_osb_eth(cmd):
    try:
        (command, hostname) = cmd.split()
    except:
        print_error("# Command must be: t_osb_eth <hostname>")
        return False

    FileName = "/usr/share/pydiag/oth/ethernet_tmp.txt"

    # hostname = "10.33.17.16"
    # response = os.system("ping -I eth1 -c 1 " + hostname)

    os_cmde = "ping -c 1 " + hostname + "> " + FileName

    response = os.system(os_cmde)

    #print("reponse is %s" %response)

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    # and then check the response...
    if response == 0:
        pingstatus = "Network Active"
        print(pingstatus)
        return True
    else:
        pingstatus = "Network Error"
        print(pingstatus)
        return False
#
#=============================================================================================
#
@DIAG("OSB i2c devices detection")
def t_osb_i2cdetect(cmd):

    debug = False
    verbose = False

    try:
        if ' ' not in cmd:
            pass
        else:
            (command,consigne) = cmd.split()
            print("arg exist : %s" %consigne)
            if consigne == "verbose":
                verbose = True
            else:
                print_error("   Syntax error : arg must be : verbose\n")

    except:
        print_error("   Syntax error : Command must be : t_osb_i2cdetect <consigne>")
        print_error("                                                              ")
        return False

    osb_i2c_bus_nb = sysfs_read_string("/sys/platform/uts_scan/i2c_master_pw")

    i2c_to_detect_list = ['20', '50', '54', '71', '72', '74', '75']

    if debug:
        print("osb_i2c_bus_nb is ", osb_i2c_bus_nb)

    i2c_cmde = "i2cdetect -y "+osb_i2c_bus_nb

    if verbose:
        print("\n================")
        print("**** i2c-%s bus:" %osb_i2c_bus_nb)
        print("================\n")

        (os.system(i2c_cmde))
        print("")

    file_name = "/usr/share/pydiag/oth/osb_i2c_detect_tmp.txt"

    os_cmde = i2c_cmde+ " > "+file_name
    (os.system(os_cmde))

    osb_i2c_detected_list = i2c_table_parser(file_name, debug)

    if debug:
        print("\nosb_i2c_detected_list is ",osb_i2c_detected_list)

    os_cmde = "rm -f "+file_name
    (os.system(os_cmde))

    pass_flag = True

    for elt in enumerate (i2c_to_detect_list):
        if (elt[1]) in osb_i2c_detected_list:
            if verbose:
                print("I2C device at addr 0x%s" %elt[1], "detected")
        else:
            if verbose:
                print("I2C device at addr 0x%s" %elt[1], "not detected")
            pass_flag = False

    return pass_flag
#
#=============================================================================================
# OSB board has 2 EEPROM :
#
# * Legacy (ISU) EEPROM @ address 0x50
# * HETNET EEPROM @ address 0x54
#
# i2cdump -y -f <bus_nb> <i2c_address> c        -- c if for 'char' dump
#                                               -- -f if for 'force' : needed for HETNET EEPROM
#
#=============================================================================================
#
@DIAG("OSB i2c eeprom read")
def t_osb_eeprom_read(cmd):

    try:
        (command, eeprom_i2c_addr) = cmd.split()

        if eeprom_i2c_addr != "0x50" and eeprom_i2c_addr != "0x54":
            print_error("eeprom_i2c_addr : "+eeprom_i2c_addr+" is an invalid value : MUST be 0x50 or 0x54")
            return False

    except:
        print_error("   Syntax error : Command must be : t_osb_eeprom_read <eeprom_i2c_addr>")
        print_error("   where        <eeprom_i2c_addr> = 0x50 or 0x54")
        return False

    eeprom_i2c_bus_nb = sysfs_read_string("/sys/platform/uts_scan/i2c_master_pw")

    i2c_cmde = "i2cdump -y -f "+eeprom_i2c_bus_nb+" "+eeprom_i2c_addr+" c"

    print("i2c cmde is :", i2c_cmde)
    (os.system(i2c_cmde))

    return True
#
#=============================================================================================
#
#   FPGA_VER        = MSR=$(( 0x$BAR_1 + 0x01400 )) ; mem $MSR w
#   FPGA_SYNTH_DATE = MSR=$(( 0x$BAR_1 + 0x01404 )) ; mem $MSR w
#
#=============================================================================================
#
@DIAG("OSB FPGA VERSION")
def t_osb_fpga_ver(cmd):

    osb_fpga_version = sysfs_read_string("/sys/platform/osb_fpga/fpga_ver")
    print("OSB FPGA version is", osb_fpga_version)

    osb_fpga_synth_date = sysfs_read_string("/sys/platform/osb_fpga/fpga_synt_date")
    print("OSB FPGA synthesis date is", osb_fpga_synth_date)

    return True
#
#=============================================================================================
#
# Latteral SYS fans :
#
# t_osb_peripheral_fans REAR-PWM-255    SYS_FAN1_CTRL : MSR=$(( 0x$BAR_1 + 0x01418 )) ; mem $MSR w  => @reset 0x0009003A
# t_osb_peripheral_fans MIDDLE-PWM-255  SYS_FAN2_CTRL : MSR=$(( 0x$BAR_1 + 0x0141C )) ; mem $MSR w  => @reset 0x0009003A
# t_osb_peripheral_fans FRONT-PWM-255   SYS_FAN3_CTRL : MSR=$(( 0x$BAR_1 + 0x01420 )) ; mem $MSR w  => @reset 0x0009003A
#
# t_osb_peripheral_fans REAR-TACHI-0    SYS_FAN1_TACHI_VAL : MSR=$(( 0x$BAR_1 + 0x01440 )) ; mem $MSR w
# t_osb_peripheral_fans MIDDLE-TACHI-0  SYS_FAN2_TACHI_VAL : MSR=$(( 0x$BAR_1 + 0x01444 )) ; mem $MSR w
# t_osb_peripheral_fans FRONT-TACHI-0   SYS_FAN3_TACHI_VAL : MSR=$(( 0x$BAR_1 + 0x01448 )) ; mem $MSR w
#
#=============================================================================================
#
@DIAG("OSB lateral fans set using hwmon drivers")
def t_osb_periph_fans_set(cmd):

    debug = False

    try:
        (command,consigne) = cmd.split()

        if len(consigne.split('-')) == 2:
            consigne_lgth = 2
            (fan_position, access_type) = consigne.split('-')
        else:
            consigne_lgth = 3
            (fan_position, access_type, fan_pwm_value_str) = consigne.split('-')

        if debug == True:
            print("   command = ", command, " ; consigne = ", consigne)
            print("   fan_position = %s ; access_type = %s ; fan_pwm_value = %s" %(fan_position, access_type, fan_pwm_value_str))

    except:
        print_error("   Syntax error : Command must be : t_osb_periph_fans_set <consigne>")
        print_error("                                                                    ")
        print_error("   where <consigne> is : <FAN_POSITION>-<ACCESS_TYPE>-<PWM_VALUE>")
        print_error("   With                : <FAN_POSITION> = FRONT|MIDDLE|REAR|ALL")
        print_error("                         <ACCESS_TYPE>  = PWM|TACHI")
        print_error("                         <PWM_VALUE>    = {0 to 255}")
        return False

    error_flag = False

    if fan_position == "FRONT" or fan_position == "MIDDLE" or fan_position == "REAR" or fan_position == "ALL":
        print_info("fan_position is %s" %fan_position)
    else:
        print("fan_position : incorrect value : MUST be FRONT or MIDDLE or REAR or ALL")
        error_flag = True

    if access_type == "PWM" or access_type == "TACHI":
        print_info("access_type is %s" %access_type)
    else:
        print("fan access_type is undefined : must be PWM or TACHI")
        error_flag = True

    if access_type == "PWM":
        if consigne_lgth == 2:
            print_error("   * Error : PWM_VALUE : unknown value : MUST be 0 or 0 < value < 256")
            error_flag = True
        else:
            fan_pwm_value = int(fan_pwm_value_str)
            if fan_pwm_value == 0 or (fan_pwm_value > 0 and fan_pwm_value < 256):
                print_info("fan_pwm_value is %d" %fan_pwm_value)
            else:
                print("fan_pwm_value : incorrect value: MUST be 0 or 0 < value < 256")
                error_flag = True

    if error_flag:
        return False

    fan_topo_list = OSB_fan_topology()

    if len(fan_topo_list) == 0:
        print_error("Error : Fan topology CAN'T be extracted")
        return False

    if debug:
        print ("\nfan_topo_list is ", fan_topo_list)

    # enable_list : to validate this test for fan_0 (REAR) , fan_1 (MIDDLE), fan_2 (FRONT) ONLY :
    # enable_list = ["FAN_EN", "FAN_EN", "FAN_DIS","FAN_DIS","FAN_EN"]

    glu_reg_base_addr = get_glu_base_addr()

    driver_list = []
    fan_list = []
    position_list = []
    enable_list = []
    osb_fpga_reg_list = []
    osb_fpga_reg_str_list = []
    i = 0
    for elt in enumerate(fan_topo_list):
        if debug == True:
            print("elt_%d = " %i, elt)
        if i == 0 or i%2 == 0:
            driver_list.append(elt[1])
        else:
            fan_list.append(elt[1])
            if "fan0" in elt[1]:
                position_list.append("REAR")
                enable_list.append("FAN_EN")
                osb_fpga_reg_list.append(REG_SYS_FAN1_CTRL)
                osb_fpga_reg_str_list.append("REG_SYS_FAN1_CTRL")

            elif "fan1" in elt[1]:
                position_list.append("MIDDLE")
                enable_list.append("FAN_EN")
                osb_fpga_reg_list.append(REG_SYS_FAN2_CTRL)
                osb_fpga_reg_str_list.append("REG_SYS_FAN2_CTRL")

            elif "fan2" in elt[1]:
                position_list.append("FRONT")
                enable_list.append("FAN_EN")
                osb_fpga_reg_list.append(REG_SYS_FAN3_CTRL)
                osb_fpga_reg_str_list.append("REG_SYS_FAN3_CTRL")

            elif "fan4" in elt[1]:
                position_list.append("RIGHT")
                enable_list.append("FAN_DIS")
                osb_fpga_reg_list.append(REG_OSB_FAN1_CTRL)
                osb_fpga_reg_str_list.append("REG_OSB_FAN1_CTRL")
            else:
                position_list.append("LEFT")
                enable_list.append("FAN_DIS")
                osb_fpga_reg_list.append(REG_OSB_FAN1_CTRL)
                osb_fpga_reg_str_list.append("REG_OSB_FAN2_CTRL")

        if debug == True:
            print("driver_list is", driver_list)
            print("fan_list is", fan_list)
            print("position_list is", position_list)

        i += 1

    debug = False

    i = 0
    while i < 5:
        hwmon_val = driver_list[i]
        fan_num   = fan_list[i]
        position  = position_list[i]

        if debug:
            print("\nhwmon_val[%d] :" %i, hwmon_val)
            print("fan_num[%d] :" %i, fan_num)
            print("position[%d] :" %i, position)
            print("enable_list[%d] :" %i, enable_list[i])

        if enable_list[i] == 'FAN_EN' :

            if (position == fan_position or fan_position == 'ALL'):

                if access_type == "TACHI":
                    fan_tachi_cmde = "/sys/platform/osb_tachi/"+fan_num
                    if debug:
                        print("sys_cmde is :", "cat "+fan_tachi_cmde)

                    tachi_value = sysfs_read(fan_tachi_cmde)
                    print(" * ", fan_num, "fan tachi_value is 0x%x" %tachi_value)
                else:
                    fan_pwm_cmde = "/sys/class/hwmon/"+hwmon_val+"/pwm1"
                    print("     * sys_cmde is :", "echo "+fan_pwm_value_str+" > "+fan_pwm_cmde)
                    sysfs_write(fan_pwm_cmde, fan_pwm_value_str)
                    pwm_rdback = sysfs_read_string(fan_pwm_cmde)
                    pwm_value_reg = get_reg_value(glu_reg_base_addr + osb_fpga_reg_list[i])
                    print("         - %s readback value is" %hwmon_val, pwm_rdback)
                    print("         - %s (%s) pwm_value_reg value is 0x%x" %(fan_num, position, pwm_value_reg))
        i += 1

    return None
#
#=============================================================================================
#
#
#=============================================================================================
#
@DIAG("OSB functional fans test")
def t_osb_func_fans_test(cmd):

    debug = False
    error_flag = False

    try:
        (command, consigne) = cmd.split()

        if len(consigne.split('-')) == 2:
            consigne_lgth = 2
            (fan_localisation, verbose_flag) = consigne.split('-')
        else:
            consigne_lgth = 1
            fan_localisation = consigne

        if fan_localisation == "LATERAL" or fan_localisation == "FRONT_PANEL":
            print_info("fan_localisation is %s" %fan_localisation)
            if fan_localisation == "LATERAL":
                res_consigne = 6
            elif fan_localisation == "FRONT_PANEL":
                res_consigne = 4
            else:
                res_consigne = 10
        else:
            print_error("fan_localisation : incorrect value : MUST be LATERAL or FRONT_PANEL")
            error_flag = True

        if consigne_lgth == 2:
            if verbose_flag == "verbose":
                verbose = True
            else:
                print_error("   Syntax error : 2nd arg must be : verbose")
                error_flag = True
        else:
            verbose = False

        if error_flag:
            return False

        if debug == True:
            print("   command = ", command, " ; consigne = ", consigne)
            print("   fan_localisation = %s ; function_type = %s" %(fan_localisation, fct_type))

    except:
        print_error("   Syntax error : Command must be : t_osb_func_fans_test <consigne>")
        print_error("                                                                    ")
        print_error("   where <consigne> is : <FAN_LOCALISATION>")
        print_error("   With                : <FAN_LOCALISATION> = LATERAL|FRONT_PANEL")

        return False

    fan_topo_list = OSB_fan_topology()

    if verbose:
        print ("\nfan_topo_list is ", fan_topo_list, "\n")

    if len(fan_topo_list) == 0:
        print_error("Error : Fan topology CAN'T be extracted")
        return False

    print("")

    enable_list = []
    driver_list = []
    fan_list = []
    position_list = []
    osb_fpga_pwm_reg_list = []

    glu_reg_base_addr = get_glu_base_addr()

    if fan_localisation == "FRONT_PANEL":

        psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)
        psu_mask_FAN_12V = 0x200

        if (psu_ctrl_value  & psu_mask_FAN_12V) == psu_mask_FAN_12V:
            psu_currently_en = 'Yes'
            print("FAN_12V is currently enable (psu_ctrl_value = 0x%x" %psu_ctrl_value+")")
        else:
            psu_currently_en = 'No'
            print("FAN_12V need to be enable (psu_ctrl_value = 0x%x" %psu_ctrl_value+")")
            data_val = psu_ctrl_value | psu_mask_FAN_12V
            reg_write(glu_reg_base_addr + REG_PSU_CTRL, data_val)
            time.sleep(0.5)

    i = 0
    for elt in enumerate(fan_topo_list):
        if debug:
            print("elt_%d = " %i, elt)
        if i == 0 or i%2 == 0:
            driver_list.append(elt[1])
        else:
            fan_list.append(elt[1])
            if "fan0" in elt[1]:
                position_list.append("REAR")
                osb_fpga_pwm_reg_list.append(REG_SYS_FAN1_CTRL)
                if fan_localisation == "LATERAL":
                    enable_list.append("FAN_EN")
                else:
                    enable_list.append("FAN_DIS")
            elif "fan1" in elt[1]:
                position_list.append("MIDDLE")
                if fan_localisation == "LATERAL":
                    enable_list.append("FAN_EN")
                else:
                    enable_list.append("FAN_DIS")
                osb_fpga_pwm_reg_list.append(REG_SYS_FAN2_CTRL)
            elif "fan2" in elt[1]:
                position_list.append("FRONT")
                if fan_localisation == "LATERAL":
                    enable_list.append("FAN_EN")
                else:
                    enable_list.append("FAN_DIS")
                osb_fpga_pwm_reg_list.append(REG_SYS_FAN3_CTRL)
            elif "fan4" in elt[1]:
                position_list.append("RIGHT")
                if fan_localisation == "FRONT_PANEL":
                    enable_list.append("FAN_EN")
                else:
                    enable_list.append("FAN_DIS")
                osb_fpga_pwm_reg_list.append(REG_OSB_FAN2_CTRL)
            else:
                position_list.append("LEFT")
                if fan_localisation == "FRONT_PANEL":
                    enable_list.append("FAN_EN")
                else:
                    enable_list.append("FAN_DIS")
                osb_fpga_pwm_reg_list.append(REG_OSB_FAN1_CTRL)

        if debug:
            print("driver_list is", driver_list)
            print("fan_list is", fan_list)
            print("position_list is", position_list)

        i += 1

    res = 0

    temp_cmde = "/sys/class/thermal/thermal_zone0/emul_temp"
    temp_value_str = "50000"
    if verbose:
        print("sys_cmde is :", "echo "+temp_value_str+" > "+temp_cmde)
    sysfs_write(temp_cmde, temp_value_str)

    time.sleep(5)

    temp_cmde = "/sys/class/thermal/thermal_zone0/emul_temp"
    temp_value_str = "79000"
    if verbose:
        print("sys_cmde is :", "echo "+temp_value_str+" > "+temp_cmde)
    sysfs_write(temp_cmde, temp_value_str)

    time.sleep(3)

    i = 0
    while i < 5:
        hwmon_val = driver_list[i]
        fan_num   = fan_list[i]
        position  = position_list[i]

        if enable_list[i] == 'FAN_EN' :
            if debug:
                print("\nhwmon_val[%d] :" %i, hwmon_val)
                print("fan_num[%d]     :" %i, fan_num)
                print("position[%d]    :" %i, position)
                print("enable_list[%d] :" %i, enable_list[i])

            pwm_reg_value = get_reg_value(glu_reg_base_addr + osb_fpga_pwm_reg_list[i])
            speed_percent = pwm_reg_value & 0xff
            #print("     *", fan_num, "(", position, ")", "pwm_reg_value   = 0x%x (%d" %(pwm_reg_value, speed_percent), "%)")

            fan_tachi_cmde = "/sys/platform/osb_tachi/"+fan_num
            if verbose:
                print("sys_cmde is :", "cat "+fan_tachi_cmde)

            tachi_value = sysfs_read(fan_tachi_cmde)
            tachi_value_str = sysfs_read_string(fan_tachi_cmde)

            display_message = "     * "+fan_num+" ("+position+") pwm_speed = "+str(speed_percent)+"% : tachi_value is "+tachi_value_str+":"

            if tachi_value > 0x120:
                res += 1
                ret_message = display_message+" PASS"
                print_ok(ret_message)
            else:
                msg = " FAIL"
                ret_message = display_message+" FAIL"
                print_error(ret_message)

        i += 1

    temp_value_str = "50000"
    if verbose:
        print("sys_cmde is :", "echo "+temp_value_str+" > "+temp_cmde)
    sysfs_write(temp_cmde, temp_value_str)

    time.sleep(7)

    temp_value_str = "25000"
    if verbose:
        print("sys_cmde is :", "echo "+temp_value_str+" > "+temp_cmde)
    sysfs_write(temp_cmde, temp_value_str)

    time.sleep(7)

    temp_value_str = "10000"
    if verbose:
        print("sys_cmde is :", "echo "+temp_value_str+" > "+temp_cmde)
    sysfs_write(temp_cmde, temp_value_str)

    time.sleep(7)

    i = 0
    while i < 5:
        hwmon_val = driver_list[i]
        fan_num   = fan_list[i]
        position  = position_list[i]

        if enable_list[i] == 'FAN_EN' :

            pwm_reg_value = get_reg_value(glu_reg_base_addr + osb_fpga_pwm_reg_list[i])
            speed_percent = pwm_reg_value & 0xff
#           print("     *", fan_num, "(", position, ")", "pwm_reg_value   = 0x%x (%d" %(pwm_reg_value, speed_percent), "%)")

            fan_tachi_cmde = "/sys/platform/osb_tachi/"+fan_num
            if verbose:
                print("sys_cmde is :", "cat "+fan_tachi_cmde)

            tachi_value = sysfs_read(fan_tachi_cmde)
            tachi_value_str = sysfs_read_string(fan_tachi_cmde)

            display_message = "     * "+fan_num+" ("+position+") pwm_speed = "+str(speed_percent)+"% : tachi_value is "+tachi_value_str+":"

            if tachi_value > 0x50 and tachi_value < 0x120:
                res += 1
                ret_message = display_message+" PASS"
                print_ok(ret_message)
            else:
                ret_message = display_message+" FAIL"
                print_error(ret_message)

        i += 1

    # Set PWM value @ 25%

    fan_pwm_value_str = "64"
    i = 0
    while i < 5:
        fan_pwm_cmde = "/sys/class/hwmon/"+driver_list[i]+"/pwm1"
        if verbose:
            print("sys_cmde is :", "echo "+fan_pwm_value_str+" > "+fan_pwm_cmde)
        sysfs_write(fan_pwm_cmde, fan_pwm_value_str)

        i += 1

    if debug:
        print("res = ", res)

    if res == res_consigne:
        ret_flag = True
    else:
        ret_flag = False

    return ret_flag
#
#==================================================================================================
#
# OSB Fans at front panel :
#
# ENABLE_OSB_FAN_12V : PSU_CTRL(9) : MSR=$(( 0x$BAR_1 + 0x01408 )) ; mem $MSR w 0x2FF
#
# OSB_FAN1_CTRL : MSR=$(( 0x$BAR_1 + 0x01424 )) ; mem $MSR w  => @reset x0013001E <= Fan3 (LEFT)
# OSB_FAN2_CTRL : MSR=$(( 0x$BAR_1 + 0x01428 )) ; mem $MSR w  => @reset x00130028 <= Fan4 (RIGHT)
#
# OSB_FAN1_TACH : MSR=$(( 0x$BAR_1 + 0x0144C )) ; mem $MSR w
# OSB_FAN1_TACH : MSR=$(( 0x$BAR_1 + 0x01450 )) ; mem $MSR w
#
# root@afth-00005:# echo 255 > /sys/class/hwmon/hwmon9/pwm1
# root@afth-00005:# echo 255 > /sys/class/hwmon/hwmon10/pwm1*
#
# t_osb_fans LEFT-PWM-255
# t_osb_fans RIGHT-PWM-255
#
# t_osb_fans LEFT-TACHI
# t_osb_fans RIGHT-TACHI
#
#==================================================================================================
#
@DIAG("OSB front panel fans set using hwmon drivers")
def t_osb_front_panel_fans_set(cmd):

    debug = False

    try:
        (command,consigne) = cmd.split()

        if len(consigne.split('-')) == 2:
            consigne_lgth = 2
            (fan_position, access_type) = consigne.split('-')
        else:
            consigne_lgth = 3
            (fan_position, access_type, fan_pwm_value_str) = consigne.split('-')

        if debug == True:
            print("   command = ", command, " ; consigne = ", consigne)
            print("   fan_position = %s ; access_type = %s ; fan_pwm_value = %s" %(fan_position, access_type, fan_pwm_value_str))

    except:
        print_error("   Syntax error : Command must be : t_osb_front_panel_fans_set <consigne>")
        print_error("                                                                     ")
        print_error("   where <consigne> is : <FAN_POSITION>-<ACCESS_TYPE>-<PWM_VALUE>")
        print_error("   With                : <FAN_POSITION> = RIGHT|LEFT|ALL")
        print_error("                         <ACCESS_TYPE>  = PWM|TACHI")
        print_error("                         <PWM_VALUE>    = {0 to 255}")
        return False

    #print_info("# VERSION: 0.0.1")
    time.sleep(0.6)

    error_flag = False

    if fan_position == "RIGHT" or fan_position == "LEFT" or fan_position == "ALL":
        print("   * fan_position is %s" %fan_position)
    else:
        print("   * Error : undefined fan_position : MUST be RIGHT or LEFT or ALL")
        error_flag = True

    if access_type == "PWM" or access_type == "TACHI":
        print("   * access_type is ", access_type)
    else:
        print("   * Error : undefined access_type : must be PWM or TACHI")
        error_flag = True

    if access_type == "PWM":
        if consigne_lgth == 2:
            print_error("   * Error : PWM_VALUE : unknown value : MUST be 0 or 0 < value < 256")
            error_flag = True
        else:
            fan_pwm_value = int(fan_pwm_value_str)
            if (fan_pwm_value == 0 or (fan_pwm_value > 0 and fan_pwm_value < 256)):
                print("   * fan_pwm_value is %d" %fan_pwm_value)
            else:
                print("   * Error : fan_pwm_value : incorrect value : MUST be 0 or 0 < value < 256")
                error_flag = True

    if error_flag:
        return False

    fan_topo_list = OSB_fan_topology()

    if len(fan_topo_list) == 0:
        print_error("Error : Fan topology CAN'T be extracted")
        return False

    if debug:
        print ("\nfan_topo_list is ", fan_topo_list, "\n")

    # enable_list : to validate this test for fan_3 (LEFT) , fan_4 (RIGHT) ONLY :
    enable_list = []
    driver_list = []
    fan_list = []
    position_list = []

    i = 0
    for elt in enumerate(fan_topo_list):
        if debug == True:
            print("elt_%d = " %i, elt)
        if i == 0 or i%2 == 0:
            driver_list.append(elt[1])
        else:
            fan_list.append(elt[1])
            if "fan0" in elt[1]:
                position_list.append("REAR")
                enable_list.append("FAN_DIS")
            elif "fan1" in elt[1]:
                position_list.append("MIDDLE")
                enable_list.append("FAN_DIS")
            elif "fan2" in elt[1]:
                position_list.append("FRONT")
                enable_list.append("FAN_DIS")
            elif "fan4" in elt[1]:
                position_list.append("RIGHT")
                enable_list.append("FAN_EN")
            else:
                position_list.append("LEFT")
                enable_list.append("FAN_EN")

        if debug == True:
            print("driver_list is", driver_list)
            print("fan_list is", fan_list)
            print("position_list is", position_list)

        i += 1

    debug = False

    i = 0
    while i < 5:
        hwmon_val = driver_list[i]
        fan_num   = fan_list[i]
        position  = position_list[i]

        if debug:
            print("\nhwmon_val[%d] :" %i, hwmon_val)
            print("fan_num[%d] :" %i, fan_num)
            print("position[%d] :" %i, position)
            print("enable_list[%d] :" %i, enable_list[i])

        if enable_list[i] == 'FAN_EN' and (position == fan_position or fan_position == 'ALL'):

            if access_type == "TACHI":
                if fan_num == 'fann':
                    fan_num = 'fan3'
                fan_tachi_cmde = "/sys/platform/osb_tachi/"+fan_num
                if debug:
                    print("sys_cmde is :", "cat "+fan_tachi_cmde)

                tachi_value = sysfs_read(fan_tachi_cmde)
                print(" * ", fan_num, "("+position+")", "tachi_value is 0x%x" %tachi_value)
            else:
                fan_pwm_cmde = "/sys/class/hwmon/"+hwmon_val+"/pwm1"
                print("* sys_cmde is :", "echo "+fan_pwm_value_str+" > "+fan_pwm_cmde, "("+fan_num+" : "+position+")")
                sysfs_write(fan_pwm_cmde, fan_pwm_value_str)
                pwm_rdback = sysfs_read_string(fan_pwm_cmde)
                print("* hwmon readback is ", pwm_rdback)
        i += 1

    return None
#
#=============================================================================================
#
@DIAG("OSB Fan functionality test")
def t_osb_fan_func(cmd):

    OSB_fan_func()

    return None
#
#=============================================================================================
#
# See OSB_fan_topology() function
#
#=============================================================================================
#
@DIAG("OSB Fan topology function")
def t_osb_fan_topology(cmd):

    debug = True

    fan_topo_list = OSB_fan_topology()

    if debug:
        print ("\nfan_topo_list is ", fan_topo_list, " : length = ", len(fan_topo_list))
    else:
        print ("")

#   fan_list = ["fan0", "fan1", "fan2", "fann", "fan4"]
    fan_list = ["fan0", "fan1", "fan2", "fan3", "fan4"]
    fan_indice = []

    for elt in enumerate(fan_list):
        fan_name = elt[1]
        if debug:
            print("fan_name is", fan_name)
        for elt1 in enumerate(fan_topo_list):
            if elt1[1] == fan_name:
                fan_indice.append(elt1[0])
                if debug:
                    print("elt1 is", elt1, "fan_indice is ", fan_indice)

    if debug:
        print ("\nfan_indice is ", fan_indice, " : length = ", len(fan_indice))

    if len(fan_indice) == 0:
        print_error("Error : Fan topology CAN'T be extracted")
        return False
    else:
        i = 0
        for elt in enumerate(fan_list):
            fan_name = elt[1]
            hwmon_indice = fan_indice[i]
            hwmon = fan_topo_list[hwmon_indice-1]
            print("*", fan_name, " : hwmon is", hwmon)
            i += 1

    return None
#
#=============================================================================================
#
# PWM_value is the pourcentage (%) of the PWM speed : 100% means max speed
# ex : PWM_VALUE = 25 => 25% of max_speed
#
#=============================================================================================
#
@DIAG("OSB lateral fans Set using OSB_FPGA registers")
def t_osb_periph_fans_reg_set(cmd):

    debug = False

    try:
        (command, consigne) = cmd.split()

        if len(consigne.split('-')) == 2:
            consigne_lgth = 2
            (fan_position, access_type) = consigne.split('-')
        else:
            consigne_lgth = 3
            (fan_position, access_type, fan_pwm_value_str) = consigne.split('-')

        if debug:
            if len(consigne.split('-')) == 2:
                print("   command = ", command, " ; consigne = ", consigne)
                print("   fan_position = %s ; access_type = %s" %(fan_position, access_type))
            else:
                print("   command = ", command, " ; consigne = ", consigne)
                print("   fan_position = %s ; access_type = %s ; fan_pwm_value = %s" %(fan_position, access_type, fan_pwm_value_str))

    except:
        print_error("   Syntax error : Command must be : t_osb_periph_fans_reg_set <consigne>")
        print_error("                                                                            ")
        print_error("   where <consigne> is : <FAN_POSITION>-<ACCESS_TYPE>-<PWM_VALUE>")
        print_error("   with                : <FAN_POSITION> = FRONT|MIDDLE|REAR|ALL")
        print_error("                         <ACCESS_TYPE>  = PWM|TACHI")
        print_error("                         <PWM_VALUE>    = {0 to 100}")
        return False

    error_flag = False

    position_list = ["FRONT", "MIDDLE", "REAR", "ALL"]
    access_list = ["PWM", "TACHI"]

    if fan_position not in position_list:
        print_error("   * Error : undefined FAN_POSITION : MUST be FRONT or MIDDLE or REAR or ALL")
        error_flag = True
    else:
        print_info("   * fan_position is %s" %fan_position)

    if access_type not in access_list:
        print_error("   * Error : undefined ACCESS_TYPE : must be PWM or TACHI")
        error_flag = True
    else:
        print_info("   * access_type is %s" %access_type)

    if access_type == "PWM":
        if consigne_lgth == 2:
            print_error("   * Error : PWM_VALUE : unknown value : MUST be 0 or 0 < value < 101")
            error_flag = True
        else:
            fan_pwm_value = int(fan_pwm_value_str)
            if (fan_pwm_value == 0 or (fan_pwm_value > 0 and fan_pwm_value < 101)):
                print_info("   * fan_pwm_value is %d (0x%x)" %(fan_pwm_value, fan_pwm_value))
            else:
                print_error("   * Error : PWM_VALUE : incorrect value : MUST be 0 or 0 < value < 101")
                error_flag = True

    if error_flag:
        return False

    glu_reg_base_addr = get_glu_base_addr()
    if debug:
        print("glu_reg_base_addr = 0x%x" % glu_reg_base_addr)

    if access_type == "PWM":

        print("")
        pwm_duty_value = 0x90000 + fan_pwm_value
        if debug:
            print("pwm_duty_value = 0x%x" % pwm_duty_value)

        if fan_position == "REAR" or fan_position == "ALL":
            reg_write(glu_reg_base_addr + REG_SYS_FAN1_CTRL, pwm_duty_value)
            sys_fan1_ctrl_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_CTRL)
            print("sys_fan1_ctrl_reg value (REAR) = 0x%x" % sys_fan1_ctrl_value)

        if fan_position == "MIDDLE" or fan_position == "ALL":
            reg_write(glu_reg_base_addr + REG_SYS_FAN2_CTRL, pwm_duty_value)
            sys_fan2_ctrl_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_CTRL)
            print("sys_fan2_ctrl_reg value (MIDDLE) = 0x%x" % sys_fan2_ctrl_value)

        if fan_position == "FRONT" or fan_position == "ALL":
            reg_write(glu_reg_base_addr + REG_SYS_FAN3_CTRL, pwm_duty_value)
            sys_fan3_ctrl_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_CTRL)
            print("sys_fan3_ctrl_reg value (FRONT) = 0x%x" % sys_fan3_ctrl_value)

    else:
        print("")
        if fan_position == "REAR" or fan_position == "ALL":
            sys_fan1_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_TACHI_VAL)
            print("sys_fan1_tachi_reg value (REAR) = 0x%x" % sys_fan1_tachi_value)

        if fan_position == "MIDDLE" or fan_position == "ALL":
            sys_fan2_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_TACHI_VAL)
            print("sys_fan2_tachi_reg value (MIDDLE) = 0x%x" % sys_fan2_tachi_value)

        if fan_position == "FRONT" or fan_position == "ALL":
            sys_fan3_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_TACHI_VAL)
            print("sys_fan3_tachi_reg value (FRONT) = 0x%x" % sys_fan3_tachi_value)

    return None
#
#=============================================================================================
#
# PWM_value is the pourcentage (%) of the PWM speed : 100% means max speed
# ex : PWM_VALUE = 25 => 25% of max_speed
#
#=============================================================================================
#
@DIAG("OSB front_panel fans Set using OSB_FPGA registers")
def t_osb_front_panel_fans_reg_set(cmd):

    debug = False

    try:
        (command, consigne) = cmd.split()

        if len(consigne.split('-')) == 2:
            consigne_lgth = 2
            (fan_position, access_type) = consigne.split('-')
        else:
            consigne_lgth = 3
            (fan_position, access_type, fan_pwm_value_str) = consigne.split('-')

        if debug:
            if len(consigne.split('-')) == 2:
                print("   command = ", command, " ; consigne = ", consigne)
                print("   fan_position = %s ; access_type = %s" %(fan_position, access_type))
            else:
                print("   command = ", command, " ; consigne = ", consigne)
                print("   fan_position = %s ; access_type = %s ; fan_pwm_value = %s" %(fan_position, access_type, fan_pwm_value_str))

    except:
        print_error("   Syntax error : Command must be : t_osb_front_panel_fans_reg_set <consigne>")
        print_error("                                                                            ")
        print_error("   where <consigne> is : <FAN_POSITION>-<ACCESS_TYPE>-<PWM_VALUE>")
        print_error("   with                : <FAN_POSITION> = RIGHT|LEFT|ALL")
        print_error("                         <ACCESS_TYPE>  = PWM|TACHI")
        print_error("                         <PWM_VALUE>    = {0 to 100}")
        return False

    error_flag = False

    position_list = ["RIGHT", "LEFT", "ALL"]
    access_list = ["PWM", "TACHI"]

    if fan_position not in position_list:
        print_error("   * Error : undefined FAN_POSITION : MUST be RIGHT or LEFT or ALL")
        error_flag = True
    else:
        print_info("   * fan_position is %s" %fan_position)

    if access_type not in access_list:
        print_error("   * Error : undefined ACCESS_TYPE : must be PWM or TACHI")
        error_flag = True
    else:
        print_info("   * access_type is %s" %access_type)

    if access_type == "PWM":
        if consigne_lgth == 2:
            print_error("   * Error : PWM_VALUE : unknown value : MUST be 0 or 0 < value < 101")
            error_flag = True
        else:
            fan_pwm_value = int(fan_pwm_value_str)
            if (fan_pwm_value == 0 or (fan_pwm_value > 0 and fan_pwm_value < 101)):
                print_info("   * fan_pwm_value is %d (0x%x)" %(fan_pwm_value, fan_pwm_value))
            else:
                print_error("   * Error : PWM_VALUE : incorrect value : MUST be 0 or 0 < value < 101")
                error_flag = True

    if error_flag:
        return False

    glu_reg_base_addr = get_glu_base_addr()
    if debug:
        print("glu_reg_base_addr = 0x%x" % glu_reg_base_addr)

    psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)
    psu_mask_FAN_12V = 0x200

    if (psu_ctrl_value  & psu_mask_FAN_12V) == psu_mask_FAN_12V:
        psu_currently_en = 'Yes'
        print("FAN_12V is currently enable (psu_ctrl_value = 0x%x" %psu_ctrl_value+")")
    else:
        psu_currently_en = 'No'
        print("FAN_12V need to be enable (psu_ctrl_value = 0x%x" %psu_ctrl_value+")")
        data_val = psu_ctrl_value | psu_mask_FAN_12V
        reg_write(glu_reg_base_addr + REG_PSU_CTRL, data_val)
        time.sleep(0.5)

    if access_type == "PWM":

        print("")
        pwm_duty_value = 0x90000 + fan_pwm_value
        if debug:
            print("pwm_duty_value = 0x%x" % pwm_duty_value)

        if fan_position == "RIGHT" or fan_position == "ALL":
            reg_write(glu_reg_base_addr + REG_OSB_FAN2_CTRL, pwm_duty_value)
            osb_fan2_ctrl_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_CTRL)
            print("osb_fan2_ctrl_value value (RIGHT) = 0x%x" % osb_fan2_ctrl_value)

        if fan_position == "LEFT" or fan_position == "ALL":
            reg_write(glu_reg_base_addr + REG_OSB_FAN1_CTRL, pwm_duty_value)
            osb_fan1_ctrl_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_CTRL)
            print("osb_fan1_ctrl_value value (LEFT) = 0x%x" % osb_fan1_ctrl_value)

    else:
        print("")
        if fan_position == "RIGHT" or fan_position == "ALL":
            osb_fan2_tachi_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_TACHI_VAL)
            print("osb_fan2_tachi_reg value (RIGHT) = 0x%x" % osb_fan2_tachi_value)

        if fan_position == "LEFT" or fan_position == "ALL":
            osb_fan1_tachi_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_TACHI_VAL)
            print("osb_fan1_tachi_reg value (LEFT) = 0x%x" % osb_fan1_tachi_value)

    return None

#
#=============================================================================================
#
# PWM_value is the pourcentage (%) of the PWM max speed : i.e 100% means max speed
#   ex : PWM_VALUE = 25 => 25% of max_speed
#
#=============================================================================================
#
@DIAG("OSB fans test using OSB_FPGA registers")
def t_osb_fans_test(cmd):

    debug = False
    error_nb = 0

    side_list     =["LATERAL", "FRONT_PANEL"]
    position_list = ["FRONT", "MIDDLE", "REAR", "RIGHT", "LEFT", "ALL"]

    try:
        error_flag = False

        if len(cmd.split()) == 2:
            (command, consigne) = cmd.split()
            verbose_str = "desactivated"
            verbose = False
        else:
            (command, consigne, verbose_info) = cmd.split()
            if verbose_info == "verbose":
                verbose_str = "activated"
                verbose = True
            else:
                verbose_str = "not recognized (Must be verbose)"
                verbose = False

        (fan_info, precision) = consigne.split(';')
        (fan_side, fan_pwm_value_str) = fan_info.split('-')
        (min_value_str, max_value_str) = precision.split('_')

        min_value = int(min_value_str)
        max_value = int(max_value_str)

        if debug:
            print("   command = ", command, " ; consigne = ", consigne)
            print("   fan_side = %s ; fan_pwm_value = %s" %(fan_side, fan_pwm_value_str))

        if fan_side not in side_list:
            print_error("   * Error : undefined FAN_SIDE : MUST be LATERAL or FRONT_PANEL")
            error_flag = True
        else:
            print_info("   * FAN_SIDE is %s" %fan_side)

        fan_pwm_value = int(fan_pwm_value_str)

        if (fan_pwm_value == 0 or (fan_pwm_value > 0 and fan_pwm_value < 101)):
            print_info("   * PWM value is %d" %fan_pwm_value+"%")
        else:
            print_error("   * Error : PWM_VALUE : incorrect value : MUST be 0 or 0 < value < 101")
            error_flag = True

        if min_value > max_value:
            print_error("   * Error : min_value > max_value")
            error_flag = True
        else:
            print_info("   * TACHI value : min_value : %d ; max_value = %d" %(min_value, max_value))

        print_info("# Note : verbose mode is %s\n" %verbose_str)

    except:
        print_error("   Syntax error : Command must be : t_osb_fans_test <consigne>")
        print_error("                                                                            ")
        print_error("   where <consigne> is : <FAN_SIDE>-<PWM_VALUE>;<Min_value>_<Max_value>")
        print_error("   with                : <FAN_SIDE>  = LATERAL|FRONT_PANEL")
        print_error("                         <PWM_VALUE> = {0 to 100}%")
        print_error("                         <Min_value>_<Max_value> = resp. min and max values of")
        print_error("                                                   expected TACHI return value")
        return False

    if error_flag:
        return False

    os_cmde = "modprobe -r pwm-fan"
    (os.system(os_cmde))

    glu_reg_base_addr = get_glu_base_addr()

    fan_pwm_val = int((fan_pwm_value * 255)/100)
    pwm_duty_value = 0x90000 + fan_pwm_val

    if debug:
        print("pwm_duty_value = 0x%x" % pwm_duty_value)

    if fan_side == "FRONT_PANEL":
        psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)
        psu_mask_FAN_12V = 0x200

        if (psu_ctrl_value  & psu_mask_FAN_12V) == psu_mask_FAN_12V:
            if verbose:
                print("\n * Note : FRONT_PANEL fans test : ")
                print("            FAN_12V is currently enable (psu_ctrl_reg value = 0x%x)" %psu_ctrl_value)
        else:
            if verbose:
                print("\n * Note : FRONT_PANEL fans test : ")
                print("            FAN_12V need to be enable (psu_ctrl_reg value = 0x%x)" %psu_ctrl_value)
            data_val = psu_ctrl_value | psu_mask_FAN_12V
            reg_write(glu_reg_base_addr + REG_PSU_CTRL, data_val)
            time.sleep(0.5)

    if fan_pwm_value > 74:
        delay = 15
    elif fan_pwm_value > 49:
         delay = 10
    else:
        delay = 8

    if fan_side == "LATERAL":
        position_list = ["FRONT", "MIDDLE", "REAR"]
    else:
        position_list = ["RIGHT", "LEFT"]

    i = 0
    for elt in enumerate(position_list):

        if "RIGHT" in elt[1]:
            osb_fpga_pwm_reg = REG_OSB_FAN2_CTRL
            osb_fpga_tachi_reg = REG_OSB_FAN2_TACHI_VAL
            fan_ctrl_reg_str   = "osb_fan2_ctrl"
            fan_tachi_reg_str  = "osb_fan2_tachi"
        elif "LEFT" in elt[1]:
            osb_fpga_pwm_reg = REG_OSB_FAN1_CTRL
            osb_fpga_tachi_reg = REG_OSB_FAN1_TACHI_VAL
            fan_ctrl_reg_str   = "osb_fan1_ctrl"
            fan_tachi_reg_str  = "osb_fan1_tachi"
        elif "FRONT" in elt[1]:
            osb_fpga_pwm_reg = REG_SYS_FAN3_CTRL
            osb_fpga_tachi_reg = REG_SYS_FAN3_TACHI_VAL
            fan_ctrl_reg_str   = "sys_fan3_ctrl"
            fan_tachi_reg_str  = "sys_fan3_tachi"
        elif "MIDDLE" in elt[1]:
            osb_fpga_pwm_reg = REG_SYS_FAN2_CTRL
            osb_fpga_tachi_reg = REG_SYS_FAN2_TACHI_VAL
            fan_ctrl_reg_str   = "sys_fan2_ctrl"
            fan_tachi_reg_str  = "sys_fan2_tachi"
        else:
            osb_fpga_pwm_reg = REG_SYS_FAN1_CTRL
            osb_fpga_tachi_reg = REG_SYS_FAN1_TACHI_VAL
            fan_ctrl_reg_str   = "sys_fan1_ctrl"
            fan_tachi_reg_str  = "sys_fan1_tachi"

        # force PWM_value

        reg_write(glu_reg_base_addr + osb_fpga_pwm_reg, pwm_duty_value)
        fan_ctrl_value = get_reg_value(glu_reg_base_addr + osb_fpga_pwm_reg)
        if verbose:
            print("*** %s fan : " %elt[1])
            print("     * force PWM value = 0x%x into %s reg." %(fan_ctrl_value, fan_ctrl_reg_str))

        if verbose:
            print("     * Wait for %ds" %delay)
        time.sleep(delay)

        # Read TACHI value

        tachi_value = get_reg_value(glu_reg_base_addr + osb_fpga_tachi_reg)

        if verbose:
            print("     * %s fan : TACHI read value from %s reg. = 0x%x" %(elt[1], fan_tachi_reg_str, tachi_value))

        display_msg = "* "+ elt[1] + " fan : PWM value = " + str(fan_pwm_value) + "% : TACHI value = " + \
                      str(tachi_value) + " : "

        if (tachi_value >= min_value) and (tachi_value <= max_value):
            msg = "PASS"
            print_ok("%s %s" %(display_msg, msg))
        else:
            error_nb += 1
            msg = "FAIL"
            print_error("%s %s" %(display_msg, msg))

        # Reset PWM_value

        pwm_0_value = 0x90000 + 0x0

        reg_write(glu_reg_base_addr + osb_fpga_pwm_reg, pwm_0_value)
        fan_ctrl_value = get_reg_value(glu_reg_base_addr + osb_fpga_pwm_reg)
        if verbose:
            print("     * %s fan : reset PWM value = 0x%x into %s reg." %(elt[1], fan_ctrl_value, fan_ctrl_reg_str))
    i += 1

    if error_nb == 0:
        return True
    else:
        return False
#
#=============================================================================================
#
# Exist /sys/platform/osb_fpga/fpga_ver
# Exist /sys/platform/osb_fpga/fpga_synt_date
#
#=============================================================================================
#
@DIAG("OSB FPGA VERSION using FPGA register")
def t_osb_fpga_ver_reg(cmd):

    debug = False

    glu_reg_base_addr = get_glu_base_addr()
    if debug:
        print("glu_reg_base_addr = 0x%x" % glu_reg_base_addr)

    osb_fpga_version = get_reg_value(glu_reg_base_addr + REG_FPGA_VERSION)
    print("* OSB FPGA version = 0x%x" % osb_fpga_version)

    osb_fpga_synth_date = get_reg_value(glu_reg_base_addr + REG_FPGA_SYNTH_DATE)
    print("* OSB FPGA synthesis date = 0x%x" % osb_fpga_synth_date)

    return True
#
#=============================================================================================
#
# BAR_64K=$(lspci -s 06:00.0 -v | grep size=128K | cut -f 3 -d " ")
# BAR_1=$(echo $BAR_64K | awk '{print $1}');echo -n "BAR 1 : ";echo  $BAR_1
#
#=============================================================================================
#
@DIAG("OSB FPGA registers check")
def t_osb_reg_check(cmd):

    glu_reg_base_addr = get_glu_base_addr()
    print("glu_reg_base_addr = 0x%x" % glu_reg_base_addr)

    fpga_version_value = get_reg_value(glu_reg_base_addr + REG_FPGA_VERSION)
    print("fpga_version_value = 0x%x" % fpga_version_value)

    fpga_synth_date_value = get_reg_value(glu_reg_base_addr + REG_FPGA_SYNTH_DATE)
    print("fpga_synth_date_value = 0x%x" % fpga_synth_date_value)

    psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)
    print("psu_ctrl_value = 0x%x" % psu_ctrl_value)

    shifter_ctrl_value = get_reg_value(glu_reg_base_addr + REG_SHIFTER_CTRL)
    print("shifter_ctrl_value = 0x%x" % shifter_ctrl_value)

    mux_ctrl_value = get_reg_value(glu_reg_base_addr + REG_MUX_CTRL)
    print("mux_ctrl_value = 0x%x" % mux_ctrl_value)

    msel_value = get_reg_value(glu_reg_base_addr + REG_MSEL)
    print("msel_value = 0x%x" % msel_value)

    sys_fan1_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_CTRL)
    print("sys_fan1_value = 0x%x" % sys_fan1_value)

    sys_fan2_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_CTRL)
    print("sys_fan2_value = 0x%x" % sys_fan2_value)

    sys_fan3_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_CTRL)
    print("sys_fan3_value = 0x%x" % sys_fan3_value)

    sys_fan1_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_TACHI_VAL)
    sys_fan1_freq = float(sys_fan1_tachi_value)
    print("sys_fan1_tachi_value = 0x%x (freq = %fHz)" % (sys_fan1_tachi_value, sys_fan1_freq))

    sys_fan2_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_TACHI_VAL)
    sys_fan2_freq = float(sys_fan2_tachi_value)
    print("sys_fan2_tachi_value = 0x%x (freq = %5.3fHz)" % (sys_fan2_tachi_value, sys_fan2_freq))

    sys_fan3_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_TACHI_VAL)
    sys_fan3_freq = float(sys_fan3_tachi_value)
    print("sys_fan3_tachi_value = 0x%x (freq = %fHz)" % (sys_fan3_tachi_value, sys_fan3_freq))

    osb_fan1_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_CTRL)
    print("osb_fan1_value = 0x%x" % osb_fan1_value)

    osb_fan2_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_CTRL)
    print("osb_fan2_value = 0x%x" % osb_fan2_value)

    osb_fan1_tachi_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_TACHI_VAL)
    osb_fan1_freq = float(osb_fan1_tachi_value)
    print("osb_fan1_tachi_value = 0x%x (%fHz)" % (osb_fan1_tachi_value, osb_fan1_freq))

    osb_fan2_tachi_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_TACHI_VAL)
    osb_fan2_freq = float(osb_fan2_tachi_value)
    print("osb_fan2_tachi_value = 0x%x (%fHz)" % (osb_fan2_tachi_value, osb_fan2_freq))

    osb_pusb_value = get_reg_value(glu_reg_base_addr + REG_OSB_PUSHB_VAL)
    print("osb_pusb_value = 0x%x" % osb_pusb_value)

    psync_700K_value = get_reg_value(glu_reg_base_addr + REG_SYS_PSYNC_700K_MEAS)
    psync_700K_freq = float(psync_700K_value)
    print("psync_700K_value = 0x%x (freq = %fHz)" % (psync_700K_value, psync_700K_freq))

    module_fan_rdback_value = get_reg_value(glu_reg_base_addr + REG_MOD_FAN_RDBCK)
    print("module_fan_rdback_value = 0x%x" % module_fan_rdback_value)

    module_presence_rdback_value = get_reg_value(glu_reg_base_addr + REG_MOD_PRESENCE_RDBCK)
    print("module_presence_readback_value = 0x%x" % module_presence_rdback_value)

    reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_RST_CTRL)
    print("reset_ctrl_value = 0x%x" % reset_ctrl_value)

    return True
#
#=============================================================================================
#
@DIAG("OSB Periph reset test")
def t_osb_periph_reset_test(cmd):

    try:
        (command,consigne) = cmd.split()

    except:
        print_error("   Syntax error : Command must be : t_osb_periph_reset_test <consigne>")
        print_error("                                                                      ")
        print_error("   where            <consigne> is : <ENABLE|DISABLE>")
        return False

    if consigne == "ENABLE" or consigne == "DISABLE":
        print("   * consigne is %s\n" %consigne)
    else:
        print("   * Error : undefined consigne : MUST be ENABLE or DISABLE")
        return False

    glu_reg_base_addr = get_glu_base_addr()
    print("glu_reg_base_addr = 0x%x" % glu_reg_base_addr)

    reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_RST_CTRL)
    print("Current reset_ctrl_value = 0x%x" % reset_ctrl_value)

    if consigne == "ENABLE":
        if reset_ctrl_value == 0:
            print("\nPeriph reset allready Enable")
        else:
            data_val =  0x0
            reg_write(glu_reg_base_addr + REG_RST_CTRL, data_val)

            print ("\n* Enable reset")
            reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_RST_CTRL)
            print("     => reset_ctrl_value = 0x%x" % reset_ctrl_value)

    if consigne == "DISABLE":
        if reset_ctrl_value == 1:
            print("\nPeriph reset allready Disable")
        else:
            print ("\n* Disable reset")
            data_val =  0x1
            reg_write(glu_reg_base_addr + REG_RST_CTRL, data_val)

            reset_ctrl_value = get_reg_value(glu_reg_base_addr + REG_RST_CTRL)
            print("     => reset_ctrl_value = 0x%x" % reset_ctrl_value)

    return True
#
#=============================================================================================
#
#   OSB_FPGA_PSU_CTRL(0) = ENABLE_VEE_12V8_EXT
#   OSB_FPGA_PSU_CTRL(1) = ENABLE_VDD_12V8_EXT
#   OSB_FPGA_PSU_CTRL(2) = ENABLE_VDD_5V_EXT
#   OSB_FPGA_PSU_CTRL(3) = ENABLE_VDD_3V3_EXT
#   OSB_FPGA_PSU_CTRL(4) = ENABLE_VDD_1V8_EXT
#   OSB_FPGA_PSU_CTRL(5) = ENABLE_VDD_5V8_EXT
#   OSB_FPGA_PSU_CTRL(6) = ENABLE_VEE_5V8_EXT
#   OSB_FPGA_PSU_CTRL(7) = Unused
#   OSB_FPGA_PSU_CTRL(8) = ENABLE_RS485_PWR_FPGA
#   OSB_FPGA_PSU_CTRL(9) = ENABLE_OSB_FAN_12V
#
# See also /sys/platform/osb_fpga/fpga_pwr_ctl
#
#=============================================================================================
#
@DIAG("OSB PSU check")
def t_osb_psu_check(cmd):

    debug = False

    psu_name_list = ["VEE_12V8", "VDD_12V8", "VDD_5V", "VDD_3V3", "VDD_1V8", "VDD_5V8", "VEE_5V8",
                     "RS485_PWR", "FAN_12V"]

    psu_mask_list = [0x01, 0x02, 0x4, 0x8, 0x10, 0x20, 0x40, 0x100, 0x200]

    glu_reg_base_addr = get_glu_base_addr()

    psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)
    print("\n* psu_ctrl_value = 0x%x" % psu_ctrl_value)

    i = 0
    while i < len(psu_name_list):

        val = psu_ctrl_value & psu_mask_list[i]

        if val == psu_mask_list[i]:
            print_ok("%s : ENABLE" %psu_name_list[i])
        else:
            print_ok("%s : DISABLE" %psu_name_list[i])
        i += 1

    return None
#
#=============================================================================================
#
#=============================================================================================
#
@DIAG("OSB PSU SET")
def t_osb_psu_set(cmd):

    debug = False

    try:
        (command, consigne) = cmd.split()

    except:
        print_error("   Syntax error : Command must be : t_osb_psu_set <consigne>")
        print_error("                                                            ")
        print_error("   where            <consigne> is : <PSU_LIST>-<FLAG>")
        print_error("   with                <PSU_LIST> = VDD_12V8|VEE_12V8|VDD_5V8|VEE_5V8|")
        print_error("                                    VDD_5V|VDD_3V3|VDD_1V8|RS485_PWR|FAN_12V|ALL")
        print_error("                       <FLAG>     = ON|OFF")
        return False

    print_info("* consigne is "+consigne)
    (psu_list, en_flag) = consigne.split('-')
    print_info("     * psu_list is "+psu_list)
    print_info("     * en_flag  is "+en_flag)

    arg_list = psu_list.split('|')
    if debug:
        print("     * arg_list is ", arg_list,"\n")

    psu_name_list =     ["VEE_12V8", "VDD_12V8", "VDD_5V", "VDD_3V3", "VDD_1V8", "VDD_5V8", "VEE_5V8", "RS485_PWR", "FAN_12V"]
    psu_mask_list =     [ 0x01,       0x02,       0x4,      0x8,       0x10,      0x20,      0x40,      0x100,       0x200]
    psu_mask_list_bar = [0xffe,       0xffd,      0xffb,    0xff7,     0xfef,     0xfdf,     0xfbf,     0xeff,       0xdff]

    psu_todo_list =     ["NO",        "NO",       "NO",     "NO",      "NO",      "NO",      "NO",      "NO",        "NO"]

    for elt in enumerate(arg_list):
        if debug:
            print(elt)
        if elt[1] not in psu_name_list:
            if elt[1] != 'ALL':
                print("Error : ", elt[1], "is not a correct arg")
                return False

    if 'ALL' in arg_list:
        i = 0
        while i < len(psu_name_list):
            psu_todo_list[i] = "YES"
            i += 1
    else:
        i = 0
        for elt in enumerate(psu_name_list):
            if elt[1] in arg_list:
                psu_todo_list[i] = "YES"
            i += 1

    glu_reg_base_addr = get_glu_base_addr()

    i = 0
    while i < len(psu_name_list):
        #debug = True
        if debug:
            print("     * psu_name_list[%d] = "%i, psu_name_list[i], "psu_mask_list[%d] = 0x%x" %(i, psu_mask_list[i]), "psu_mask_list_bar[%d] = 0x%x" %(i,psu_mask_list_bar[i]))
            print("     * psu_todo_list[%d] = "%i, psu_todo_list[i])
        #debug = False

        if psu_todo_list[i] == 'YES':
            psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)

            if (psu_ctrl_value  & psu_mask_list[i]) == psu_mask_list[i]:
                psu_currently_en = 'Yes'
            else:
                psu_currently_en = 'No'

            to_reprogram = 'No'

            if en_flag == "ON":
                if psu_currently_en == 'No':
                    to_reprogram = 'Yes'
                    data_val = psu_ctrl_value | psu_mask_list[i]
                    msg = " need to be ENABLE (psu_reg_val = 0x%x)" %data_val
                else:
                    msg = " is currently ENABLE"
            else:
                if psu_currently_en == 'Yes':
                    to_reprogram = 'Yes'
                    data_val = psu_ctrl_value & psu_mask_list_bar[i]
                    msg = " need to be DISABLE (psu_reg_val = 0x%x)" %data_val
                else:
                    msg = " is currently DISABLE"

            print_ok("* %s" %psu_name_list[i]+msg)

            if to_reprogram == 'Yes':
                reg_write(glu_reg_base_addr + REG_PSU_CTRL, data_val)
        i += 1

    return True
#
#=============================================================================================
#
# VAUX_0  : K_VDD_12V8 = 8.25/108.25 = 0.07621    ; expected : 976mV
# VAUX_8  : K_VDD_5V8  = 1.78/11.78  = 0.15110    ; expected : 876mV
# VAUX_1  : K_VDD_5V   = 2.26/12.26  = 0.18434    ; expected : 922mV
# VAUX_9  : K_VDD_3V3  = 3.32/13.32  = 0.24925    ; expected : 823mV
# VAUX_2  : K_VDD_1V8  = 6.81/16.81  = 0.40512    ; expected : 729mV
# VAUX_10 : K_VEE_12V8 = 2.26/12.26  = 0.18434    ; expected : 861mV <- warning : partir de 4.68V au lieu de 12.8V
# VAUX_11 : K_VEE_5V8  = 2.26/12.26  = 0.18434    ; expected : 842mV <- warning : partir de 4.57V au lieu de 5.8V
#
#=============================================================================================
#
@DIAG("OSB PSU TEST")
def t_osb_psu_test(cmd):

    try:
        (command, consigne) = cmd.split()

    except:
        print_error("   Syntax error : Command must be : t_osb_psu_test <consigne>")
        print_error("                                                            ")
        print_error("   where            <consigne> is : <PSU_NAME>;<Min_value>;<Max_value>")
        print_error("   with                <PSU_LIST> = VDD_12V8|VEE_12V8|VDD_5V8|VEE_5V8|")
        print_error("                                    VDD_5V|VDD_3V3|VDD_1V8")
        #print_error("   And <Min_value>, <Max_value> MUST be decimal values")
        return False

    print_info("* consigne is "+consigne)
    (psu_name, min_value_str, max_value_str) = consigne.split(';')

    print_info("     * psu_name  is "+psu_name)
    print_info("     * min_value is "+min_value_str+"V")
    print_info("     * max_value is "+max_value_str+"V")

    min_value = float(min_value_str)
    max_value = float(max_value_str)
#   print(" min_value : %3.3fV ;  coeff_value = %3.3f" %(min_value, max_value))

    if min_value > max_value:
        print_error(" Error : min_value > max_value")
        return False

    psu_name_list = ["VDD_12V8", "VDD_5V", "VDD_1V8", "VEE_5V8", "VDD_5V8", "VDD_3V3", "VEE_12V8"]
    psu_mask_list = [0x02, 0x4, 0x10, 0x40, 0x20, 0x8, 0x01]

    xadc_aux_list = [XADC_VAUX_0_REG, XADC_VAUX_1_REG, XADC_VAUX_2_REG, XADC_VAUX_11_REG, XADC_VAUX_8_REG,
                     XADC_VAUX_9_REG, XADC_VAUX_10_REG]

    coeff_list = [0.07621, 0.18434, 0.40512, -0.18434*(4.57/5.8), 0.15110, 0.24925, -0.18434*(4.68/12.8)]

    i = 0
    for elt in enumerate(psu_name_list):
        if elt[1] == psu_name:
            xadc_aux_addr = xadc_aux_list[i]
            coeff_value = coeff_list[i]
            psu_mask = psu_mask_list[i]
        i += 1

    glu_reg_base_addr = get_glu_base_addr()

    # Enable PSU into PSU_CTRL register

#   psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)

#   data_val = psu_ctrl_value | psu_mask
#   reg_write(glu_reg_base_addr + REG_PSU_CTRL, data_val)

    # Read PSU_Value from XADC

    xadc_vaux_value = int(get_reg_value(glu_reg_base_addr + xadc_aux_addr)/16)
    vaux_value = float(xadc_vaux_value * 0.244)/1000

    psu_value = vaux_value / coeff_value

    print(" PSU value measured : %3.3fV" %psu_value)
    if psu_value > min_value and psu_value < max_value:
        return True
    else:
        return False

#
#=============================================================================================
#
# To test PSU when they are ENABLE
#
#=============================================================================================
#
@DIAG("OSB XADC TEST using /sys/platform drivers")
def t_osb_sys_xadc_test(cmd):

    res = 0

    R1 = 100000
    R2 = 8250
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux0")
    print(" * aux0 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VDD12V8 / AUX0 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux0", "tension VCC12V8",
                            vmin=int(CONST("mtsps.aux0.min", ((((12650 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux0.max", ((((12950 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD12V8. res = %d" %res)

    R1 = 10000
    R2 = 2260
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux1")
    print(" * aux1 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VDD5V0 / AUX1 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux1", "tension VCC5V",
                            vmin=int(CONST("mtsps.aux1.min", ((((4950 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux1.max", ((((5050 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD5V0. res = %d" %res)

    R1 = 10000
    R2 = 6810
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux2")
    print(" * aux2 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VCC1V8 / AUX2 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux2", "tension VCC1V8",
                            vmin=int(CONST("mtsps.aux2.min", ((((1750 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux2.max", ((((1850 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD1V8. res = %d" %res)

    R1 = 10000
    R2 = 1780
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux8")
    print(" * aux8 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VDD5V8 / AUX8 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux8", "tension VCC5V8",
                            vmin=int(CONST("mtsps.aux8.min", ((((5750 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux8.max", ((((5950 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD5V8. res = %d" %res)

    R1 = 10000
    R2 = 3320
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux9")
    print(" * aux9 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VDD3V3 / AUX9 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux9", "tension VCC3V3",
                            vmin=int(CONST("mtsps.aux9.min", ((((3250 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux9.max", ((((3380 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD3V3. res = %d" %res)

    R1 = 10000
    R2 = 2260
    R1P = 22600
    R2P = 8250
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux10")
    print(" * aux10 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    valn_mv = ((R2 + R1) * val_mv) / R2
    print("   VEE12V8 / AUX10 = -%d mV" % ( ((R1P * valn_mv) / R2P)))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux10", "tension VEE12V8",
                            vmin=int(CONST("mtsps.aux10.min",
                                           (((((R2P * (12600) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux10.max",
                                           (((((R2P * (12900) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VEE12V8. res = %d" %res)

    R1 = 10000
    R2 = 2260
    R1P = 22600
    R2P = 17800
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux11")
    print(" * aux11 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    valn_mv = ((R2 + R1) * val_mv) / R2
    print("   VEE5V8 / AUX11 = -%d mV" % ( (R1P * valn_mv) / R2P))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux11", "tension VEE5V8",
                            vmin=int(CONST("mtsps.aux11.min",
                                           (((((R2P * (5600) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux11.max",
                                           (((((R2P * (5900) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print ("     VEE5V8. res = %d" %res)

    if res == 7:
        return True
    else:
        return False
#
#=============================================================================================
#
# Same as t_osb_sys_xadc_test, but to test PSU when they are DISABLE
#
#=============================================================================================
#
@DIAG("OSB XADC TEST1 (/sys))")
def t_osb_sys_xadc1_test(cmd):

    res = 0

    R1 = 100000
    R2 = 8250
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux0")
    print(" * aux0 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VDD12V8 / AUX0 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux0", "tension VCC12V8",
                            vmin=int(CONST("mtsps.aux0.min", ((((-150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux0.max", ((((150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD12V8. res = %d" %res)

    R1 = 10000
    R2 = 2260
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux1")
    print(" * aux1 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VDD5V0 / AUX1 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux1", "tension VCC5V",
                            vmin=int(CONST("mtsps.aux1.min", ((((-150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux1.max", ((((150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD5V0. res = %d" %res)

    R1 = 10000
    R2 = 6810
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux2")
    print(" * aux2 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VDD1V8 / AUX2 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux2", "tension VCC1V8",
                            vmin=int(CONST("mtsps.aux2.min", ((((-150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux2.max", ((((150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD1V8. res = %d" %res)

    R1 = 10000
    R2 = 1780
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux8")
    print(" * aux8 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VDD5V8 / AUX8 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux8", "tension VCC5V8",
                            vmin=int(CONST("mtsps.aux8.min", ((((-150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux8.max", ((((150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD5V8. res = %d" %res)

    R1 = 10000
    R2 = 3320
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux9")
    print(" * aux9 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    print("   VDD3V3 / AUX9 = %d mV" % ( ((R2 + R1) * val_mv) / R2))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux9", "tension VCC3V3",
                            vmin=int(CONST("mtsps.aux9.min", ((((-150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux9.max", ((((150 * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VDD3V3. res = %d" %res)

    R1 = 10000
    R2 = 2260
    R1P = 22600
    R2P = 8250
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux10")
    print(" * aux10 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    valn_mv = ((R2 + R1) * val_mv) / R2
    print("   VEE12V8 / AUX10 = -%d mV" % ( ((R1P * valn_mv) / R2P)))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux10", "tension VEE12V8",
                            vmin=int(CONST("mtsps.aux10.min",
                                           (((((R2P * (-150) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux10.max",
                                           (((((R2P * (150) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print("     VEE12V8. res = %d" %res)

    R1 = 10000
    R2 = 2260
    R1P = 22600
    R2P = 17800
    val_hex = sysfs_read_string("/sys/platform/osb-xadc/aux11")
    print(" * aux11 : val_hex = ", val_hex, "; %d" % int(val_hex, 16))
    val = int(val_hex, 16) / 16
    val_mv = (val * 1000 / 4095)
    valn_mv = ((R2 + R1) * val_mv) / R2
    print("   VEE5V8 / AUX11 = -%d mV" % ( (R1P * valn_mv) / R2P))
    res = res + sysfs_check("/sys/platform/osb-xadc/aux11", "tension VEE5V8",
                            vmin=int(CONST("mtsps.aux11.min",
                                           (((((R2P * (-150) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)),
                            vmax=int(CONST("mtsps.aux11.max",
                                           (((((R2P * (150) / R1P) * R2) / (R1 + R2)) * 4095) / 1000) * 16)))
    print ("    VEE5V8 = %d" %res)

    if res == 7:
        return True
    else:
        return False
#
#=============================================================================================
#
#=============================================================================================
#
@DIAG("OSB UART test")
def t_osb_uart_test(cmd):

    debug = True

    try:
        (command,consigne) = cmd.split()

        if consigne != "MOD" and consigne != "RS485":
            print_error("Error : consigne undefined : MUST be MOD or RS485")
            return False

    except:
        print_error("   Syntax error : Command must be : t_osb_uart_test <consigne>")
        print_error("   where            <consigne> is : <MOD|RS485>")

        return False

    glu_reg_base_addr = get_glu_base_addr()
    if debug:
        print("glu_reg_base_addr = 0x%x" %glu_reg_base_addr, "\n")

    if consigne == "RS485":

        UART_BASE_ADDR = UART_RS485_BASE_ADDR

        # OSB_FPGA_PSU_CTRL(8) = ENABLE_RS485_PWR_FPGA Must be set to '1'

        psu_ctrl_value = get_reg_value(glu_reg_base_addr + REG_PSU_CTRL)
        if debug:
            print("psu_ctrl_value = 0x%x" %psu_ctrl_value)

        data_val = psu_ctrl_value | 0x100
        reg_write(glu_reg_base_addr + REG_PSU_CTRL, data_val)
        if debug:
            print("Write = 0x%x" %data_val, "@ addr = 0x%x" %REG_PSU_CTRL)
    else:
        # Verify Internal Optical module present : MOD_PRESENCE_RDBK reg

        # MOD_PRESENCE_RDBK(0) = SWITCH_PRESENCE ( value = 1 => Switch_module present ; value = 0 => Switch_module absent)
        # MOD_PRESENCE_RDBK(1) = DTSS_PRESENCE ( value = 1 => DTSS_module present   ; value = 0 => DTSS_module absent)
        # MOD_PRESENCE_RDBK(2) = DTSS_FPGA_READY  ( value = 1 => DTSS_FPGA present ?   ; value = 0 => DTSS_FPGA absent ?)

        UART_BASE_ADDR = UART_MOD_BASE_ADDR

        if debug:
            print("UART_BASE_ADDR = 0x%x" % UART_BASE_ADDR, "\n")

        module_presence_rdback_value = get_reg_value(glu_reg_base_addr + REG_MOD_PRESENCE_RDBCK)
        if debug:
            print("module_presence_rdback_value = 0x%x" % module_presence_rdback_value)

    # Read LCR register (0x100C) : verify LCR(7) = 0
    LCR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR))
    if debug:
        print("Read LCR_value = 0x%x" % LCR_value, "@ addr = 0x%x (LCR reg.)" %UART_LCR)

    # LCR(7) = 0 : read @ 0x01000 addr i.e RBR register
    RBR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL))
    if debug:
        print("Read RBR_value = 0x%x" % RBR_value, "@ addr = 0x%x (RBR_DLL reg.)" %UART_RBR_DLL)

    # LCR register (0x100C) : set LCR(7) = 1
    data_val =  0x80
    if debug:
        print("Write = 0x%x" % data_val, " @ addr = 0x%x (LCR reg.)" %UART_LCR)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR), data_val)

    LCR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR))
    if debug:
        print("Read LCR_value = 0x%x" % LCR_value, "@ addr = 0x%x (LCR reg.)" %UART_LCR)

    # LCR(7) = 1 : read @ 0x01000 addr i.e DLL register
    DLL_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL))
    if debug:
        print("Read DLL_value = 0x%x" % DLL_value, "@ addr = 0x%x (RBR_DLL reg.)" %UART_RBR_DLL)

    # Program uart baudrate :

    data_val =  0x8b
    if debug:
        print("Write = 0x%x" % data_val, "@ addr = 0x%x (LCR reg.)" %UART_RBR_DLL)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL), data_val)

    DLL_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL))
    if debug:
        print("Read DLL_value = 0x%x" % DLL_value, "@ addr = 0x%x (RBR_DLL reg.)" %UART_RBR_DLL)

    # LCR(7) = 1 : read à 0x01004 addr i.e DLM register
    DLM_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_DLM))
    if debug:
        print("Read DLM_value = 0x%x" % DLM_value, "@ addr = 0x%x" %UART_DLM)

    data_val =  0x2
    if debug:
        print("Write = 0x%x" % data_val, "@ addr = 0x%x (LCR reg.)" %UART_DLM)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_DLM), data_val)

    DLM_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_DLM))
    if debug:
        print("Read DLM_value = 0x%x" % DLM_value, "@ addr = 0x%x" %UART_DLM)

    # Write 0x3 into LCR register (0x100C) : clear LCR(7)
    #
    # LCR(1:0) = WLS (Word Length Select) = b'11 => 8 bits/character
    # LCR(2) = STB (STop Bit) => 0 = 1 Stop bit

    data_val =  0x3
    if debug:
        print("Write = 0x%x" % data_val, "@ addr = 0x%x (LCR reg.)" %UART_LCR)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR), data_val)

    LCR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR))
    if debug:
        print("Read LCR_value = 0x%x" % LCR_value, "@ addr = 0x%x (LCR reg.)" %UART_LCR)

    # LCR(7) = 0 : write 0x1 @ 0x01008 addr i.e FCR register (Fifo Control register)
    data_val =  0x1
    if debug:
        print("Write = 0x%x" % data_val, "@ addr = 0x%x (FCR reg.)" %UART_FCR)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_FCR), data_val)

    if debug:
        print("**** Launch Tx values :" )

    Tx_value = [0x1, 0x2, 0x4, 0x8, 0x10, 0x20, 0x40, 0x80]
    i = 0
    while i < len(Tx_value):
        reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL), Tx_value[i])
        if debug:
            print("Write = 0x%x" % Tx_value[i], "@ addr = 0x%x (RBR reg.)" %UART_RBR_DLL)

        if i == 0:
            time.sleep(10)
        else:
            time.sleep(2)
        i += 1

    time.sleep(2)

    Tx_value = 0x71
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL), Tx_value)
    if debug:
        print("Write = 0x%x" % Tx_value, "@ addr = 0x%x (RBR reg.)" %UART_RBR_DLL)

    return True
#
#=============================================================================================
#
#
#=============================================================================================
#
@DIAG("OSB-B200 UART test")
def t_osb200_uart_test(cmd):

    debug = False
    verbose = False

    glu_reg_base_addr = get_glu_base_addr()
    if debug:
        print("glu_reg_base_addr = 0x%x" %glu_reg_base_addr, "\n")

    my_verbose = False
    periph_reset_gen("release", my_verbose)

    UART_BASE_ADDR = UART_MOD_BASE_ADDR

    module_presence_rdback_value = get_reg_value(glu_reg_base_addr + REG_MOD_PRESENCE_RDBCK)
    switch_bit = module_presence_rdback_value & 0x1
    dtss_bit = (module_presence_rdback_value & 0x2)/2
    dtss_fpga_rdy_bit = (module_presence_rdback_value & 0x4)/4

    if debug:
        print("* OSB module_presence_rdback register details :")
        print("  module_presence_rdback_value read = 0x%x" % module_presence_rdback_value)
        print("     * module_presence_rdback(2) = DTSS_FPGA_RDY = %d ('0' = ABSENT ; '1' = present)" %dtss_fpga_rdy_bit)
        print("     * module_presence_rdback(1) = DTSS_PRES     = %d ('0' = ABSENT ; '1' = present)" %dtss_bit)
        print("     * module_presence_rdback(0) = SWITCH_PRES   = %d ('0' = ABSENT ; '1' = present)\n" %switch_bit, "\n")

    if debug:
        print("UART_BASE_ADDR = 0x%x" % UART_BASE_ADDR, "\n")

    # Use spi_switch

    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    value = Utsspi.read_mod_register8(SPI_ADDR_BOARD_PRES)
    switch_pres_b_bit = value & 0x1
    dtss_pres_b_bit   = (value & 0x2)/2
    toggle_bit        = (value & 0x4)/4
    board_id          = (value & 0x80)/128

    if debug:
        print("* B200 module_presence_rdback register details (after write) :")
        print("     * module_presence_rdback read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_BOARD_PRES, value))
        print("         - BOARD_ID           = %d (2 = SWITCH ; 1 = DTSS)" %board_id)
        print("         - toggle_board_state = %d " %toggle_bit)
        print("         - DTSS_PRES_B        = %d ('0' = Selected ; '1' = DeSelected)" %dtss_pres_b_bit)
        print("         - SWITCH_PRES_B      = %d ('0' = Selected ; '1' = DeSelected)\n" %switch_pres_b_bit)

    if board_id == 1:
        print ("* Run b200_module_type_set(SWITCH)")
        b200_module_type_set("SWITCH", verbose)

    b200_UART_reg = Utsspi.read_mod_register8(SPI_ADDR_UART)
    ctsn_bit  = b200_UART_reg & 0x1
    rtsn_bit  = (b200_UART_reg & 0x2)/2
    dtrn_bit  = (b200_UART_reg & 0x4)/4

    if debug:
        print("* B200_UART_reg at init = 0x%x" % b200_UART_reg, "@ addr = 0x%x" %SPI_ADDR_UART)
        print("     * bit_2 = OSB_UART_DTR_N     = %d" %dtrn_bit)
        print("     * bit_1 = OSB_UART_RTS_N     = %d" %rtsn_bit)
        print("     * bit_0 = OSB_UART_CTS_N_bit = %d\n" %ctsn_bit)

    # Modem Status Register (MSR)  : to read CTS = MSR(4)

    print("* CTSn test")

    MSR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_MSR))
    DCTS_bit = MSR_value & 0x1
    DDSR_bit = (MSR_value & 0x2)/2
    TERI_bit = (MSR_value & 0x4)/4
    DDCR_bit = (MSR_value & 0x8)/8
    CTS_bit  = (MSR_value & 0x10)/16
    DSR_bit  = (MSR_value & 0x20)/32

    if debug:
        print("\nRead OSB MSR_value at init = 0x%x" % MSR_value, "@ addr = 0x%x (MSR reg.)" %UART_MSR)
        print("     * DSR_bit  = %d" %DSR_bit)
        print("     * CTS_bit  = %d" %CTS_bit)
        print("     * DDCD_bit = %d" %DDCR_bit)
        print("     * TERI_bit = %d" %TERI_bit)
        print("     * DDSR_bit = %d" %DDSR_bit)
        print("     * DCTS_bit = %d\n" %DCTS_bit)

    wr_data = 0x1
    ver = Utsspi.write_mod_register8(SPI_ADDR_UART, wr_data)

    b200_UART_reg = Utsspi.read_mod_register8(SPI_ADDR_UART)
    ctsn_bit  = b200_UART_reg & 0x1
    rtsn_bit  = (b200_UART_reg & 0x2)/2
    dtrn_bit  = (b200_UART_reg & 0x4)/4

    if debug:
        print("* B200_UART_reg  after writing '1' = 0x%x" % b200_UART_reg, "@ addr = 0x%x" %SPI_ADDR_UART)
        print("     * bit_2 = OSB_UART_DTR_N     = %d" %dtrn_bit)
        print("     * bit_1 = OSB_UART_RTS_N     = %d" %rtsn_bit)
        print("     * bit_0 = OSB_UART_CTS_N_bit = %d\n" %ctsn_bit)

    MSR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_MSR))
    DCTS_bit = MSR_value & 0x1
    DDSR_bit = (MSR_value & 0x2)/2
    TERI_bit = (MSR_value & 0x4)/4
    DDCR_bit = (MSR_value & 0x8)/8
    CTS_bit  = (MSR_value & 0x10)/16
    DSR_bit  = (MSR_value & 0x20)/32

    if debug:
        print("Read OSB MSR_value after writing '1' = 0x%x" % MSR_value, "@ addr = 0x%x (MSR reg.)" %UART_MSR)
        print("     * DSR_bit  = %d" %DSR_bit)
        print("     * CTS_bit  = %d" %CTS_bit)
        print("     * DDCD_bit = %d" %DDCR_bit)
        print("     * TERI_bit = %d" %TERI_bit)
        print("     * DDSR_bit = %d" %DDSR_bit)
        print("     * DCTS_bit = %d\n" %DCTS_bit)

    wr_data = 0x0
    ver = Utsspi.write_mod_register8(SPI_ADDR_UART, wr_data)

    b200_UART_reg = Utsspi.read_mod_register8(SPI_ADDR_UART)
    ctsn_bit  = b200_UART_reg & 0x1
    rtsn_bit  = (b200_UART_reg & 0x2)/2
    dtrn_bit  = (b200_UART_reg & 0x4)/4

    if debug:
        print("* B200_UART_reg  after writing '0' = 0x%x" % b200_UART_reg, "@ addr = 0x%x" %SPI_ADDR_UART)
        print("     * bit_2 = OSB_UART_DTR_N     = %d" %dtrn_bit)
        print("     * bit_1 = OSB_UART_RTS_N     = %d" %rtsn_bit)
        print("     * bit_0 = OSB_UART_CTS_N_bit = %d\n" %ctsn_bit)

    MSR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_MSR))
    DCTS_bit = MSR_value & 0x1
    DDSR_bit = (MSR_value & 0x2)/2
    TERI_bit = (MSR_value & 0x4)/4
    DDCR_bit = (MSR_value & 0x8)/8
    CTS_bit  = (MSR_value & 0x10)/16
    DSR_bit  = (MSR_value & 0x20)/32

    if debug:
        print("Read OSB MSR_value after writing '0' = 0x%x" % MSR_value, "@ addr = 0x%x (LCR reg.)" %UART_MSR)
        print("     * DSR_bit  = %d" %DSR_bit)
        print("     * CTS_bit  = %d" %CTS_bit)
        print("     * DDCD_bit = %d" %DDCR_bit)
        print("     * TERI_bit = %d" %TERI_bit)
        print("     * DDSR_bit = %d" %DDSR_bit)
        print("     * DCTS_bit = %d" %DCTS_bit)

    # Modem Control Register (MCR) : to pilot RTS = MCR(0), DTR = MCR(1) bits

#   print("\n*=====================")
    print("* RTSn and DTRn test")
#   print("*=====================\n")

    MCR_value  = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_MCR))
    DTR_bit = MSR_value & 0x1
    RTS_bit = (MCR_value & 0x2)/2
    Out1_bit = (MCR_value & 0x4)/4
    Out2_bit = (MCR_value & 0x8)/8
    loop_bit  = (MCR_value & 0x10)/16

    if debug:
        print("Read OSB MCR_value = 0x%x" % MCR_value, "@ addr = 0x%x (MCR reg.)" %UART_MCR)
        print("     * bit_4 = loop_bit = %d" %loop_bit)
        print("     * bit_3 = Out2_bit = %d" %Out2_bit)
        print("     * bit_2 = Out1_bit = %d" %Out1_bit)
        print("     * bit_1 = RTS_bit  = %d" %RTS_bit)
        print("     * bit_0 = DTR_bit  = %d\n" %DTR_bit)

    b200_UART_reg = Utsspi.read_mod_register8(SPI_ADDR_UART)
    ctsn_bit  = b200_UART_reg & 0x1
    rtsn_bit  = (b200_UART_reg & 0x2)/2
    dtrn_bit  = (b200_UART_reg & 0x4)/4
    dtrn_re_bit  = (b200_UART_reg & 0x8)/8
    dtrn_fe_bit  = (b200_UART_reg & 0x10)/16

    if debug:
        print("* B200_UART_reg  after writing '0' = 0x%x" % b200_UART_reg, "@ addr = 0x%x" %SPI_ADDR_UART)
        print("     * bit_4 = OSB_UART_DTR_N_FE  = %d" %dtrn_fe_bit)
        print("     * bit_3 = OSB_UART_DTR_N_RE  = %d" %dtrn_re_bit)
        print("     * bit_2 = OSB_UART_DTR_N     = %d" %dtrn_bit)
        print("     * bit_1 = OSB_UART_RTS_N     = %d" %rtsn_bit)
        print("     * bit_0 = OSB_UART_CTS_N_bit = %d\n" %ctsn_bit)

    data_val =  0x3     # drive RTS_N and DTR_N low
    if debug:
        print("Write = 0x%x" % data_val, " @ addr = 0x%x (MCR reg.)\n" %UART_MCR)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_MCR), data_val)

    MCR_value  = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_MCR))
    DTR_bit = MSR_value & 0x1
    RTS_bit = (MCR_value & 0x2)/2
    Out1_bit = (MCR_value & 0x4)/4
    Out2_bit = (MCR_value & 0x8)/8
    loop_bit  = (MCR_value & 0x10)/16

    if debug:
        print("Read OSB MCR_value = 0x%x" % MCR_value, "@ addr = 0x%x (MCR reg.)" %UART_MCR)
        print("     * bit_4 = loop_bit = %d" %loop_bit)
        print("     * bit_3 = Out2_bit = %d" %Out2_bit)
        print("     * bit_2 = Out1_bit = %d" %Out1_bit)
        print("     * bit_1 = RTS_bit  = %d" %RTS_bit)
        print("     * bit_0 = DTR_bit  = %d\n" %DTR_bit)

    b200_UART_reg = Utsspi.read_mod_register8(SPI_ADDR_UART)
    ctsn_bit  = b200_UART_reg & 0x1
    rtsn_bit  = (b200_UART_reg & 0x2)/2
    dtrn_bit  = (b200_UART_reg & 0x4)/4
    dtrn_re_bit  = (b200_UART_reg & 0x8)/8
    dtrn_fe_bit  = (b200_UART_reg & 0x10)/16

    if debug:
        print("* B200_UART_reg  after writing '0' = 0x%x" % b200_UART_reg, "@ addr = 0x%x" %SPI_ADDR_UART)
        print("     * bit_4 = OSB_UART_DTR_N_FE  = %d" %dtrn_fe_bit)
        print("     * bit_3 = OSB_UART_DTR_N_RE  = %d" %dtrn_re_bit)
        print("     * bit_2 = OSB_UART_DTR_N     = %d" %dtrn_bit)
        print("     * bit_1 = OSB_UART_RTS_N     = %d" %rtsn_bit)
        print("     * bit_0 = OSB_UART_CTS_N_bit = %d\n" %ctsn_bit)

    data_val =  0x0     # drive RTS_N and DTR_N high
    if debug:
        print("Write = 0x%x" % data_val, " @ addr = 0x%x (MCR reg.)\n" %UART_MCR)
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_MCR), data_val)

    MCR_value  = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_MCR))
    DTR_bit = MSR_value & 0x1
    RTS_bit = (MCR_value & 0x2)/2
    Out1_bit = (MCR_value & 0x4)/4
    Out2_bit = (MCR_value & 0x8)/8
    loop_bit  = (MCR_value & 0x10)/16

    if debug:
        print("Read OSB MCR_value = 0x%x" % MCR_value, "@ addr = 0x%x (MCR reg.)" %UART_MCR)
        print("     * bit_4 = loop_bit = %d" %loop_bit)
        print("     * bit_3 = Out2_bit = %d" %Out2_bit)
        print("     * bit_2 = Out1_bit = %d" %Out1_bit)
        print("     * bit_1 = RTS_bit  = %d" %RTS_bit)
        print("     * bit_0 = DTR_bit  = %d\n" %DTR_bit)

    b200_UART_reg = Utsspi.read_mod_register8(SPI_ADDR_UART)
    ctsn_bit  = b200_UART_reg & 0x1
    rtsn_bit  = (b200_UART_reg & 0x2)/2
    dtrn_bit  = (b200_UART_reg & 0x4)/4
    dtrn_re_bit  = (b200_UART_reg & 0x8)/8
    dtrn_fe_bit  = (b200_UART_reg & 0x10)/16

    if debug:
        print("* B200_UART_reg  after writing '0' = 0x%x" % b200_UART_reg, "@ addr = 0x%x" %SPI_ADDR_UART)
        print("     * bit_4 = OSB_UART_DTR_N_FE  = %d" %dtrn_fe_bit)
        print("     * bit_3 = OSB_UART_DTR_N_RE  = %d" %dtrn_re_bit)
        print("     * bit_2 = OSB_UART_DTR_N     = %d" %dtrn_bit)
        print("     * bit_1 = OSB_UART_RTS_N     = %d" %rtsn_bit)
        print("     * bit_0 = OSB_UART_CTS_N_bit = %d\n" %ctsn_bit)

    my_verbose = False
    periph_reset_gen("release", my_verbose)

    ver = Utsspi.read_mod_register8(SPI_ADDR_VERSION)

    if debug:
        print("* SWITCH CPLD Version 0x%2x\n" % ver)

    b200_UART_reg = Utsspi.read_mod_register8(SPI_ADDR_UART)
    ctsn_bit  = b200_UART_reg & 0x1
    rtsn_bit  = (b200_UART_reg & 0x2)/2
    dtrn_bit  = (b200_UART_reg & 0x4)/4
    dtrn_re_bit  = (b200_UART_reg & 0x8)/8
    dtrn_fe_bit  = (b200_UART_reg & 0x10)/16

    if debug:
        print("* B200_UART_reg  after reset = 0x%x" % b200_UART_reg, "@ addr = 0x%x" %SPI_ADDR_UART)
        print("     * bit_4 = OSB_UART_DTR_N_FE  = %d" %dtrn_fe_bit)
        print("     * bit_3 = OSB_UART_DTR_N_RE  = %d" %dtrn_re_bit)
        print("     * bit_2 = OSB_UART_DTR_N     = %d" %dtrn_bit)
        print("     * bit_1 = OSB_UART_RTS_N     = %d" %rtsn_bit)
        print("     * bit_0 = OSB_UART_CTS_N_bit = %d\n" %ctsn_bit)

#   my_verbose = True
    my_verbose = False
    osb_uart_mod_init (my_verbose)

    b200_UART_reg = Utsspi.read_mod_register8(SPI_ADDR_UART)
    ctsn_bit  = b200_UART_reg & 0x1
    rtsn_bit  = (b200_UART_reg & 0x2)/2
    dtrn_bit  = (b200_UART_reg & 0x4)/4
    dtrn_re_bit  = (b200_UART_reg & 0x8)/8
    dtrn_fe_bit  = (b200_UART_reg & 0x10)/16

    if debug:
        print("* B200_UART_reg  after uart_mode_init = 0x%x" % b200_UART_reg, "@ addr = 0x%x" %SPI_ADDR_UART)
        print("     * bit_4 = OSB_UART_DTR_N_FE  = %d" %dtrn_fe_bit)
        print("     * bit_3 = OSB_UART_DTR_N_RE  = %d" %dtrn_re_bit)
        print("     * bit_2 = OSB_UART_DTR_N     = %d" %dtrn_bit)
        print("     * bit_1 = OSB_UART_RTS_N     = %d" %rtsn_bit)
        print("     * bit_0 = OSB_UART_CTS_N_bit = %d\n" %ctsn_bit)

    # Read LCR register (0x100C) : verify LCR(7) = 0
    LCR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR))
    if debug:
        print("Verify LCR(7) = 0 before sending Tx value")
        print(" LCR_value Read = 0x%x" % LCR_value, "@ addr = 0x%x (LCR reg.)" %UART_LCR)

    Tx_value = 0x71
    reg_write(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL), Tx_value)
    if debug:
        print("****** Write = 0x%x" % Tx_value, "@ addr = 0x%x (RBR reg.)" %UART_RBR_DLL)

    b200_UART_reg = Utsspi.read_mod_register8(SPI_ADDR_UART)
    ctsn_bit  = b200_UART_reg & 0x1
    rtsn_bit  = (b200_UART_reg & 0x2)/2
    dtrn_bit  = (b200_UART_reg & 0x4)/4
    dtrn_re_bit  = (b200_UART_reg & 0x8)/8
    dtrn_fe_bit  = (b200_UART_reg & 0x10)/16

    if debug:
        print("* B200_UART_reg  after reset = 0x%x" % b200_UART_reg, "@ addr = 0x%x" %SPI_ADDR_UART)
        print("     * bit_4 = OSB_UART_DTR_N_FE  = %d" %dtrn_fe_bit)
        print("     * bit_3 = OSB_UART_DTR_N_RE  = %d" %dtrn_re_bit)
        print("     * bit_2 = OSB_UART_DTR_N     = %d" %dtrn_bit)
        print("     * bit_1 = OSB_UART_RTS_N     = %d" %rtsn_bit)
        print("     * bit_0 = OSB_UART_CTS_N_bit = %d\n" %ctsn_bit)

    # Read LCR register (0x100C) : verify LCR(7) = 0
    LCR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_LCR))
    if debug:
        print("Read LCR_value = 0x%x" % LCR_value, "@ addr = 0x%x (LCR reg.)" %UART_LCR)

    # LCR(7) = 0 : read @ 0x01000 addr i.e RBR register (Receive Buffer Register)
    RBR_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_RBR_DLL))
    if debug:
        print("Read RBR_value = 0x%x" % RBR_value, "@ addr = 0x%x (RBR_DLL reg.)" %UART_RBR_DLL)

    # LCR(7) = 0 : read @ 0x01004 addr i.e IER register (Interrupt Enable register)

    IER_value = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_DLM))
    EDSSI_bit = (IER_value & 0x8)/8
    ELSI_bit  = (IER_value & 0x4)/4
    ETBEI_bit = (IER_value & 0x2)/2
    ERBFI_bit = IER_value & 0x1

    if debug:
        print("Read IER_value = 0x%x" % IER_value, "@ addr = 0x%x" %UART_DLM)
        print("     * bit_3 = EDSSI_bit = %d" %EDSSI_bit)
        print("     * bit_2 = ELSI_bit  = %d" %ELSI_bit)
        print("     * bit_1 = ETBEI     = %d" %ETBEI_bit)
        print("     * bit_0 = ERBFI_bit = %d\n" %ERBFI_bit)

    # LCR(7) = 0 : read @ 0x01008 addr i.e IIR register (Interrupt Identification Register)

    IIR_value  = get_reg_value(glu_reg_base_addr + (UART_BASE_ADDR + UART_FCR))
    INTPEND_bit    = IIR_value & 0x1
    INTID2_bit1to3 = (IIR_value & 0xe)/2
    FIFOEN_bit6to7 = (IIR_value & 0xc)/64

    if debug:
        print("Read IIR_value = 0x%x" % IIR_value, "@ addr = 0x%x" %UART_FCR)

    return True
#
#=============================================================================================
#
## FPGA XADC block address = 0x00_4000_2000
#
#=============================================================================================
#
@DIAG("OSB FPGA XADC On-Chip Temperature test")
def t_osb_temp_fpga_test(cmd):

    debug = False
    res = 0

    try:
        (command, consigne) = cmd.split()

    except:
        print_error("   Syntax error : Command must be : t_osb_xadc_temp_test <consigne>")
        print_error("                                                            ")
        print_error("   where            <consigne> is : <Min_value>;<Max_value>")
        return False

    print_info("* consigne is "+consigne)
    (min_value_str, max_value_str) = consigne.split(';')

    print_info("     * min_value is "+min_value_str+"C")
    print_info("     * max_value is "+max_value_str+"C")

    val_hex = sysfs_read_string("/sys/platform/osb-xadc/temperature")
    val = int(val_hex, 16) / 16
    val_temperature = (val * 504 / 4095) - 273

    min_value = int(min_value_str)
    max_value = int(max_value_str)
    if val_temperature == min_value or (val_temperature > min_value and val_temperature < (max_value+1)):
        msg= "PASS"
        res += 1
    else:
        msg= "FAIL"

    print("\n   * Using osb-xadc driver, XADC temperature = %d C : %s" %(val_temperature, msg))

    glu_reg_base_addr = get_glu_base_addr()

    if debug:
        print("glu_reg_base_addr = 0x%x" %glu_reg_base_addr, "\n")

    xadc_temp_value = int(get_reg_value(glu_reg_base_addr + XADC_TEMP_REG)/16)
    val_temperature_reg = (xadc_temp_value * 504 / 4095) - 273

    if val_temperature_reg == min_value or (val_temperature_reg > min_value and val_temperature_reg < (max_value+1)):
        msg= "PASS"
        res += 1
    else:
        msg= "FAIL"
    print("   * Using OSB_FPGA register, XADC temperature = %d C : %s" %(val_temperature_reg, msg))

    if res == 2:
        return True
    else:
        return False
#
#=============================================================================================
#
# FPGA JTAG registers :
#
# * REG_URJTAG_FREQ(31:0)
#
#   REG_URJTAG_IO :
#
#       REG_URJTAG_IO(0)    = TDI_MASK
#       REG_URJTAG_IO(1)    = TDI_VAL
#       REG_URJTAG_IO(2)    = TMS_MASK
#       REG_URJTAG_IO(3)    = TMS_VAL
#       REG_URJTAG_IO(31:4) = TCK_NBVAL
#
# Note : writing into REG_URJTAG_IO register generates ajtag_start (if jtag_busy = 0)
#
# * REG_URJTAG_STATUS :
#
#       RD_BCK = SLV2(31:2) & (busy or start) & tdo
#
# * REG_URJTAG_FREQ(31:0)
#
#=============================================================================================
#
@DIAG("OSB JTAG registers test")
def t_osb_jtag_reg_test(cmd):

    debug = True

    glu_reg_base_addr = get_glu_base_addr()
    if debug:
        print("glu_reg_base_addr = 0x%x" %glu_reg_base_addr, "\n")

    print("REG_URJTAG_FREQ register Test\n")

    wr_value = 0xffffffff
    reg_write(glu_reg_base_addr + REG_URJTAG_FREQ, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_FREQ)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    wr_value = 0x0
    reg_write(glu_reg_base_addr + REG_URJTAG_FREQ, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_FREQ)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    wr_value = 0xa5a55a5a
    reg_write(glu_reg_base_addr + REG_URJTAG_FREQ, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_FREQ)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    print("")
    #

    print("REG_URJTAG_IO register Test\n")

    wr_value = 0xffffffff
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_IO)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    wr_value = 0x0
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_IO)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    wr_value = 0x5a5aa5a5
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_IO)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    print("")
    #

    print("REG_URJTAG_DUMMY register Test\n")

    wr_value = 0xffffffff
    reg_write(glu_reg_base_addr + REG_URJTAG_DUMMY, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_DUMMY)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    wr_value = 0x0
    reg_write(glu_reg_base_addr + REG_URJTAG_DUMMY, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_IO)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    wr_value = 0x5a5aa5a5
    reg_write(glu_reg_base_addr + REG_URJTAG_DUMMY, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_DUMMY)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    print("")
    #

    print("REG_URJTAG_STATUS register Test\n")

    wr_value = 0x0
    reg_write(glu_reg_base_addr + REG_URJTAG_STATUS, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_STATUS)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    wr_value = 0xffffffff
    reg_write(glu_reg_base_addr + REG_URJTAG_STATUS, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_STATUS)
    print("rd_value = 0x%x (exp-data = 0x%x)" %(rd_value, wr_value))

    print("")

    return None
#
#=============================================================================================
#
# MUX_CTRL(0) = mux_sel = 0 : switch selected
# MUX_CTRL(0) = mux_sel = 1 : DTSS selected
# MUX_CTRL(1) = mux_oe = 0
#
# REG_URJTAG_IO wr_value = tck_nb | tms_val | tms_msk | tdi_val | tdi_msk
#=============================================================================================
#
@DIAG("OSB module board detection")
def t_osb_module_detect(cmd):

    debug = True

    glu_reg_base_addr = get_glu_base_addr()
    if debug:
        print("glu_reg_base_addr = 0x%x" %glu_reg_base_addr, "\n")

    # Verify REG_MUX_CTRL reg. correctly set :

    rd_value = get_reg_value(glu_reg_base_addr + REG_MUX_CTRL)
    mux_sel = rd_value & 0x1
    mux_oe  = (rd_value & 0x2)/2
    if mux_sel == 1:
        msg = "(DTSS)"
    else:
        msg = "(SWITCH)"

    print("REG_MUX_CTRL reg. rd_value = 0x%x => mux_sel = %d %s ; mux_oe = %d\n" %(rd_value, mux_sel, msg, mux_oe))

    if mux_oe == 1:
        print("mux_oe = %d => no module detected" %mux_oe)
        return False
    else:
        print("************************************")
        if mux_sel == 0:
            print("****** SWITCH module detected ******")
        else:
            print("****** DTSS module detected ******")
        print("************************************\n")
#
#=============================================================================================
#
@DIAG("OSB JTAG test")
def t_osb_jtag_test(cmd):

    debug = True

    glu_reg_base_addr = get_glu_base_addr()
    if debug:
        print("glu_reg_base_addr = 0x%x" %glu_reg_base_addr, "\n")

    # Verify REG_MUX_CTRL reg. correctly set :

    rd_value = get_reg_value(glu_reg_base_addr + REG_MUX_CTRL)
    mux_sel = rd_value & 0x1
    mux_oe  = (rd_value & 0x2)/2
    if mux_sel == 1:
        msg = "(DTSS)"
    else:
        msg = "(SWITCH)"

    print("REG_MUX_CTRL reg. rd_value = 0x%x => mux_sel = %d %s ; mux_oe = %d\n" %(rd_value, mux_sel, msg, mux_oe))

    if mux_oe == 1:
        print("mux_oe = %d => no module detected" %mux_oe)
        return False
    else:
        print("************************************")
        if mux_sel == 0:
            print("****** SWITCH module detected ******")
        else:
            print("****** DTSS module detected ******")
        print("************************************\n")

    # Verify busy = 0

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_STATUS)

    busy_or_start = (rd_value & 0x2)/2
    tdo = rd_value & 0x1

    if busy_or_start == 0:
        print("REG_URJTAG_STATUS reg. : rd_value = 0x%x : busy_or_start_bit = %d ; tdo = %d" %(rd_value, busy_or_start, tdo))
    else:
        print("Error (busy = 1) : REG_URJTAG_STATUS reg. : rd_value = 0x%x : busy_or_start_bit = %d ; tdo = %d" %(rd_value, busy_or_start, tdo))

    # Set jtag_tck frequency (period = 700ns)

    freq_value = 0x20
    reg_write(glu_reg_base_addr + REG_URJTAG_FREQ, freq_value)

    # lauch a jtag_start :

    tdi_msk_0 = 0x0 # TDI masked
    tdi_msk_1 = 0x1 # TDI unmasked
    tdi_val_0 = 0x0
    tdi_val_1 = 0x2
    tms_msk_0 = 0x0
    tms_msk_1 = 0x4
    tms_val_0 = 0x0
    tms_val_1 = 0x8

    # lauch 4 TCK pulses

    tck_nb    = 0x40

    wr_value = tck_nb | tms_val_0 | tms_msk_1 | tdi_val_0 | tdi_msk_1
    print("REG_URJTAG_IO reg. wr_value = 0x%x => jtag_start generated" %wr_value)
    print("     * tck_nb | tms_val_1 | tms_msk_1 | tdi_val_0 | tdi_msk_1")
    print("         TCK : %d pulses ; TMS = 0 (unmasked); TDI = 0 (unmasked)" %(tck_nb/16))

    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    rd_value = get_reg_value(glu_reg_base_addr + REG_URJTAG_STATUS)

    busy_or_start = (rd_value & 0x2)/2
    tdo = rd_value & 0x1

    if busy_or_start == 0:
        print("REG_URJTAG_STATUS reg. : rd_value = 0x%x : busy_or_start_bit = %d ; tdo = %d" %(rd_value, busy_or_start, tdo))
    else:
        print("Error (busy = 1) : REG_URJTAG_STATUS reg. : rd_value = 0x%x : busy_or_start_bit = %d ; tdo = %d" %(rd_value, busy_or_start, tdo))

    input("\nEnter")

    # lauch 16 TCK pulses

    tck_nb    = 0x100

    print("\n*************")
    print("* Toggle TDI")
    print("*************")

    wr_value = tck_nb | tms_val_0 | tms_msk_0 | tdi_val_1 | tdi_msk_1
    print("\nI.1. REG_URJTAG_IO reg : wr_value = 0x%x" %wr_value)
    print("     * tck_nb | tms_val_0 | tms_msk_0 | tdi_val_1 | tdi_msk_1")
    print("         TCK : %d pulses ; TMS = unchanged (masked); TDI toggle = 1 (toggle ; unmasked)" %(tck_nb/16))
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    input("\nEnter")

    tck_nb    = 0x20

    wr_value = tck_nb | tms_val_0 | tms_msk_0 | tdi_val_1 | tdi_msk_0
    print("\nI.2. REG_URJTAG_IO reg : wr_value = 0x%x" %wr_value)
    print("     * tck_nb | tms_val_0 | tms_msk_0 | tdi_val_1 | tdi_msk_0")
    print("         TCK : %d pulses ; TMS = unchanged (masked) ; TDI = unchanged (masked)" %(tck_nb/16))
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    input("\nEnter")

    tck_nb    = 0x100

    wr_value = tck_nb | tms_val_0 | tms_msk_0 | tdi_val_0 | tdi_msk_0
    print("\nI.3. REG_URJTAG_IO reg : wr_value = 0x%x" %wr_value)
    print("     * tck_nb | tms_val_0 | tms_msk_0 | tdi_val_0 | tdi_msk_0")
    print("         TCK : %d pulses ; TMS = unchanged (masked) ; TDI = unchanged (masked)" %(tck_nb/16))
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    input("\nEnter")

    wr_value = tck_nb | tms_val_0 | tms_msk_0 | tdi_val_0 | tdi_msk_1
    print("\nI.4. REG_URJTAG_IO reg : wr_value = 0x%x" %wr_value)
    print("     * tck_nb | tms_val_0 | tms_msk_0 | tdi_val_0 | tdi_msk_1")
    print("         TCK : %d pulses ; TMS = unchanged (masked) ; TDI = 1 (toggle ; unmasked)" %(tck_nb/16))
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    print("\n***********")
    print("Toggle TMS")
    print("***********")

    # lauch 8 TCK pulses

    tck_nb    = 0x080

    wr_value = tck_nb | tms_val_1 | tms_msk_1 | tdi_val_1 | tdi_msk_1
    print("\nII.1. REG_URJTAG_IO reg : wr_value = 0x%x" %wr_value)
    print("     * tck_nb | tms_val_1 | tms_msk_1 | tdi_val_1 | tdi_msk_1")
    print("         TCK : %d pulses ; TMS = 1 (unmasked) ; TDI = 1 (unmasked)" %(tck_nb/16))
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    input("\nEnter")

    wr_value = tck_nb | tms_val_0 | tms_msk_1 | tdi_val_0 | tdi_msk_0
    print("\nII.2. REG_URJTAG_IO reg : wr_value = 0x%x" %wr_value)
    print("     * tck_nb | tms_val_0 | tms_msk_1 | tdi_val_1 | tdi_msk_1")
    print("         TCK : %d pulses ; TMS = 0 (toggle ; unmasked) ; TDI = unchanged (masked)" %(tck_nb/16))
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    input("\nEnter")

    wr_value = tck_nb | tms_val_1 | tms_msk_0 | tdi_val_0 | tdi_msk_1
    print("\nII.3. REG_URJTAG_IO reg : wr_value = 0x%x" %wr_value)
    print("     * tck_nb | tms_val_1 | tms_msk_0 | tdi_val_1 | tdi_msk_1")
    print("         TCK : %d pulses ; TMS = unchanged (masked) ; TDI = 0 (toggle ; unmasked)" %(tck_nb/16))
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

    input("\nEnter")

    wr_value = tck_nb | tms_val_1 | tms_msk_1 | tdi_val_1 | tdi_msk_1
    print("\nII.4. REG_URJTAG_IO reg : wr_value = 0x%x" %wr_value)
    print("     * tck_nb | tms_val_1 | tms_msk_1 | tdi_val_1 | tdi_msk_1")
    print("         TCK : %d pulses ; TMS = 1 (toggle ; unmasked) ; TDI = 1 (toggle ; unmasked)" %(tck_nb/16))
    reg_write(glu_reg_base_addr + REG_URJTAG_IO, wr_value)

@DIAG("OSB sicutil init")
def t_osb_sicutil_init(cmd):

    try:
        (command) = cmd.split()

    except:
        print_error("# Command must be: t_osb_sicutil_init")
        return False

    Hetnet_Module = "4"
    fab_date = datetime.today().strftime('%Y%m%d')
    err = 0
    p = subprocess.Popen("sicutil -w -sys %s  HardwareSetName OSB" % (Hetnet_Module), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  HardwareSetId 703100B01-001" % (Hetnet_Module), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  Revision 2" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  SerialNumber 00001" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  ManufactureDate %s" % (Hetnet_Module, fab_date), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  CalibrationDate 20000101" % (Hetnet_Module), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  MinimumPower 1" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  CriticalTemp 85" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  PcieEndpoints 4" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -w -sys %s  ParentSerialNumber none" % (Hetnet_Module), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", end='', flush=True)
    p = subprocess.Popen("sicutil -r -sys %s" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    err += p.returncode;
    print(".", flush=True)
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
    return err == 10
#
#================================================================================
#
# t_osb_sicutil_r 4 all

#lines is [b'HardwareSetName\t: CPB\n', b'HardwareSetId  \t: 703100B11-001\n', b'Revision       \t: 0\n', b'SerialNumber   \t: EBAP00005\n', #b'ManufactureDate\t: 20210412\n', b'CalibrationDate\t: 20220110\n', b'MinimumPower   \t: 6\n', b'CriticalTemp   \t: 85\n', b'PcieEndpoints  \t: 3\n', #b'ParentSerialNumber\t: none\n']

#================================================================================
#
@DIAG("OSB sicutil read")
def t_osb_sicutil_r(cmd):
    try:
        (command, field) = cmd.split()
    except:
        print_error("# Command must be: t_osb_sicutil_r field/'all'")
        return False

    Hetnet_Module = "4"
    if (field == "all"):
        p = subprocess.Popen("sicutil -r -sys %s" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    else:
        print(field)
        p = subprocess.Popen("sicutil -r -sys %s %s" % (Hetnet_Module, field), shell=True, stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    lines = p.stdout.readlines()
    #print("err is %s; lines is %s" %(err,lines))
    l = False
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
    #print("l is", l)
    if l:
        return True
    else:
        return False

@DIAG("OSB sicutil write")
def t_osb_sicutil_w(cmd):
    try:
        (command, field, field_value) = cmd.split()
    except:
        print_error("# Command must be: t_osb_sicutil_w field field_value")
        print_error(
            "# Possible field are: # HardwareSetName  # HardwareSetId # Revision # SerialNumber # ManufactureDate # CalibrationDate # MinimumPower # CriticalTemp # PcieEndpoints # ParentSerialNumber")
        return False
    print(field)
    Hetnet_Module = "4"

    p = subprocess.Popen("sicutil -w -sys %s %s %s" % (Hetnet_Module, field, field_value), shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    print(err)
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
            return False
    return True

@DIAG("OSB sicutil solution read")
def t_osb_sicutil_sol_r(cmd):
    try:
        (command, field) = cmd.split()
    except:
        print_error("# Command must be: t_osb_sicutil_sol_r field/'all'")
        return False

    Hetnet_Module = "4"
    if (field == "all"):
        p = subprocess.Popen("sicutil -r -sol %s" % (Hetnet_Module), shell=True, stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    else:
        print(field)
        p = subprocess.Popen("sicutil -r -sol %s %s" % (Hetnet_Module, field), shell=True, stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    lines = p.stdout.readlines()
    #print("err is %s; lines is %s" %(err,lines))
    l = False
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
    # print(l)
    if l:
        return True
    else:
        return False

@DIAG("Check if i2c module eeprom is present at 0x52")
def t_osb_i2c_switch(cmd):
    try:
        (command) = cmd.split()
    except:
        print_error("# Command must be: t_osb_i2c_switch  ")
    f = open("/sys/platform/uts_scan/i2c_master_s3_pw", "r")
    i2c_bus = f.readline()
    f.close()
    i2c_bus = int(i2c_bus)
    print(i2c_bus)
    if i2c_bus <= 0:
        print("Bad i2c_bus detected %d in /sys/platform/uts_scan/i2c_master_s3_pw" )
        return False

    eeprom = smbus.SMBus(i2c_bus)
    address = 0x52
    try:
        val = eeprom.read_i2c_block_data(address, 0, 4)
        print("B200 SWITCH EEPROM detected")
        return True
    except:
        print("B200 SWITCH EEPROM Not detected")
        return False

@DIAG("Check if i2c module eeprom is present at 0x52")
def t_osb_i2c_dtss(cmd):
    try:
        (command) = cmd.split()
    except:
        print_error("# Command must be: t_osb_i2c_dtss  ")
    f = open("/sys/platform/uts_scan/i2c_master_s4_pw", "r")
    i2c_bus = f.readline()
    f.close()
    i2c_bus = int(i2c_bus)
    print(i2c_bus)
    if i2c_bus <= 0:
        print("Bad i2c_bus detected %d in /sys/platform/uts_scan/i2c_master_s4_pw" )
        return False

    eeprom = smbus.SMBus(i2c_bus)
    address = 0x52
    try:
        val = eeprom.read_i2c_block_data(address, 0, 4)
        print("B200 DTSS EEPROM detected")
        return True
    except:
        print("B200 DTSS EEPROM Not detected")
        return False


@DIAG("Read B200 SPI ")
def t_osb200_spi_test(cmd):
    try:
        (command, version) = cmd.split()
    except:
        print_error("# Command must be: t_osb200_spi_test version(hex)")
        return False

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    # Read VERSION reg at@0x00
    ver = Utsspi.read_mod_register8(0x00)
    print("B200 CPLD Version 0x%2x" % ver)

    if ver == int(version, 16):
        return True
    else:
        return False


@DIAG("Reset B200 CPLD")
def t_osb200_reset_gen(cmd):

    verbose = True
    try:
        (command, release_flag) = cmd.split()
    except:
        print_error("# Command must be: t_osb200_reset_gen <release_flag>")
        print_error("   where            <release_flag> is : release or no_release")
        return False

    print(" release_flag is", release_flag)
    periph_reset_gen(release_flag, verbose)

    return None

#
#===============================================================================================
#
#constant cAddress_Version          : std_logic_vector(6 downto 0) := "0000000";--"0"   -- RO
#constant cAddress_RegLvds_7_0      : std_logic_vector(6 downto 0) := "0000001";--"1"   -- R/W
#constant cAddress_RegLvds_15_8     : std_logic_vector(6 downto 0) := "0000010";--"2"   -- R/W
#constant cAddress_RegLvds_20_16    : std_logic_vector(6 downto 0) := "0000011";--"3"   -- R/W
#constant cAddress_Misc             : std_logic_vector(6 downto 0) := "0000100";--"4"   -- R/W
#constant cAddress_SyncMeas1LSB     : std_logic_vector(6 downto 0) := "0000101";--"5"   -- RO
#constant cAddress_SyncMeas1MSB     : std_logic_vector(6 downto 0) := "0000110";--"6"   -- RO
#constant cAddress_SyncMeas2LSB     : std_logic_vector(6 downto 0) := "0000111";--"7"   -- RO
#constant cAddress_SyncMeas2MSB     : std_logic_vector(6 downto 0) := "0001000";--"8"   -- RO
#constant cAddress_SyncMeas3LSB     : std_logic_vector(6 downto 0) := "0001001";--"9"   -- RO
#constant cAddress_SyncMeas3MSB     : std_logic_vector(6 downto 0) := "0001010";--"A"   -- RO
#constant cAddress_SyncMeas4LSB     : std_logic_vector(6 downto 0) := "0001011";--"B"   -- RO
#constant cAddress_SyncMeas4MSB     : std_logic_vector(6 downto 0) := "0001100";--"C"   -- RO
#constant cAddress_BoardPresence    : std_logic_vector(6 downto 0) := "0001101";--"D"   -- RO
#
#===============================================================================================
#
@DIAG("Read B200 registers (DTSS)")
def t_osb200_spi_reg_check(cmd):

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    # Read VERSION reg at@0x00
    ver = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    print("CPLD VERSION 0x%2x" % ver)

    addr = 0x1
    value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_7_0)
    print("reg. value @ 0x%1x (cAddress_RegLvds_7_0) : 0x%x" %(SPI_ADDR_REGLVDS_7_0, value))

    wr_data = 0x55
    ver = Utsspi.write_mod_register8(SPI_ADDR_REGLVDS_7_0, wr_data)

    value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_7_0)
    print("reg. value after write @ 0x%1x (cAddress_RegLvds_7_0) : 0x%x" %(SPI_ADDR_REGLVDS_7_0, value))

    addr = 0x2
    value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_15_8)
    print("reg. value @ 0x%1x (cAddress_RegLvds_15_8) : 0x%x" %(SPI_ADDR_REGLVDS_15_8, value))

    wr_data = 0xaa
    ver = Utsspi.write_mod_register8(SPI_ADDR_REGLVDS_15_8, wr_data)

    value = Utsspi.read_mod_register8(addr)
    print("reg. value after write @ 0x%1x (cAddress_RegLvds_15_8) : 0x%x" %(SPI_ADDR_REGLVDS_15_8, value))

    addr = 0x3
    value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_20_16)
    print("reg. value @ 0x%1x (cAddress_RegLvds_20_16) : 0x%x" %(SPI_ADDR_REGLVDS_20_16, value))

    wr_data = 0x3f
    ver = Utsspi.write_mod_register8(SPI_ADDR_REGLVDS_20_16, wr_data)

    value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_20_16)
    print("reg. value after write @ 0x%1x (cAddress_RegLvds_20_16) : 0x%x\n" %(SPI_ADDR_REGLVDS_20_16, value))

    addr = 0x4
    value = Utsspi.read_mod_register8(SPI_ADDR_MISC)
    print("reg. value @ 0x%2x (cAddress_Misc) : 0x%x" %(SPI_ADDR_MISC, value))

    # following register are RO

    addr = 0x5
    value = Utsspi.read_mod_register8(addr)
    print("reg. value @ 0x%2x (cAddress_SyncMeas1LSB) : 0x%x" %(addr, value))

    addr = 0x6
    value = Utsspi.read_mod_register8(addr)
    print("reg. value @ 0x%2x (cAddress_SyncMeas1MSB) : 0x%x" %(addr, value))

    addr = 0x7
    value = Utsspi.read_mod_register8(addr)
    print("reg. value @ 0x%2x (cAddress_SyncMeas2LSB) : 0x%x" %(addr, value))

    addr = 0x8
    value = Utsspi.read_mod_register8(addr)
    print("reg. value @ 0x%2x (cAddress_SyncMeas2MSB) : 0x%x" %(addr, value))

    addr = 0x9
    value = Utsspi.read_mod_register8(addr)
    print("reg. value @ 0x%2x (cAddress_SyncMeas3LSB) : 0x%x" %(addr, value))

    addr = 0xa
    value = Utsspi.read_mod_register8(addr)
    print("reg. value @ 0x%2x (cAddress_SyncMeas3MSB) : 0x%x" %(addr, value))

    addr = 0xb
    value = Utsspi.read_mod_register8(addr)
    print("reg. value @ 0x%2x (cAddress_SyncMeas4LSB) : 0x%x" %(addr, value))

    addr = 0xc
    value = Utsspi.read_mod_register8(addr)
    print("reg. value @ 0x%2x (cAddress_SyncMeas4MSB) : 0x%x" %(addr, value))

    # cAddress_BoardPresence reg. : x"0" & "00" & DTSS_PRES_B & SWITCH_PRES_B

    addr = 0xd
    value = Utsspi.read_mod_register8(addr)
    switch_pres_b = value & 0x1
    dtss_pres_b = (value & 0x2)/2

    if value == 0x2 :
        msg = "SWITCH module present"
    else:
        msg = "DTSS module present"

    print("reg. value @ 0x%1x (cAddress_BoardPresence) : 0x%x => %s" %(addr, value, msg))
    print("     * SWITCH_PRES_B = %d" %switch_pres_b)
    print("     * DTSS_PRES_B   = %d" %dtss_pres_b)
#
#================================================================================
#
#
#================================================================================
#
@DIAG("B200 module set")
def t_osb200_module_type_set(cmd):

    verbose = False

    try:
        (command, module_to_select) = cmd.split()
    except:
        print_error("# Command must be : t_osb200_module_type_set <module_to_select>")
        print_error("#            with : module_to_select = SWITCH or DTSS")
        return False

    print_info("  module_to_select = %s" %module_to_select)

    if module_to_select == "DTSS" or module_to_select == "SWITCH" or\
       module_to_select == "DTSS-VERBOSE" or module_to_select == "SWITCH-VERBOSE" :
        error_flag = False
        if module_to_select == "DTSS-VERBOSE" or module_to_select == "SWITCH-VERBOSE":
            verbose = True
    else:
        error_flag = True
        print_error("Error : module_to_select MUST be : DTSS or SWITCH")
        return False

    b200_module_type_set(module_to_select, verbose)
#
#================================================================================
#
#
#================================================================================
#
@DIAG("B200 module type detect")
def t_osb200_module_type_detect(cmd):

    verbose = False

    try:
        if ' ' not in cmd:
            pass
        else:
            (command,consigne) = cmd.split()
#           print("arg exist : %s" %consigne)
            if consigne == "verbose":
                verbose = True
            else:
                print_error("   Syntax error : arg must be : verbose")
                print_error("                                       ")
                return False

    except:
        print_error("   Syntax error : Command must be : t_osb200_module_type_detect <consigne>")
        print_error("                                                                          ")
        return False

    glu_reg_base_addr = get_glu_base_addr()
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    module_presence_rdback_value = get_reg_value(glu_reg_base_addr + REG_MOD_PRESENCE_RDBCK)

    switch_bit        = module_presence_rdback_value & 0x1
    dtss_bit          = (module_presence_rdback_value & 0x2)/2
    dtss_fpga_rdy_bit = (module_presence_rdback_value & 0x4)/4

    if verbose == 1:
        print("* OSB module_presence_rdback register details :")
        print("     * module_presence_rdback_value read = 0x%x" % module_presence_rdback_value)
        print("         - module_presence_rdback(2) = DTSS_FPGA_RDY = %d ('0' = ABSENT ; '1' = present)" %dtss_fpga_rdy_bit)
        print("         - module_presence_rdback(1) = DTSS_PRES     = %d ('0' = ABSENT ; '1' = present)" %dtss_bit)
        print("         - module_presence_rdback(0) = SWITCH_PRES   = %d ('0' = ABSENT ; '1' = present)" %switch_bit, "\n")


    value = Utsspi.read_mod_register8(SPI_ADDR_BOARD_PRES)

    switch_pres_b_bit = value & 0x1
    dtss_pres_b_bit   = (value & 0x2)/2
    toggle_bit        = (value & 0x4)/4
    board_id          = (value & 0xc0)/64

    if verbose == 1:
        print("* B200 module_presence_rdback register details :")
        print("     * module_presence_rdback read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_BOARD_PRES, value))
        print("         - BOARD_ID           = %d (2 = SWITCH ; 1 = DTSS)" %board_id)
        print("         - toggle_board_state = %d " %toggle_bit)
        print("         - DTSS_PRES_B        = %d ('0' = Selected ; '1' = DeSelected)" %dtss_pres_b_bit)
        print("         - SWITCH_PRES_B      = %d ('0' = Selected ; '1' = DeSelected)\n" %switch_pres_b_bit)

    if switch_bit == 1 and switch_pres_b_bit == 0:
        print ("Module SWITCH config. detected" )
        return_flag = True
    else:
        if dtss_bit == 1 and dtss_pres_b_bit == 0:
            print ("Module DTSS config. detected" )
            return_flag = True
        else:
            print ("Module config. NOT detected" )
            return_flag = False

    return return_flag
#
#================================================================================
#
# Module board MUST be set to 2 (SWITCH board selected) to run this test
#
#  REAR   = Fan1 : Measured with cAddress_SyncMeas1LSB/cAddress_SyncMeas1MSB
#  MIDDLE = Fan2 : Measured with cAddress_SyncMeas2LSB/cAddress_SyncMeas2MSB
#  FRONT  = Fan3 : Measured with cAddress_SyncMeas3LSB/cAddress_SyncMeas3MSB
#
#  pwm1_duty_value = 0x70000 + fan1_pwm_value   => meas = 0x420
#  pwm1_duty_value = 0x90000 + fan1_pwm_value   => meas = 0x528
#  pwm1_duty_value = 0xb0000 + fan1_pwm_value   => meas = 0x630
#
#  PWM_reg = x"0" & PWM_DIV(11:0) & (x"00" & '0' & PWM_DUTY(6:0)
#
#================================================================================
#
@DIAG("B200 peripheral fans Test")
def t_osb200_periph_fans_test(cmd):

    verbose = False

    try:
        if ' ' not in cmd:
            pass
        else:
            (command,consigne) = cmd.split()
#           print("arg exist : %s" %consigne)
            if consigne == "verbose":
                verbose = True
            else:
                print_error("   Syntax error : arg must be : verbose")
                print_error("                                       ")
                return False

    except:
        print_error("   Syntax error : Command must be : t_osb200_periph_fans_test <consigne>")
        print_error("                                                                        ")
        return False

    return_flag = True
    error_nb = 0

    glu_reg_base_addr = get_glu_base_addr()
    b200_module_type_set("SWITCH", verbose)

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    fan1_pwm_value = 0x1e
    fan2_pwm_value = 0x32
    fan3_pwm_value = 0x4b
    fan1_ctrl_value = 0x90000 + fan1_pwm_value
#   fan1_ctrl_value = 0xb0000 + fan1_pwm_value

    fan2_ctrl_value = 0xc0000 + fan2_pwm_value
#   fan2_ctrl_value = 0x70000 + fan3_pwm_value

    fan3_ctrl_value = 0xd0000 + fan3_pwm_value
#   fan3_ctrl_value = 0x90000 + fan1_pwm_value

    periph_reset_gen("release", verbose)

    if verbose:
        value = Utsspi.read_mod_register8(SPI_ADDR_MISC)
        print(" Read cAddress_Misc reg. value @ 0x%1x after reset : 0x%x" %(SPI_ADDR_MISC, value))

        enable_loop_uart_bit   = value & 0x1
        select_2V5_VIN_bit     = (value & 0x2)/2
        nIRQ_SX_bit            = (value & 0x4)/4
        nIRQ_Sn                = (value & 0x8)/8
        synchro_meas_start_bit = (value & 0x10)/16
        synchro_meas_ok_bit    = (value & 0x80)/128

        print("         - REGISTER_MISC(0) = ENABLE_LOOP_UART = %d" %enable_loop_uart_bit)
        print("         - REGISTER_MISC(1) = SELECT_2V5_VIN = %d" %select_2V5_VIN_bit)
        print("         - REGISTER_MISC(2) = nIRQ_SX = %d" %nIRQ_SX_bit)
        print("         - REGISTER_MISC(3) = nIRQ_Sn = %d" %nIRQ_Sn)
        print("         - REGISTER_MISC(4) = SYNCHRO_MEASURE_START = %d" %synchro_meas_start_bit)
        print("")
        print("         - REGISTER_MISC(7) = SYNCHRO_MEASURE_OK = %d\n" %synchro_meas_ok_bit)

    # fan_position = "REAR"
    reg_write(glu_reg_base_addr + REG_SYS_FAN1_CTRL, fan1_ctrl_value)
    sys_fan1_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_TACHI_VAL)

    # fan_position = "MIDDLE"
    reg_write(glu_reg_base_addr + REG_SYS_FAN2_CTRL, fan2_ctrl_value)
    sys_fan2_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_TACHI_VAL)

    # fan_position = "FRONT"
    reg_write(glu_reg_base_addr + REG_SYS_FAN3_CTRL, fan3_ctrl_value)
    sys_fan3_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_TACHI_VAL)

    time.sleep(1)

    # Set misc_reg(4) = '1' to launch the measure
    #
    wr_data = 0x10
    if verbose:
        print("* Write data = 0x%x into cAddress_Misc reg (@ addr 0x%1x ) : " %(wr_data, SPI_ADDR_MISC))

    ver = Utsspi.write_mod_register8(SPI_ADDR_MISC, wr_data)

    if verbose:
        value = Utsspi.read_mod_register8(SPI_ADDR_MISC)
        print(" Read cAddress_Misc reg. value @ 0x%1x after launching measure: 0x%x" %(SPI_ADDR_MISC, value))

        enable_loop_uart_bit   = value & 0x1
        select_2V5_VIN_bit     = (value & 0x2)/2
        nIRQ_SX_bit            = (value & 0x4)/4
        nIRQ_Sn                = (value & 0x8)/8
        synchro_meas_start_bit = (value & 0x10)/16
        synchro_meas_ok_bit    = (value & 0x80)/128

        print("         - REGISTER_MISC(0) = ENABLE_LOOP_UART = %d" %enable_loop_uart_bit)
        print("         - REGISTER_MISC(1) = SELECT_2V5_VIN = %d" %select_2V5_VIN_bit)
        print("         - REGISTER_MISC(2) = nIRQ_SX = %d" %nIRQ_SX_bit)
        print("         - REGISTER_MISC(3) = nIRQ_Sn = %d" %nIRQ_Sn)
        print("         - REGISTER_MISC(4) = SYNCHRO_MEASURE_START = %d" %synchro_meas_start_bit)
        print("")
        print("         - REGISTER_MISC(7) = SYNCHRO_MEASURE_OK = %d\n" %synchro_meas_ok_bit)

    time.sleep(1)

    # fan1 measure :

    fan1_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS1_LSB)
    fan1_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS1_MSB)

    fan1_val = (float(fan1_msb_value) * 256) + float(fan1_lsb_value)
    fan1_period = (fan1_val * 30.3)/1000000000

    if fan1_period == 0.0:
        fan1_freq = 0.0
    else:
        fan1_freq = 1.0/fan1_period

    if fan1_freq > 24000.0 and fan1_freq < 25100.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - fan1 PWM frequency = %5.3fHz : %s" %(fan1_freq, msg))

    sys_fan1_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_TACHI_VAL)
#   print("     => sys_fan1_tachi_value read = 0x%x" % sys_fan1_tachi_value)
    fan1_tachi = float(sys_fan1_tachi_value)

    if fan1_tachi > 24000.0 and fan1_tachi < 25100.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - fan1 TACHI frequency = %5.3fHz : %s" %(fan1_tachi, msg))

    # fan2 measure :

    fan2_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS2_LSB)
    fan2_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS2_MSB)

    fan2_val = (float(fan2_msb_value) * 256) + float(fan2_lsb_value)
    fan2_period = (fan2_val * 30.3)/1000000000

    if fan2_period == 0.0:
        fan2_freq = 0.0
    else:
        fan2_freq = 1.0/fan2_period

    if fan2_freq > 19100.0 and fan2_freq < 19300.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - fan2 PWM frequency = %5.3fHz : %s" %(fan2_freq, msg))

    sys_fan2_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_TACHI_VAL)
#   print("     => sys_fan2_tachi_value read = 0x%x" % sys_fan2_tachi_value)
    fan2_tachi = float(sys_fan2_tachi_value)

    if fan2_tachi > 19100.0 and fan2_tachi < 19300.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - fan2 TACHI frequency = %5.3fHz : %s" %(fan2_tachi, msg))

    # fan3 measure :

    fan3_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS3_LSB)
    fan3_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS3_MSB)

    fan3_val = (float(fan3_msb_value) * 256) + float(fan3_lsb_value)
    fan3_period = (fan3_val * 30.3)/1000000000

    if fan3_period == 0.0:
        fan3_freq = 0.0
    else:
        fan3_freq = 1.0/fan3_period

    if fan3_freq > 17700.0 and fan3_freq < 17900.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - fan3 PWM frequency = %5.3fHz : %s" %(fan3_freq, msg))

    sys_fan3_tachi_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_TACHI_VAL)
#   print("     => sys_fan3_tachi_value read = 0x%x" % sys_fan3_tachi_value)
    fan3_tachi = float(sys_fan3_tachi_value)

    if fan3_tachi > 17700.0 and fan3_tachi < 17900.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - fan3 TACHI frequency = %5.3fHz : %s" %(fan3_tachi, msg))

    wr_data = 0x00
    if verbose:
        print("\n* Write data = 0x%x into cAddress_Misc reg (@ addr 0x%1x ) : " %(wr_data, SPI_ADDR_MISC))

    ver = Utsspi.write_mod_register8(SPI_ADDR_MISC, wr_data)

    if verbose:
        value = Utsspi.read_mod_register8(SPI_ADDR_MISC)
        print(" Read cAddress_Misc reg. value @ 0x%1x: 0x%x" %(SPI_ADDR_MISC, value))

        enable_loop_uart_bit   = value & 0x1
        select_2V5_VIN_bit     = (value & 0x2)/2
        nIRQ_SX_bit            = (value & 0x4)/4
        nIRQ_Sn                = (value & 0x8)/8
        synchro_meas_start_bit = (value & 0x10)/16
        synchro_meas_ok_bit    = (value & 0x80)/128

        print("         - REGISTER_MISC(0) = ENABLE_LOOP_UART = %d" %enable_loop_uart_bit)
        print("         - REGISTER_MISC(1) = SELECT_2V5_VIN = %d" %select_2V5_VIN_bit)
        print("         - REGISTER_MISC(2) = nIRQ_SX = %d" %nIRQ_SX_bit)
        print("         - REGISTER_MISC(3) = nIRQ_Sn = %d" %nIRQ_Sn)
        print("         - REGISTER_MISC(4) = SYNCHRO_MEASURE_START = %d" %synchro_meas_start_bit)
        print("")
        print("         - REGISTER_MISC(7) = SYNCHRO_MEASURE_OK = %d\n" %synchro_meas_ok_bit)

#   print(" error_nb = %d" %error_nb)

    if error_nb > 0:
        return_flag = False

    return return_flag
#
#================================================================================
#
# OSB_DTSS_SYNC_641K        : DTSS_SYNC_MEASURE     : SYNCHRO_MEASURE1
# PSYNC_EXPCB_700KHZ_CLK    : EXPCB_SYNC_MEASURE    : SYNCHRO_MEASURE2
# P0_PCON_M1_PS_SYNC_CLK    : M1_SYNC_MEASURE       : SYNCHRO_MEASURE3
# P0_PCON_M2_PS_SYNC_CLK    : M2_SYNC_MEASURE       : SYNCHRO_MEASURE4
#
#================================================================================
#
@DIAG("B200 freq_measTest")
def t_osb200_freq_meas_test(cmd):

    verbose = False

    try:
        if ' ' not in cmd:
            pass
        else:
            (command,consigne) = cmd.split()
#           print("arg exist : %s" %consigne)
            if consigne == "verbose":
                verbose = True
            else:
                print_error("   Syntax error : arg must be : verbose")
                print_error("                                       ")
                return False

    except:
        print_error("   Syntax error : Command must be : t_osb200_freq_meas_test <consigne>")
        print_error("                                                                      ")
        return False

    return_flag = True
    error_nb = 0

    glu_reg_base_addr = get_glu_base_addr()
    b200_module_type_set("DTSS", verbose)

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    wr_data = 0x10
    Utsspi.write_mod_register8(SPI_ADDR_MISC, wr_data)

    if verbose:
        value = Utsspi.read_mod_register8(SPI_ADDR_MISC)
        print(" Read cAddress_Misc reg. value @ 0x%1x after launching measure: 0x%x" %(SPI_ADDR_MISC, value))

        enable_loop_uart_bit   = value & 0x1
        select_2V5_VIN_bit     = (value & 0x2)/2
        nIRQ_SX_bit            = (value & 0x4)/4
        nIRQ_Sn                = (value & 0x8)/8
        synchro_meas_start_bit = (value & 0x10)/16
        synchro_meas_ok_bit    = (value & 0x80)/128

        print("         - REGISTER_MISC(0) = ENABLE_LOOP_UART = %d" %enable_loop_uart_bit)
        print("         - REGISTER_MISC(1) = SELECT_2V5_VIN = %d" %select_2V5_VIN_bit)
        print("         - REGISTER_MISC(2) = nIRQ_SX = %d" %nIRQ_SX_bit)
        print("         - REGISTER_MISC(3) = nIRQ_Sn = %d" %nIRQ_Sn)
        print("         - REGISTER_MISC(4) = SYNCHRO_MEASURE_START = %d" %synchro_meas_start_bit)
        print("")
        print("         - REGISTER_MISC(7) = SYNCHRO_MEASURE_OK = %d\n" %synchro_meas_ok_bit)

    time.sleep(1)

    # OSB_DTSS_SYNC_641K measure : 641KHz

    sync1_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS1_LSB)
    sync1_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS1_MSB)

    if verbose:
        print("sync1_msb_value = 0x%x ; sync1_lsb_value = 0x%x " %(sync1_msb_value, sync1_lsb_value))

    sync1_val = (float(sync1_msb_value) * 256) + float(sync1_lsb_value)
    sync1_period = (sync1_val * 30.3)/1000000000

    if sync1_period == 0.0:
        sync1_freq = 0.0
    else:
        sync1_freq = 1.0/sync1_period

    if sync1_freq > 630000.0 and sync1_freq < 650000.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - OSB_DTSS_SYNC_641K frequency = %5.3fHz : %s" %(sync1_freq, msg))

    # PSYNC_EXPCB_700KHZ_CLK measure : 700KHz

#   os_cmde = "echo 1 > /sys/class/i2c-dev/i2c-5/device/5-0021/gpiochip6/gpio/gpio481/value"
    os_cmde = "qdbus --system com.viavisolutions.platform.PowerBudget /com/viavisolutions/platform/PowerBudget com.viavisolutions.platform.PowerBudget.RequestSubmodulePowerOn 0 1 > /dev/null"
    (os.system(os_cmde))

    sync2_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS2_LSB)
    sync2_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS2_MSB)

    if verbose:
        print("sync2_msb_value = 0x%x ; sync2_lsb_value = 0x%x " %(sync2_msb_value, sync2_lsb_value))

    sync2_val = (float(sync2_msb_value) * 256) + float(sync2_lsb_value)
    sync2_period = (sync2_val * 30.3)/1000000000

    if sync2_period == 0.0:
        sync2_freq = 0.0
    else:
        sync2_freq = 1.0/sync2_period

    if sync2_freq > 690000.0 and sync2_freq < 720000.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - PSYNC_EXPCB_700KHZ_CLK frequency = %5.3fHz : %s" %(sync2_freq, msg))

    # P0_PCON_M1_PS_SYNC_CLK measure : 1.4MHz

    sync3_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS3_LSB)
    sync3_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS3_MSB)

    if verbose:
        print("sync3_msb_value = 0x%x ; sync3_lsb_value = 0x%x " %(sync3_msb_value, sync3_lsb_value))

    sync3_val = (float(sync3_msb_value) * 256) + float(sync3_lsb_value)
    sync3_period = (sync3_val * 30.3)/1000000000

    if sync3_period == 0.0:
        sync3_freq = 0.0
    else:
        sync3_freq = 1.0/sync3_period

    if sync3_freq > 1300000.0 and sync3_freq < 1500000.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - P0_PCON_M1_PS_SYNC_CLK frequency = %5.3fHz : %s" %(sync3_freq, msg))

    # P0_PCON_M2_PS_SYNC_CLK measure : 1.4MHz

    sync4_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS4_LSB)
    sync4_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS4_MSB)

    if verbose:
        print("sync4_msb_value = 0x%x ; sync4_lsb_value = 0x%x " %(sync4_msb_value, sync4_lsb_value))

    sync4_val = (float(sync4_msb_value) * 256) + float(sync4_lsb_value)
    sync4_period = (sync4_val * 30.3)/1000000000

    if sync4_period == 0.0:
        sync4_freq = 0.0
    else:
        sync4_freq = 1.0/sync4_period

    if sync4_freq > 1300000.0 and sync4_freq < 1500000.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - P0_PCON_M2_PS_SYNC_CLK frequency = %5.3fHz : %s" %(sync4_freq, msg))

    # SCD_EXPCB_PCIE_CLK measure :

    sync5_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS5_LSB)
    sync5_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS5_MSB)

    if verbose:
        print("sync5_msb_value = 0x%x ; sync5_lsb_value = 0x%x " %(sync5_msb_value, sync5_lsb_value))

    sync5_val = (float(sync5_msb_value) * 256) + float(sync5_lsb_value)
    sync5_freq = (sync5_val) * 10000

    if sync5_freq > 99000000.0 and sync5_freq < 110000000.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - SCD_EXPCB_PCIE_CLK frequency = %5.3fHz : %s" %(sync5_freq, msg))

    # SCD_M1_REF_CLK measure :

    sync6_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS6_LSB)
    sync6_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS6_MSB)

    if verbose:
        print("sync6_msb_value = 0x%x ; sync6_lsb_value = 0x%x " %(sync6_msb_value, sync6_lsb_value))

    sync6_val = (float(sync6_msb_value) * 256) + float(sync6_lsb_value)
    sync6_freq = (sync6_val) * 10000

    if sync6_freq > 99000000.0 and sync6_freq < 110000000.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - SCD_M1_REF_CLK frequency = %5.3fHz : %s" %(sync6_freq, msg))

    # SCD_M2_REF_CLK measure :

    sync7_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS7_LSB)
    sync7_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS7_MSB)

    if verbose:
        print("sync7_msb_value = 0x%x ; sync7_lsb_value = 0x%x " %(sync7_msb_value, sync7_lsb_value))

    sync7_val = (float(sync7_msb_value) * 256) + float(sync7_lsb_value)
    sync7_freq = (sync7_val) * 10000

    if sync7_freq > 99000000.0 and sync7_freq < 110000000.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - SCD_M2_REF_CLK frequency = %5.3fHz : %s" %(sync7_freq, msg))

#   print(" error_nb = %d" %error_nb)

    if error_nb > 0:
        return_flag = False

    return return_flag
#
#================================================================================
#
# MSEL_NB_NOR_CS <= slv_reg15(4 downto 0);
# MSEL_START     <= slv_reg15(5);
# MSEL_CS_RESET  <= slv_reg15(7);
#
# reg_data_out <= slv_reg15(31 downto 10) & MOD_OUT_LOC & MSEL_STATUS & slv_reg15(7 downto 0);
#
#================================================================================
#
@DIAG("B200 module selection (MSEL) Test")
def t_osb200_msel_test(cmd):

    try:
        (command, module_to_select) = cmd.split()
    except:
        print_error("# Command must be : t_osb200_msel_test <module_to_select>")
        print_error("#            with : module_to_select = SWITCH or DTSS")
        return False

    if module_to_select == "DTSS" or module_to_select == "SWITCH":
        error_flag = False
    else:
        print_error("Error : module_to_select MUST be : DTSS or SWITCH")
        return False

    verbose = False

    return_flag = True
    error_nb = 0

    glu_reg_base_addr = get_glu_base_addr()

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    b200_module_type_set(module_to_select, verbose)

    periph_reset_gen("release", verbose)

    if module_to_select == "DTSS":
        msel_nb_nor_cs = 0x5
    else:
        msel_nb_nor_cs = 0x15

    msel_start     = 0x0
    msel_cs_reset  = 0x0

    msel_reg_value = (msel_cs_reset * 0x80) | (msel_start * 0x20) | msel_nb_nor_cs
    reg_write(glu_reg_base_addr + REG_MSEL, msel_reg_value)

    msel_value = get_reg_value(glu_reg_base_addr + REG_MSEL)
#   print("msel_value = 0x%x" % msel_value)

    msel_cnter_value = Utsspi.read_mod_register8(SPI_ADDR_MSEL_REG)

    if msel_cnter_value == 0x0:
        pass
    else:
        error_nb += 1

    periph_reset_gen("release", verbose)

    msel_start = 0x1

    msel_reg_value = (msel_cs_reset * 0x80) | (msel_start * 0x20) | msel_nb_nor_cs
    reg_write(glu_reg_base_addr + REG_MSEL, msel_reg_value)

    msel_start = 0x0

    msel_reg_value = (msel_cs_reset * 0x80) | (msel_start * 0x20) | msel_nb_nor_cs
    reg_write(glu_reg_base_addr + REG_MSEL, msel_reg_value)

    msel_cnter_value = Utsspi.read_mod_register8(SPI_ADDR_MSEL_REG)

    exp_value = msel_nb_nor_cs

    if msel_cnter_value == exp_value:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print("msel_value = 0x%x (expected 0x%x) : %s" %(msel_cnter_value, exp_value, msg))

    if error_nb > 0:
        return_flag = False

    return return_flag
#
#===============================================================================
#
#   slv_reg14(31) <= dbg_gpio_i(0);     -- ADP
#   slv_reg14(30) <= dbg_gpio_i(1);     -- SWITCH_IRQ
#   slv_reg14(29) <= dbg_gpio_i(2);     -- DTSS_IRQ
#   slv_reg14(28) <= PUSHB;
#
#===============================================================================
#
@DIAG("B200 IRQs Test")
def t_osb200_irq_test(cmd):

    debug = True

    try:
        (command, module_to_select) = cmd.split()
    except:
        print_error("# Command must be : t_osb200_irq_test <module_to_select>")
        print_error("#            with : module_to_select = SWITCH or DTSS")
        return False

    if module_to_select == "DTSS" or module_to_select == "SWITCH":
        error_flag = False
    else:
        print_error("Error : module_to_select MUST be : DTSS or SWITCH")
        return False

    verbose = False

    return_flag = True
    error_nb = 0

    glu_reg_base_addr = get_glu_base_addr()

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    b200_module_type_set(module_to_select, verbose)

    periph_reset_gen("release", verbose)

    dbg_reg_value = get_reg_value(glu_reg_base_addr + REG_DBG_CTRL)

    # Write data = 0x4 into cAddress_Misc reg to set IRQn = 0

    wr_data = 0x4
    ver = Utsspi.write_mod_register8(SPI_ADDR_MISC, wr_data)

    dbg_reg_value = get_reg_value(glu_reg_base_addr + REG_DBG_CTRL)

#   print("     => dbg_reg_value read = 0x%x" % dbg_reg_value)

    dtss_IRQ_bit29 = dbg_reg_value &0x20000000 >> 29
    switch_IRQ_bit30 = dbg_reg_value &0x40000000 >> 30

    if module_to_select == "DTSS":
        IRQ_bit = dtss_IRQ_bit29
    else:
        IRQ_bit = switch_IRQ_bit30

    if IRQ_bit == 0:
        pass
    else:
        error_nb =+ 1

    wr_data = 0x0
    ver = Utsspi.write_mod_register8(SPI_ADDR_MISC, wr_data)

    dbg_reg_value = get_reg_value(glu_reg_base_addr + REG_DBG_CTRL)

#   print("     => dbg_reg_value read = 0x%x" % dbg_reg_value)

    dtss_IRQ_bit29 = dbg_reg_value &0x20000000 >> 29
    switch_IRQ_bit30 = (dbg_reg_value & 0x40000000) >> 30

    if module_to_select == "DTSS":
        IRQ_bit = dtss_IRQ_bit29
    else:
        IRQ_bit = switch_IRQ_bit30

    if IRQ_bit == 1:
        pass
    else:
        error_nb =+ 1

    return return_flag
#
#================================================================================
#
#  osb_fan_1 (LEFT) , osb_fan_2 (RIGHT)
#
# OSB_FAN1_PWM              :
# OSB_FAN2_PWM              :
#
#
#================================================================================
#
@DIAG("OSB front panel fans test")
def t_osb200_front_fans_test(cmd):

    verbose = False

    try:
        if ' ' not in cmd:
            pass
        else:
            (command,consigne) = cmd.split()
#           print("arg exist : %s" %consigne)
            if consigne == "verbose":
                verbose = True
            else:
                print_error("   Syntax error : arg must be : verbose")
                print_error("                                       ")
                return False

    except:
        print_error("   Syntax error : Command must be : t_osb200_periph_fans_test <consigne>")
        print_error("                                                                          ")
        return False

    return_flag = True
    error_nb = 0

    glu_reg_base_addr = get_glu_base_addr()
    b200_module_type_set("SWITCH", verbose)

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    fan1_pwm_value = 0x1e
    fan2_pwm_value = 0x32
    osb_fan1_ctrl_value = 0x90000 + fan1_pwm_value
#   osb_fan1_ctrl_value = 0xb0000 + fan1_pwm_value

    osb_fan2_ctrl_value = 0xc0000 + fan2_pwm_value
#   osb_fan2_ctrl_value = 0x70000 + fan3_pwm_value

    periph_reset_gen("release", verbose)

    # fan_position = "RIGHT"
    reg_write(glu_reg_base_addr + REG_SYS_FAN1_CTRL, osb_fan1_ctrl_value)
    sys_fan1_tachi_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_CTRL)

    # fan_position = "LEFT"
    reg_write(glu_reg_base_addr + REG_SYS_FAN2_CTRL, osb_fan2_ctrl_value)
    sys_fan2_tachi_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_CTRL)

    wr_data = 0x10
    if verbose:
        print("* Write data = 0x%x into cAddress_Misc reg @ addr 0x%1x : " %(wr_data, SPI_ADDR_MISC))

    ver = Utsspi.write_mod_register8(SPI_ADDR_MISC, wr_data)

    if verbose:
        value = Utsspi.read_mod_register8(SPI_ADDR_MISC)
        print(" Read cAddress_Misc reg. value @ 0x%1x after launching measure: 0x%x" %(SPI_ADDR_MISC, value))

        enable_loop_uart_bit   = value & 0x1
        select_2V5_VIN_bit     = (value & 0x2)/2
        nIRQ_SX_bit            = (value & 0x4)/4
        nIRQ_Sn                = (value & 0x8)/8
        synchro_meas_start_bit = (value & 0x10)/16
        synchro_meas_ok_bit    = (value & 0x80)/128

        print("         - REGISTER_MISC(0) = ENABLE_LOOP_UART = %d" %enable_loop_uart_bit)
        print("         - REGISTER_MISC(1) = SELECT_2V5_VIN = %d" %select_2V5_VIN_bit)
        print("         - REGISTER_MISC(2) = nIRQ_SX = %d" %nIRQ_SX_bit)
        print("         - REGISTER_MISC(3) = nIRQ_Sn = %d" %nIRQ_Sn)
        print("         - REGISTER_MISC(4) = SYNCHRO_MEASURE_START = %d" %synchro_meas_start_bit)
        print("")
        print("         - REGISTER_MISC(7) = SYNCHRO_MEASURE_OK = %d\n" %synchro_meas_ok_bit)

    time.sleep(1)

    # osb_fan1 measure :

    osb_fan1_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS5_LSB)
    osb_fan1_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS5_MSB)

    osb_fan1_val = (float(osb_fan1_msb_value) * 256) + float(osb_fan1_lsb_value)
    osb_fan1_period = (osb_fan1_val * 30.3)/1000000000

    if osb_fan1_period == 0.0:
        osb_fan1_freq = 0.0
    else:
        osb_fan1_freq = 1.0/osb_fan1_period

    if osb_fan1_freq > 24000.0 and osb_fan1_freq < 25100.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - fan1 PWM frequency = %5.3fHz : %s" %(osb_fan1_freq, msg))

    # osb_fan2 measure :
    osb_fan2_lsb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS6_LSB)
    osb_fan2_msb_value = Utsspi.read_mod_register8(SPI_ADDR_SYNC_MEAS6_MSB)

    osb_fan2_val = (float(osb_fan2_msb_value) * 256) + float(osb_fan2_lsb_value)
    osb_fan2_period = (osb_fan2_val * 30.3)/1000000000

    if osb_fan2_period == 0.0:
        osb_fan2_freq = 0.0
    else:
        osb_fan2_freq = 1.0/osb_fan2_period

    if osb_fan2_freq > 24000.0 and osb_fan2_freq < 25100.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    print(" - fan2 PWM frequency = %5.3fHz : %s" %(osb_fan2_freq, msg))

    wr_data = 0x00
    if verbose:
        print("\n* Write data = 0x%x into cAddress_Misc reg (@ addr 0x%1x ) : " %(wr_data, SPI_ADDR_MISC))

    ver = Utsspi.write_mod_register8(SPI_ADDR_MISC, wr_data)

    if verbose:
        value = Utsspi.read_mod_register8(SPI_ADDR_MISC)
        print(" Read cAddress_Misc reg. value @ 0x%1x: 0x%x" %(SPI_ADDR_MISC, value))

        enable_loop_uart_bit   = value & 0x1
        select_2V5_VIN_bit     = (value & 0x2)/2
        nIRQ_SX_bit            = (value & 0x4)/4
        nIRQ_Sn                = (value & 0x8)/8
        synchro_meas_start_bit = (value & 0x10)/16
        synchro_meas_ok_bit    = (value & 0x80)/128

        print("         - REGISTER_MISC(0) = ENABLE_LOOP_UART = %d" %enable_loop_uart_bit)
        print("         - REGISTER_MISC(1) = SELECT_2V5_VIN = %d" %select_2V5_VIN_bit)
        print("         - REGISTER_MISC(2) = nIRQ_SX = %d" %nIRQ_SX_bit)
        print("         - REGISTER_MISC(3) = nIRQ_Sn = %d" %nIRQ_Sn)
        print("         - REGISTER_MISC(4) = SYNCHRO_MEASURE_START = %d" %synchro_meas_start_bit)
        print("")
        print("         - REGISTER_MISC(7) = SYNCHRO_MEASURE_OK = %d\n" %synchro_meas_ok_bit)

#   print(" error_nb = %d" %error_nb)

    if error_nb > 0:
        return_flag = False

    return return_flag

@DIAG("Verify JTAG ID")
def t_osb200_jtag_test(cmd):

    verbose = False

    try:
        (command, module_to_select) = cmd.split()
    except:
        print_error("# Command must be : t_osb200_jtag_test <module_to_select>")
        print_error("#            with : module_to_select = SWITCH or DTSS")
        return False

    if module_to_select == "DTSS" or module_to_select == "SWITCH":
        error_flag = False
    else:
        print_error("Error : module_to_select MUST be : DTSS or SWITCH")
        return False

    res = 0

    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    b200_module_type_set(module_to_select, verbose)

    FileName = "/usr/share/pydiag/oth/"+module_to_select+"_jtag.txt"

    f = open(FileName, "w")
    f.write('cable osb\n')
    f.write('detect\n')
    f.close()
    cmde = "jtag "+FileName

    p = subprocess.Popen(cmde, shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                         stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
            # print_info("col %s" % columns[0])
            if module_to_select == "DTSS":
                if ((columns[0] == "Part(0)") and (columns[1] == "10M04SCU169 (0x318A)")):
                    res = 1
            else:
                if ((columns[0] == "Device Id") and (columns[1] == "00010100100101010001000111000011 (0x149511C3)")):
                    print("columns[1] is" , columns[1])
                    res = 1

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    return res == 1
#
#================================================================================
#
#
#================================================================================
#
@DIAG("OSB-B200 : Load B200 CPLD")
def t_osb200_load_cpld(cmd):

    verbose = False

    glu_reg_base_addr = get_glu_base_addr()
    b200_module_type_set("DTSS", verbose)

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    module_presence_info(OSB_flag = 1, B200_flag = 1, verbose_flag = verbose)

    svf_file = "/usr/share/pydiag/oth/oth_b200.svf"
    FileName = "/usr/share/pydiag/oth/cpld_load.txt"

    f = open(FileName, "w")
    f.write('cable osb\n')
    f.write('detect\n')
    f.close()

    cmde = "jtag "+FileName

    p = subprocess.Popen(cmde, shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                         stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
            # print_info("col %s" % columns[0])
            if ((columns[0] == "Part(0)") and (columns[1] == "10M04SCU169 (0x318A)")):
                res = 1

    if  res == 1:
        print("\n=> CPLD Detect : PASS\n")
    else:
        print("\n=> CPLD Detect : FAIL\n")
        print("\n   CPLD not programmed\n")
        return False

    # Verify svf_file exist

    f = open(FileName, "w")
    f.write('cable osb\n')
    f.write('detect\n')
    f.write('svf %s progress\n' %svf_file)
    f.close()

    cmde = "jtag "+FileName

    res = 0

    p = subprocess.Popen(cmde, shell=True, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                         stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]
            # print_info("col %s" % columns[0])
            if ((columns[0] == "Part(0)") and (columns[1] == "10M04SCU169 (0x318A)")):
                ver = Utsspi.read_mod_register8(0x00)
                res += 1

            if ((columns[0] == "detail") and (columns[1] == "Scanned device output matched expected TDO values.")):
                res += 1

    os_cmde = "rm -f "+FileName
    (os.system(os_cmde))

    if res == 2:
        print("\n=> CPLD programation (cpld_version = %2x) : PASS\n" % ver)
        return True
    else:
        print("\n=> CPLD programation : FAIL\n")
        return False
#
#================================================================================
#
# To test DTSS_OSB_FAN_ON and DTSS_OSB_FPGA_READY_N bit coming from dtss
# connector
#
#================================================================================
#
#
@DIAG("OSB-B200 DTSS_FAN_ON and DTSS_FPGA_READY_N test")
def t_osb200_dtss_test(cmd):

    debug = False
    verbose = False

    glu_reg_base_addr = get_glu_base_addr()
    if debug:
        print("glu_reg_base_addr = 0x%x" %glu_reg_base_addr, "\n")

    periph_reset_gen("release", verbose)

    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    value = Utsspi.read_mod_register8(SPI_ADDR_BOARD_PRES)
    switch_pres_b_bit = value & 0x1
    dtss_pres_b_bit   = (value & 0x2)/2
    toggle_bit        = (value & 0x4)/4
    board_id          = (value & 0x80)/128

    if debug:
        print("* B200 module_presence_rdback register details :")
        print("     * module_presence_rdback read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_BOARD_PRES, value))
        print("         - BOARD_ID           = %d (2 = SWITCH ; 1 = DTSS)" %board_id)
        print("         - toggle_board_state = %d " %toggle_bit)
        print("         - DTSS_PRES_B        = %d ('0' = Selected ; '1' = DeSelected)" %dtss_pres_b_bit)
        print("         - SWITCH_PRES_B      = %d ('0' = Selected ; '1' = DeSelected)\n" %switch_pres_b_bit)

    if board_id == 2:
        if debug:
            print (" Run b200_module_type_set(DTSS)")
        b200_module_type_set("DTSS", my_verbose)

    error_flag = 0

    wr_data = 0x0
    ver = Utsspi.write_mod_register8(SPI_ADDR_DTSS_REG, wr_data)
    value = Utsspi.read_mod_register8(SPI_ADDR_DTSS_REG)

    if value == wr_data:
        pass
    else:
        error_flag += 1

    wr_data = 0x3
    ver = Utsspi.write_mod_register8(SPI_ADDR_DTSS_REG, wr_data)
    value = Utsspi.read_mod_register8(SPI_ADDR_DTSS_REG)

    if value == wr_data:
        pass
    else:
        error_flag += 1

    if error_flag > 0:
        return False
    else:
        return True
#
#================================================================================
#
# To test EXPB_PWR_EN signal coming from CPB board
#
#================================================================================
#
@DIAG("B200 EXPCB_PWR_EN  Test")
def t_osb200_expb_pwr_en_test(cmd):

    verbose = False
    error_nb = 0

    try:
        if ' ' not in cmd:
            pass
        else:
            (command,consigne) = cmd.split()
#           print("arg exist : %s" %consigne)
            if consigne == "verbose":
                verbose = True
            else:
                print_error("   Syntax error : arg must be : verbose")
                print_error("                                       ")
                return False

    except:
        print_error("   Syntax error : Command must be : t_osb200_expb_pwr_en_test <consigne>")
        print_error("                                                                        ")
        return False

    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    # Allumage :

    msg = "* Enable expb_pwr_en  :"
    if verbose:
        print(msg)

    os_cmde = "qdbus --system com.viavisolutions.platform.PowerBudget /com/viavisolutions/platform/PowerBudget com.viavisolutions.platform.PowerBudget.RequestSubmodulePowerOn 0 1 > /dev/null"
    (os.system(os_cmde))

    expb_pwr_en_value = Utsspi.read_mod_register8(SPI_ADDR_EXPCB)

    if verbose:
        print("     - SPI_ADDR_EXPCB register read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_EXPCB, expb_pwr_en_value))

    if expb_pwr_en_value == 0:
        flag_msg = "FAIL"
        error_nb += 1
    else:
        flag_msg = "PASS"

    print("%s %s" %(msg, flag_msg))

    # Extinction :

    msg = "* Disable expb_pwr_en :"
    if verbose:
        print(msg)

    os_cmde = "qdbus --system com.viavisolutions.platform.PowerBudget /com/viavisolutions/platform/PowerBudget com.viavisolutions.platform.PowerBudget.SubmodulePowerOff 0 1 > /dev/null"
    (os.system(os_cmde))

    expb_pwr_en_value = Utsspi.read_mod_register8(SPI_ADDR_EXPCB)

    if verbose:
        print("     - SPI_ADDR_EXPCB register read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_EXPCB, expb_pwr_en_value))

    if expb_pwr_en_value == 1:
        flag_msg = "FAIL"
        error_nb += 1
    else:
        flag_msg = "PASS"

    print("%s %s" %(msg, flag_msg))

    if error_nb > 0:
        return False
    else:
        return True

@DIAG("B200 EXPCB_PWR_EN check")
def t_osb200_expb_pwr_en_check(cmd):

    # Query :

    os_cmde = "qdbus --system com.viavisolutions.platform.PowerBudget /com/viavisolutions/platform/PowerBudget com.viavisolutions.platform.PowerBudget.IsSubmodulePoweredOn 0 1"
    (os.system(os_cmde))

    return

@DIAG("B200 CPLD CLOCK 33MHz test")
def t_osb200_cpld_33M_test(cmd):

    verbose = False
    res = 0

    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    periph_reset_gen("release", verbose)

    cpld33M_cnter_value = Utsspi.read_mod_register8(SPI_ADDR_CPLD33_REG)

    if verbose:
        print("     - SPI_ADDR_CPLD33 register read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_CPLD33_REG, cpld33M_cnter_value))

    if cpld33M_cnter_value == 0xf:
        pass
    else:
        res = res + 1

    wr_data = 0x1
    ver = Utsspi.write_mod_register8(SPI_ADDR_CPLD33_REG, wr_data)

    time.sleep(0.1)

    cpld33M_cnter_value = Utsspi.read_mod_register8(SPI_ADDR_CPLD33_REG)

    if verbose:
        print("     - SPI_ADDR_CPLD33 register read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_CPLD33_REG, cpld33M_cnter_value))

    if cpld33M_cnter_value == 0xf7:
        pass
    else:
        res = res + 1

    periph_reset_gen("release", verbose)

    cpld33M_cnter_value = Utsspi.read_mod_register8(SPI_ADDR_CPLD33_REG)

    if verbose:
        print("     - SPI_ADDR_CPLD33 register read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_CPLD33_REG, cpld33M_cnter_value))

    if cpld33M_cnter_value == 0xf:
        pass
    else:
        res = res + 1

    if res == 0:
        return True
    else:
        return False

@DIAG("Write OSB QSPI Flash if necessary")
def t_osb_prog_flash(cmd):
    try:
        (command) = cmd.split()
    except:
        print_error("# Command must be: t_osb_prog_flash ")
        return False

    p = subprocess.Popen("advisordff ste1 -sf 4 /usr/share/firmware/osb/osb.bin" , shell=True, stdin=subprocess.PIPE,
                    stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    print(err)

    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]

    if err == 0 or err == 11:
        return True
    else:
        return False


@DIAG("Write OSB QSPI Flash with all '1' (erase)")
def t_osb_erase_flash(cmd):
    try:
        (command) = cmd.split()
    except:
        print_error("# Command must be: t_osb_erase_flash ")
        return False
    byte_arr = [255]
    binary_format = bytearray(byte_arr)
    print_info("# Creating /tmp/erase.bin")
    f = open('/tmp/erase.bin', 'w+b')
    for i in range(2*1024*1024):
        f.write(binary_format)
    f.close()

    print_info("# Flashing /tmp/erase.bin")
    p = subprocess.Popen("advisordff ste1 -sf 4 /tmp/erase.bin" , shell=True, stdin=subprocess.PIPE,
                         stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait()
    err = p.returncode
    print(err)

    lines = p.stdout.readlines()
    for l in lines:
        l = l.decode("utf-8").strip()
        if l:
            print_info("# %s" % (l))
            columns = [item.strip() for item in l.split(':')]

    if err == 0 or err == 11 or err == 7:
        return True
    else:
        return False
#
#================================================================================
#
# lateral_fans are :
#
#   fan0 : lateral_fan0 (REAR)
#   fan1 : lateral_fan1 (MIDDLE)
#   fan2 : lateral_fan2 (FRONT)
#
# OSB_front_fans are :

#   fan3 : OSB_front_fan1 (RIGHT/LEFT)
#   fan4 : OSB_front_fan2 (RIGHT/LEFT)
#
# /sys/class/hwmon/hwmon<x>/pwm1 = 64  => osb_fan_value = 0x90019
# /sys/class/hwmon/hwmon<x>/pwm1 = 128 => osb_fan_value = 0x90032
# /sys/class/hwmon/hwmon<x>/pwm1 = 255 => osb_fan_value = 0x90063
#
#================================================================================
#
@DIAG("OSB periph and front fans debug")
def t_osb_fans_dbg(cmd):

    verbose = True
    debug   = False

    glu_reg_base_addr = get_glu_base_addr()

    fan_topo_list = OSB_fan_topology()

    if verbose:
        print ("\nfan_topo_list is ", fan_topo_list, "\n")

    driver_list = []
    fan_list = []
    position_list = []
    enable_list = []
    osb_fpga_reg_str_list = []
    osb_fpga_reg_list = []

    i = 0
    for elt in enumerate(fan_topo_list):
        if debug:
            print("****** elt_%d = " %i, elt)
        if i == 0 or i%2 == 0:
            driver_list.append(elt[1])
        else:
            fan_list.append(elt[1])
            if "fan0" in elt[1]:
                position_list.append("REAR")
                enable_list.append("FAN_EN")
                osb_fpga_reg_str_list.append("REG_SYS_FAN1_CTRL")
                osb_fpga_reg_list.append(REG_SYS_FAN1_CTRL)

            elif "fan1" in elt[1]:
                position_list.append("MIDDLE")
                enable_list.append("FAN_EN")
                osb_fpga_reg_str_list.append("REG_SYS_FAN2_CTRL")
                osb_fpga_reg_list.append(REG_SYS_FAN2_CTRL)

            elif "fan2" in elt[1]:
                position_list.append("FRONT")
                enable_list.append("FAN_EN")
                osb_fpga_reg_str_list.append("REG_SYS_FAN3_CTRL")
                osb_fpga_reg_list.append(REG_SYS_FAN3_CTRL)

            elif "fan4" in elt[1]:
                position_list.append("RIGHT")
                enable_list.append("FAN_DIS")
                osb_fpga_reg_str_list.append("REG_OSB_FAN1_CTRL")
                osb_fpga_reg_list.append(REG_OSB_FAN1_CTRL)
            else:
                position_list.append("LEFT")
                enable_list.append("FAN_DIS")
                osb_fpga_reg_str_list.append("REG_OSB_FAN2_CTRL")
                osb_fpga_reg_list.append(REG_OSB_FAN2_CTRL)

        if debug:
            print("\ndriver_list is", driver_list)
            print("fan_list is", fan_list)
            print("position_list is", position_list,"\n")
        i += 1

    i = 0
    fan_pwm_value_str = "0"

    print("* Reset all fans pwm value @ 0 :")

    for elt in enumerate(fan_list):
        hwmon_val = driver_list[i]
        fan_pwm_cmde = "/sys/class/hwmon/"+hwmon_val+"/pwm1"
        if verbose:
            print("     * sys_cmde is :", "echo "+fan_pwm_value_str+" > "+fan_pwm_cmde)
        sysfs_write(fan_pwm_cmde, fan_pwm_value_str)
        i += 1

    print("\n* hwmon readback all fans pwm value :")

    i = 0
    for elt in enumerate(fan_list):
        fan = fan_list[i]
        hwmon_val = driver_list[i]
        fan_pwm_cmde = "/sys/class/hwmon/"+hwmon_val+"/pwm1"
        pwm_rdback = sysfs_read_string(fan_pwm_cmde)
        if verbose:
            print("     *", fan, hwmon_val, "pwm_rdback is", pwm_rdback)
        i += 1

    print("\n* OSB FPGA reg. readback all fans pwm value :")
    sys_fan1_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_CTRL)
    print("     * sys_fan1_value = 0x%x" % sys_fan1_value)
    sys_fan2_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_CTRL)
    print("     * sys_fan2_value = 0x%x" % sys_fan2_value)
    sys_fan3_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_CTRL)
    print("     * sys_fan3_value = 0x%x\n" % sys_fan3_value)

    osb_fan1_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_CTRL)
    print("     * osb_fan1_value = 0x%x" % osb_fan1_value)
    osb_fan2_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_CTRL)
    print("     * osb_fan2_value = 0x%x\n" % osb_fan2_value)

    i = 0
    for elt in enumerate(fan_list):
        if fan_list[i] == "fan0":
            fan = fan_list[i]
            hwmon_val = driver_list[i]
            osb_fpga_reg_str = osb_fpga_reg_str_list[i]
            osb_fpga_reg = osb_fpga_reg_list[i]
            position = position_list[i]
        i += 1

    if verbose:
        print("for %s (%s)" %(fan, position), ": hwmon_val =", hwmon_val)
        print("                    OSB FPGA reg. is %s @ 0x%x\n" %(osb_fpga_reg_str, osb_fpga_reg))
    fan_pwm_value_str = "0"

    fan_pwm_cmde = "/sys/class/hwmon/"+hwmon_val+"/pwm1"

    if verbose:
        print("sys_cmde is :", "echo "+fan_pwm_value_str+" > "+fan_pwm_cmde)

    sysfs_write(fan_pwm_cmde, fan_pwm_value_str)
    pwm_rdback = sysfs_read_string(fan_pwm_cmde)
    if verbose:
        print(hwmon_val,"pwm_rdback is", pwm_rdback)

    osb_fan_value = get_reg_value(glu_reg_base_addr + osb_fpga_reg)
    print("osb_fan_ctrl_reg = 0x%x\n" % osb_fan_value)

    fan_pwm_value_str = "1"

    if verbose:
        print("sys_cmde is :", "echo "+fan_pwm_value_str+" > "+fan_pwm_cmde)

    sysfs_write(fan_pwm_cmde, fan_pwm_value_str)
    pwm_rdback = sysfs_read_string(fan_pwm_cmde)
    if verbose:
        print(hwmon_val, "pwm_rdback is", pwm_rdback)

    osb_fan_value = get_reg_value(glu_reg_base_addr + osb_fpga_reg)
    print("osb_fan_ctrl_reg = 0x%x\n" % osb_fan_value)

    sys_fan1_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_CTRL)
    print("     * sys_fan1_value = 0x%x" % sys_fan1_value)
    sys_fan2_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_CTRL)
    print("     * sys_fan2_value = 0x%x" % sys_fan2_value)
    sys_fan3_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_CTRL)
    print("     * sys_fan3_value = 0x%x\n" % sys_fan3_value)

    osb_fan1_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_CTRL)
    print("     * osb_fan1_value = 0x%x" % osb_fan1_value)
    osb_fan2_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_CTRL)
    print("     * osb_fan2_value = 0x%x\n" % osb_fan2_value)

    fan_pwm_value_str = "64"

    if verbose:
        print("sys_cmde is :", "echo "+fan_pwm_value_str+" > "+fan_pwm_cmde)

    sysfs_write(fan_pwm_cmde, fan_pwm_value_str)

    pwm_rdback = sysfs_read_string(fan_pwm_cmde)
    if verbose:
        print(hwmon_val, "pwm_rdback is", pwm_rdback)

    osb_fan_value = get_reg_value(glu_reg_base_addr + osb_fpga_reg)
    print("osb_fan_ctrl_reg = 0x%x\n" % osb_fan_value)

    sys_fan1_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_CTRL)
    print("     * sys_fan1_value = 0x%x" % sys_fan1_value)
    sys_fan2_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_CTRL)
    print("     * sys_fan2_value = 0x%x" % sys_fan2_value)
    sys_fan3_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_CTRL)
    print("     * sys_fan3_value = 0x%x\n" % sys_fan3_value)

    osb_fan1_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_CTRL)
    print("     * osb_fan1_value = 0x%x" % osb_fan1_value)
    osb_fan2_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_CTRL)
    print("     * osb_fan2_value = 0x%x\n" % osb_fan2_value)

    fan_pwm_value_str = "128"

    if verbose:
        print("sys_cmde is :", "echo "+fan_pwm_value_str+" > "+fan_pwm_cmde)

    sysfs_write(fan_pwm_cmde, fan_pwm_value_str)
    pwm_rdback = sysfs_read_string(fan_pwm_cmde)
    if verbose:
        print(hwmon_val, "pwm_rdback is", pwm_rdback)

    osb_fan_value = get_reg_value(glu_reg_base_addr + osb_fpga_reg)
    print("osb_fan_ctrl_reg = 0x%x\n" % osb_fan_value)

    sys_fan1_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_CTRL)
    print("     * sys_fan1_value = 0x%x" % sys_fan1_value)
    sys_fan2_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_CTRL)
    print("     * sys_fan2_value = 0x%x" % sys_fan2_value)
    sys_fan3_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_CTRL)
    print("     * sys_fan3_value = 0x%x\n" % sys_fan3_value)

    osb_fan1_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_CTRL)
    print("     * osb_fan1_value = 0x%x" % osb_fan1_value)
    osb_fan2_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_CTRL)
    print("     * osb_fan2_value = 0x%x\n" % osb_fan2_value)

    fan_pwm_value_str = "255"

    if verbose:
        print("sys_cmde is :", "echo "+fan_pwm_value_str+" > "+fan_pwm_cmde)

    sysfs_write(fan_pwm_cmde, fan_pwm_value_str)
    pwm_rdback = sysfs_read_string(fan_pwm_cmde)
    if verbose:
        print(hwmon_val, "pwm_rdback is", pwm_rdback)

    osb_fan_value = get_reg_value(glu_reg_base_addr + osb_fpga_reg)
    print("osb_fan_ctrl_reg value = 0x%x\n" % osb_fan_value)

    sys_fan1_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN1_CTRL)
    print("     * sys_fan1_value = 0x%x" % sys_fan1_value)
    sys_fan2_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN2_CTRL)
    print("     * sys_fan2_value = 0x%x" % sys_fan2_value)
    sys_fan3_value = get_reg_value(glu_reg_base_addr + REG_SYS_FAN3_CTRL)
    print("     * sys_fan3_value = 0x%x\n" % sys_fan3_value)

    osb_fan1_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN1_CTRL)
    print("     * osb_fan1_value = 0x%x" % osb_fan1_value)
    osb_fan2_value = get_reg_value(glu_reg_base_addr + REG_OSB_FAN2_CTRL)
    print("     * osb_fan2_value = 0x%x\n" % osb_fan2_value)

    print("\n******************************************************")
    print(" print now into register and read corresponding hwmon")
    print("******************************************************\n")

    osb_fan_ctrl_value = 0x90025
    reg_write(glu_reg_base_addr + osb_fpga_reg, osb_fan_ctrl_value)

    if verbose:
        print("Write 0x%x at %s reg. FPGA_addr = 0x%x" %(osb_fan_ctrl_value, osb_fpga_reg_str, osb_fpga_reg))

    hwmon_pwm_rdback = sysfs_read_string(fan_pwm_cmde)
    if verbose:
        print("hwmon pwm_rdback is", hwmon_pwm_rdback, "\n")

    i = 0
    for elt in enumerate(fan_list):
        fan = fan_list[i]
        hwmon_val = driver_list[i]
        fan_pwm_cmde = "/sys/class/hwmon/"+hwmon_val+"/pwm1"
        pwm_rdback = sysfs_read_string(fan_pwm_cmde)
        if verbose:
            print(fan, hwmon_val, "pwm_rdback is", pwm_rdback)
        i += 1

    return
#
#================================================================================
#
#
#================================================================================
#
@DIAG("OSB periphal and front_panel fans PWM/TACHI check")
def t_osb_fans_check(cmd):

    verbose = False
    debug   = False

    error_flag = False

    try:
        (command, consigne) = cmd.split()

        if len(consigne.split('-')) == 2:
            (fan_localisation, fct_type) = consigne.split('-')

            if fan_localisation == "LATERAL" or fan_localisation == "FRONT_PANEL" or fan_localisation == "ALL":
                print_info("fan_localisation is %s" %fan_localisation)
            else:
                print_error("fan_localisation : incorrect value : MUST be LATERAL or FRONT_PANEL or ALL")
                error_flag = True

            if fct_type == "PWM" or fct_type == "TACHI":
                print_info("function_type is %s" %fct_type)
            else:
                print_error("function_type : incorrect value : must be PWM or TACHI")
                error_flag = True

            if error_flag:
                return False

        if debug == True:
            print("   command = ", command, " ; consigne = ", consigne)
            print("   fan_localisation = %s ; function_type = %s" %(fan_localisation, fct_type))

    except:
        print_error("   Syntax error : Command must be : t_osb_fans_check <consigne>")
        print_error("                                                                    ")
        print_error("   where <consigne> is : <FAN_LOCALISATION>-<FUNCTION_TYPE>")
        print_error("   With                : <FAN_LOCALISATION> = LATERAL|FRONT_PANEL|ALL")
        print_error("                         <FUNCTION_TYPE> = PWM|TACHI")

        return False

    glu_reg_base_addr = get_glu_base_addr()

    fan_topo_list = OSB_fan_topology()

    if verbose:
        print ("\n  * fan_topo_list is ", fan_topo_list, "\n")

    if len(fan_topo_list) == 0:
        print_error("WARNING : Fan topology CAN'T be extracted")
        driver_state = "OFF"
        fan_topo_list =  ['hwmon_x', 'fan0', 'hwmon_y', 'fan1', 'hwmon_z', 'fan2', 'hwmon_f', 'fan3',  'hwmon_g', 'fan4']
#       return False
    else:
        driver_state = "ON"

    driver_list = []
    fan_list = []
    localisation_list = []
    position_list = []
    enable_list = []
    osb_fpga_reg_str_list = []
    osb_fpga_pwm_reg_list = []
    osb_fpga_tachi_reg_list = []

    i = 0
    for elt in enumerate(fan_topo_list):
        if debug:
            print("****** elt_%d = " %i, elt)
        if i == 0 or i%2 == 0:
            driver_list.append(elt[1])
        else:
            fan_list.append(elt[1])
            if "fan0" in elt[1]:
                localisation_list.append("LATERAL")
                position_list.append("REAR")
                enable_list.append("FAN_EN")
                osb_fpga_reg_str_list.append("REG_SYS_FAN1_CTRL")
                osb_fpga_pwm_reg_list.append(REG_SYS_FAN1_CTRL)
                osb_fpga_tachi_reg_list.append(REG_SYS_FAN1_TACHI_VAL)

            elif "fan1" in elt[1]:
                localisation_list.append("LATERAL")
                position_list.append("MIDDLE")
                enable_list.append("FAN_EN")
                osb_fpga_reg_str_list.append("REG_SYS_FAN2_CTRL")
                osb_fpga_pwm_reg_list.append(REG_SYS_FAN2_CTRL)
                osb_fpga_tachi_reg_list.append(REG_SYS_FAN2_TACHI_VAL)

            elif "fan2" in elt[1]:
                localisation_list.append("LATERAL")
                position_list.append("FRONT")
                enable_list.append("FAN_EN")
                osb_fpga_reg_str_list.append("REG_SYS_FAN3_CTRL")
                osb_fpga_pwm_reg_list.append(REG_SYS_FAN3_CTRL)
                osb_fpga_tachi_reg_list.append(REG_SYS_FAN3_TACHI_VAL)

            elif "fan4" in elt[1]:
                localisation_list.append("FRONT_PANEL")
                position_list.append("RIGHT")
                enable_list.append("FAN_DIS")
                osb_fpga_reg_str_list.append("REG_OSB_FAN2_CTRL")
                osb_fpga_pwm_reg_list.append(REG_OSB_FAN2_CTRL)
                osb_fpga_tachi_reg_list.append(REG_OSB_FAN2_TACHI_VAL)
            else:
                localisation_list.append("FRONT_PANEL")
                position_list.append("LEFT")
                enable_list.append("FAN_DIS")
                osb_fpga_reg_str_list.append("REG_OSB_FAN1_CTRL")
                osb_fpga_pwm_reg_list.append(REG_OSB_FAN1_CTRL)
                osb_fpga_tachi_reg_list.append(REG_OSB_FAN1_TACHI_VAL)

        if debug:
            print("\ndriver_list is", driver_list)
            print("fan_list is", fan_list)
            print("position_list is", position_list,"\n")
        i += 1

    if fct_type == "PWM":



        print("\nReadback of selected fans PWM value :\n")

        if driver_state == "ON":
            print("> using driver (hwmon) readback :")

            if fan_localisation == "LATERAL" or fan_localisation == "ALL":
                i = 0
                for elt in enumerate(fan_list):
                    if localisation_list[i] == "LATERAL":
                        hwmon_val = driver_list[i]
                        fan_pwm_cmde = "/sys/class/hwmon/"+hwmon_val+"/pwm1"
                        pwm_rdback = int(sysfs_read(fan_pwm_cmde))
                        speed_val = pwm_rdback & 0xff
                        speed_percent = int((speed_val * 100) / 255)
                        print("     *", localisation_list[i], fan_list[i], "(%s)" %position_list[i], hwmon_val,  "pwm_rdback is", pwm_rdback, "(speed = %d" %speed_percent, "%)" )
                    i += 1

            if fan_localisation == "FRONT_PANEL" or fan_localisation == "ALL":
                i = 0
                for elt in enumerate(fan_list):
                    if localisation_list[i] == "FRONT_PANEL":
                        hwmon_val = driver_list[i]
                        fan_pwm_cmde = "/sys/class/hwmon/"+hwmon_val+"/pwm1"
                        pwm_rdback = int(sysfs_read_string(fan_pwm_cmde))
                        speed_val = pwm_rdback & 0xff
                        speed_percent = int((speed_val * 100) / 255)
                        print("     *", localisation_list[i], fan_list[i], "(%s)" %position_list[i], hwmon_val,  "pwm_rdback is", pwm_rdback, "(speed = %d" %speed_percent, "%)")
                    i += 1

        print("\n> using OSB_FPGA fans PWM_reg. readback :")

        if fan_localisation == "LATERAL" or fan_localisation == "ALL":
            i = 0
            for elt in enumerate(fan_list):
                if localisation_list[i] == "LATERAL":
                    osb_fpga_reg = osb_fpga_pwm_reg_list[i]
                    sys_fan_value = get_reg_value(glu_reg_base_addr + osb_fpga_reg)
                    speed_val = sys_fan_value & 0xff
                    print("     * LATERAL", fan_list[i], "(%s)" %position_list[i], "PWM value = 0x%x (speed = %d" %(sys_fan_value, speed_val), "%)")
                i += 1

        if fan_localisation == "FRONT_PANEL" or fan_localisation == "ALL":
            i = 0
            for elt in enumerate(fan_list):
                if localisation_list[i] == "FRONT_PANEL":
                    osb_fpga_reg = osb_fpga_pwm_reg_list[i]
                    osb_fan_value = get_reg_value(glu_reg_base_addr + osb_fpga_reg)
                    speed_val = osb_fan_value & 0xff
                    print("     * FRONT_PANEL", fan_list[i], "(%s)" %position_list[i], "PWM value = 0x%x (speed = %d" %(osb_fan_value, speed_val), "%)")
                i += 1

    else:

        print("\nReadback of selected fans TACHI value :\n")

        if driver_state == "ON":
            print("> using driver (/sys/platform) readback :")

            if fan_localisation == "LATERAL" or fan_localisation == "ALL":

                i = 0
                for elt in enumerate(fan_list):
                    if localisation_list[i] == "LATERAL":
                        fan_num = fan_list[i]
                        fan_tachi_cmde = "/sys/platform/osb_tachi/"+fan_num
                        #print("    * sys_cmde is :", "cat "+fan_tachi_cmde)
                        tachi_value = sysfs_read(fan_tachi_cmde)
                        print("     *", localisation_list[i], fan_list[i], "(%s)" %position_list[i], "fan tachi_value is 0x%x" %tachi_value)
                    i += 1

            if fan_localisation == "FRONT_PANEL" or fan_localisation == "ALL":
                i = 0
                for elt in enumerate(fan_list):
                    if localisation_list[i] == "FRONT_PANEL":
                        fan_num = fan_list[i]
                        fan_tachi_cmde = "/sys/platform/osb_tachi/"+fan_num
                        #print("    * sys_cmde is :", "cat "+fan_tachi_cmde)
                        tachi_value = sysfs_read(fan_tachi_cmde)
                        print("     *", localisation_list[i], fan_list[i], "(%s)" %position_list[i], "fan tachi_value is 0x%x" %tachi_value)
                    i += 1

        print("\n> using OSB_FPGA fans TACHI_reg. readback :")

        if fan_localisation == "LATERAL" or fan_localisation == "ALL":

            i = 0
            for elt in enumerate(fan_list):
                if localisation_list[i] == "LATERAL":
                    fan_num = fan_list[i]
                    sys_fan_tachi_value = get_reg_value(glu_reg_base_addr + osb_fpga_tachi_reg_list[i])
                    print("     *", localisation_list[i], fan_list[i], "(%s)" %position_list[i], "tachi_value is 0x%x" %sys_fan_tachi_value)
                i += 1

        if fan_localisation == "FRONT_PANEL" or fan_localisation == "ALL":

            i = 0
            for elt in enumerate(fan_list):
                if localisation_list[i] == "FRONT_PANEL":
                    fan_num = fan_list[i]
                    osb_fan_tachi_value = get_reg_value(glu_reg_base_addr + osb_fpga_tachi_reg_list[i])
                    print("     *", localisation_list[i], fan_list[i], "(%s)" %position_list[i], "tachi_value is 0x%x" %osb_fan_tachi_value)
                i += 1

    return None

def udev_add_rules():
    lines = ['SUBSYSTEM=="tty", KERNELS=="3-1.1.1:1.0", SYMLINK="ttyUSB_M1"','SUBSYSTEM=="tty", KERNELS=="3-1.1.4:1.0", SYMLINK="ttyUSB_M2"']
    with open('/etc/udev/rules.d/99-tmp-pydiag.rules', 'w') as f:
        for line in lines:
            f.write(line)
            f.write('\n')
    f.close()

    #Reload rules
    p = subprocess.Popen("udevadm control --reload-rules", shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    p = subprocess.Popen("udevadm test $(udevadm info -q path -n /dev/ttyUSB0)", shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    p = subprocess.Popen("udevadm test $(udevadm info -q path -n /dev/ttyUSB1)", shell=True,
                         stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    p.wait();
    return True;

@DIAG("Test RS485")
def t_osb_rs485(cmd):
    buffer = b'{aM\\\xe1\x8a\xe5L\xc4+rW9\xed\x8eFR\x10\xff\xe9\xcf;[' \
             b'\xe3\x1d\x1b\xa3\xab\x84\xdf\xe0NA\xad\xd5VQ\xeb\x15\x15"\xdf\x14\xe4\xc4\xa1\x83\xaa\x87\xd0\x1e&7' \
             b'\x8b\xc1\xbd\x04\xa0/:\xfb\xa0\xbb\xf0!\xa8\x08<\xe7%\x99\xb2#\xd8y{\xebzMt\xce| ' \
             b'\x01\xed\xb8\x99\xbb\xa3\xb1j;\xbe\xc7J`\xe3\xe1\x8eg\xfdC\x97@&D\x10\x08Vd4R"\n\x9b\xeaz\x07\xc7>5' \
             b'\xe9\xdd\xcfmK\xbb\xc3\xbb!\xf1\x83\x81]N\xe6=\xe3\xd6\x84\xb1;\xac\xde\x87\x91\x7fq#\xe3\xc4\xf00[' \
             b'\xd3\x8e\x9at\xa3\x10\xbf\x8a\xc7\xd8\xa9\x15\xcfK\xfc\x0b\x86\x1a\xfa\xfb\x17iX,' \
             b'\r\x96\x18\xbf\xd7\xb7=\xad\xc0\xfd{' \
             b'\x94\xecG\x03\xf7\x89\xaa\xc9\xe6\x89\x94l\x85\x1aU\x13\x07\xd7\'\x0b\xfc!*\x89\xad\xe6\x1eK\xd9E\xed' \
             b'\x01\x86C\x84\xdc\xe7\x0f\x95\xc9\x9d"z\x9a^7\xd1RI:\xbb\xa1\x84$\xe4\x1e\x8c\xed\x90\xe3\\?e3\xc1S' \
             b'\xe5\xf9|\x1a\xc4\xefC\xee\xab\xb4\xb3\x81\xd6\n\x17\xf1\xaaJPm\xb9\xf5\xec\xbb\xe87\xfe\xaf\x9c\xc5' \
             b'\x8d\xbd\xed2/\xb9\xe9\x8d\x9a\x99\xf8\xe4\xd5L\xc9\xe0\xac3\xf2\xe3}\xa2@?\xfc\xde>^$C&\xcd\xe7\xb2' \
             b'\xa9\xddY/a\xcbE\x92sJ\xc0\xb4Y\x84\x14\xbaR!\xb1!\xbe\xcdv!\xa4\xde\x85\xd0\x0b\xa8\\p~\xfd\x807\x17' \
             b'\xd8\xcc\xc3\x07\xea_\xab\x8fF8\x1e\x0ed\x86\x88\x14\xb4\xbe{\xa9\xbd\x0b6O\x0cB\xf9\x8e\xf8# ' \
             b'\xd5\xbb\xeb\x15\x10f\x9b\x1axx\x98\xc8y]\xee\xde\xae\xbfh\xa6\x12\xb9\x18\xb8N\x17\x7f\xaau\x9dp\x99' \
             b'\x93\xa1<_F\x11\x154O\xb9\x97<nYz>j\x91\t\xdc2\xc3\xfc[' \
             b'\xdd\xc6\x0b1I\x83\x99\xf8\xf2BI\x97\r\x8a\xd2}!\xa13\x0e\x9b1\xbf\xb5 ' \
             b'\x92\xfe\xd9\xbfrtW\xe7\xe3\x12\xb1\x84\xf7z\x15\xcdRGa\xf6\x9c\xf3\xfc\xe7\x7f\xbb\xefq\xc9BRAc\x9e' \
             b'\x8b\x8a\x99\xde\xed\xa1\x14\x8a\xf6C@\x17\x114\x92\xd0\xbf\xb1$1\xb0lS\x86\x02\x83\xb9\xbf\x1b\xb3' \
             b'\x1bE}\x94chO\xf9yNs\xe5Lm*\xf5\x80\xba\xf9\x97wrh"T\x81a\x0b\xaf\xa6\x8f,' \
             b'\xfe\x8a].\xd1\x16Fi\x14\x8b\xe3\x06;\xdeGMq\x1e\xbe{' \
             b'\xcd\xb3\xd5\x95\'9\xeew\x06\x12s\xf2\x1f\\E\xb2\xa3\x9bP\xfc\x05\xca\x81\x03\xcb\x1f\xe1\xa2\xc0R\xbe' \
             b'\x07N\x0c\xe0\xb7\xb4$\xf9\xdf\x87\xbc\x86\x05U\xc7\x00\xd5\xac\x96\xd4sz\xd0\x9d7\x10\xf2\x01Z~\xed' \
             b'\xf5\xc6\x8aUA{\x17\x8a\xdc\x02GA\x8d\x98\x0f\xb9p\xef9\x91\x19k\x0f\x8c\x0eS\x17\x03\x12\xdf\x11\xcd' \
             b'\x1f\xc3V\x8e=M\x1f\x97\x0fW\xde\x04\x1e!y6 ' \
             b'\n\xbe\xca\x86\xe3\x13D\x89\x84N\xfbK\xf2L\xc9\xe0\x1f\xb4\xc1r\r\x91\x1c\xe7P\xae\x11\xe2^Z\xf9\xa5' \
             b'-\n\xcf\xad\x97Wl\xa3\x94\xfc\xffD\xf0\xc0\xb5\x88\xb0>\x16(' \
             b'\xdb\xd9\x82\x13\x9d\x82^O\xa4\xc0E\xea\xf8\xd5i)\x03\x1e\x01\xc3N\x89S\x97\x94\xeabs\x85K\xb7\xf0\x8e' \
             b'\xf2\xd3\xe7s!\xa1bp\xd70kgilz\x04\x07\xf8\xa4]D\xb9x9\x83\x88H\xe1P\x8c%S\xfb\x12\x14\xd1\x845`\x92' \
             b'\x84W\xfdO\x82\xc6\xa1Q\x01\x1e\xa8P\x17mS\x14\xf2\x8e\xc1\xbeu\xc0Q\x96\x1d\xdb\xbb\xf0\x9b\xa7i\xb6' \
             b'\xf5\xf4\x01AJf<\xeb;\x86\xea\x05J\x02\x14.\xd0@\xe1\xd9\xd8D4\xd5z"\r\nbh\x19\xec\x18\xe4W\xd0\x83' \
             b'\x95\xe8\x84\x05\xc0/s\xa1\xbbB\xdc\x7f\x91\xf0\xe5\x89\xc7\r\xef\xc2\x1e\\\xf8?\x1c\xf5\x82\xd1' \
             b'}\xf4NL\x85\x15\xf3\xb1\x89\xa8\xa5e\x1aJ\x9f\xfa\x0bL\xdb\xd7\rS\n\xcf\xba|\xe2\xae\x0fd\xb9\xc5\xe05' \
             b'\x7f\x0bQ\xaaO>C\x84G\xa0\xd8\xc9\x9e\x13:2Ni\x12\xb3\xd2\xab+gh\xb4\xf3\xecwm\xda\xfam\xea4\x1a\x966h' \
             b'=\xe9\x8a\xcc\xca\xe4g\xaa6\x89\x90\x0bF\xb4\xe3\xa9\x1b\x1a\xc6\x17\x8c\xba\xcb)\xdb\xd9\x17\xd0\xb7' \
             b'\xd4=\x91\x8f\x04'
    buffer_size = len(buffer)

    udev_add_rules()

    serial_device = "/dev/ttyUSB_M1"
    rs485_device = "/dev/ttyRS485"

    if not os.path.exists(rs485_device, ):
        print_error("# RS485 not detected")
    if not os.path.exists(serial_device):
        print_error("# Testing serial port not detected")

    b200_serial = serial.Serial(serial_device, baudrate=115200, bytesize=8, parity='N', stopbits=1, timeout=15,
                               xonxoff=0, rtscts=0)
    b01_serial = serial.Serial(rs485_device, baudrate=115200, bytesize=8, parity='N', stopbits=1, timeout=15, xonxoff=0,
                               rtscts=0)

    serials = [b200_serial, b01_serial]
    for _ in range(1, 3):
        serials[0].flushOutput()
        serials[0].flushInput()
        serials[1].flushOutput()
        serials[1].flushInput()
        print_info("# transferring buffer from {} to {}".format(serials[0].name, serials[1].name))
        serials[0].write(buffer)
        read_buff = serials[1].read(buffer_size)
        if buffer != read_buff:
            print_error("# Communication error for {} -> {}, send {} - received {}".format(
                serials[0].name, serials[1].name, len(buffer), len(read_buff)))
            file = open("/var/tmp/buffer.bin", 'wb')
            file.write(buffer)
            file.flush()
            file = open("/var/tmp/read_buffer.bin", 'wb')
            file.write(read_buff)
            file.flush()
            return False
        serials[0], serials[1] = serials[1], serials[0]

    b200_serial.close()
    b01_serial.close()

    return True

@DIAG("OSB Master SPI debug")
def t_osb_master_spi_test(cmd):

    verbose_flag = True

    module_presence_info(OSB_flag = 1, B200_flag = 0, verbose_flag = verbose_flag)

    glu_reg_base_addr = get_glu_base_addr()

    if verbose_flag:
        mux_ctrl_value = get_reg_value(glu_reg_base_addr + REG_MUX_CTRL)

        print("* OSB mux_ctrl_value = 0x%x" % mux_ctrl_value)
        print("     - MUX_CTRL(0) = mux_sel = %d (0 : switch selected ; 1 : DTSS selected)" %(mux_ctrl_value & 0x1))
        print("     - MUX_CTRL(1) = mux_oe  = %d\n" %((mux_ctrl_value & 0x2)/2))

    spi_ctrl_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_CTRL)
    print("* OSB SPI Master SPI_CTRL register = 0x%x" % spi_ctrl_value)
    print("     - SPI_CTRL(0) = spi_start = %d" %(spi_ctrl_value & 0x1))
    print("     - SPI_CTRL(1) = spi_sel16 = %d" %(spi_ctrl_value & 0x2))
    print("     - SPI_CTRL(2) = spi_cpol  = %d" %(spi_ctrl_value & 0x4))
    print("     - SPI_CTRL(3) = spi_cpha  = %d" %(spi_ctrl_value & 0x8))
    spi_ctrl_freq = (spi_ctrl_value & 0xff00) / 256
    print("     - SPI_CTRL(15:8) = spi_freq  = %d" %spi_ctrl_freq)

    input("\nB200 analysis : Enter")

    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    print("\n* B200 CPLD version is 0x%2x (using utsspi driver)" % cpld_version)

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    print("\n* 2nd read of B200 CPLD version is 0x%2x (using utsspi driver)" % cpld_version)

    spi_ctrl_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_CTRL)
    print("\n* OSB SPI Master SPI_CTRL register = 0x%x" % spi_ctrl_value)
    print("     - SPI_CTRL(0) = spi_start = %d" %(spi_ctrl_value & 0x1))
    print("     - SPI_CTRL(1) = spi_sel16 = %d" %(spi_ctrl_value & 0x2))
    print("     - SPI_CTRL(2) = spi_cpol  = %d" %(spi_ctrl_value & 0x4))
    print("     - SPI_CTRL(3) = spi_cpha  = %d" %(spi_ctrl_value & 0x8))
    spi_ctrl_freq = (spi_ctrl_value & 0xff00) / 256
    print("     - SPI_CTRL(15:8) = spi_freq = %d" %spi_ctrl_freq)

    input("\nEnter")

    print("\nRead/Write OSB FPGA SPI_Master registers : ")

    data_wr = 0xa5a55a5a
    print("\nSPI Master slv0 write value = 0x%x" % data_wr)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_SLV0, data_wr)

    spi_slv0 = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_SLV0)
    print("SPI Master slv0 read = 0x%x" % spi_slv0)

    data_wr = 0x12345678
    print("SPI Master slv1 write value = 0x%x" % data_wr)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_SLV1, data_wr)

    spi_slv1 = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_SLV1)
    print("SPI Master slv1 read = 0x%x" % spi_slv1)

    input("\nEnter")
    print("\n Initialize SPI Master SPI_CTRL register (SPI_START = '0')")

    # SPI_CTRL (slv2) write access :
    #
    # SPI_CTRL(0)    = MOD_SPI_START
    # SPI_CTRL(1)    = MOD_SPI_SEL16
    # SPI_CTRL(2)    = MOD_SPI_CPOL
    # SPI_CTRL(3)    = MOD_SPI_CPHA
    # SPI_CTRL(15:8) = MOD_SPI_FREQ
    #
    # SPI_CTRL read access : spi_ctrl(31:8) & spi_status & spi_ctrl(6:0)

    mod_spi_start = 0x0
    mod_spi_sel16 = 0x1
    mod_spi_cpol  = 0x1
    mod_spi_cpha  = 0x1
    mod_spi_freq  = 0x5
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    print("SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    spi_ctrl_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_CTRL)
    print("SPI Master SPI_CTRL read value = 0x%x" % spi_ctrl_value)

    # SPI_TRANSMIT reg (slv3) Write access : slv3(31:0)

    spi_slv3 = 0x2468ace2
    print("SPI Master SPI_TRANSMIT write value = 0x%x" % spi_slv3)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, spi_slv3)

    #
    # SPI_TRANSMIT reg read access : spi_slv3(31:16) & spi_receive_value(15:0)
    #                                where spi_slv3(23:16) = spi_transmit_value(7:0)

    spi_transmit_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
    print("SPI Master SPI_TRANSMIT read value = 0x%x" % spi_transmit_value)

    input("\nEnter")

    print("\nLaunch spi_tfer (SPI_START = '1') with data = 0x68")

    mod_spi_sel16 = 0x0
    mod_spi_cpol  = 0x0
    mod_spi_cpha  = 0x0
    mod_spi_freq  = 0x5
    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    print("SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    input("\nEnter")

    print("\n Initialize SPI Master SPI_CTRL register with new FREQ (SPI_START = '0')")
    mod_spi_start = 0x0
    mod_spi_freq  = 0x20
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    print("\nSPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    print("\nLaunch spi_tfer (SPI_START = '1') with data = 0x68")

    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    print("SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    input("\nEnter")

    print("\n Initialize SPI Master SPI_CTRL register with new FREQ (SPI_START = '0')")
    mod_spi_start = 0x0
    mod_spi_sel16 = 0x0
    mod_spi_cpol  = 0x0
    mod_spi_cpha  = 0x0
    mod_spi_freq  = 0xf
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    print("\nSPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    print("\nLaunch spi_tfer (SPI_START = '1') with data = 0x68")
    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    print("SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    # B200

    input("\nB200 analysis : Enter")

    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    print("\n* B200 CPLD version is 0x%2x (using utsspi driver)" % cpld_version)

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    print("\n* 2nd B200 CPLD version read : 0x%2x (using utsspi driver)" % cpld_version)

    input("\nEnter")

    print("\nRead CPLD reg. cAddress_RegLvds_7_0 using Utsspi driver :")

    value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_7_0)
    print("\nreg. value @ 0x%1x (cAddress_RegLvds_7_0) : 0x%x" %(SPI_ADDR_REGLVDS_7_0, value))

    input("\nEnter")

    wr_data = 0x55
    print("\nWrite 0x%x into CPLD reg. cAddress_RegLvds_7_0 using Utsspi driver:" %wr_data)
    ver = Utsspi.write_mod_register8(SPI_ADDR_REGLVDS_7_0, wr_data)

    input("\nEnter")
    print("\nRead CPLD reg. cAddress_RegLvds_7_0 using Utsspi driver:")

    value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_7_0)
    print("reg. value after write @ 0x%1x (cAddress_RegLvds_7_0) : 0x%x" %(SPI_ADDR_REGLVDS_7_0, value))

    input("\nEnter")
    wr_data = 0x1
    print("\nWrite 0x%x into CPLD reg. cAddress_RegLvds_7_0 using Utsspi driver:" %wr_data)
    ver = Utsspi.write_mod_register8(SPI_ADDR_REGLVDS_7_0, wr_data)

    input("\nEnter")
    print("\nRead CPLD reg. cAddress_RegLvds_7_0 using Utsspi driver:")
    value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_7_0)
    print("reg. value after reset @ 0x%1x (cAddress_RegLvds_7_0) : 0x%x" %(SPI_ADDR_REGLVDS_7_0, value))

    input("\n header transfer : Enter")

    # header trfer :
    #================

    header = 0xc0

    print("\nSPI Master SPI_TRANSMIT write value = 0x%x" % header)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, (header * 65536))

    header_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
    print("SPI Master SPI_TRANSMIT read value = 0x%x" % header_value)

    time.sleep(0.5)

    mod_spi_sel16 = 0x0
    mod_spi_cpol  = 0x0
    mod_spi_cpha  = 0x0
    mod_spi_freq  = 0xf

    mod_spi_start = 0x1
    spi_ctrl = (mod_spi_freq * 256) | (mod_spi_cpha * 8) | (mod_spi_cpol * 4) | (mod_spi_sel16 *2) | mod_spi_start

    print("SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    input("\n reg_addr transfer : Enter")

    # reg_addr trfer :
    #==================

    reg_addr_value = SPI_ADDR_REGLVDS_7_0 * 2

    print("\nSPI Master SPI_TRANSMIT write value = 0x%x" % reg_addr_value)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, ( reg_addr_value * 65536))

    addr_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT)
    print("SPI Master SPI_TRANSMIT read value = 0x%x" % addr_value)

    print("SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    # reg_addr_value = SPI_ADDR_VERSION

    input("\n data transfer : Enter")

    # data trfer :
    #==============

    data_reg_value = 0x55

    print("\nSPI Master SPI_TRANSMIT write value = 0x%x" % data_reg_value)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_TRANSMIT, (data_reg_value * 65536))

    print("SPI Master SPI_CTRL write value = 0x%x" % spi_ctrl)
    reg_write(glu_reg_base_addr + REG_OSB_SPI_CTRL, spi_ctrl)

    input("\n Verify spi transfer using utsspi driver : Enter")

    time.sleep(3)

    spi_ctrl_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_CTRL)
    print("\n\n* OSB SPI Master SPI_CTRL register = 0x%x" % spi_ctrl_value)
    print("     - SPI_CTRL(0) = spi_start = %d" %(spi_ctrl_value & 0x1))
    print("     - SPI_CTRL(1) = spi_sel16 = %d" %(spi_ctrl_value & 0x2))
    print("     - SPI_CTRL(2) = spi_cpol  = %d" %(spi_ctrl_value & 0x4))
    print("     - SPI_CTRL(3) = spi_cpha  = %d" %(spi_ctrl_value & 0x8))
    spi_ctrl_freq = (spi_ctrl_value & 0xff00) / 256
    print("     - SPI_CTRL(15:8) = spi_freq  = %d" %spi_ctrl_freq)

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    print("\n* B200 CPLD version read : 0x%2x (using utsspi driver)" % cpld_version)

    time.sleep(0.5)

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    print("\n* B200 CPLD version read : 0x%2x (using utsspi driver)" % cpld_version)

    input("\nEnter")

    print("\nRead CPLD reg. cAddress_RegLvds_7_0 using Utsspi driver:")
    value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_7_0)
    print("reg. value after reset @ 0x%1x (cAddress_RegLvds_7_0) : 0x%x" %(SPI_ADDR_REGLVDS_7_0, value))

    input("\nEnter")

    cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
    print("\n* B200 CPLD version read : 0x%2x (using utsspi driver)" % cpld_version)

    input("\nEnter")

    print("\n* OSB SPI Master SPI_CTRL register = 0x%x" % spi_ctrl_value)
    print("     - SPI_CTRL(0) = spi_start = %d" %(spi_ctrl_value & 0x1))
    print("     - SPI_CTRL(1) = spi_sel16 = %d" %(spi_ctrl_value & 0x2))
    print("     - SPI_CTRL(2) = spi_cpol  = %d" %(spi_ctrl_value & 0x4))
    print("     - SPI_CTRL(3) = spi_cpha  = %d" %(spi_ctrl_value & 0x8))
    spi_ctrl_freq = (spi_ctrl_value & 0xff00) / 256
    print("     - SPI_CTRL(15:8) = spi_freq  = %d" %spi_ctrl_freq)

    return True
#
#================================================================================
#
#
#================================================================================
#
@DIAG("DTSS SPI Debug")
def t_osbdtss_spi_dbg(cmd):

    verbose_flag = True
    error_flag = False

    Read_f = 1
    Write_f = 0

    try:
        (command, consigne) = cmd.split()

        (board_type, spi_driver_flag) = consigne.split('-')

        if board_type == "B200" or board_type == "DTSS":
            pass
        else:
            print_error("Error : board_type MUST be : B200 or DTSS")
            error_flag = True

        if spi_driver_flag == "TRUE":
            Utsspi_driver = True
        elif spi_driver_flag == "FALSE":
            Utsspi_driver = False
        else:
            print_error("Error : spi_driver_flag MUST be : TRUE or FALSE")
            error_flag = True

        if error_flag == True:
            return False

    except:
        print_error("# Command must be: t_osbdtss_spi_dbg <board_type>-<spi_driver_flag>")
        print_error("#           with : <board_type> = B200|DTSS")
        print_error("#           with : <spi_driver_flag> = TRUE|FALSE")
        return False

    # Select SPI_DTSS

#   module_presence_info(OSB_flag = 1, B200_flag = 0, verbose_flag = verbose_flag)

    glu_reg_base_addr = get_glu_base_addr()

    module_presence_rdback_value = get_reg_value(glu_reg_base_addr + REG_MOD_PRESENCE_RDBCK)

    switch_bit        = module_presence_rdback_value & 0x1
    dtss_bit          = (module_presence_rdback_value & 0x2)/2
    dtss_fpga_rdy_bit = (module_presence_rdback_value & 0x4)/4

    print("* OSB module_presence_rdback register details :")
    print("     * module_presence_rdback_value read = 0x%x" % module_presence_rdback_value)
    print("         - module_presence_rdback(2) = DTSS_FPGA_RDY = %d ('0' = ABSENT ; '1' = present)" %dtss_fpga_rdy_bit)
    print("         - module_presence_rdback(1) = DTSS_PRES     = %d ('0' = ABSENT ; '1' = present)" %dtss_bit)
    print("         - module_presence_rdback(0) = SWITCH_PRES   = %d ('0' = ABSENT ; '1' = present)" %switch_bit, "\n")

    # pwrside S3/S4
    if board_type == "B200":
        side = 0
        slice = 3
    else:
        side = 0
        slice = 4

    Utsspi = utsspi.SpiModule(side, slice, "/dev/uts_spi")

    if board_type == "DTSS":

        if Utsspi_driver:
            if 1 == 0:
                ver_low = Utsspi.read_mod_register8(DTSS_SPI_XVERSION)
                ver_middle0 = Utsspi.read_mod_register8(DTSS_SPI_XVERSION+1)
                ver_middle1 = Utsspi.read_mod_register8(DTSS_SPI_XVERSION+2)
                ver_high    = Utsspi.read_mod_register8(DTSS_SPI_XVERSION+3)

                print("DTSS ver is", ver_high, ver_middle1, ver_middle0, ver_low)

                tnumber_value = Utsspi.read_mod_register32(DTSS_SPI_TNUMBER)
                print(" * 1. tnumber_reg(24:0) before reset read_32 = 0x%x" %tnumber_value)

                wr_data = 0x0

                print(" * 2. DTSS tnumber_reg(24:0) write_32 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register32(DTSS_SPI_TNUMBER, wr_data)

                tnumber_value = Utsspi.read_mod_register32(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_32 : 0x%x" %tnumber_value)

                wr_data = 0xbb5566aa
                print(" * 3. DTSS tnumber_reg(24:0) write_32 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register32(DTSS_SPI_TNUMBER, wr_data)

                # read_data = '1' & 0x66aa & x"00"= wr_data(15:0)
                tnumber_value = Utsspi.read_mod_register32(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_32 : 0x%x" %tnumber_value)

                tnumber_value = Utsspi.read_mod_register24(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_24 : 0x%x" %tnumber_value)

                tnumber_value = Utsspi.read_mod_register16(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_16 : 0x%x" %tnumber_value)

                tnumber_value = Utsspi.read_mod_register8(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_8 : 0x%x" %tnumber_value)

                wr_data = 0xbb5466aa
                print(" * 4. DTSS tnumber_reg(24:0) write_32 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register32(DTSS_SPI_TNUMBER, wr_data)

                # TNUMBER(24:0)
                # read_data(31 downto 0) =  "0000000" & TNUMBER(24:0)
                # tnumber_value = '1' & 0x7f64 & x"00" = wr_data(16) & wr_data(15:0) & x"00"

                tnumber_value = Utsspi.read_mod_register32(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_32 0x%x" %tnumber_value)

                wr_data = 0x27f64
                print(" * 6. DTSS tnumber_reg(24:0) write_32 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register32(DTSS_SPI_TNUMBER, wr_data)

                # read_data = '1' & 0x7f64 = wr_data(16) & wr_data(15:0) & x"00"
                tnumber_value = Utsspi.read_mod_register32(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_32 : 0x%x" %tnumber_value)

                # DTSS_SPI_PULSE register

                wr_data = 0x9a57e681
                print(" * 7. DTSS pulse_reg(31:0) write : 0x%x" %wr_data)
                test = Utsspi.write_mod_register32(DTSS_SPI_PULSE, wr_data)

                # read_data(31 downto 0) = PULSE_VALUE(31 downto 0) =
                # pulse_value = 0xe6810000 = wr_data(15:0) & x"0000"
                pulse_value = Utsspi.read_mod_register32(DTSS_SPI_PULSE)
                print("      DTSS pulse_reg read_32 : 0x%x" %pulse_value)

                # DTSS_SPI_NDURATION register

                wr_data = 0x1234feef
                print(" * 8. DTSS NOISE_DURATION_reg(16:0) write : 0x%x" %wr_data)
                # NOISE_DURATION(16 downto 0)

                # read_data(31 downto 8) = "0000000" & NOISE_DURATION (16 downto 0)
                # duration_value = 0xfeef00 = wr_data(15:0) & x"00"

                Utsspi.write_mod_register32(DTSS_SPI_NDURATION, wr_data)
                duration_value = Utsspi.read_mod_register32(DTSS_SPI_NDURATION)
                print("      DTSS duration_reg read_32 : 0x%x" %duration_value)

                wr_data = 0xb975feef
                print(" * 9. DTSS NOISE_DURATION_reg(16:0) write : 0x%x" %wr_data)
                # NOISE_DURATION(16 downto 0)

                # read_data(31 downto 8) = "0000000" & NOISE_DURATION (16 downto 0)
                # duration_value = 0x01feef00 = wr_data(16) & wr_data(15:0) & x"00"

                Utsspi.write_mod_register32(DTSS_SPI_NDURATION, wr_data)
                duration_value = Utsspi.read_mod_register32(DTSS_SPI_NDURATION)
                print("      DTSS duration_reg read_32 : 0x%x" %duration_value)

                duration_value = Utsspi.read_mod_register24(DTSS_SPI_NDURATION)
                print("      DTSS duration_reg read_24 : 0x%x" %duration_value)

                duration_value = Utsspi.read_mod_register16(DTSS_SPI_NDURATION)
                print("      DTSS duration_reg read_16 : 0x%x" %duration_value)

                duration_value = Utsspi.read_mod_register8(DTSS_SPI_NDURATION)
                print("      DTSS duration_reg read_8 : 0x%x" %duration_value)

                print("****************************")

                wr_data = 0x0

                print(" * 2. DTSS tnumber_reg(24:0) write_16 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register16(DTSS_SPI_TNUMBER, wr_data)

                tnumber_value = Utsspi.read_mod_register16(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_16 : 0x%x" %tnumber_value)

                wr_data = 0xb6a5

                print(" * 2. DTSS tnumber_reg(24:0) write_16 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register16(DTSS_SPI_TNUMBER, wr_data)

                tnumber_value = Utsspi.read_mod_register16(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_16 : 0x%x" %tnumber_value)

                tnumber_value = Utsspi.read_mod_register16(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_16 : 0x%x" %tnumber_value)

                tnumber_value = Utsspi.read_mod_register24(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_24 : 0x%x" %tnumber_value)

                tnumber_value = Utsspi.read_mod_register32(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_32 : 0x%x" %tnumber_value)

                print("****************************")

                wr_data = 0x0

                print(" * 1. DTSS tnumber_reg(24:0) write_8 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register8(DTSS_SPI_TNUMBER, wr_data)

                tnumber_value = Utsspi.read_mod_register8(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_8 : 0x%x" %tnumber_value)

                wr_data = 0x5500
                print(" * 2. DTSS tnumber_reg(24:0) write_8 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register8(DTSS_SPI_TNUMBER, wr_data)

                tnumber_value = Utsspi.read_mod_register8(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_8 : 0x%x" %tnumber_value)

                wr_data = 0xaa
                print(" * 3. DTSS tnumber_reg(24:0) write_8 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register8(DTSS_SPI_TNUMBER, wr_data)

                tnumber_value = Utsspi.read_mod_register8(DTSS_SPI_TNUMBER)
                print("      DTSS tnumber_reg read_8 : 0x%x" %tnumber_value)

            else:
                wr_data = 0x3355aa
                print("\n *** 1. DTSS tnumber_reg(24:0) write_8 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register8(DTSS_SPI_TNUMBER, wr_data)

                input("\nEnter")

                tnumber_value = Utsspi.read_mod_register8(DTSS_SPI_TNUMBER)
                print("\n      DTSS tnumber_reg read_8 : 0x%x" %tnumber_value)

                input("\nEnter")

                tnumber_value = Utsspi.read_mod_register16(DTSS_SPI_TNUMBER)
                print("\n      DTSS tnumber_reg read_16 : 0x%x" %tnumber_value)

                input("\nEnter")

                tnumber_value = Utsspi.read_mod_register32(DTSS_SPI_TNUMBER)
                print("\n      DTSS tnumber_reg read_32 : 0x%x" %tnumber_value)

                input("\nEnter")

                wr_data = 0x55aa
                print("\n *** 2. DTSS tnumber_reg(24:0) write_16 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register16(DTSS_SPI_TNUMBER, wr_data)

                input("\nEnter")

                tnumber_value = Utsspi.read_mod_register8(DTSS_SPI_TNUMBER)
                print("\n      DTSS tnumber_reg read_8 : 0x%x" %tnumber_value)

                input("\nEnter")

                tnumber_value = Utsspi.read_mod_register16(DTSS_SPI_TNUMBER)
                print("\n      DTSS tnumber_reg read_16 : 0x%x" %tnumber_value)

                input("\nEnter")

                tnumber_value = Utsspi.read_mod_register32(DTSS_SPI_TNUMBER)
                print("\n      DTSS tnumber_reg read_32 : 0x%x" %tnumber_value)

                input("\nEnter")

                wr_data = 0x3355aa
                print("\n *** 3. DTSS tnumber_reg(24:0) write_32 : 0x%x" %wr_data)
                test = Utsspi.write_mod_register32(DTSS_SPI_TNUMBER, wr_data)

                input("\nEnter")

                tnumber_value = Utsspi.read_mod_register8(DTSS_SPI_TNUMBER)
                print("\n      DTSS tnumber_reg read_8 : 0x%x" %tnumber_value)

                input("\nEnter")

                tnumber_value = Utsspi.read_mod_register16(DTSS_SPI_TNUMBER)
                print("\n      DTSS tnumber_reg read_16 : 0x%x" %tnumber_value)

                input("\nEnter")

                tnumber_value = Utsspi.read_mod_register32(DTSS_SPI_TNUMBER)
                print("\n      DTSS tnumber_reg read_32 : 0x%x" %tnumber_value)

        else:
            data_wr = 0x73
            print(" * DTSS tnumber_reg(24:0) write_8 : 0x%x" %data_wr)
            value = spi_master_transfer(Write_f, DTSS_SPI_TNUMBER, data_wr)
            time.sleep(1)
            value = spi_master_transfer(Read_f, DTSS_SPI_TNUMBER, 0)
            print("\nSPI Received Data Value = 0x%x" %value)

    #===============
    # B200 board :
    #===============

    else:
        if Utsspi_driver:
            cpld_version = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
            print("\n* B200 CPLD version read : 0x%2x (using utsspi driver)" % cpld_version)

            input("\nEnter")

            wr_data = 0x55
            print("\nWrite 0x%x into CPLD reg. cAddress_RegLvds_7_0 using Utsspi driver:" %wr_data)
            ver = Utsspi.write_mod_register8(SPI_ADDR_REGLVDS_7_0, wr_data)

            input("\nEnter")

            print("\nRead CPLD reg. cAddress_RegLvds_7_0 using Utsspi driver:")

            value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_7_0)
            print("reg. value after writing @ addr 0x%1x (cAddress_RegLvds_7_0) : 0x%x" %(SPI_ADDR_REGLVDS_7_0, value))

            input("\nEnter")

            wr_data = 0xaa
            print("\nWrite 0x%x into CPLD reg. cAddress_RegLvds_15_8 using Utsspi driver:" %wr_data)
            ver = Utsspi.write_mod_register8(SPI_ADDR_REGLVDS_15_8, wr_data)

            input("\nEnter")

            print("\nRead CPLD reg. cAddress_RegLvds_15_8 using Utsspi driver:")

            value = Utsspi.read_mod_register8(SPI_ADDR_REGLVDS_15_8)
            print("reg. value after writing @ addr 0x%1x (cAddress_RegLvds_5_8) : 0x%x" %(SPI_ADDR_REGLVDS_15_8, value))

            input("\nEnter")

            print("\nRead/Write OSB FPGA SPI_Master registers : ")

            spi_ctrl_value = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_CTRL)
            print("\n* OSB SPI Master SPI_CTRL register = 0x%x" % spi_ctrl_value)
            print("     - SPI_CTRL(0) = spi_start = %d" %(spi_ctrl_value & 0x1))
            print("     - SPI_CTRL(1) = spi_sel16 = %d" %(spi_ctrl_value & 0x2))
            print("     - SPI_CTRL(2) = spi_cpol  = %d" %(spi_ctrl_value & 0x4))
            print("     - SPI_CTRL(3) = spi_cpha  = %d" %(spi_ctrl_value & 0x8))
            spi_ctrl_freq = (spi_ctrl_value & 0xff00) / 256
            print("     - SPI_CTRL(15:8) = spi_freq = %d" %spi_ctrl_freq)

            data_wr = 0xa5a55a5a
            print("\nSPI Master slv0 write value = 0x%x" % data_wr)
            reg_write(glu_reg_base_addr + REG_OSB_SPI_SLV0, data_wr)

            spi_slv0 = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_SLV0)
            print("SPI Master slv0 read = 0x%x" % spi_slv0)

            data_wr = 0x12345678
            print("SPI Master slv1 write value = 0x%x" % data_wr)
            reg_write(glu_reg_base_addr + REG_OSB_SPI_SLV1, data_wr)

            spi_slv1 = get_reg_value(glu_reg_base_addr + REG_OSB_SPI_SLV1)
            print("SPI Master slv1 read = 0x%x" % spi_slv1)
        else:
            Read_f = 1
            Write_f = 0
            data_wr = 0x37
            value = spi_master_transfer(Write_f, SPI_ADDR_REGLVDS_7_0, data_wr)
            time.sleep(1)
            value = spi_master_transfer(Read_f, SPI_ADDR_REGLVDS_7_0, 0)
            print("\nSPI Received Data Value = 0x%x" %value)

    return None
#*****************************************
#
# REG_BDT_CTRL(1) = fifo_reset
# REG_BDT_CTRL(0) = OSAnotOTDR
#
# REG_BDT_FIFO_RD_CNT
# REG_BDT_FIFO_OUT
#
# REG_BDT_SEQ_MISC(2) = RESET_PLL
#
#*****************************************

@DIAG("OSB BDT")
def t_osb_bdt_test(cmd):

    verbose = True

    print()

    glu_reg_base_addr = get_glu_base_addr()

    if verbose:
        print("\nglu_reg_base_addr = 0x%x" % glu_reg_base_addr)

    bdt_reg_base_addr = get_pci_bdt_base_addr()

    if verbose:
        print("\nbdt_reg_base_addr = 0x%x" % bdt_reg_base_addr)

    pci_bdt_slv0 = get_reg_value(bdt_reg_base_addr + REG_BDT_IP_VERSION)
    print("PCI_BDT slv0 read = 0x%x" % pci_bdt_slv0)

    pci_bdt_slv1 = get_reg_value(bdt_reg_base_addr + REG_BDT_CTRL)
    print("PCI_BDT slv1 read = 0x%x" % pci_bdt_slv1)

    data_wr = 0xa5a55a5a
    print("\nPCI_BDT slv1 write value = 0x%x" % data_wr)
    reg_write(bdt_reg_base_addr + REG_BDT_CTRL, data_wr)

    pci_bdt_slv1 = get_reg_value(bdt_reg_base_addr + REG_BDT_CTRL)
    print("PCI_BDT slv1 readback = 0x%x" % pci_bdt_slv1)

    print("\n*****************************")
    print("sequencer_core registers test")
    print("*****************************\n")

    pci_bdt_seq_core_ver = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_VERSION)
    print("PCI_BDT seq_core_version readback = 0x%x" % pci_bdt_seq_core_ver)

    data_wr = 0x73738282
    print("\nPCI_BDT SEQ_CORE slv1 write value = 0x%x" % data_wr)
    reg_write(bdt_reg_base_addr + REG_BDT_SEQ_SLV1, data_wr)

    pci_bdt_seq_core_slv1 = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV1)
    print("PCI_BDT seq_core slv1 readback = 0x%x" % pci_bdt_seq_core_slv1)

    # SEQ_MISC reg.

    pci_bdt_seq_core_misc = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_MISC)
    print("\nPCI_BDT SEQ_MISC reg. readback before write = 0x%x" % pci_bdt_seq_core_misc)

    data_wr = 0xffffffff
    print("\nPCI_BDT SEQ_MISC write value = 0x%x" % data_wr)
    reg_write(bdt_reg_base_addr + REG_BDT_SEQ_MISC, data_wr)

    pci_bdt_seq_core_misc = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_MISC)
    print("PCI_BDT SEQ_MISC readback after write = 0x%x" % pci_bdt_seq_core_misc)

    #
    # to verify misc(2) settings :
    # misc(2) = 0 => SLV30_reg = 0xffff0000
    # misc(2) = 1 => SLV30_reg = 0xffff1111
    #

    data_wr = 0x0
    print("\nPCI_BDT SEQ_MISC reset (SEQ_MISC(2) = RESET_PLL = 0) : write value = 0x%x" % data_wr)
    reg_write(bdt_reg_base_addr + REG_BDT_SEQ_MISC, data_wr)

    pci_bdt_seq_core_misc = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_MISC)
    print("PCI_BDT SEQ_MISC readback : 0x%x" % pci_bdt_seq_core_misc)

    pci_bdt_seq_core_slv28 = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV28)
    print("PCI_BDT seq_core slv28 readback = 0x%x" % pci_bdt_seq_core_slv28)

    pci_bdt_seq_core_slv29 = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV29)
    print("PCI_BDT seq_core slv29 readback = 0x%x" % pci_bdt_seq_core_slv29)

    pci_bdt_seq_core_slv30 = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV30)
    print("PCI_BDT seq_core slv30 readback = 0x%x" % pci_bdt_seq_core_slv30)

    data_wr = 0x4
    print("\nSet PCI_BDT SEQ_MISC(2) = RESET_PLL = 1 : write value = 0x%x" % data_wr)
    reg_write(bdt_reg_base_addr + REG_BDT_SEQ_MISC, data_wr)

    pci_bdt_seq_core_misc = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_MISC)
    print("PCI_BDT SEQ_MISC readback after write = 0x%x" % pci_bdt_seq_core_misc)

    pci_bdt_seq_core_slv29 = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV29)
    print("PCI_BDT seq_core slv29 readback = 0x%x" % pci_bdt_seq_core_slv29)

    pci_bdt_seq_core_slv30 = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV30)
    print("PCI_BDT seq_core slv30 readback = 0x%x" % pci_bdt_seq_core_slv30)

    return None
#
#***********************************************
#
# LVDS_DATA(2:0) = slv_reg28(2:0)
# LVDS_CLK : freuency value read by slv_reg27
# LVDS_CCAN = slv_reg26(0)
# LVDS_PLAS = slv_reg26(1)
#
#***********************************************
#
@DIAG("OSB LVDS TEST")
def t_osb200_lvds_test(cmd):

    verbose = False

    try:
        if ' ' not in cmd:
            pass
        else:
            (command, consigne) = cmd.split()
            print("arg exist : %s" %consigne)
            if consigne == "verbose":
                verbose = True
            else:
                print_error("   Syntax error : arg must be : verbose\n")

    except:
        print_error("   Syntax error : Command must be : t_osb200_lvds_test <consigne>")
        print_error("                                                                 ")
        return False

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    if verbose:
        # Read B200 CPLD VERSION reg
        ver = Utsspi.read_mod_register8(SPI_ADDR_VERSION)
        print("B200 CPLD Version 0x%2x" % ver)

    glu_reg_base_addr = get_glu_base_addr()
    bdt_reg_base_addr = get_pci_bdt_base_addr()

    if verbose:
        print("glu_reg_base_addr = 0x%x" %glu_reg_base_addr)
        print("bdt_reg_base_addr = 0x%x" %bdt_reg_base_addr, "\n")

    # cAddress_BoardPresence reg. : x"0" & "00" & DTSS_PRES_B & SWITCH_PRES_B

    value = Utsspi.read_mod_register8(SPI_ADDR_BOARD_PRES)
    switch_pres_b_bit = value & 0x1
    dtss_pres_b_bit   = (value & 0x2)/2
    toggle_bit        = (value & 0x4)/4
    board_id          = (value & 0x80)/128

    if verbose:
        print("* B200 module_presence_rdback register details (after write) :")
        print("     * module_presence_rdback read (@ addr 0x%1x) = 0x%x" %(SPI_ADDR_BOARD_PRES, value))
        print("         - BOARD_ID           = %d (2 = SWITCH ; 1 = DTSS)" %board_id)
        print("         - toggle_board_state = %d " %toggle_bit)
        print("         - DTSS_PRES_B        = %d ('0' = Selected ; '1' = DeSelected)" %dtss_pres_b_bit)
        print("         - SWITCH_PRES_B      = %d ('0' = Selected ; '1' = DeSelected)\n" %switch_pres_b_bit)

    if board_id == 2:
        print ("* Run b200_module_type_set(DTSS)")
        b200_module_type_set("DTSS", verbose)

# TEST of LVDS_DATA lines :

    error_nb = 0
    wr_data = 0x0
    j = 0

    while j < 8:

        ver = Utsspi.write_mod_register8(SPI_ADDR_REGLVDS_7_0, wr_data)

        read_data = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV28)


        if read_data == wr_data:
            msg = "PASS"
        else:
            msg = "FAIL"
            error_nb += 1

        if msg == "PASS":
            print_ok("LVDS_data reg. : read_data = 0x%x : %s" %(read_data, msg))
        else:
            print_error("LVDS_data reg. : read_data = 0x%x : expected_data = 0x%x : %s" %(read_data, wr_data, msg))

        wr_data += 1
        j += 1

# TEST of LVDS_CLK line :

    lvds_clk_reg_value = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV27)

    if verbose:
        print("lvds_clk_reg_value = 0x%x = %d" %(lvds_clk_reg_value, lvds_clk_reg_value))

    lvds_clk_val = float(lvds_clk_reg_value)
    lvds_clk_freq = lvds_clk_val * 1000

    expected_value = 40000000.0

    if lvds_clk_freq > 38000000.0 and lvds_clk_freq < 42000000.0:
        msg = "PASS"
    else:
        msg = "FAIL"
        error_nb += 1

    if msg == "PASS":
        print_ok("lvds_clk_freq  : %5.3fHz  : %s" %(lvds_clk_freq, msg))
#       print_ok("lvds_clk_freq : %5.3fHz : expected_value = %5.3fHz " %(lvds_clk_freq, expected_value), "+/- 5% :", msg)
    else:
        print_error("lvds_clk_freq  : %5.3fHz : expected_value = %5.3fHz : %s" %(lvds_clk_freq, expected_value, msg))

# TEST of LVDS_CCAN and LVDS_PLAS lines :

#   data_wr = 0xabcd1234
#   print("\nPCI_BDT SEQ_CORE slv26 write value = 0x%x" % data_wr)
#   reg_write(bdt_reg_base_addr + REG_BDT_SEQ_SLV26, data_wr)

#   read_data = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV26)
#   print("PCI_BDT SEQ_CORE slv26 readback value = 0x%x" % read_data)

    j = 0
    wr_data = 0x3

    while j < 4:

        reg_write(bdt_reg_base_addr + REG_BDT_SEQ_SLV26, wr_data)
        rd_data = get_reg_value(bdt_reg_base_addr + REG_BDT_SEQ_SLV26)

        if rd_data == wr_data:
            msg = "PASS"
        else:
            msg = "FAIL"
            error_nb += 1

        read_data = Utsspi.read_mod_register8(SPI_ADDR_CCAN_PLAS)

        if verbose:
            ccan_bit = read_data & 0x1
            plas_bit = (read_data & 0x2)/2

            print("lvds_ccan_plas reg. readback value = 0x%x" % read_data)
            print("     * ccan_bit = %d" %ccan_bit)
            print("     * plas_bit = %d" %plas_bit)

        if read_data == wr_data:
            msg = "PASS"
        else:
            msg = "FAIL"
            error_nb += 1

        if msg == "PASS":
            print_ok("LVDS_CCAN_PLAS reg. : read_data = 0x%x : %s" %(read_data, msg))
        else:
            print_error("LVDS_CCAN_PLAS reg. : read_data = 0x%x : expected_data = 0x%x : %s" %(read_data, wr_data, msg))

        wr_data -= 1
        j += 1

    if error_nb == 0:
        return True
    else:
        return False

#
#===============================================================================
#
#
#===============================================================================
#
@DIAG("Test DTSS module connector")
def t_osb_Z112_test(cmd):

    try:
        (command, version) = cmd.split()
    except:
        print_error("# Command must be: t_osb200_spi_test version(hex)")
        return False

    error_nb = 0

    print()
    module_presence_info(OSB_flag = 1, B200_flag = 0, verbose_flag = 0)

    # pwrside S3/S4
    Utsspi = utsspi.SpiModule(0, 3, "/dev/uts_spi")

    # Read VERSION reg at@0x00
    ver = Utsspi.read_mod_register8(0x00)

    if ver == int(version, 16):
        print_ok("Z112 CPLD Version 0x%2x : PASS" % ver)
    else:
        error_nb += 1
        print_error("Z112 : version readback : 0x%x (expected = 0x%x)" %(ver, int(version, 16)))

    # Z112 : REG LVDS registers

    expected_val = 0x90
    ver = Utsspi.read_mod_register8(0x0)


    print("Z112 : Check with pattern 0xAA55")
    data_wrL = 0x55
    Utsspi.write_mod_register8(0x01, data_wrL)
    data_wrL_rdbck =  Utsspi.read_mod_register8(0x01)

    if data_wrL_rdbck == data_wrL:
        print_ok("Z112 : reg. readback @ address 0x01 : 0x%x : PASS" %(data_wrL_rdbck))
    else:
        print_error("Z112 : reg. readback @ address 0x02 : 0x%x (expected = 0x%x) : FAIL" %(data_wrL_rdbck, data_wrL))
        error_nb += 1

    data_wrH = 0xaa
    Utsspi.write_mod_register8(0x02, data_wrH)
    data_wrH_rdbck =  Utsspi.read_mod_register8(0x02)

    if data_wrH_rdbck == data_wrH:
        print_ok("Z112 : reg. readback @ address 0x01 : 0x%x : PASS" %(data_wrH_rdbck))
    else:
        print_error("Z112 : reg. readback @ address 0x01 : 0x%x (expected = 0x%x): FAIL " %(data_wrH_rdbck, data_wrH))
        error_nb += 1

    if error_nb == 0:
        return True
    else:
        return False
