import QtQuick 2.1
import graph_plugin 1.0

Flickable {
    id: flick

    property ZoomRect zoomRect
    default property alias mouseArea: pinchArea.children
    readonly property alias pinching: pinchArea.pinching
    property alias pinchEnabled: pinchArea.enabled
    property alias multipleDirectionPinch: pinchArea.multipleDirectionPinch

    anchors.fill: parent
    boundsBehavior: Flickable.StopAtBounds
    interactive: !pinching && enabled
    pixelAligned: false

    // The PinchArea will scale with the Flickable's contentItem so we have to
    //  clip the PinchArea to keep it from covering the entire screen, but only
    //  when it's not in use.
    clip: !pinching

    PinchArea { id: pinchArea
        width: flick.contentWidth
        height: flick.contentHeight
        pinch.dragAxis: Pinch.NoDrag

        property bool multipleDirectionPinch: false
        property bool pinching: false

        // Determine zoom direction from the angle between the two touch points
        // Three possible outputs: horizontal, vertical or diagonal (zoom both directions)
        //TODO: replace these 4 values with real QML enums when switching to Qt 5.10+
        readonly property int zoomNone: 0
        readonly property int zoomHorizontal: zoomNone + 1
        readonly property int zoomVertical: zoomHorizontal + 1
        readonly property int zoomAll: zoomVertical + 1
        function getZoomDirection(pinch) {
            var maxAngle = 180
            var rightAngle = maxAngle/2
            var arcAngle = rightAngle/4
            var zoomDirection = zoomNone

            //Absolute value to simplify conditions. True direction of a diagonal is of no interest
            var angle = Math.abs(pinch.angle)

            function inRange(x, min, max) {
                var res = (x >= min && x <= max)
                return res
            }

            if (inRange(angle, 0, arcAngle) || inRange(angle, maxAngle-arcAngle, maxAngle)) {
                zoomDirection = zoomHorizontal
            }
            else if (inRange(angle, rightAngle-arcAngle, rightAngle+arcAngle)) {
                zoomDirection = zoomVertical
            }
            else {
                zoomDirection = zoomAll
            }
            return zoomDirection
        }

        onPinchStarted: {
            pinching = true
        }
        onPinchFinished: {
            pinching = false
        }

        onPinchUpdated: {
            var scale = 1.0 + pinch.scale - pinch.previousScale;
            var invScale = Qt.vector2d(zoomRect.width / scale, zoomRect.height / scale);
            var center = zoomRect.mapFromContent(pinch.center);
            if (multipleDirectionPinch)  {
                var crtZoomDirection = getZoomDirection(pinch)
                switch (crtZoomDirection) {
                case zoomHorizontal:
                    zoomRect.zoomX(invScale.x, center.x)
                    break
                case zoomVertical:
                    zoomRect.zoomY(invScale.y, center.y)
                    break
                default:
                    zoomRect.zoom(invScale, center)
                    break;
                }
            }
            else {
                switch (flick.flickableDirection) {
                case Flickable.HorizontalFlick:
                    zoomRect.zoomX(invScale.x, center.x);
                    break;
                case Flickable.VerticalFlick:
                    zoomRect.zoomY(invScale.y, center.y);
                    break;
                default:
                    zoomRect.zoom(invScale, center);
                }
            }
        }
    }
}
