import shutil, tarfile, os, io, subprocess, glob, hashlib
import urllib.request as urlreq
from tempfile import mkstemp
from os import fdopen, remove
from shutil import move, copymode
from viavi.stdhw.bootutils import bootenv
from viavi.stdhw.blockutils import mount, umount, mount_bind, mounted_point, sync
from viavi.upgrade.flashutils import mount_chroot, umount_chroot, chroot_exec, format_device
from viavi.upgrade import PrintException, run_hook, split_url, md5, get_user_agent
from urllib.error import HTTPError, URLError
import struct

def synchronize(source, destination, overwrite=False):
    if os.path.isdir(source):
        if os.path.isdir(destination):
            for item in os.listdir(source):
                s = os.path.join(source, item)
                d = os.path.join(destination, item)
                synchronize(s, d)
        else:
            shutil.copytree(source, destination, symlinks=True)
    else:
        # If a file (or link) is marked as to be synchronized and overwrite is enabled,
        # we want to delete it if it disappeared from the source but still exists in the destination
        if overwrite and not os.path.lexists(source) and os.path.lexists(destination):
            os.remove(destination)
        else:
            if not os.path.lexists(destination) or overwrite:
                if os.path.islink(source):
                    linkto = os.readlink(source)
                    # We try to create the symlink
                    # If it already exists AND overwrite is enabled,
                    # we have to remove the old symlink and create a new one - if they are different
                    if overwrite and os.path.lexists(destination) and (linkto != os.readlink(destination)):
                        os.remove(destination)
                    # If it already exists - do nothing
                    if not os.path.lexists(destination):
                        os.symlink(linkto, destination)
                else:
                    if not os.path.exists(os.path.dirname(destination)):
                        os.makedirs(os.path.dirname(destination))
                    if os.path.exists(source):
                        shutil.copy2(source, destination)


class MyFileObj(io.FileIO):
    def setCls(self, cls, fsize, name):
        self.__cls = cls
        self.__fsize = fsize
        self.__rname = name
        self.lastpercent = 1000

    def read(self, size):
        percent = int(self.tell() * 100 / self.__fsize)
        if percent != self.lastpercent:
            if self.__cls is not None:
                self.__cls.expose("Upgrade", int(50 + (percent * 0.4)), self.__rname, "Installation", int(percent))
            self.lastpercent = percent
        return io.FileIO.read(self, size)


def download_file(src, dest, proxy, chunk_size=8192, report_hook=None):
    with open(dest, "wb") as destination:
        md5 = hashlib.md5()
        if proxy is not None and proxy.strip() != "DIRECT":
            print("Enable proxy %s" % proxy)
            proxy_handler = urlreq.ProxyHandler({'http': proxy})
        else:
            print("Disable proxy")
            proxy_handler = urlreq.ProxyHandler({})
        opener = urlreq.build_opener(proxy_handler)
        urlreq.install_opener(opener)
        response = urlreq.urlopen(src);
        total_size = response.info().get('Content-Length').strip()
        total_size = int(total_size)
        bytes_so_far = 0

        while 1:
            chunk = response.read(chunk_size)
            md5.update(chunk)
            bytes_so_far += len(chunk)
            destination.write(chunk)

            if not chunk:
                break

            if report_hook:
                report_hook(bytes_so_far, chunk_size, total_size)

    return (bytes_so_far, md5)


# smartmanager:
#  expose
#  set_action
#  log
#  upgrade_by_tar_terminate

# options:
#  proxy
#  daemon
#  flash_block_device
#  auto_switch
#  test_update_part
def start_upgrade_tar(xmlrelease, hook=None, options={}, smartmanager=None):
    if not run_hook(hook, 'pre_tar_upgrade'):
        return False

#    if xmlrelease.is_vui():
#        return start_upgrade_vui(xmlrelease, hook, options, smartmanager)

    # Option default values
    if 'auto_switch' not in options.keys():
        options['auto_switch'] = True
    if 'test_update_part' not in options.keys():
        options['test_update_part'] = False
    if 'proxy' not in options.keys():
        options['proxy'] = ''
    if 'current_rootfs' not in options.keys():
        options['current_rootfs'] = ''
    if 'syncronise' not in options.keys():
        options['syncronise'] = True

    result = True
    tarf = xmlrelease.xpath("repositories/archive/@file")[0]
    try:
        md5f = xmlrelease.xpath("repositories/archive/@md5")[0]
        print("Use MD5 auto-signed check")
    except:
        md5f = None
        print("Use SIG TAR check")
    name = xmlrelease.xpath("@name")[0]
    try:
        sizef = int(xmlrelease.xpath("repositories/archive/@size")[0])
    except:
        sizef = None

    if sizef is not None:
        size = sizef
        size = int(size) / (1024 * 1024)
        size = int(size * 1.1)
    else:
        size = 256
    if size < 256:
        size = 256
    url = xmlrelease.url()
    remote = xmlrelease.remote()
    (xmlpath, slot) = split_url(url)
    path = "/".join(xmlpath.split("/")[:-1]) + "/" + tarf
    if remote:
        try:
            if not os.path.exists("/mnt/tmp"):
                os.mkdir("/mnt/tmp")
            subprocess.check_output(["mount", "-t", "tmpfs", "-o", "size={}M".format(size), "tmpfs", "/mnt/tmp"])
        except:
            PrintException()
            return False
        lpath = "/mnt/tmp"
    else:
        lpath = "/".join(xmlpath.split("/")[:-1]) + "/"

    def chunk_report(bytes_so_far, chunk_size, total_size):
        percent = float(bytes_so_far) / total_size
        percent = round(percent * 100, 2)
        if int(percent) != int(chunk_report.lastpercent):
            if smartmanager is not None:
                # Dowload during 0%~45% of total
                smartmanager.expose("Upgrade", int(percent * 0.45), name, "Download release", int(percent))
            chunk_report.lastpercent = percent

    chunk_report.lastpercent = 1000
    run_hook(hook, 'pre_tar_download')

    dlmd5 = None
    if remote:
        if smartmanager is not None:
            smartmanager.set_action("1: Download release")
            smartmanager.log("Download %s" % path)
        (sz, dlmd5) = download_file(path, "%s/%s" % (lpath, tarf), options['proxy'], report_hook=chunk_report)
    else:
        with open("%s/%s" % (lpath, tarf), "rb") as f:
            dlmd5 = hashlib.md5()
            chunk = f.read(8192)
            while chunk:
                dlmd5.update(chunk)
                chunk = f.read(8192)

    if smartmanager is not None:
        smartmanager.set_action("2: Check release")
    try:
        if not os.path.exists("/etc/release/no_check"):
            if remote:
                if md5f is None:
                    if smartmanager is not None:
                        smartmanager.log("Download %s.sig" % path)
                    download_file(path + ".sig", "%s/%s.sig" % (lpath, tarf), options['proxy'], report_hook=chunk_report)

            if smartmanager is not None:
                smartmanager.expose("Upgrade", 46, name, "Check integrity", 0)
                if md5f is None:
                    print(subprocess.check_output(["gpg", "--verify", "%s/%s.sig" % (lpath, tarf), "%s/%s" % (lpath, tarf)]))
                else:
                    if dlmd5.hexdigest().lower() != md5f.lower():
                        raise subprocess.CalledProcessError

            if smartmanager is not None:
                smartmanager.expose("Upgrade", 46, name, "Check integrity", 100)
    except subprocess.CalledProcessError as err:
        print("BAD SIGNATURE")
        if options['daemon']:
            if smartmanager is not None:
                smartmanager.upgrade_by_tar_terminate("BAD SIGNATURE")
        return
    try:
        try:
            bootenv.refresh()
            mmc_part = int(bootenv.mmc_part)
            if mmc_part == options['parts'][0]:
                mmc_part_new = options['parts'][1]
            else:
                mmc_part_new = options['parts'][0]
        except:
            mmc_part_new = options['parts'][1]

        print("mmc_part is %s" % mmc_part_new)

        # Unmount the partition about to be formatted (all mounts)
        for mount_point in mounted_point(options['flash_block_device'] + "p%s" % mmc_part_new):
            umount(mount_point)

        run_hook(hook, 'pre_tar_install')
        if smartmanager is not None:
            smartmanager.set_action("3: Format partitions")
        if not os.path.exists("/mnt/fs"):
            os.makedirs("/mnt/fs")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 47, name, "Prepare filesystem %s" % mmc_part_new, 10)
        format_device(mmc_part_new)
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 48, name, "Prepare filesystem", 50)
        mount(options['flash_block_device'] + "p%s" % mmc_part_new, "/mnt/fs/", "ext4")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 49, name, "Prepare filesystem", 60)
        if not os.path.exists("/mnt/fs/user/disk"):
            os.makedirs("/mnt/fs/user/disk")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 50, name, "Prepare filesystem", 70)
        mount_bind("/user/disk", "/mnt/fs/user/disk")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 50, name, "Prepare filesystem", 100)
            smartmanager.set_action("4: Install release")

        run_hook(hook, 'pre_tar_untar')
        total_size = sizef
        if total_size is None:
            total_size = os.path.getsize("%s/%s" % (lpath, tarf))

        with MyFileObj("%s/%s" % (lpath, tarf)) as fileobj:
            fileobj.setCls(smartmanager, total_size, name)
            with tarfile.open(fileobj=fileobj, mode="r|*", errorlevel=1) as tar:
                tar.extractall("/mnt/fs")
        run_hook(hook, 'post_tar_untar')

        if smartmanager is not None:
            smartmanager.set_action("5: Setup release")
        try:
            shutil.copyfile("/etc/resolv.conf", "/mnt/fs/etc/resolv.conf")
        except shutil.SameFileError:
            pass
        except FileNotFoundError:
            try:
                open("/mnt/fs/etc/resolv.conf", 'a').close()
            except:
                print("Warning: Can't create /mnt/fs/etc/resolv.conf")
                pass

        # Copy all file explicitly listed in /etc/release/backup
        run_hook(hook, 'pre_synchronize')
        if options['syncronise']:
            if smartmanager is not None:
                smartmanager.set_action("6: Synchronize")
            backup = []
            if os.path.exists("{}/etc/release/backup".format(options['current_rootfs'])):
                for f in os.listdir("{}/etc/release/backup".format(options['current_rootfs'])):
                    with open("{}/etc/release/backup/{}".format(options['current_rootfs'], f)) as fi:
                        backup += [line.strip() for line in fi if line.strip() != '']
            if os.path.exists("/mnt/fs/etc/release/backup"):
                for f in os.listdir("/mnt/fs/etc/release/backup"):
                    with open("/mnt/fs/etc/release/backup/%s" % f) as fi:
                        backup += [line.strip() for line in fi if line.strip() != '']
            run_hook(hook, 'pre_synchronize_list')

            for bk in set(backup):
                override = False
                if bk.startswith("@"):
                    override = True
                    bk = bk[1:].strip()

                # Enable deleting files
                # glob.glob returns nothing but we want to delete the file pointed by bk anyway
                files = glob.glob(options['current_rootfs'] + bk)
                if(override and (len(files) == 0)):
                    files = [ options['current_rootfs'] + bk ]

                for fn in files:
                    if len(options['current_rootfs']) > 0:
                        print("Sync {} {}".format(fn, fn.replace(options['current_rootfs'], "/mnt/fs")))
                        synchronize(fn, fn.replace(options['current_rootfs'], "/mnt/fs"), override)
                    else:
                        print("Sync {} {}".format(fn, "/mnt/fs/%s" % fn))
                        synchronize(fn, "/mnt/fs/%s" % fn, override)
        run_hook(hook, 'post_synchronize')

        if smartmanager is not None:
            smartmanager.set_action("7: Run post install (don't turn off device)")
        id = xmlrelease.xpath("@id")[0]
        xmlrelease.save("/mnt/fs/etc/release/%s.xml" % id)
        os.system("rm -f /mnt/fs/etc/release/current.xml")
        os.system("ln -s /etc/release/%s.xml /mnt/fs/etc/release/current.xml" % id)

        # Remove old files form /users/disk declared in /etc/release/remove
        if os.path.isdir("/mnt/fs/etc/release/remove"):
            for f in os.listdir("/mnt/fs/etc/release/remove"):
                with open("/mnt/fs/etc/release/remove/%s" % f) as rf:
                    for line in [l.strip() for l in rf.readlines() if len(l.strip()) > 0 and not l.startswith("#")]:
                        tab = line.split(" ")
                        if os.path.isfile(tab[1]):
                            if md5(tab[1]) == tab[0]:
                                print("Delete removable file: %s" % tab[1])
                                os.unlink(tab[1])

        # Manage password changes
        #  1. Get a list of users before and after the upgrade (only user not system role)
        def extract_user_from_passwd(filename):
            users = []
            if not os.path.isfile(filename):
                return []
            with open(filename, mode="r") as fp:
                for line in fp:
                    line_part = line.split(":")
                    if len(line_part) >= 3:
                        user_id = int(line_part[2])
                        if user_id == 0 or user_id >= 1000:
                            users.append(line_part[0])
            return users

        current_users = extract_user_from_passwd("/etc/passwd")
        new_users = extract_user_from_passwd("/mnt/fs/etc/passwd")

        #  2. Keep only password to update
        user_to_synchronize = [user for user in new_users if user in current_users]
        current_password = {}
        with open("/etc/shadow", mode="r") as fp:
            for line in fp:
                user = line.split(":")[0]
                if user in user_to_synchronize:
                    current_password[user] = line.split(":")[1]

        new_shadow, abs_path = mkstemp()
        with fdopen(new_shadow, 'w') as new_file:
            with open("/mnt/fs/etc/shadow", mode="r") as fp:
                for line in fp:
                    split_line = line.split(":")
                    user = split_line[0]
                    if user in user_to_synchronize:
                        new_password = split_line[1]
                        if user in current_password.keys() and new_password != current_password[user]:
                            split_line[1] = current_password[user]
                            new_file.write(":".join(split_line))
                        else:
                            new_file.write(line)
                    else:
                        new_file.write(line)
        copymode("/mnt/fs/etc/shadow", abs_path)
        # Remove original file
        remove("/mnt/fs/etc/shadow")
        # Move new file
        move(abs_path, "/mnt/fs/etc/shadow")

        # Manage Downgrade to fido (cortexa9_vfp_neon architecture)
        if "cortexa9_vfp_neon" in xmlrelease.xpath("/repositories/rpm/@id"):
            bootenv.addmmc = 'setenv bootargs ${bootargs} root=/dev/mmcblk0p${mmc_part} rootwait'
            bootenv.bootcmd = 'run upgenv; run defaultboot'
            bootenv.bootdelay = '0'
            bootenv.defaultboot = 'get_vext 6000 8000 || run mmcboot'
            bootenv.env_version = '1'
            bootenv.getitb = 'ext2load mmc 0:${mmc_part} ${kernel_addr_r} /boot/mts1000.itb'
            bootenv.kernel_addr_r = '0x2000000'
            bootenv.kernel_options = 'sdhci.debug_quirks=0x8000 coherent_pool=0x400000'
            bootenv.mmcboot = 'run getitb; run addtty addmmc addfixes;bootm ${kernel_addr_r}'
            bootenv.rescue = 'setenv mmc_part 7; run mmcboot'
            bootenv.rescuefs = 'run addtty addfixes;bootm ${kernel_addr_r}'
            bootenv.rescuefs_version = '1'
            bootenv.stderr = 'serial'
            bootenv.stdin = 'serial'
            bootenv.stdout = 'serial'
            bootenv.upgenv = 'ext2load mmc 0:${mmc_part} ${kernel_addr_r} /boot/configs.txt && env import ${kernel_addr_r} ${filesize}'

        try:
            mount_chroot()
            os.system("mount > /tmp/mount.log")
            chroot_exec("/mnt/fs", ["/usr/sbin/run-postinsts"])
        except:
            result = False
            PrintException()
        finally:
            umount_chroot()
    except subprocess.CalledProcessError as err:
        result = False
        PrintException()
    except:
        result = False
        PrintException()
    finally:
        if smartmanager is not None:
            smartmanager.set_action("8: Clean installation")
            smartmanager.expose("Upgrade", 90, name, "Clean installation", 100)
        umount("/mnt/fs/user/disk")
        if not os.path.isdir("/mnt/fs/var/lib/release-manager"):
            os.mkdir("/mnt/fs/var/lib/release-manager")
        with open("/mnt/fs/var/lib/release-manager/upgrade_status", "w") as f:
            f.write("Success")
        umount("/mnt/fs")
        if remote:
            umount("/mnt/tmp")
        if result:
            run_hook(hook, 'pre_enable')
            if smartmanager is not None:
                smartmanager.set_action("8: Enable new installation")
                smartmanager.expose("Upgrade", 100, name, "Enable new installation", 100)
            if options['auto_switch']:
                print("Setting permanent boot part")
                bootenv.mmc_part = mmc_part_new
            if options['test_update_part']:
                print("Setting temp boot part")
                bootenv.mmc_part_volatile = mmc_part_new
            if smartmanager is not None:
                smartmanager.expose("Upgrade", 100, name, "Finished successful", 100)
            run_hook(hook, 'post_enable')
    if options['daemon']:
        if smartmanager is not None:
            smartmanager.set_action("Upgrade Terminate")
        if result:
            if smartmanager is not None:
                smartmanager.upgrade_by_tar_terminate("OK")
        else:
            if smartmanager is not None:
                smartmanager.upgrade_by_tar_terminate("FAIL")
    if not run_hook(hook, 'post_tar_upgrade'):
        return False
    return result


def start_upgrade_vui(xmlrelease, hook=None, options={}, smartmanager=None):
    # Option default values

    if 'auto_switch' not in options.keys():
        options['auto_switch'] = True
    if 'test_update_part' not in options.keys():
        options['test_update_part'] = False
    if 'proxy' not in options.keys():
        options['proxy'] = ''
    if 'current_rootfs' not in options.keys():
        options['current_rootfs'] = ''
    if 'syncronise' not in options.keys():
        options['syncronise'] = True

    result = True
    tarf = xmlrelease.xpath("repositories/archive/@file")[0]
    md5f = xmlrelease.xpath("repositories/archive/@md5")[0]
    name = xmlrelease.xpath("@name")[0]
    sizef = int(xmlrelease.xpath("repositories/archive/@size")[0])
    url = xmlrelease.url()
    remote = xmlrelease.remote()
    (xmlpath, slot) = split_url(url)
    path = "/".join(xmlpath.split("/")[:-1]) + "/" + tarf

    try:
        try:
            bootenv.refresh()
            mmc_part = int(bootenv.mmc_part)
            if mmc_part == options['parts'][0]:
                mmc_part_new = options['parts'][1]
            else:
                mmc_part_new = options['parts'][0]
        except:
            mmc_part_new = options['parts'][1]

        print("mmc_part is %s" % mmc_part_new)

        # Unmount the partition about to be formatted (all mounts)
        for mount_point in mounted_point(options['flash_block_device'] + "p%s" % mmc_part_new):
            umount(mount_point)

        run_hook(hook, 'pre_tar_install')
        if smartmanager is not None:
            smartmanager.set_action("1: Format partitions")
        if not os.path.exists("/mnt/fs"):
            os.makedirs("/mnt/fs")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 1, name, "Prepare filesystem %s" % mmc_part_new, 20)
        format_device(mmc_part_new)
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 2, name, "Prepare filesystem", 40)
        mount(options['flash_block_device'] + "p%s" % mmc_part_new, "/mnt/fs/", "ext4")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 3, name, "Prepare filesystem", 60)
        if not os.path.exists("/mnt/fs/user/disk"):
            os.makedirs("/mnt/fs/user/disk")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 4, name, "Prepare filesystem", 80)
        mount_bind("/user/disk", "/mnt/fs/user/disk")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 5, name, "Prepare filesystem", 100)
            smartmanager.set_action("2: Install release")

        vui = None
        if remote:
            # Download XML file
            proxy = options['proxy']
            if proxy is not None and proxy.strip() != "DIRECT":
                print("Enable proxy %s" % proxy)
                proxy_handler = urlreq.ProxyHandler({'http': proxy})
            else:
                print("Disable proxy")
                proxy_handler = urlreq.ProxyHandler({})
            opener = urlreq.build_opener(proxy_handler)
            opener.addheaders = [('User-Agent', get_user_agent())]
            urlreq.install_opener(opener)

            # Check mymirror.php
            release = xmlrelease.url()
            base_url = release.replace("http://", "").split("/")[0]
            mymirror_url = "http://%s/mymirror.php" % (base_url)
            req = urlreq.Request(mymirror_url)
            try:
                mirror = urlreq.urlopen(req).read().decode('ascii')
                release = release.replace(base_url, mirror)
                url = url.replace(base_url, mirror)
                print("Release with mirror: %s" % release)
            except HTTPError:
                pass

            req = urlreq.Request(release)
            vui = urlreq.urlopen(req)
        else:
            vui = open("%s" % (release), "rb")

        header = vui.read(12)
        (v,u,i,version,lenxml,lensig) = struct.unpack("<cccBII", header)
        xmlc = vui.read(lenxml)
        sigc = vui.read(lensig)

        def print_gui(self, percent):
            if smartmanager is not None:
                smartmanager.expose("Upgrade", 6 + (percent * 90/100), name, "Installation", percent)

        def StreamDecorator(original, tarsize, notify):
            setattr(original.__class__, "oldread", original.read)
            setattr(original.__class__, "notify", notify)
            original.lastpercent = 0
            original.tarsize = int(tarsize)
            original.currentsize = 0
            original.dlmd5 = hashlib.md5()

            def myread(self, size):
                percent = int(self.currentsize * 100 / self.tarsize)
                if percent != self.lastpercent:
                    self.notify(percent)
                    self.lastpercent = percent

                if self.currentsize == 0:
                    key = struct.pack("cccB", b'V', b'U', b'I', 1)
                    data = self.oldread(4)
                    chuck = struct.pack("<I", struct.unpack('<I', data)[0] ^ struct.unpack('<I', key)[0])
                    chuck += self.oldread(size - 4)
                    self.currentsize += size
                else:
                    chuck = self.oldread(size)

                self.dlmd5.update(chuck)
                self.currentsize += size
                return chuck

            myread.__doc__ = "My read"
            myread.__name__ = "read"
            setattr(original.__class__, "read", myread)
            return original

        run_hook(hook, 'pre_tar_untar')
        f = StreamDecorator(vui, sizef, print_gui)
        tar = tarfile.open(fileobj=f, mode='r|gz', errorlevel=1)
        tar.extractall("/mnt/fs")
        sync()
        run_hook(hook, 'post_tar_untar')


        if smartmanager is not None:
            smartmanager.set_action("5: Check release")
            smartmanager.expose("Upgrade", 96, name, "Check integrity", 100)
        if md5f.lower() != f.dlmd5.hexdigest().lower():
            print("BAD SIGNATURE %s %s", md5f.lower(), f.dlmd5.hexdigest().lower())
            if options['daemon']:
                if smartmanager is not None:
                    smartmanager.upgrade_by_tar_terminate("BAD SIGNATURE")
            return

        try:
            shutil.copyfile("/etc/resolv.conf", "/mnt/fs/etc/resolv.conf")
        except shutil.SameFileError:
            pass
        except FileNotFoundError:
            try:
                open("/mnt/fs/etc/resolv.conf", 'a').close()
            except:
                print("Warning: Can't create /mnt/fs/etc/resolv.conf")
                pass

        # Copy all file explicitly listed in /etc/release/backup
        run_hook(hook, 'pre_synchronize')
        if options['syncronise']:
            if smartmanager is not None:
                smartmanager.set_action("6: Synchronize")
            backup = []
            if os.path.exists("{}/etc/release/backup".format(options['current_rootfs'])):
                for f in os.listdir("{}/etc/release/backup".format(options['current_rootfs'])):
                    with open("{}/etc/release/backup/{}".format(options['current_rootfs'], f)) as fi:
                        backup += [line.strip() for line in fi if line.strip() != '']
            if os.path.exists("/mnt/fs/etc/release/backup"):
                for f in os.listdir("/mnt/fs/etc/release/backup"):
                    with open("/mnt/fs/etc/release/backup/%s" % f) as fi:
                        backup += [line.strip() for line in fi if line.strip() != '']
            run_hook(hook, 'pre_synchronize_list')

            for bk in set(backup):
                override = False
                if bk.startswith("@"):
                    override = True
                    bk = bk[1:].strip()
                for fn in glob.glob(options['current_rootfs'] + bk):
                    if len(options['current_rootfs']) > 0:
                        print("Sync {} {}".format(fn, fn.replace(options['current_rootfs'], "/mnt/fs")))
                        synchronize(fn, fn.replace(options['current_rootfs'], "/mnt/fs"), override)
                    else:
                        print("Sync {} {}".format(fn, "/mnt/fs/%s" % fn))
                        synchronize(fn, "/mnt/fs/%s" % fn, override)
        run_hook(hook, 'post_synchronize')

        if smartmanager is not None:
            smartmanager.set_action("7: Run post install (don't turn off device)")
        id = xmlrelease.xpath("@id")[0]
        xmlrelease.save("/mnt/fs/etc/release/%s.xml" % id)
        os.system("rm -f /mnt/fs/etc/release/current.xml")
        os.system("ln -s /etc/release/%s.xml /mnt/fs/etc/release/current.xml" % id)

        # Remove old files form /users/disk declared in /etc/release/remove
        if os.path.isdir("/mnt/fs/etc/release/remove"):
            for f in os.listdir("/mnt/fs/etc/release/remove"):
                with open("/mnt/fs/etc/release/remove/%s" % f) as rf:
                    for line in [l.strip() for l in rf.readlines() if len(l.strip()) > 0 and not l.startswith("#")]:
                        tab = line.split(" ")
                        if os.path.isfile(tab[1]):
                            if md5(tab[1]) == tab[0]:
                                print("Delete removable file: %s" % tab[1])
                                os.unlink(tab[1])

        # Manage password changes
        #  1. Get a list of users before and after the upgrade (only user not system role)
        def extract_user_from_passwd(filename):
            users = []
            if not os.path.isfile(filename):
                return []
            with open(filename, mode="r") as fp:
                for line in fp:
                    line_part = line.split(":")
                    if len(line_part) >= 3:
                        user_id = int(line_part[2])
                        if user_id == 0 or user_id >= 1000:
                            users.append(line_part[0])
            return users

        current_users = extract_user_from_passwd("/etc/passwd")
        new_users = extract_user_from_passwd("/mnt/fs/etc/passwd")

        #  2. Keep only password to update
        user_to_synchronize = [user for user in new_users if user in current_users]
        current_password = {}
        with open("/etc/shadow", mode="r") as fp:
            for line in fp:
                user = line.split(":")[0]
                if user in user_to_synchronize:
                    current_password[user] = line.split(":")[1]

        new_shadow, abs_path = mkstemp()
        with fdopen(new_shadow, 'w') as new_file:
            with open("/mnt/fs/etc/shadow", mode="r") as fp:
                for line in fp:
                    split_line = line.split(":")
                    user = split_line[0]
                    if user in user_to_synchronize:
                        new_password = split_line[1]
                        if user in current_password.keys() and new_password != current_password[user]:
                            split_line[1] = current_password[user]
                            new_file.write(":".join(split_line))
                        else:
                            new_file.write(line)
                    else:
                        new_file.write(line)
        copymode("/mnt/fs/etc/shadow", abs_path)
        # Remove original file
        remove("/mnt/fs/etc/shadow")
        # Move new file
        move(abs_path, "/mnt/fs/etc/shadow")

        try:
            mount_chroot()
            os.system("mount > /tmp/mount.log")
            chroot_exec("/mnt/fs", ["/usr/sbin/run-postinsts"])
        except:
            result = False
            PrintException()
        finally:
            umount_chroot()
    except subprocess.CalledProcessError as err:
        result = False
        PrintException()
    except:
        result = False
        PrintException()
    finally:
        if smartmanager is not None:
            smartmanager.set_action("8: Clean installation")
            smartmanager.expose("Upgrade", 90, name, "Clean installation", 100)
        umount("/mnt/fs/user/disk")
        if os.path.isdir("/mnt/fs/var/lib") and not os.path.isdir("/mnt/fs/var/lib/release-manager"):
            os.mkdir("/mnt/fs/var/lib/release-manager")
        with open("/mnt/fs/var/lib/release-manager/upgrade_status", "w") as f:
            f.write("Success")
        umount("/mnt/fs")
        if result:
            run_hook(hook, 'pre_enable')
            if smartmanager is not None:
                smartmanager.set_action("8: Enable new installation")
                smartmanager.expose("Upgrade", 100, name, "Enable new installation", 100)
            if options['auto_switch']:
                print("Setting permanent boot part")
                bootenv.mmc_part = mmc_part_new
            if options['test_update_part']:
                print("Setting temp boot part")
                bootenv.mmc_part_volatile = mmc_part_new
            if smartmanager is not None:
                smartmanager.expose("Upgrade", 100, name, "Finished successful", 100)
            run_hook(hook, 'post_enable')
    if options['daemon']:
        if smartmanager is not None:
            smartmanager.set_action("Upgrade Terminate")
        if result:
            if smartmanager is not None:
                smartmanager.upgrade_by_tar_terminate("OK")
        else:
            if smartmanager is not None:
                smartmanager.upgrade_by_tar_terminate("FAIL")
    if not run_hook(hook, 'post_tar_upgrade'):
        return False
    return result
