import socket, sys, time, select

class EsrError(Exception):
    strings = {\
    16:"Impossible to execute",\
    32:"Syntax error",\
    }

    def __init__(self, value):
        self.value = value

    def __str__(self):
        return "ESR("+str(self.value)+"):"+self.strings[self.value]

    def displayError(self):
        print(self.strings[self.value])


class ConnectionError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return str(self.value)



class ScpiAccess(object):
    """ SCPI Socket access """
    __max_buffer_size = 4096

    def __init__(self, host, port, timeout = 5, login = "*rem\n", tektronix = False):
        self.__host         = host
        self.__port         = int(port)
        self.__isConnected  = False
        self.__timeout      = timeout
        self.__socket       = None
        self.__buff         = []
        self.__login        = login
        self.__doubleRead   = tektronix

    def connect(self):
        if not self.__isConnected:
            self.__socket = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
            self.__socket.settimeout(self.__timeout)
            self.__socket.connect((self.__host,self.__port))
            if not self.__doubleRead:
                if (sys.version_info < (3,0,0)):
                    self.__socket.send(self.__login)
                else:
                    self.__socket.send(bytes(self.__login,'ascii'))
            self.__isConnected = True
            self.verify()

    def __readline(self):
        if self.__isConnected:
            if len(self.__buff) == 0:
                fullbuff = []
                inbuff = "0"
                while 1:
                    #print("@ readline read")
                    self.__socket.setblocking(0)
                    ready = select.select([self.__socket], [], [], self.__timeout)
                    if ready[0]:
                        inbuff = self.__socket.recv(self.__max_buffer_size)
                    else:
                        print("TIMEOUT")
                        break
                    #print("@ readline read %i: %s"%(len(inbuff), inbuff))
                    fullbuff.append(inbuff)

                    if b"\n" in inbuff:
                        break

                if (sys.version_info < (3,0,0)):
                    buff = "".join(fullbuff)
                    self.__buff = [l.strip() for l in buff.split("\n") if len(l) > 0]
                    self.__buff.reverse()
                else :
                    buff = b"".join(fullbuff)
                    self.__buff = [l.strip() for l in str(buff, 'utf-8').split("\n") if len(l) > 0]
                    self.__buff.reverse()
            if len(self.__buff):
                return self.__buff.pop()
            else:
                return ""
        else:
            raise ConnectionError("Not connected")

    def __writeline(self,val):
        self.connect()
        if (sys.version_info < (3,0,0)):
            self.__socket.send(val.strip()+"\n")
        else:
            self.__socket.send(bytes(val.strip()+"\n",'ascii'))

    def __purge(self):
        if self.__socket:
            self.__socket.setblocking(False)
            try:
                while self.__socket.recv(self.__max_buffer_size):
                    pass
            except BlockingIOError:
                pass
            self.__buff = []
            self.__socket.setblocking(True)

    def disconnect(self):
        if self.__isConnected:
            self.__socket.close()
            self.__socket = None
            self.__isConnected = False

    def verify(self):
        tempo = 0
        cont = 1
        count = 0
        while (cont == 1) and (count < 3):
            cont = 0
            self.__writeline("*esr?\n")
            time.sleep(tempo)
            try:
                r = self.__readline()
                if self.__doubleRead:
                    self.__readline()
                res = int(r)
            except ValueError as err:
                print("Error in verify (%s)"%r)
                self.__purge()
                tempo = 0.2
                count += 1
                cont = 1
                res=-1
        if res != 0:
            raise EsrError(res)

    def receive(self, size):
        return self.__socket.recv(size)

    def send(self, msg):
       self.__writeline(msg)

    def SendCommand(self,command,timeout=0,verbose=0):
        """ Send a SCPI command and check the ESR register"""
        if verbose:
            print(command)
        self.__writeline(command+"\n")
        if(timeout != 0):
            oldtimeout = self.__timeout
            self.__timeout = timeout
        self.verify()
        if(timeout != 0):
            self.__timeout = oldtimeout


    def SendCommandHalt(self,command,verbose=0):
        if verbose:
            print(command)
        self.__writeline(command+"\n")

    def SendAndReadCommand(self,command,timeout=0,verbose=0):
        """ Send a SCPI query, check the ESR egister and return the query result"""
        tempo = 0
        cont = 1
        count = 0
        pterr = None
        while (cont == 1) and (count < 5):
            cont = 0
            self.__purge()
            self.__writeline(command+"\n")
            if(timeout != 0):
                oldtimeout = self.__timeout
                self.__timeout = timeout
            res = self.__readline()
            if self.__doubleRead:
                self.__readline()
            time.sleep(tempo)
            try :
                self.__purge()
                self.verify()
                if(timeout != 0):
                    self.__timeout = oldtimeout
                if verbose:
                    print(command + "\t\t : " + res)
            except ValueError as err:
                tempo += 0.1
                count += 1
                cont = 1
                pterr = err
                print("")
                print(pterr)
                print("")
            except EsrError as err:
                tempo += 0.1
                count += 1
                cont = 1
                pterr = err
                print("")
                print(pterr)
                print("")
        if (count >= 5) :
            if verbose:
                for l in res:
                    print(l.strip())
            raise pterr
        else :
            if verbose:
                print(command + "\t\t : " + res)
            return res.strip()

    def query(self, command, *args):
        if '?' in command.split(" ")[0]:
            return self.SendAndReadCommand(command, *args)
        else:
            return self.SendCommand(command, *args)

if __name__ == "__main__":
    u = ScpiAccess(sys.argv[1], 8000, tektronix = False)
    u.connect()
    print(u.SendAndReadCommand("*idn?"))
