#!/usr/bin/env bash
#
# Initialize all otdr modules and do the low level i2c scan using uts-scan driver
#
# This is a modular scripts running functions in others scripts like plugins. It searches within /usr/lib/otdr/
# directory for the modules specific code
#
# The init scripts are launched following the alphabetical order
# If some scripts HAVE TO be executed before others, use some prefix (digits) to sort them accordingly
# Ex : '010-test.sh' will be handled before '020-another-test.sh'
#      But 'another-test.sh' will be handled before 'test.sh'

usage() {
	echo "Usage: $0 [-d <path for functions>] [-s]" 1>&2
	echo "           -d <path for functions> -- Path where the functions are located" 1>&2
	echo "           -s                      -- Stop mode - deinit instead of init" 1>&2
	exit 0
}

FUNCTIONS_PATH="/usr/lib/otdr/"

# By default, this script inits the OTDR modules (STOP=0)
# If it's called with the right parameter, it instead de-inits the OTDR modules (STOP=1)
STOP=0

while getopts "d:s" o
do
	case "${o}" in
	d)
		FUNCTIONS_PATH=${OPTARG}
		if ! [ -d ${FUNCTIONS_PATH} ] ; then
			echo "Functions directory doesn't exist, exiting"
			exit 1
		fi
		;;
	s)
		STOP=1
		;;
	*)
		usage
		;;
	esac
done
shift $((OPTIND-1))

# Sourcing all otdr functions files
function_list=()
for func_file in ${FUNCTIONS_PATH}/*.sh
do
	source $func_file
	func=$(echo $func_file | sed 's/.*\/[0-9]*-*//' | sed 's/\.sh//')
	function_list+=("${func}")
done

if [ "${STOP}" -eq "0" ]
then
	# Init all modules
	for func in "${function_list[@]}"
	do
		function_init="${func}_init"
		[[ $(type -t ${function_init}) == function ]] && echo "Initializing $func" && $function_init
	done

	# Do the scan
	if [ -d "/sys/platform/uts_scan" ]; then
		echo 1 > /sys/platform/uts_scan/rescan
		# Wait for the scan to be completed
		cat /sys/platform/uts_scan/ready
	else
		echo "No scan driver found"
	fi

	# Executing all post init functions
	for func in "${function_list[@]}"
	do
		function_init_post="${func}_init_post"
		[[ $(type -t ${function_init_post}) == function ]] && echo "Post init for $func" && $function_init_post
	done
else
	# De-init all modules
	for func in "${function_list[@]}"
	do
		function_deinit="${func}_deinit"
		[[ $(type -t ${function_deinit}) == function ]] && echo "De-initializing $func" && $function_deinit
	done
fi

exit 0
