<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\sharedmemory;

use app\util\SMTLogger;

/**
 * Store, update and fetch Dtos from APC shared memory.
 *
 * @author Sylvain Desplat
 *
 */
class SMTMemoryManager
{       
    /**
     * Fetch Dto from shared memory 
     * 
     * @param string $dtoClass dto class implementing SMTIMemorySupport
     * @param string $id dto identifier
     * 
     * @throws \InvalidArgumentException
     * @throws SMTAPCMemoryException
     * 
     * @return SMTIMemorySupport
     */
    static function fetch( $dtoClass, $id )
    {
    	$success = FALSE;
    	$dtos = NULL;
   	
    	$reflectionClass = new \ReflectionClass( $dtoClass );
    	if ( !$reflectionClass->implementsInterface( SMTIMemorySupport::SMTIMEMORY_SUPPORT_FULL_QUALIFIED_NAME ) )
    	{
    		throw new \InvalidArgumentException( sprintf( "Dto class %s should be an instance of SMTIMemorySupport", $dtoClass ) );
    	}
    	
    	if ( SMTAPCMemoryUtil::acquireLock() )
    	{
    		try
    		{
    			$dtos = SMTAPCMemoryUtil::readFromApcStore( $dtoClass );
    			SMTAPCMemoryUtil::releaseLock();
    		}
    		catch(SMTAPCMemoryException $e)
    		{
    			SMTAPCMemoryUtil::releaseLock();
    			throw $e;
    		}    		
    		catch(\Exception $e)
    		{
    			SMTAPCMemoryUtil::releaseLock();
    			throw $e;
    		}
    	}
    	else
    	{
    	    SMTLogger::getInstance()->trace("Couldn't acquire lock for APC store for :".$dtoClass, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_WRITING_APC );
    	}    	
    	
    	return ($dtos != NULL)? ( (array_key_exists( $id, $dtos) )? $dtos[$id] : NULL ) : NULL;
    }
    
    /**
     * Fetch all Dtos of this object type from shared memory
     *
     * @param string $dtoClass dto class implementing SMTIMemorySupport
     *
     * @throws SMTAPCMemoryException
     * @throws \InvalidArgumentException
     *
     * @return SMTIMemorySupport[]
     */
    static function fetchAll( $dtoClass )
    {
    	$success = FALSE;
    	$dtos = NULL;
    
    	$reflectionClass = new \ReflectionClass( $dtoClass );
    	if ( !$reflectionClass->implementsInterface( SMTIMemorySupport::SMTIMEMORY_SUPPORT_FULL_QUALIFIED_NAME ) )
    	{
    		throw new \InvalidArgumentException( sprintf( "Dto class %s should be an instance of SMTIMemorySupport", $dtoClass ) );
    	}
    
    	if ( SMTAPCMemoryUtil::acquireLock() )
    	{
    		try
    		{
    			$dtos = SMTAPCMemoryUtil::readFromApcStore( $dtoClass );
    			SMTAPCMemoryUtil::releaseLock();
    		}
    		catch(SMTAPCMemoryException $e)
    		{
    			SMTAPCMemoryUtil::releaseLock();
    			throw $e;
    		}
    		catch(\Exception $e)
    		{
    			SMTAPCMemoryUtil::releaseLock();
    			throw $e;
    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace("Couldn't acquire lock for APC store for :".$dtoClass, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_WRITING_APC );
    	}
    
    	return $dtos;
    }    
    
    /**
     * Fetch all Dtos from shared memory
     *
     * @throws SMTAPCMemoryException
     * @throws \InvalidArgumentException
     * 
     * @return array of SMTIMemorySupport[] for all dtos class types
    */
    static function fetchAllCache( )
    {
        $success = FALSE;
        $cache_user = NULL;            
        
        if ( SMTAPCMemoryUtil::acquireLock() )
        {
        	try
        	{
        		$cache_user = SMTAPCMemoryUtil::fetchAllFromApcStore();
        		SMTAPCMemoryUtil::releaseLock();
        	}
        	catch(SMTAPCMemoryException $e)
        	{
        		SMTAPCMemoryUtil::releaseLock();
        		throw $e;
        	}        	
        	catch(\Exception $e)
        	{
        		SMTAPCMemoryUtil::releaseLock();
        		throw $e;
        	}
        }
        else
        {
            SMTLogger::getInstance()->trace("Couldn't acquire lock for fetching all dtos stored in APC.", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_WRITING_APC );
        }        
        
    	return $cache_user["cache_list"];
    }       
    
    /**
     * Save all given Dtos objects of the given type to shared memory or remove it if they are obsolete
     *
     * @param SMTIMemorySupport[] $dtos dtos array of the same type
     *
     * @throws SMTAPCMemoryException
     * @throws \InvalidArgumentException
     */
    static function saveDtos( array &$dtos )
    {
    	$dtoClassName = NULL;
    	
    	foreach ( $dtos as $dto )
    	{
    	    if ( !( $dto instanceof SMTIMemorySupport ) )
    	    {
    	    	throw new \InvalidArgumentException( sprintf( "Dto class %s should be an instance of SMTIMemorySupport", get_class( $dto ) ) );
    	    }
    	    
    	    if ( $dtoClassName == NULL )
    	    {
    	        $dtoClassName = get_class( $dto );
    	    }
    	    else if ( $dtoClassName != get_class( $dto ))
    	    {
    	        throw new \InvalidArgumentException( sprintf( "Wrong dto class type %s for dtos array of type: %s ", get_class( $dto ), $dtoClassName ) );
    	    }
    	}
    	    	
    	if ( SMTAPCMemoryUtil::acquireLock() )
    	{
    		try
    		{
    		    foreach( $dtos as $key=>$dto )
    		    {
    		        //all dtos implements SMTIMemorySupport
    		        if ( $dto->isObsolete() )
    		        {
    		            unset( $dtos[$key] );
    		        }
    		    }
            	SMTAPCMemoryUtil::writeToApcStore( $dtoClassName, $dtos ); 	
            	SMTAPCMemoryUtil::releaseLock();
        	}
        	catch(SMTAPCMemoryException $e)
        	{
        		SMTAPCMemoryUtil::releaseLock();
        		throw $e;
        	}        	
        	catch(\Exception $e)
        	{
        		SMTAPCMemoryUtil::releaseLock();
        		throw $e;
        	}
    	}
    	else
    	{
    		throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_WRITING_APC );
    	}    	
    }    
    
    /**
     * Update all dtos with last access property stored in APC memory with the current date: 
     * required when changing the OTU date.
     * 
     */
    public static function touchAllDtos()
    {
        try 
        {
            if ( SMTAPCMemoryUtil::acquireLock() )
            {                
                $cache_list = self::fetchAllCache();
                        
                foreach ( $cache_list as $key=>$entry )
                {
                    $dto_class_key = $entry[SMTAPCMemoryUtil::APC_CACHE_INFO_DTOS_KEY];
                    $dtos = apc_fetch( $dto_class_key );
                    //SMTLogger::getInstance()->trace( $dto_class_key." => ".var_export($dtos, true), SMTLogger::DEBUG);
                    if ( $dtos !== NULL && is_array( $dtos) )
                    {                        
                        foreach ( $dtos as $key=>$dto )
                        {                
                        	if ( $dto instanceof SMTIMemorySupport )
                        	{
                        		$dto->touchAccess();
                        		SMTLogger::getInstance()->trace("touch dto: ".$dto_class_key, SMTLogger::DEBUG );
                        	}
                        }    
                  		SMTAPCMemoryUtil::writeToApcStore( $dto_class_key, $dtos );
                    }    
                }        
            }
        }
        catch(\Exception $e)
        {
            SMTAPCMemoryUtil::releaseLock();
        	SMTLogger::getInstance()->traceException($e);
        }        
    }
    
    /**
     * Save Dto in shared memory or remove it if it's obsolete
     * 
     * @param $dto SMTIMemorySupport
     * 
     * @throws SMTAPCMemoryException
     */
    static function saveDto( SMTIMemorySupport $dto )
    {
        $data_name = $dto->getDtoClassName();
        
        if ( SMTAPCMemoryUtil::acquireLock() )
        {
            try 
            {
                $dtos = SMTAPCMemoryUtil::readFromApcStore( $data_name );

                //all dtos implements SMTIMemorySupport
                if ( $dto->isObsolete() )
                {
                	unset( $dtos[ $dto->getId() ] );
                }
                else
                {
                    $dtos[ $dto->getId() ] = $dto ;
                }
                
                SMTAPCMemoryUtil::writeToApcStore( $data_name, $dtos );
                SMTAPCMemoryUtil::releaseLock();
            }
            catch(SMTAPCMemoryException $e)
            {
                SMTAPCMemoryUtil::releaseLock();
            	throw $e;
            }            
            catch(\Exception $e)
            {
                SMTLogger::getInstance()->trace("Couldn't write for APC store for :".$data_name." with dto id: ".$dto->getId()." and values ".var_dump($dto->getJsonData()), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                SMTAPCMemoryUtil::releaseLock();
                throw $e;
            }
        }
        else
        {
            SMTLogger::getInstance()->trace("Couldn't acquire lock for APC store for :".$dtoClass, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_WRITING_APC );
        }        
    }
        
    /**
     * Delete Dto from shared memory
     * 
     * @param $dto SMTIMemorySupport
     *
     * @throws SMTAPCMemoryException
    */
    static function deleteDto( SMTIMemorySupport $dto )
    {        
        self::delete( $dto->getDtoClassName(), $dto->getId() );
    }    
    
    /**
     * Delete Dto from shared memory
     *
     * @param string $dtoClass dto class implementing SMTIMemorySupport
     * @param string $id dto identifier
     *
     * @throws SMTAPCMemoryException
     */
    static function delete( $dtoClass, $id )
    {        
        $success = FALSE;
        
        if ( SMTAPCMemoryUtil::acquireLock() )
        {
        	try
        	{               
        	    $dtos = SMTAPCMemoryUtil::readFromApcStore( $dtoClass );
        	    if ( $dtos != NULL && (array_key_exists( $id, $dtos ) ) )
        	    {
                    unset( $dtos[ $id ] );
                    SMTAPCMemoryUtil::writeToApcStore( $dtoClass, $dtos );
        	    }
        	    SMTAPCMemoryUtil::releaseLock();
            }
            catch(SMTAPCMemoryException $e)
            {
            	SMTAPCMemoryUtil::releaseLock();
            	throw $e;
            }            
            catch(\Exception $e)
            {
            	SMTAPCMemoryUtil::releaseLock();
            	throw $e;
            }
        }
        else
        {
            SMTLogger::getInstance()->trace("Couldn't acquire lock for APC store for :".$dtoClass, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_WRITING_APC );
        }
    }    
    
    /**
     * Delete All Dto of the given type from shared memory
     *
     * @param string $dtoClass dto class implementing SMTIMemorySupport
     *
     * @throws SMTAPCMemoryException
     */
    static function deleteAll( $dtoClass )
    {
    	$success = FALSE;
    
    	if ( SMTAPCMemoryUtil::acquireLock() )
    	{
    		try
    		{
            	SMTAPCMemoryUtil::writeToApcStore( $dtoClass, array() );
            	SMTAPCMemoryUtil::releaseLock();
        	}
        	catch(SMTAPCMemoryException $e)
        	{
        		SMTAPCMemoryUtil::releaseLock();
        		throw $e;
        	}        	
        	catch(\Exception $e)
        	{
        		SMTAPCMemoryUtil::releaseLock();
        		throw $e;
        	}
       	}    
       	else
       	{
       	    SMTLogger::getInstance()->trace("Couldn't acquire lock for APC store for :".$dtoClass, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
       	    throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_WRITING_APC );
       	}        	
    }    
}
?>