<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\module;

use app\admin\SMTOtuType;

use app\serviceshelper\sensing\SMTDTSFunctionDto;

use app\serviceshelper\sensing\SMTDTSSFunctionDto;

use app\util\SMTLogger;

use app\serviceshelper\otdr\SMTOtdrRangeDto;

use app\serviceshelper\otdr\SMTOtdrPulseDto;

use app\sharedmemory\SMTMemoryManager;

use app\serviceshelper\otdr\SMTOtdrFunctionDto;

use app\serviceshelper\monitoring\SMTLinkTestException;

use app\serviceshelper\SMTServiceHelper;

use app\parser\SMTOtuApi;
use app\services\setup\SMTConfDetModuleDto;
use app\services\setup\SMTModuleDto;
use app\admin\SMTOtuMode;

/**
 * 
 * @author Sylvain Desplat
 *
 */
class SMTModule extends SMTServiceHelper
{
    const INDEX_MODULE_POSITION = 0;
    const INDEX_MODULE_TYPE = 1;
    const MODULE_COUNT = 3;
    const OTU_DTSS_FUNCTION = 'COBRA';
    const OTU_DTS_FUNCTION  = 'DTS';
    const HIGH_RESOLUTION   = '-2';
    const HIGH_DYNAMIC      = '-1';
    const DWDM_Laser        = '0';
    
    /**
     * Check module number: 0 or 1. Throw exception if the test fails.
     *
     * @param number $moduleRunningNumber Module Running Number
     *
     * @throws SMTLinkTestException
     */
    static function checkModuleRunningNumber( $moduleRunningNumber )
    {
    	if ( $moduleRunningNumber === NULL || $moduleRunningNumber <= 0 || $moduleRunningNumber > 3 )
    	{
    		throw new SMTLinkTestException( SMTLinkTestException::ERROR_INVALID_MODULE_SELECTED );
    	}
    }
    
    /**
     * Fetch OTDR module config
     *
     * @return \app\services\setup\SMTModuleDto
     */
    function fetchOTDRModuleConfig()
    {
    	list($mod1_registered, $mod2_registered, $mod3_registered) = explode(";", $this->sendReceive(SMTOtuApi::getModuleConfig()) );
    
    	//only one module managed by standalone OTU:
    	//fetch MOD1
    	$module = self::parseModuleString($mod1_registered, FALSE);
    	//if no module found, fetch MOD2 module
    	if ( $module == NULL )
    	{
    		$module = self::parseModuleString($mod2_registered, FALSE);
    	}
    	//if no module found, fetch MOD3 module
    	if ( $module == NULL )
    	{
    		$module = self::parseModuleString($mod3_registered, FALSE);
    	}
    	return $module;
    }
    
    /**
     * Fetch optical modules config
     * List all modules available (not only one): required for SmartOEM
     * 
     * @param app\services\setup\SMTConfDetModuleDto
     * 
     */
    function fetchModulesConfig( SMTConfDetModuleDto $configDetectModuleDto )
    {
    	list($mod1_registered, $mod2_registered, $mod3_registered) = explode(";", $this->sendReceive(SMTOtuApi::getModuleConfig()) );
    		
    	//only one module managed by standalone OTU:
    	//fetch MOD1
    	$module = self::parseModuleString($mod1_registered, FALSE);
    	if ( $module != NULL )
    	{
    	    $configDetectModuleDto->addConfiguredModule($module);
    	}
    	//if no module found, fetch MOD2 module
  		$module = self::parseModuleString($mod2_registered, FALSE);
  		if ( $module != NULL )
  		{
  		    $configDetectModuleDto->addConfiguredModule($module);
  		}
   		$module = self::parseModuleString($mod3_registered, FALSE);
      	if ( $module != NULL )
  		{
  		    $configDetectModuleDto->addConfiguredModule($module);
  		}    	
    }
    
    /**
     * Fetch optical modules detected
     * List all modules available (not only one): required for SmartOEM
     *
     * @param app\services\setup\SMTConfDetModuleDto
     *
     */
    function fetchModulesDetected( SMTConfDetModuleDto $configDetectModuleDto )
    {
        list($mod1_detected, $mod2_detected, $mod3_detected) = explode(";", $this->sendReceive(SMTOtuApi::CMD_detect_config) );
    
    	//parse MOD1
    	$module = self::parseModuleString($mod1_detected, TRUE);
    	if ( $module != NULL )
    	{
    		$configDetectModuleDto->addDetectedModule($module);
    	}
    	// parse MOD2 module
    	$module = self::parseModuleString($mod2_detected, TRUE);
    	if ( $module != NULL )
    	{
    		$configDetectModuleDto->addDetectedModule($module);
    	}
    	// parse MOD3
    	$module = self::parseModuleString($mod3_detected, TRUE);
    	if ( $module != NULL )
    	{
    		$configDetectModuleDto->addDetectedModule($module);
    	}
    }    
    
    /**
     * Check whether modules configuration != detection
     */
    function isDifference()
    {        
    	//module found in MOD1 or MOD2 or MOD3
    	$difference = ( $this->sendReceive( sprintf( SMTOtuApi::CMD_check_MOD, SMTModuleDto::MOD1 ) ) != SMTOtuApi::RES_OK );
    	$difference |= ( $this->sendReceive( sprintf( SMTOtuApi::CMD_check_MOD, SMTModuleDto::MOD2 ) ) != SMTOtuApi::RES_OK );
    	if ( (SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2()) && !SMTOtuMode::isONMS() && !SMTOtuMode::isROTAU() && !SMTOtuMode::isUFOM() )
    	{
    		$difference |= ( $this->sendReceive( sprintf( SMTOtuApi::CMD_check_MOD, SMTModuleDto::MOD3 ) ) != SMTOtuApi::RES_OK );
    	}
    	
    	return $difference;    	
    }
    
    /**
     * Fetch optical modules config (without full functions parameters) and compare the result with the detection
     * 
     * @return \app\services\setup\SMTConfDetModuleDto
     * @see SMTModule::fetchOtdrModuleFunctions
     */
    function fetchModules()
    {
        $configDetectModuleDto = new SMTConfDetModuleDto();
        $this->fetchModulesConfig( $configDetectModuleDto );
        $difference = FALSE;
        $index = 0;
        $hasModuleDifference = FALSE;
        //no MOD3 on compactOTU and in ONMS mode
        $moduleCount = (SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2())&&( !SMTOtuMode::isONMS() )? self::MODULE_COUNT : self::MODULE_COUNT - 1;
        
        while ( $index < $moduleCount )
        {
            $module = $configDetectModuleDto->getConfiguredModule($index);
            $position = SMTModuleDto::MOD_PREFIX.($index + 1);
            //module found in MOD1 or MOD2 or MOD3
            $difference = ( $this->sendReceive( sprintf( SMTOtuApi::CMD_check_MOD, $position) ) != SMTOtuApi::RES_OK ); //$module->getPosition() ) ) != "OK" );
            $hasModuleDifference |= $difference;
            $configDetectModuleDto->setModuleDifference($position, $difference);
                        
            //retrieve module settings
            if ( $module != NULL )
            {
            	$this->fetchCalibrationModuleSettings( $module );
            }
            $index++;
        }

        if ( $hasModuleDifference || ($configDetectModuleDto->getConfiguredModuleCount() == 0) )
        {
            $index=0;
            $this->fetchModulesDetected( $configDetectModuleDto );
            while ( $index < $configDetectModuleDto->getDetectedModuleCount() )
            {
                $module = $configDetectModuleDto->getDetectedModule($index);
                //retrieve module settings
                if ( $module != NULL )
                {
                	$this->fetchDetectionModuleSettings( $module );
                }
                $index++;
            }
        }
        
        //check whether there are link-tests
        $monitoringTests = TRUE;
        $linksId = explode(",", $this->sendReceive( SMTOtuApi::getLinkListCommand() ) );
        //if no link exists
        if ( !is_array( $linksId ) || count( $linksId ) <= 0 || $linksId[0] == NULL || $linksId[0] == "" )
        {
            $monitoringTests = FALSE;
        }
        $configDetectModuleDto->setMonitoringTests($monitoringTests);
        
        $moduleCompatibleWithSwitch= strcasecmp( 'TRUE', $this->sendReceive( SMTOtuApi::CMD_check_module_compatibility_with_switch ) ) == 0;
        $configDetectModuleDto->setModuleCompatibleWithSwitch($moduleCompatibleWithSwitch);
        
        $index=0;
        $difference = FALSE;
        while ( $index < $moduleCount)
        {
        	$position = SMTModuleDto::MOD_PREFIX.($index + 1);
        	$difference = $configDetectModuleDto->isModuleDifference($position);
        	$difference = ( $difference || (($configDetectModuleDto->getConfiguredModuleByPosition($position) == NULL) && ($configDetectModuleDto->getDetectedModuleByPosition($position) != NULL)));
        	$configDetectModuleDto->setModuleDifference($position, $difference);
        	$index++;
        }
        
    	return $configDetectModuleDto;
    }    
    
    /**
     * Retrieve OTDR Module functions and their wavelengths,pulses,ranges,resolutions.
     * Retrieve OTDR Module functions from cache and query OTU parser if they are not found in cache and save it in cache.
     * 
     * @param app\serviceshelper\otdr\SMTModuleDto $moduleDto If NULL, the default OTDR module is retrieved
     * 
     * @return app\serviceshelper\otdr\SMTModuleDto   
     */
    function fetchOtdrModuleFunctions( SMTModuleDto $moduleDto = NULL )
    {
    	if ( ( $modules = SMTMemoryManager::fetchAll( SMTModuleDto::getClass() ) ) == NULL || count ( $modules ) <= 0  )
    	{
    	    if ( $moduleDto == NULL )
    	    {
    	        $moduleDto = $this->fetchOTDRModuleConfig();
    	    }
    	    $functions = self::parseFunctionString( $this->sendReceive( SMTOtuApi::getModuleFunctions( $moduleDto->getPosition() ) ) );
    	    foreach ( $functions as $function )
    	    {
    	        //on se limite à l'OTDR
//     	        if ( $function === self::OTU_DTSS_FUNCTION)
//     	        {
//     	        	$functionDto = new SMTDTSSFunctionDto();
//     	        	$functionDto->setName($function);
//     	        	$moduleDto->setDTSSFunction( $functionDto );
//     	        }
//     	        else if ( $function === self::OTU_DTS_FUNCTION)
//     	        {
//     	        	$functionDto = new SMTDTSFunctionDto();
//     	        	$functionDto->setName($function);
//     	        	$moduleDto->setDTSFunction( $functionDto );
//     	        }
//     	        else
//     	        {
        	        $functionDto = new SMTOtdrFunctionDto();
        	        $functionDto->setName( $function );
        	        
        	        $moduleWavelengthString = $this->sendReceive( SMTOtuApi::getModuleWavelengths( $moduleDto->getPosition(), $function ) );
        	        if ( strcasecmp( $moduleWavelengthString, self::DWDM_Laser ) == 0 )
        	        {
        	        	$moduleFrequencyParametersString = $this->sendReceive( SMTOtuApi::getModuleFrequencyParameters( $moduleDto->getPosition() ) );
        	        	self::parseFrequencyParameters( $functionDto, $moduleFrequencyParametersString);
        	        	
        	        }
        	        else
        	        {
        	        	$functionDto->setWavelengthArray( self::parseWavelengthsString( $moduleWavelengthString) );
        	        }
        	        
            	    $pulses = self::parsePulsesString( $this->sendReceive( SMTOtuApi::getModulePulses( $moduleDto->getPosition(), $function )) );
            	    foreach ( $pulses as $pulse )
            	    {
            	        $pulseDto = new SMTOtdrPulseDto();
            	        $pulseDto->setPulseNs( $pulse );
            	        
            	        $ranges = self::parseRangesString( $this->sendReceive( SMTOtuApi::getModuleRanges( $moduleDto->getPosition(), $function, $pulse ) ) );        	        
            	        foreach ( $ranges as $range )
            	        {
            	        	$rangeDto = new SMTOtdrRangeDto();
            	        	$rangeDto->setRangeKm( $range );
            	        		
            	        	$resolutions = self::parseResolutionsString( $this->sendReceive( SMTOtuApi::getModuleResolutions( $moduleDto->getPosition(), $function, $pulse, $range ) ) );
            	        	$rangeDto->setOtdrResolutionCmArray( $resolutions );
            	        	$pulseDto->addOtdrRange( $rangeDto );
            	        }
            	        $functionDto->addOtdrPulse( $pulseDto );
            	    }       	    
            	    $moduleDto->addFunction( $functionDto );
//     	        }
    	    }
    	    
    	    //save module full config in memory:
    	    SMTMemoryManager::saveDto( $moduleDto );
    	}
    	else
    	{
    	    //only one module in smartOTU; WARNING: modify the original array
    	    $moduleDto = array_shift( $modules );
    	    $functions = $moduleDto->getFunctions();
    	    
    	    //always fetch dwdm frequencies, because they can have changed
    	    foreach ( $functions as &$functionDto )
    	    { 
    	    	if ( count( $functionDto->getDwdmFrequenciesArray() ) > 0 ) 
    	    	{
    	    		$moduleFrequencyParametersString = $this->sendReceive( SMTOtuApi::getModuleFrequencyParameters( $moduleDto->getPosition() ) );
    	    		$functionDto->setDwdmFrequenciesArray( array() );
    	    		self::parseFrequencyParameters( $functionDto, $moduleFrequencyParametersString);
    	    	}
    	    }
    	}
    	
    	return $moduleDto;
    }
    
    /**
     * Parse the OTDR module string and create the module dto if the module exists.
     *
     * @param string $moduleString
     * @param bool $detected
     *
     * @return \app\services\setup\SMTModuleDto or NULL if the module doesn't exist
     */
    private static function parseModuleString( $moduleString, $detected )
    {
    	$module = NULL;
    	if ( strlen($moduleString) > 0)
    	{
	    	//retrieve module attributes
	    	$array_module = preg_split("/[:,]+/", $moduleString );
	    	//remove spaces in array values
	    	$array_module = array_map ( "trim", $array_module );
	    	
	    	SMTLogger::getInstance()->trace("MODULE_POSITION ".$array_module[self::INDEX_MODULE_POSITION], SMTLogger::DEBUG);
	    	SMTLogger::getInstance()->trace("MODULE_TYPE ".$array_module[self::INDEX_MODULE_TYPE], SMTLogger::DEBUG );
	    	if ( is_array($array_module) && (strlen($array_module[self::INDEX_MODULE_TYPE]) > 0) && $array_module[self::INDEX_MODULE_TYPE] != "-1")
	    	{
	    		$module = new SMTModuleDto();
	    		$module->setDetected($detected);
	    		$module->setPosition( $array_module[self::INDEX_MODULE_POSITION] );
	    		$module->setType( $array_module[self::INDEX_MODULE_TYPE] );
	    	}
    	}
    	return $module;
    }
    
    /**
     * Parse the OTDR module function string.
     *
     * @param string $functionString
     *
     * @return string
     */
    private static function parseFunctionString( $functionsString )
    {
    	return explode(",", $functionsString );
    }    
    
    /**
     * Parse the OTDR module wavelengths string.
     *
     * @param string $functionString
     *
     * @return int[]
     */
    private static function parseWavelengthsString( $wavelengthsString )
    {
    	$wavelengths = explode( ",", $wavelengthsString );
    	$wavelengths = array_map ( "trim", $wavelengths );
    	return array_map( 'intval', $wavelengths );    	
    }    
    
    private static function parseFrequencyParameters( SMTOtdrFunctionDto $functionDto, $moduleFrequencyParametersString )
    {
    	$frequencies = explode( ",", $moduleFrequencyParametersString);
    	$frequencies= array_map ( "trim", $frequencies);
    	$frequencies = array_map( 'intval', $frequencies);
    	
    	list($minFreqGhz, $maxFreqGhz, $spacingGhz) = $frequencies;
    	for ( $channel= $minFreqGhz; $channel<= $maxFreqGhz; $channel+=$spacingGhz )
    	{
    		$functionDto->addDwdmFrequenciesArray ( $channel );
    	}
    }
    
    /**
     * Parse the OTDR module wavelengths string.
     *
     * @param string $pulsesString
     *
     * @return int[]
     */
    private static function parsePulsesString( $pulsesString )
    {
    	$pulses = explode(",", $pulsesString );
    	$pulses = array_map ( "trim", $pulses );
    	return array_map( 'intval', $pulses );
    }    
    
    /**
     * Parse the OTDR module ranges string.
     *
     * @param string $rangesString
     *
     * @return float[]
     */
    private static function parseRangesString( $rangesString )
    {
        $ranges = explode(",", $rangesString );
        $ranges = array_map ( "trim", $ranges );
    	return array_map( 'floatval', $ranges );
    }    
    
    /**
     * Parse the OTDR module resolutions string.
     *
     * @param string $resolutionsString
     *
     * @return int[]
     */
    private static function parseResolutionsString( $resolutionsString )
    {
    	$resolutions = explode(",", $resolutionsString );
    	$resolutions = array_map ( "trim", $resolutions );
    	//add high_dyn and high_res solutions:
    	array_splice($resolutions,1,0, array(self::HIGH_DYNAMIC, self::HIGH_RESOLUTION));
    	return array_map( 'intval', $resolutions );
    }    
    
    /**
     * Fetch additional module infos from calibration
     *
     * @param SMTModuleDto $module
     * 
     * @return SMTModuleDto
     */
    private function fetchCalibrationModuleSettings( SMTModuleDto $module)
    {
    	$serialNumber = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_NMOD, $module->getPosition() ) );
    	$module->setSerialNumber( $serialNumber );
    
    	$function = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_Funct, $module->getPosition() ) );
    	$functions = explode(",", $function );
    	foreach ($functions as $func_name)
    	{
    	    if ( $func_name === self::OTU_DTSS_FUNCTION)
    	    {
    	        $functionDto = new SMTDTSSFunctionDto();
    	        $functionDto->setName($func_name);
    	        $module->setDTSSFunction( $functionDto );
    	    }
    	    else
    	    {    	             	       
        	    if ( $func_name === self::OTU_DTS_FUNCTION)
        	    {
        	        $functionDto = new SMTDTSFunctionDto();
        	        $functionDto->setName($func_name);
        	        $module->setDTSFunction( $functionDto );    	         
        	    }
        	    else 
        	    {
            	    $functionDto = new SMTOtdrFunctionDto();
            	    $functionDto->setName( $func_name );
            	    $module->addFunction( $functionDto );
        	    }
        	    
        	    //retrieve wavelengths
        	    $laser = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_Laser_List_SI, $module->getPosition(), $func_name ) );
        	    $laser = trim($laser);
        	     
        	    if ( !empty($laser) )
        	    {
        	    	$wavelengths = explode(",", $laser);
        	    	for ( $index = 0; $index < count($wavelengths); $index++ )
        	    	{
        	    		$functionDto->addWavelength( $wavelengths[$index] );
        	    	}
        	    }
    	    }
    	}
    	return $module;
    }
    
    /**
     * Fetch additional module infos from detection
     *
     * @param SMTModuleDto $module
     * 
     * @return SMTModuleDto
     */
    private function fetchDetectionModuleSettings( SMTModuleDto $module)
    {
    	$serialNumber = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_NMOD_Detected, $module->getPosition() ) );
    	$module->setSerialNumber( $serialNumber );
    	 
    	$function = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_Funct_Detected, $module->getPosition() ) );
    	$functions = explode(",", $function );
    	foreach ($functions as $func_name)
    	{
	    	if ( $func_name === self::OTU_DTSS_FUNCTION)
	    	{
	    		$functionDto = new SMTDTSSFunctionDto();
	    		$functionDto->setName($func_name);
	    		$module->setDTSSFunction( $functionDto );
	    	}
	    	else
	    	{
    	    	if ( $func_name === self::OTU_DTS_FUNCTION)
    	    	{
    	    		$functionDto = new SMTDTSFunctionDto();
    	    		$functionDto->setName($func_name);
    	    		$module->setDTSFunction( $functionDto );
    	    	}
    	    	else
    	    	{
            		$functionDto = new SMTOtdrFunctionDto();
            		$functionDto->setName( $func_name );
            		$module->addFunction( $functionDto );
    	    	}
        			
        		//retrieve $wavelengths
        		$laser = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_Laser_List_Detected_SI, $module->getPosition(), $func_name ) );    		
        		$laser = trim($laser);
        		
        		if ( !empty($laser) )
        		{
           	    $wavelengths = explode(",", $laser);
        		    for ( $index = 0; $index < count($wavelengths); $index++ )
        		    {
       				   $functionDto->addWavelength( $wavelengths[$index] );
        		    }
        		}
	    	}    		    		
    	}
    	return $module;
    }        
}

?>
