<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\alarm;

use app\util\SMTLogger;

use app\database\SMTDatabaseException;

use app\database\SMTSmartOtuDB;

use app\util\SMTResultSetDto;

use app\util\SMTUtil;

use app\util\SMTInvalidValueException;

/**
 * Base alarm event Dto 
 * 
 * @author Sylvain Desplat
 */
abstract class SMTAlarmEventDto extends SMTResultSetDto
{
    /**
     * Alarm event identifier
     * @var integer
     */
    protected $id;
    
    /**
     * @var integer
     */
    protected $alarmId;    

    /**
     * UTC date time in seconds
     * 
     * @var integer/float
     */
    protected $time;
    
    /**
     * @var integer
     */
    protected $severity;
        
    const FETCH_ALARM_EVENT_SEQUENCE = "SELECT sequence from alarm_event_sequence";
    
    const INCREMENT_ALARM_EVENT_SEQUENCE = "UPDATE alarm_event_sequence set sequence = sequence + 1";
    
    /**
     * Generate new alarm Id (moke database sequence)
     * WARNING Should be called from a critical section!
     *
     * @param SMTSmartOtuDB $dbConnection
     *
     * @return integer
     */
    public function generateId( SMTSmartOtuDB $dbConnection )
    {
        SMTLogger::getInstance()->trace( "Generate new id for alarm event: ".$this->getDtoClassName(), SMTLogger::DEBUG );
        $query = self::INCREMENT_ALARM_EVENT_SEQUENCE;
        $success = $dbConnection->execWithTrace( $query, SMTLogger::DEBUG);
         
        if ( $success === FALSE )
        {
        	SMTLogger::getInstance()->trace( "Query failed:: ".$query , SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
        }
         
        $query = self::FETCH_ALARM_EVENT_SEQUENCE;
        $sequence = $dbConnection->querySingleWithTrace( $query, SMTLogger::DEBUG);
         
        if ( $sequence == NULL || $sequence === FALSE )
        {
        	SMTLogger::getInstance()->trace( "Query failed:: ".$query , SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
        }
    	 
    	return $sequence;
    }    
    
    /**
     * Alarm event identifier
     *
     * @return integer
     */
    public function getId()
    {
    	return $this->id;
    }
    
    /**
     * Alarm Id can be set to null when parent alarm of the event is not yet saved
     * @param integer  $testId
     */
    public function setAlarmId( $alarmId )
    {
    	$this->alarmId = SMTUtil::convertToInt( $alarmId );
    }
    /**
     *
     * @return integer
     */
    public function getAlarmId()
    {
    	return $this->alarmId;
    }    
    
    /**
     * Set UTC date time in seconds
     * 
     * @param integer/float $time UTC date time in seconds
     */
    public function setTime( $time )
    {
        if ( !isset($time) || $time === NULL || !SMTUtil::isInt( $time ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "time", $time );
        }
        $this->time = SMTUtil::convertToInt( $time );
    }
    
    /**
     * Get UTC date time in seconds
     * 
     * @return integer/float
     */
    public function getTime()
    {
        return $this->time;
    }
    
    /**
     *
     * @param integer $severity
     */
    public function setSeverity( $severity )
    {
    	$this->severity = $severity;
    }
    /**
     *
     * @return integer
     */
    public function getSeverity()
    {
    	return $this->severity;
    }    
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
}



?>