"""
Module that contains schemas for the CDM configuration section for OTDR tests to
launch on the ONA product family
"""

from marshmallow import Schema, fields, post_load
from marshmallow.validate import OneOf

from rest_api.products.config_schema_common import ConfigBase, test_type_factory, config_factory

#pylint: disable=no-self-use

class FOTestLaunch():
    """
    Fiber Optics app's Test launcher details
    """
    launch = {
       'launch_type' : "exe",
       'can_launch'  : "/usr/share/job-manager/launchers/fiber/fiber_optics_ijm_tests_can_launch", # Just dummy .sh for now - Needs to change to .py so can manage scpi queries to ISU
       'launch'      : "/usr/bin/fiber_optics_ijm_test_launcher"  # Straight copy of /acterna/release/bin/otdr_launch.sh for now (Will need to change to .py script to  manage required scpi commands to isu
    }

class CdmOTDRSchema(Schema, FOTestLaunch):
    """
    Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
        'bidirOtdr',
    )
    fiber_types = (
        'Simplex',
        'Duplex',
        'MPO',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Str(
        load_from='launchCable',
        dump_to='launchCable',
        required=False,
        allow_none=True,
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='alarmThresholds',
        dump_to='alarmThresholds',
        description='The alarm thresholds'
    )
    config_file = config_factory('/user/disk/')
    setups = fields.Nested(
        Schema,
        missing=[],
        required=False,
        allow_none=True,
        load_from='setup',
        dump_to='setup',
        description='The setup(s)'
    )
    fiber = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='fiber',
        dump_to='fiber',
        description='The info on the fiber'
    )
    fiber_type = fields.Str(
        required=False,
        validate=OneOf(fiber_types),
        load_from='fiberType',
        dump_to='fiberType',
        description='The type of fiber to be tested'
    )

class OTDRSettingsSchema(Schema, FOTestLaunch):
    """
    Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    test_type = 'OTDR'

    otdr_settings = fields.Nested(
        CdmOTDRSchema,
        required=True,
        missing={},
        load_from='otdrSettings',
        dump_to='otdrSettings',
        description='The CDM configuration structure for OTDRs'
    )


CONFIG_SCHEMAS = {
                 'OTDR': OTDRSettingsSchema
                 }
