from viavi.stdhw.ioctlutil import *
import fcntl

_UTS_SPI_IO_MAGIC = 0xD2
#define SPI_SEND_AND_RECEIVE                 _IOWR( cUTS_SPI_IO_MAGIC, 0, tSpi_Send_Receive_Parameters)
#define SPI_MOD_INTERF_ACCESS                _IOWR( cUTS_SPI_IO_MAGIC, 1, tSpi_Module_Interface_Access)
#_SPI_MOD_INTERF_ACCESS = IOWR(_UTS_SPI_IO_MAGIC, 1, "ILLIPI")
#define SPI_DIRECT_SEND_AND_RECEIVE          _IOWR( cUTS_SPI_IO_MAGIC, 2, tSpi_Direct_Send_Receive_Parameters)
#define SPI_DIRECT_MOD_INTERF_ACCESS         _IOWR( cUTS_SPI_IO_MAGIC, 3, tSpi_Direct_Module_Interface_Access)
#define SPI_REQUEST_SERIAL_PORT              _IOWR( cUTS_SPI_IO_MAGIC, 4, tSpi_Module_Interface_Access)
#define SPI_FREE_SERIAL_PORT                 _IOWR( cUTS_SPI_IO_MAGIC, 5, tSpi_Module_Interface_Access)
#define SPI_NON_BLOCKING_REQUEST_SERIAL_PORT _IOWR( cUTS_SPI_IO_MAGIC, 6, tSpi_Module_Interface_Access)
#define SPI_SEND_AND_RECEIVE_DELAYED         _IOWR( cUTS_SPI_IO_MAGIC, 7, tSpi_Send_Receive_Delayed_Parameters)


class _Spi_Send_Receive_Parameters(ctypes.Structure):
    _fields_ = [
        ('Hw_Module_Side',  ctypes.c_int),
        ('Hw_Module_Slice', ctypes.c_long),
        ('Cs_Module',       ctypes.c_uint8),
        ('Frequency',       ctypes.c_uint8),
        ('Tx_Buffer',       ctypes.c_void_p),
        ('Tx_Length',       ctypes.c_uint8),
        ('Rx_Buffer',       ctypes.c_void_p),
        ('Rx_Length',       ctypes.c_uint8),
        ('Clk_Polarity',    ctypes.c_int)
        ]

class _Spi_Module_Interface_Access(ctypes.Structure):
    _fields_ = [
        ('Hw_Module_Side',  ctypes.c_int),
        ('Hw_Module_Slice', ctypes.c_long),
        ('Address',         ctypes.c_ulong),
        ('Data_Length',     ctypes.c_uint8),
        ('Data_Buffer',     ctypes.c_void_p),
        ('RW_flag',         ctypes.c_int)
        ]

_SPI_MOD_INTERF_ACCESS = IOWR(_UTS_SPI_IO_MAGIC, 1, ctypes.sizeof(_Spi_Module_Interface_Access))

cWRITE = 0
cREAD  = 1


class NotImplemented(Exception):
    pass


class UtsSpi(object):
    '''
    Class to use IOCTL from driver uts-spi
    '''
    READ = 1
    WRITE = 0

    def __init__(self, device = "/dev/uts_spi"):
        self.__dev = device
        self.__fd = None

    def open(self):
        if self.__fd is None:
            self.__fd = open(self.__dev, "rb")

    def close(self):
        if self.__fd is not None:
            self.__fd.close()
            self.__fd = None

    def send_and_receive(self):
        raise NotImplemented()


    def mod_interf_access(self, side, slice, address, data_length, data, flag):
        arg = _Spi_Module_Interface_Access()
        arg.Hw_Module_Side = side
        arg.Hw_Module_Slice = slice
        arg.Address = address
        arg.Data_Length = data_length
        arg.Data_Buffer = data
        arg.RW_flag = flag
        if self.__fd is not None:
            fcntl.ioctl(self.__fd, _SPI_MOD_INTERF_ACCESS, arg)
        else:
            with open(self.__dev, "rb") as f:
                fcntl.ioctl(f, _SPI_MOD_INTERF_ACCESS, arg)
        return arg.Data_Buffer

    def direct_send_and_receive(self):
        raise NotImplemented()

    def direct_mod_interf_access(self):
        raise NotImplemented()

    def request_serial_port(self):
        raise NotImplemented()

    def free_serial_port(self, size):
        raise NotImplemented()

    def non_blocking_request_serial_port(self):
        raise NotImplemented()

    def send_and_receive_delayed(self):
        raise NotImplemented()

    def read(self, size):
        data = None
        if self.__fd is not None:
            data = self.__fd.read(size)
        else:
            with open(self.__dev, "rb") as f:
                data = f.read(size)
        return data


class SpiModule(UtsSpi):
    '''
    Class to use directly a FPGA module throw SPI
    '''
    def __init__(self, side, slice, device = "/dev/uts_spi"):
        super().__init__(device)
        self.__dev = device
        self.__side = side
        self.__slice = slice
        self.open()

    def __del__(self):
        self.close()

    def read_mod_register8(self, address):
        num = ctypes.c_uint8(0)
        addr = ctypes.addressof(num)
        self.mod_interf_access(self.__side, self.__slice, address, 1, addr, 1)
        return num.value

    def write_mod_register8(self, address, value):
        num = ctypes.c_uint8(0)
        addr = ctypes.addressof(num)
        num.value = value
        self.mod_interf_access(self.__side, self.__slice, address, 1, addr, 0)

    def read_mod_register16(self, address):
        num = ctypes.c_uint16(0)
        addr = ctypes.addressof(num)
        self.mod_interf_access(self.__side, self.__slice, address, 2, addr, 1)
        return letoh16(num.value)

    def write_mod_register16(self, address, value):
        num = ctypes.c_uint16(0)
        addr = ctypes.addressof(num)
        num.value = htole16(value)
        self.mod_interf_access(self.__side, self.__slice, address, 2, addr, 0)

    def read_mod_register24(self, address):
        num = ctypes.c_uint32(0)
        addr = ctypes.addressof(num)
        self.mod_interf_access(self.__side, self.__slice, address, 3, addr, 1)
        return letoh32(num.value)

    def write_mod_register24(self, address, value):
        num = ctypes.c_uint32(0)
        addr = ctypes.addressof(num)
        num.value = htole32(value)
        self.mod_interf_access(self.__side, self.__slice, address, 3, addr, 0)


    def read_mod_register32(self,  address):
        num = ctypes.c_uint32(0)
        addr = ctypes.addressof(num)
        self.mod_interf_access(self.__side, self.__slice, address, 4, addr, 1)
        return letoh32(num.value)

    def write_mod_register32(self, address, value):
        num = ctypes.c_uint32(0)
        addr = ctypes.addressof(num)
        num.value = htole32(value)
        self.mod_interf_access(self.__side, self.__slice, address, 4, addr, 0)
