import QtQuick 2.5
import QtQuick.Layouts 1.2
import QtQml 2.3

import "ViaviStyle"

FiberContainer {
    id: numericDisplayContainer

    readonly property real scaleFactor: ViaviStyle.layouts.isSmallScreen ? 0.75 : 1
    readonly property real numericDisplayButtonHeight: ViaviStyle.layouts.buttonHeight * numericDisplayContainer.scaleFactor

    property string infoType: 'NumericDisplay requires this to be defined!'
    property bool isLaserActive: getDataModel().laserImagePath.length > 0
    property bool isDwdmWavelengthSelection: infoType === 'Source' && numericdisplay_res_isSourceDwdm.value
    property bool isSourceBbs: infoType === 'Source' && numericdisplay_res_isSourceBbs.value

    //TODO: all usages of 'Table', Source, Powermeter, Realtime need to be made constants
    function getDataModel() {
        switch (infoType) {
        case 'Table':
            return numericdisplay_res_dataTable.value
        case 'Source':
            return numericdisplay_res_dataSource.value
        case 'Powermeter':
            return numericdisplay_res_dataPowermeter.value
        default:
            return null
        }
    }

    function getExtraDataModel() {
        switch (infoType) {
        case 'Table':
            return numericdisplay_res_extradataTable.value
        case 'Source':
            return numericdisplay_res_extraDataSource.value
        case 'Powermeter':
            return numericdisplay_res_extraDataPowermeter.value
        default:
            return null
        }
    }

    function mainColour() {
        switch (infoType) {
        case 'Source':
            return ViaviStyle.colors.sourceColor
        case 'Powermeter':
            return ViaviStyle.colors.powermeterColor
        default:
            return ViaviStyle.colors.mainInfoColor
        }
    }

    function modeValueColour(colourName) {
        switch (colourName) {
        case 'success':
            return ViaviStyle.colors.passColor
        case 'alarm':
            return ViaviStyle.colors.warningColor
        case 'digitBlue':
            return mainColour()
        default:
            return mainColour()
        }
    }

    function getLtsDisplayHeight(){
        return  ViaviStyle.layouts.smallMargin
                + ViaviStyle.layouts.titleTextFontMetrics.height //digit type text
                + ViaviStyle.layouts.numericValueFontMetric.tightBoundingRect("0").height // value text
                + 3 * numericDisplayContainer.numericDisplayButtonHeight //comboboxes
    }

    function getTableDisplayHeight(){
        return ViaviStyle.layouts.mediumMargin + ViaviStyle.layouts.numericValueFontMetric.height
                + 2 * (ViaviStyle.layouts.activeTextFontMetrics.height + ViaviStyle.layouts.mediumMargin)
                + ViaviStyle.layouts.mediumMargin
    }

    height: 'Source' === infoType || 'Powermeter' === infoType
                ? getLtsDisplayHeight()
                : getTableDisplayHeight()

    enabled: getDataModel().active

    GridLayout {
        id: topLayout
        columns: 3

        anchors.leftMargin: ViaviStyle.layouts.mediumMargin
        anchors.rightMargin: ViaviStyle.layouts.mediumMargin
        anchors.bottomMargin: ViaviStyle.layouts.mediumMargin * numericDisplayContainer.scaleFactor
        anchors.topMargin: ViaviStyle.layouts.smallMargin * numericDisplayContainer.scaleFactor
        anchors.fill: parent

        ColumnLayout {
            id: leftLayout
            Layout.column: 0
            Layout.rowSpan: 4
            visible: infoType === 'Source' || infoType === 'Powermeter'

            Row {
                id: typeLayout
                Layout.preferredHeight: typeStr.contentHeight

                spacing: ViaviStyle.layouts.smallMargin

                Image {
                    id: typeImage
                    visible: getDataModel().typeIconPath.length !== 0
                    fillMode: Image.PreserveAspectFit
                    height: ViaviStyle.layouts.imageButtonHeight //same as the combobox
                    source: getDataModel().typeIconPath
                }

                Text {
                    id: typeStr

                    anchors.verticalCenter: typeImage.verticalCenter

                    visible: getDataModel().typeStr.length !== 0
                    text: getDataModel().typeStr
                    color:ViaviStyle.colors.mainInfoColor
                }
            }

            Item {
                id: modeSpacer
                Layout.fillHeight: true
            }

            RowLayout {
                id: modePowermeterLayout
                visible: infoType === 'Powermeter'
                height: numericDisplayContainer.numericDisplayButtonHeight
                Text {
                    text: numericdisplay_res_modeLabelText.value + ':' //'Mode: '
                    font: ViaviStyle.layouts.largeFont
                    color: ViaviStyle.colors.mainInfoColor
                }
                Text {
                    text: getDataModel().mode
                    font: ViaviStyle.layouts.titleTextFont
                    color: modeValueColour(getDataModel().modeValueColour)
                }
            }

            Column{
                id: optionColumn

                spacing: ViaviStyle.layouts.smallMargin

                property int rowMinSpacing: ViaviStyle.layouts.largeMargin

                width: Math.max(modeComboLayout.visible ? modeComboLayout.minNecessaryWidth : 0
                                ,wavelengthLayout.visible ? wavelengthLayout.minNecessaryWidth : 0
                                ,unitLayout.visible ? unitLayout.minNecessaryWidth : 0)

                RowLayout {
                    id: modeComboLayout

                    visible: infoType === 'Source'

                    width: parent.width
                    spacing: 0

                    property int minNecessaryWidth: modeText.contentWidth + optionColumn.rowMinSpacing + modeCombobox.width

                    Text {
                        id: modeText
                        text: numericdisplay_res_modeLabelText.value + ':' //'Mode:'
                        font: ViaviStyle.layouts.largeFont
                        color: ViaviStyle.colors.mainInfoColor
                    }

                    Item {
                        Layout.fillWidth: true
                        Layout.minimumWidth: optionColumn.rowMinSpacing
                    }

                    NumericDisplayComboBox {
                        id: modeCombobox
                        implicitHeight: numericDisplayContainer.numericDisplayButtonHeight
                        implicitWidth: height * 4
                        model: numericdisplay_res_choicesSource_0.value
                        activeIndex: numericdisplay_res_selectionsSource_0
                    }
                }

                RowLayout {
                    id: wavelengthLayout
                    visible: ( infoType === 'Source' && !isSourceBbs ) || infoType === 'Powermeter'

                    width: parent.width
                    spacing: 0

                    property int minNecessaryWidth: waveLengthText.contentWidth + optionColumn.rowMinSpacing + wavelengthComboBox.width

                    Text {
                        id: waveLengthText

                        text: isDwdmWavelengthSelection ? numericdisplay_res_sourceDwdmTitle.value + ':' : numericdisplay_res_wavelengthLabelText.value + ':' //'Wavelength:'
                        font: ViaviStyle.layouts.largeFont
                        color: ViaviStyle.colors.mainInfoColor
                    }

                    Item {
                        Layout.fillWidth: true
                        Layout.minimumWidth: optionColumn.rowMinSpacing
                    }

                    NumericDisplayComboBox {
                        id: wavelengthComboBox
                        visible: !isDwdmWavelengthSelection

                        implicitHeight: numericDisplayContainer.numericDisplayButtonHeight
                        implicitWidth: height * 4

                        model: {
                            if (infoType === 'Source')
                                return numericdisplay_res_choicesSource_1.value
                            else
                                return numericdisplay_res_choicesPowermeter_1.value
                        }
                        activeIndex: {
                            if (infoType === 'Source')
                                return numericdisplay_res_selectionsSource_1
                            else
                                return numericdisplay_res_selectionsPowermeter_1
                        }
                    }

                    FiberButton {
                        id: dwdmWaveLengthButton
                        visible: isDwdmWavelengthSelection

                        width: wavelengthComboBox.width
                        height: wavelengthComboBox.height

                        text: numericdisplay_res_sourceDwdmSelectedChoice.value
                        onClicked: numericDisplayDwdmWavelengthSelection.open()
                    }
                }

                RowLayout {
                    id: unitLayout
                    visible: infoType === 'Powermeter' || isDwdmWavelengthSelection

                    width: parent.width
                    spacing: 0

                    property int minNecessaryWidth: unitText.contentWidth + optionColumn.rowMinSpacing + unitComboBox.width

                    Text {
                        id: unitText

                        text: numericdisplay_res_unitLabelText.value + ':' //'Unit:'
                        font: ViaviStyle.layouts.largeFont
                        color: ViaviStyle.colors.mainInfoColor
                    }

                    Item {
                        Layout.fillWidth: true
                        Layout.minimumWidth: optionColumn.rowMinSpacing
                    }

                    NumericDisplayComboBox {
                        id: unitComboBox
                        enabled: infoType === 'Powermeter' || isDwdmWavelengthSelection

                        implicitHeight: numericDisplayContainer.numericDisplayButtonHeight
                        implicitWidth: height * 4

                        model: {
                            if ( infoType === 'Powermeter' )
                                return numericdisplay_res_choicesPowermeter_2.value
                            else
                                return numericdisplay_res_choicesSource_2.value
                        }
                        activeIndex: {
                            if ( infoType === 'Powermeter' )
                                return numericdisplay_res_selectionsPowermeter_2
                            else
                                return numericdisplay_res_selectionsSource_2
                        }
                    }
                }
            }
        }

        ColumnLayout {
            id: middleLayout
            Layout.column: 1
            Layout.row: 0
            Layout.rowSpan: 3

            RowLayout {
                id: mainValueLayout
                Text {
                    id: modeLabel
                    visible: infoType !== 'Source' && infoType !== 'Powermeter'
                    //Note: this is a visual workaround since the texts are not consistent about the space at the end
                    text: getDataModel().mode + '  '
                    font: ViaviStyle.layouts.numericConfigFont
                    color: ViaviStyle.colors.mainInfoColor
                    Layout.alignment: Qt.AlignVCenter
                }

                Text {
                    id: mainValue
                    function formatDigit(doubleValue, statusAndSign) {
                        var precision = 3
                        if (infoType === 'Powermeter' && (getDataModel().unit === 'dB' || getDataModel().unit === 'dBm')) {
                            precision = 2
                         }
                        else  {
                        	if (infoType === 'Table' && (getDataModel().unit === 'THz')) {
                            	precision = 5
                         	}
                        	else {
                        		if (infoType === 'Table' && (getDataModel().unit === 'nm')) {
                            		precision = 4
                        		}
                        	}
                        }
                        var res = statusAndSign + doubleValue.toFixed(precision)
                        return res
                    }
                    readonly property string noValueMarkerStr: '– – – –'
                    text: {
                        switch (infoType) {
                        case 'Table':
                            if (getDataModel().active ) {
                                return formatDigit(getDataModel().value, getDataModel().statusAndSign)
                            }
                            else {
                                return noValueMarkerStr
                            }

                        case 'Powermeter':
                            if (getDataModel().active && isLaserActive ) {
                                return formatDigit(getDataModel().value, getDataModel().statusAndSign)
                            }
                            else {
                                return noValueMarkerStr
                            }

                        case 'Source':
                            if (getDataModel().active && isLaserActive ) {
                                if (isDwdmWavelengthSelection) {
                                    return numericdisplay_res_sourceDwdmSelectedChoice.value
                                }
                                else {
                                		if (isSourceBbs){
                                    	return numericdisplay_res_sourceBbsTitle.value
                                		}
                                		else {
                                    	return getDataModel().wavelength
                                    }
                                }
                            }
                            else {
                                return noValueMarkerStr
                            }
                        default:
                            return getDataModel().statusAndSign + getDataModel().value
                        }
                    }
                    font: ViaviStyle.layouts.numericValueFont
                    color: ViaviStyle.colors.mainInfoColor
                    Layout.alignment: Qt.AlignBaseline
                }

                Text {
                    id: mainUnit
                    visible: getDataModel().active && ( isLaserActive || ( infoType === 'Table' ) ) && !isDwdmWavelengthSelection
                    text: getDataModel().unit
                    font: ViaviStyle.layouts.numericConfigFont
                    color: ViaviStyle.colors.mainInfoColor
                    Layout.alignment: Qt.AlignBaseline
                }
                Layout.alignment: Qt.AlignHCenter
            }
             Text {
                id: autoLambdalLabel
                visible: infoType === 'Powermeter'
                text: getDataModel().autoLambda
                font: ViaviStyle.layouts.activeTextFont
                Layout.alignment: Qt.AlignCenter
                color: modeValueColour(getDataModel().autoLambdaColour)
            }
            Text {
                id: powerLevelLabel
                visible: infoType === 'Source' && !isSourceBbs
                text: numericdisplay_res_powerLevelLabelText.value + ': ' + getDataModel().modeValue
                font: ViaviStyle.layouts.activeTextFont
                Layout.alignment: Qt.AlignCenter
            }
            Layout.alignment: Qt.AlignHCenter
        }

        Image {
            id: warningImage
            visible: infoType === 'Source'
            fillMode: Image.PreserveAspectFit
            Layout.column: 2
            Layout.row: 1
            Layout.rowSpan: 2
            Layout.preferredHeight: mainValue.height
            Layout.preferredWidth: height
            source: getDataModel().laserImagePath
        }


        //Two buttons for Powermeter, mode value for Source
        RowLayout {
            id: powermeterButtonsLayout
            visible: infoType === 'Powermeter' || infoType == 'Source'

            Layout.column: 1
            Layout.row: 3
            Layout.columnSpan: 2
            Layout.alignment: Qt.AlignRight | Qt.AlignBottom

            FiberToolButton {
                id: keepResultsButton
                text: numericdisplay_res_keepResultsLabel.value
                visible: infoType === 'Powermeter'
                enabled: numericdisplay_res_keepResultsEnabled.value
                implicitHeight: numericDisplayContainer.numericDisplayButtonHeight
                implicitWidth: height * 5
                iconPath: ViaviStyle.images.keepResultsIconPath
                onClicked: {
                    numericdisplay_act_keepResults.invoke()
                }
            }

            FiberButton {
                id: referenceButton
                text: numericdisplay_res_setAsReferenceLabel.value
                visible: infoType === 'Powermeter'
                enabled: numericdisplay_res_setAsReferenceEnabled.value
                height: numericDisplayContainer.numericDisplayButtonHeight
                width: height * 5
                onClicked: {
                    numericdisplay_act_setAsReference.invoke()
                }
            }

            FiberToolButton {
                id: startStopLaserButton
                text: numericdisplay_res_startStopLaserLabel.value
                visible: infoType === 'Source'
                enabled: numericdisplay_res_startStopLaserEnabled.value
                implicitHeight: numericDisplayContainer.numericDisplayButtonHeight
                implicitWidth: height * 5
                iconPath: numericdisplay_res_startStopLaserIconPath.value
                onClicked: {
                    numericdisplay_act_sourceStartStopLaser.invoke()
                }
            }
        }

        GridLayout {
            id: extraDataControl
            //note: the extra data is also visible in Powermeter, but in another layout and with labels
            //may or may not be needed in the final display
            visible: infoType === 'Table'
            columns: 2
            rowSpacing: ViaviStyle.layouts.mediumMargin
            columnSpacing: ViaviStyle.layouts.largeMargin

            Layout.columnSpan: 3
            Layout.alignment: Qt.AlignTop | Qt.AlignHCenter

            Repeater {
                id: extraDataRepeater
                model: getExtraDataModel()
                Text {
                    text: modelData
                    font: ViaviStyle.layouts.activeTextFont
                }
            }
        }
    }
}
