import QtQuick 2.5

import "ViaviStyle"

FiberContainer {
    id: miniSlmContainer

    property int windowHeight: height
    property int windowWidth: width

    property int centralItemWidth: windowWidth * 0.8
    property int centralItemHeight: windowHeight * 0.6

    property int centralItemTopMargin: windowHeight * 0.1
    property int centralItemBottomMargin: windowHeight * 0.1

    property int eventHeight: centralItemHeight * 0.7
    readonly property int initialEventWidth: 6
    property int eventWidth: initialEventWidth

    property int delimiterWidth: eventHeight

    readonly property int initialMinConnLength: 10
    property int minConnLength: initialMinConnLength
    readonly property int sectionHeight: 2

    property double firstItemDistance
    property double lastItemDistance
    property bool firstLastFromTrace //trace can have first last inside an actual event connection, need more calculation for the trace
    property int selectedItemIndex

    /* available space for the miniSlm list */
    property int listViewItemWidth: centralItemWidth - locAItem.width - locBItem.width

    property alias firstConnectionHighlighted: firstConnection.isHighlighted
    property alias lastConnectionHighlighted: lastConnection.isHighlighted

    SlmUtils { id: slmUtils }

    Item {
        id: centralItem /* contains the event list and the location labels*/

        visible: schematic_res_slmNotEmpty.value
        width: centralItemWidth
        height: centralItemHeight

        anchors.fill: parent
        anchors.topMargin:  centralItemTopMargin
        anchors.bottomMargin: centralItemBottomMargin
        anchors.rightMargin: windowHeight + ViaviStyle.layouts.mediumMargin /* make room for the link status information */
        anchors.leftMargin: ViaviStyle.layouts.mediumMargin

        // Item Location A: Text + MiniSlm start
        Item {
            id: locAItem

            width: locAText.width + delimiterWidth
            height: parent.height

            anchors.left: parent.left
            anchors.top: parent.top

            Text {
                id: locAText
                width: contentWidth + ViaviStyle.layouts.smallMargin

                anchors.top: parent.top
                anchors.topMargin: centralItemTopMargin
                anchors.left: parent.left

                text: schematic_res_slmOrigin.value

                color: ViaviStyle.colors.secondaryInfoColor
                font: ViaviStyle.layouts.isSmallScreen ? ViaviStyle.layouts.mediumFont : ViaviStyle.layouts.smallFont
                horizontalAlignment: Text.AlignLeft

            }

            Rectangle {

                width: Math.floor( delimiterWidth * 0.3 )
                height: width

                anchors.verticalCenter: parent.verticalCenter
                anchors.left: locAText.right

                radius: 30
                color: ViaviStyle.colors.mainInfoColor


            }
            Rectangle {
                id: firstConnection

                width: Math.floor ( delimiterWidth * 0.7 )
                height: isHighlighted ? sectionHeight * 2 : sectionHeight

                anchors.verticalCenter: parent.verticalCenter
                anchors.right: parent.right

                color: isHighlighted ? ViaviStyle.colors.activeColor : ViaviStyle.colors.borderColor

                property bool isHighlighted

            }
        }

        // Item Location B -> Text + Total Length + miniSlm End
        Item {
            id: locBItem
            width: textsItem.width + delimiterWidth
            height: parent.height

            anchors.right : parent.right
            anchors.top: parent.top

            Item {
                id: textsItem

                width: Math.ceil( Math.max(locBText.width, locBLenText.width) )
                anchors.top: parent.top
                anchors.topMargin: centralItemTopMargin
                anchors.right: parent.right
                anchors.bottom : parent.bottom
                anchors.bottomMargin: centralItemBottomMargin

                Text {
                    id: locBText
                    width: contentWidth + ViaviStyle.layouts.smallMargin
                    anchors.top: parent.top
                    anchors.right: parent.right

                    text: schematic_res_slmExtremity.value

                    color: ViaviStyle.colors.secondaryInfoColor
                    font: ViaviStyle.layouts.isSmallScreen ? ViaviStyle.layouts.mediumFont : ViaviStyle.layouts.smallFont

                    horizontalAlignment: Text.AlignRight
                }

                /* Total length text */
                Text {
                    id: locBLenText

                    width: contentWidth + ViaviStyle.layouts.smallMargin
                    anchors.bottom: parent.bottom
                    anchors.right: parent.right

                    text: schematic_res_slmTotalLength.value +
                          " " + schematic_res_slmUnitLabel.value

                    horizontalAlignment: Text.AlignRight

                    color: ViaviStyle.colors.mainInfoColor
                    font: ViaviStyle.layouts.isSmallScreen ? ViaviStyle.layouts.mediumBoldFont : ViaviStyle.layouts.smallBoldFont
                }
            }

            Rectangle {
                width: Math.floor( delimiterWidth * 0.3 )
                height: width

                anchors.verticalCenter: parent.verticalCenter
                anchors.right: textsItem.left

                radius: 30
                color: ViaviStyle.colors.mainInfoColor

            }

            Rectangle {
                id: lastConnection
                width: Math.floor( delimiterWidth * 0.7 )
                height: isHighlighted ? sectionHeight * 2 : sectionHeight

                anchors.verticalCenter: parent.verticalCenter
                anchors.left: parent.left

                color: isHighlighted ? ViaviStyle.colors.activeColor : ViaviStyle.colors.borderColor

                property bool isHighlighted
            }
        }


        // Item for displaying the event list
        Rectangle {
            id: eventsItemContainer

            visible: schematic_res_slmNotEmpty.value

            height: parent.height
            anchors.left: locAItem.right
            anchors.right: locBItem.left

            Row {

                anchors.verticalCenter: parent.verticalCenter
                anchors.left: parent.left
                anchors.right: parent.right

                Repeater {
                    id: eventsRepeater

                    model: schematic_res_slmEventList.value
                    onCountChanged:
                    {
                        var connWidth = initialMinConnLength
                        var evWidth = initialEventWidth

                        if( ( eventsItemContainer.width > 0 ) && ( eventsRepeater.count > 0 ) &&
                              ( ( eventsItemContainer.width / (connWidth+evWidth) ) <  eventsRepeater.count ) )
                        {
                            var maxVal = Math.floor( eventsItemContainer.width / eventsRepeater.count )
                            evWidth = Math.max ( 1, Math.floor( maxVal / 2))
                            connWidth = Math.max ( 1, maxVal - evWidth )
                        }

                        minConnLength = connWidth
                        eventWidth = evWidth
                    }

                    focus:true

                    /* available line length for variable distance calculations */
                    property int numberOfVisibleEvents: schematic_cfg_slmShowFirstEntryPlaceholder.value ? eventsRepeater.count - 1 : eventsRepeater.count
                    property int lineLength: eventsItemContainer.width - (eventsRepeater.count - 1) * minConnLength
                                             - numberOfVisibleEvents * eventWidth

                    // event list delegate
                    delegate: Item {

                        id: eventListDelegate

                        property bool isSelectedEvent: ( miniSlmContainer.selectedItemIndex === model.eventIndex )

                        property double modelEventDistance: model.eventDistance
                        //if the elements are in order maxBefore and Min after are equal to modelEventDistance
                        //when events are not in order discontinuities can be present on the minislm
                        //using the min max instead of the eventDistance makes sure that if an out of order element is highlighted the midle elements are as well.
                        property double modelEventMaxDistanceBefore: model.maxDistanceBefore
                        property double modelEventMinDistanceAfter: model.minDistanceAfter

                        property double modelConnectionDistance: model.connectionDistance !== "" ? model.connectionDistance: 0.0

                        property bool isHighlightedSection: (miniSlmContainer.firstItemDistance <= modelEventMaxDistanceBefore
                                                             && modelEventMinDistanceAfter <= miniSlmContainer.lastItemDistance)
                                                            || (miniSlmContainer.firstLastFromTrace && minDistanceAfter < miniSlmContainer.lastItemDistance
                                                                && miniSlmContainer.lastItemDistance <= Math.max(maxDistanceBefore,  modelEventDistance + modelConnectionDistance ))
                                                            || (miniSlmContainer.firstLastFromTrace && minDistanceAfter <= miniSlmContainer.firstItemDistance
                                                                && miniSlmContainer.firstItemDistance < Math.max(maxDistanceBefore,  modelEventDistance + modelConnectionDistance ))

                        function updateExtremitiesConnections(){
                            if(model.index === 0){
                                miniSlmContainer.firstConnectionHighlighted = isHighlightedSection
                            }

                            if(model.index === eventsRepeater.count - 1){
                                miniSlmContainer.lastConnectionHighlighted = isHighlightedSection
                            }
                        }

                        onIsHighlightedSectionChanged: updateExtremitiesConnections()

                        Component.onCompleted: {
                            eventListDelegate.updateExtremitiesConnections()
                        }

                        property bool isFirstEntryPlacholder: (model.index === 0 && schematic_cfg_slmShowFirstEntryPlaceholder.value)

                        width: event.width + connection.width
                        height: eventHeight

                        anchors.verticalCenter: parent.verticalCenter

                        Rectangle {
                            id: event
                            width: eventListDelegate.isFirstEntryPlacholder ? 0 : eventWidth
                            height: isSelectedEvent ? centralItemHeight : eventHeight

                            anchors.verticalCenter: parent.verticalCenter

                            radius: 15
                            color: isSelectedEvent ? ViaviStyle.colors.activeColor : ViaviStyle.colors.getColor(model.alarmStatus)
                            visible: !eventListDelegate.isFirstEntryPlacholder
                        }

                        Rectangle {
                            id: connection

                            width: {
                                var computedWidth = slmUtils.getMiniSlmConnSize( model.connectionDistance,
                                                                                schematic_res_totalConnectionsSum.value,
                                                                                eventsRepeater.lineLength,
                                                                                minConnLength )

                                // for the last event we need to conect the line with the locBItem delimiter
                                if( model.index === (eventsRepeater.count - 1))
                                {
                                    var missingLineWidth = eventsItemContainer.width > eventListDelegate.x ? eventsItemContainer.width - eventListDelegate.x - event.width : 0
                                    if( isNaN( computedWidth ))
                                    {
                                        //rounding up missingLineWidth ( Real) because we might miss 1 pixel
                                        computedWidth = Math.round ( missingLineWidth )
                                    } else
                                    {
                                        //rounding up missingLineWidth ( Real) because we might miss 1 pixel
                                        computedWidth += Math.round ( missingLineWidth )
                                    }
                                }

                                return Math.floor( computedWidth )
                            }
                            height: isHighlightedSection ? sectionHeight * 2 : sectionHeight

                            anchors.verticalCenter: parent.verticalCenter
                            anchors.left: eventListDelegate.isFirstEntryPlacholder ? parent.left : event.right

                            color: isHighlightedSection ? ViaviStyle.colors.activeColor : ViaviStyle.colors.borderColor
                        }
                    }
                }
            }
        }
    }

    // Link status image
    Rectangle {
        visible: schematic_res_slmGlobalAlarmStatus.value !== 0 && schematic_res_slmNotEmpty.value

        width: parent.height
        height: parent.height

        anchors.right : parent.right
        anchors.top: parent.top
        anchors.bottom: parent.bottom

        border.width: ViaviStyle.layouts.borderSmallWidth
        border.color: ViaviStyle.colors.borderColor

        Image {

            anchors.fill: parent
            anchors.topMargin: ViaviStyle.layouts.smallMargin
            anchors.bottomMargin: ViaviStyle.layouts.smallMargin

            source: schematic_res_slmGlobalAlarmStatus.value === -1 ? ViaviStyle.images.iconFailedAlarmStatus : ViaviStyle.images.iconPassedAlarmStatus
            fillMode: Image.PreserveAspectFit
            smooth: true
        }
    }
}
