import QtQuick 2.7
import QtQuick.Controls 2.0
import QtQuick.Window 2.2
import "ViaviStyle"

Menu {
    id: choiceEditorMenu

    property Item menuContainer
    property Item selectedMenuLine

    property int columnWidth
    property int maxColumnWidth

    readonly property int singleColumnNrLimit: 5
    readonly property int maxColumnsNr: 3

    onOpened: {
        choiceEditorMenu.forceActiveFocus()
    }

    closePolicy: Popup.NoAutoClose

    //Use the functions to a make a dinamic positioning and sizeing of the component.
    //Will use the following strategy
    //  1
    //Priority will be to display the choice editor to the left of the seleted menu line
    //and the first row of the choice editor will be on the same line of the selected menu line
    //      1.1
    //If more than 5(singleColumnNrLimit) choices are available
    //try to display on 3(maxColumnsNr) columns or how many columns are available to the left
    //      1.2
    //If the choice editor height is to large to fit the screen under the current selectedMenuLine
    //make sure the choice editor is fully displayed on the screen and at least keep the choices aligned to the menulines
    //  2
    //If the current selected menu line is in the leftmost of the display
    //in this situation we should have enough space to display all the ChoiceEditor to the right of the menu item.
    //  3
    //If menu is not leftmost menu and don't have enough space to display the items in the remaining columns
    //display 3 columns just make sure the choice editor is either bellow the current item and not overlaps it
    //if that is not posible display the choice edtior above the current item and make sure no overlap occurs.
    //Also make sure the choice align to the curent item left side.

    width: calculateWidth()
    height: choicesGridView.contentHeight

    x: calculateX()
    y: menu_res_textEdtiorIncremental.value ? calculateYForIncrement() : calculateY()

    background: Rectangle {
        border.width: 0
    }

    function calculateWidth(){
        if (!selectedMenuLine || !menuContainer) {
            //no menu Container than don't have any way of determening available screen surface
            return choicesGridView.cellWidth * ( choicesGridView.count > singleColumnNrLimit ? maxColumnsNr : 1 )
        } else if(menuContainer.x < choicesGridView.cellWidth) {
            //in this case the parent menu is in the leftmost extrem so we will display the choice editor on the right side
            return choicesGridView.cellWidth * ( choicesGridView.count > singleColumnNrLimit ? maxColumnsNr : 1 )
        } else {
            //check to see if availble space to the left
            var availableColumns = (menuContainer.x / choicesGridView.cellWidth) | 0
            var availableRows = (menuContainer.parent.height / choicesGridView.cellHeight) | 0
            var availableItems = availableColumns * availableRows

            if(choicesGridView.count <= availableItems) {
                //have enough space to display all the items on the left side, our priority
                var maxDisplayColumns = Math.min(availableColumns, maxColumnsNr)
                var displayColumns = choicesGridView.count > singleColumnNrLimit ? maxDisplayColumns : 1
                return displayColumns * choicesGridView.cellWidth
            } else {
                //will have to display the choice editor either below or above the selected item
                return maxColumnsNr * choicesGridView.cellWidth
            }
        }
    }

    function calculateX() {
        if(selectedMenuLine){
             if(!menuContainer){
                 //no menu Container than don't have any way of determening available screen surface
                 return selectedMenuLine.x - width
             } else {
                 if(menuContainer.x < choicesGridView.cellWidth){
                    //align it to the right
                     if(choicesGridView.cellWidth === choiceEditorMenu.columnWidth){
                        return selectedMenuLine.x + selectedMenuLine.width
                     } else {// this means there is one item and has been edited with the keyboard)
                         return Math.max(selectedMenuLine.x - width, -menuContainer.x)
                     }
                 } else {
                    if(menuContainer.x < width){
                        if(choicesGridView.cellWidth === choiceEditorMenu.columnWidth){
                            //display it bellow preferablly or above if no space
                            var availableColumns = (menuContainer.x / choicesGridView.cellWidth) | 0
                            return selectedMenuLine.x - availableColumns * choicesGridView.cellWidth
                        } else {  // this means there is one item and has been edited with the keyboard)
                            return Math.max(selectedMenuLine.x - width, -menuContainer.x)
                        }
                    } else {
                        //our priority display it to the left
                        return selectedMenuLine.x - width
                    }
                 }
             }

        } else {
             //nothing to align to
             return 0
        }
    }

    function calculateY() {
        if(selectedMenuLine){
             if(!menuContainer){
                 //no menu Container than don't have any way of determening available screen surface
                 return selectedMenuLine.y
             } else {
                 if(menuContainer.x < choicesGridView.cellWidth
                         && choicesGridView.cellWidth === choiceEditorMenu.columnWidth // otherwise means there is one item and has been edited with the keyboard
                         ){
                    //align it to the right just make sure the whole choice editor fits
                     var availableRows = ((menuContainer.parent.height - (menuContainer.y + selectedMenuLine.y))
                                             / choicesGridView.cellHeight)
                                             | 0
                     var rows = height / choicesGridView.cellHeight
                     return selectedMenuLine.y - Math.min(0, availableRows - rows) * choicesGridView.cellHeight
                 } else {
                    if(menuContainer.x < width){
                        //display it bellow preferablly or above if no space
                        if(menuContainer.y + selectedMenuLine.y + height + choicesGridView.cellHeight < menuContainer.parent.height){
                            //bellow
                            return selectedMenuLine.y + choicesGridView.cellHeight
                        } else {
                            //above
                            return selectedMenuLine.y - height
                        }
                    } else {
                        //our priority display it to the left, just make sure the whole choice editor fits
                        var maxRows = ((menuContainer.parent.height - (menuContainer.y + selectedMenuLine.y))
                                                / choicesGridView.cellHeight)
                                                | 0
                        var gridRows = height / choicesGridView.cellHeight
                        return selectedMenuLine.y + Math.min(0, maxRows - gridRows) * choicesGridView.cellHeight
                    }
                 }
             }

        } else {
             //nothing to align to
             return 0
        }
    }

    //if the text view is an increment editor, must make sure the arrows are visible on the screen
    function calculateYForIncrement(){
        var y = calculateY();

        if(menuContainer && menuContainer.parent){
            if(y + menuContainer.y < ViaviStyle.layouts.buttonHeight){
                y = ViaviStyle.layouts.buttonHeight - menuContainer.y
            }

            if(menuContainer.y + y + choicesGridView.cellHeight + ViaviStyle.layouts.buttonHeight > menuContainer.parent.height){
                y = menuContainer.parent.height - ViaviStyle.layouts.buttonHeight - choicesGridView.cellHeight - menuContainer.y
            }
        }

        return y
    }


    contentItem: GridView {
        id: choicesGridView

        anchors.fill: parent

        //Note: this is a workaround for bug FOFTSA-659 which happens on SmartOTDR and 5800, most likely because of Qt Quick graphics bugs on these platforms.
        //When showing menu elements in a grid that doesn't fill the last row, there will be a "filler" part of the popup to make it a complete rectangle.
        //This part may not disappear when changing the elements until triggering enough repaints.
        //The existance of this almost completely transparent rectangle forces a more complete repaint of the popup.
        Rectangle {
            anchors.fill: parent
            color: '#01ffffff'
        }

        focus: true
        Keys.onUpPressed: {
            if(menu_res_textEdtiorIncremental.value){
                menu_act_incrementTextDisplay.invoke()
            }else{
                if (currentItem) {
                    currentItem.handleExited()
                }
                choicesGridView.moveCurrentIndexUp()
                if (currentItem) {
                    currentItem.handleEntered()
                }
            }
        }

        Keys.onDownPressed: {
            if(menu_res_textEdtiorIncremental.value){
                menu_act_decrementTextDisplay.invoke()
            }else{
                if (currentItem) {
                    currentItem.handleExited()
                }
                choicesGridView.moveCurrentIndexDown()
                if (currentItem) {
                    currentItem.handleEntered()
                }
            }
        }

        Keys.onLeftPressed: {
            if (currentItem) {
                currentItem.handleExited()
            }
            choicesGridView.moveCurrentIndexLeft()
            if (currentItem) {
                currentItem.handleEntered()
            }
            if (count === 0) {
            	if (event.modifiers & Qt.AltModifier) {
                	menu_act_decrementFastTextDisplay.invoke()
                } else {
                	menu_act_decrementTextDisplay.invoke()
                }
            }
        }

        Keys.onRightPressed: {
            if (currentItem) {
                currentItem.handleExited()
            }
            choicesGridView.moveCurrentIndexRight()
            if (currentItem) {
                currentItem.handleEntered()
            }
            if (count === 0) {
            	if (event.modifiers & Qt.AltModifier) {
                	menu_act_incrementFastTextDisplay.invoke()
                } else {
                	menu_act_incrementTextDisplay.invoke()
                }
            }
        }
        Keys.onReturnPressed: {
            if (currentItem) {
                currentItem.handleClick()
            }
        }

        keyNavigationWraps: true

        property int columnWidth: choiceEditorMenu.columnWidth

        cellWidth: choicesGridView.columnWidth
        cellHeight: ViaviStyle.layouts.menuLineHeight
        boundsBehavior: Flickable.StopAtBounds


        model: menu_res_choiceEditor.value

        /* on model reset we have to restore the column width */
        Connections
        {
            target: menu_res_choiceEditor.value
            onModelReset: choicesGridView.columnWidth = choiceEditorMenu.columnWidth
        }

        delegate: Item {

            width: choicesGridView.columnWidth
            height: choicesGridView.cellHeight
            opacity: model.choiceColor === menu_res_choiceColor.enums.ecColor_Grey ? 0.6: 1

            Component.onCompleted:
            {
                /* increase columnWidth if text is larger than the standard size */
                if ( choicesGridView.columnWidth  < ( choiceLabel.contentWidth + ViaviStyle.layouts.mediumMargin ))
                {
                   choicesGridView.columnWidth = choiceLabel.contentWidth + ViaviStyle.layouts.mediumMargin
                }
                /* if the text is too large, set text width and wrap */
                if( choiceLabel.contentWidth > choiceEditorMenu.maxColumnWidth )
                {
                    choicesGridView.columnWidth = choiceEditorMenu.maxColumnWidth
                    choiceLabel.width = choicesGridView.columnWidth
                    choiceLabel.wrapMode = Text.Wrap
                }
            }

            AbstractButton {
                visible:  menu_res_textEdtiorIncremental.value
                z: 1
                anchors.bottom: parent.verticalCenter
                width: choiceRectangle.width
                height: ViaviStyle.layouts.buttonHeight + parent.height/2
                FiberColorImage{
                    width: ViaviStyle.layouts.buttonHeight
                    height: ViaviStyle.layouts.buttonHeight
                    anchors.top: parent.top
                    anchors.horizontalCenter: parent.horizontalCenter
                    color: parent.pressed ? ViaviStyle.colors.selectedColor : ViaviStyle.colors.activeColor
                    source: ViaviStyle.images.iconMenuUpArrow
                }

                onClicked: {
                    menu_act_incrementTextDisplay.invoke()
                }
            }

            AbstractButton {
                visible:  menu_res_textEdtiorIncremental.value
                z: 1
                anchors.top: parent.verticalCenter
                width: choiceRectangle.width
                height: ViaviStyle.layouts.buttonHeight + parent.height/2
                FiberColorImage {
                    width: ViaviStyle.layouts.buttonHeight
                    height: ViaviStyle.layouts.buttonHeight
                    anchors.bottom: parent.bottom
                    anchors.horizontalCenter: parent.horizontalCenter
                    color: parent.pressed ? ViaviStyle.colors.selectedColor : ViaviStyle.colors.activeColor
                    source: ViaviStyle.images.iconMenuDownArrow
                }

                onClicked: {
                    menu_act_decrementTextDisplay.invoke()
                }
            }

            /* line information rectangle */
            Rectangle {
                id: choiceRectangle

                width: parent.width
                height: parent.height

                anchors.centerIn: parent

                color: model.choiceColor === menu_res_choiceColor.enums.ecColor_Selected ? ViaviStyle.colors.activeColor :
                                                 model.choiceColor === menu_res_choiceColor.enums.ecColor_Grey ? ViaviStyle.colors.alternativeBackgroundColor : ViaviStyle.colors.contentBackgroundColor

                border.color: ViaviStyle.colors.borderColor
                border.width: model.choiceColor === menu_res_choiceColor.enums.ecColor_Selected ? 0: ViaviStyle.layouts.borderSmallWidth
            }

            /* Choice label */
            Text {
                id: choiceLabel

                anchors.centerIn: parent

                text: model.choiceLabel
                color: model.choiceColor === menu_res_choiceColor.enums.ecColor_Selected ? ViaviStyle.colors.activeItemTextColor : ViaviStyle.colors.mainInfoColor

                horizontalAlignment: Text.AlignHCenter
                font : ViaviStyle.layouts.largeBoldFont
                wrapMode: Text.WrapAnywhere
            }

            MouseArea {
                anchors.fill: parent

                enabled: model.choiceColor !== menu_res_choiceColor.enums.ecColor_Grey

                hoverEnabled: true

                onClicked: handleClick()
                onEntered: handleEntered()
                onExited: handleExited()
            }

            function handleEntered() {
                if( model.choiceColor === menu_res_choiceColor.enums.ecColor_Normal)
                {
                    choiceRectangle.color = ViaviStyle.colors.pageBackgroundColor
                }
            }
            function handleExited() {
                if( model.choiceColor === menu_res_choiceColor.enums.ecColor_Normal )
                {
                    choiceRectangle.color = ViaviStyle.colors.contentBackgroundColor
                }
            }
            function handleClick() {
                menu_cfg_selectedChoice.value = model.choiceIndex
            }
        }
    }
}
