<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otdr;

use app\serviceshelper\otdr\SMTBaseOtdrTraceFileUtil;

use app\serviceshelper\activity\SMTActivity;

use app\http\SMTSessionDirectoryUtil;

use app\serviceshelper\monitoring\SMTOtdrReferenceFiles;

use app\util\SMTIOException;

use app\util\SMTUtil;

use app\util\SMTLogger;


/**
 * Utility class to handle OTDR trace files on OTU file system
 * 
 * @author Sylvain Desplat
 */
class SMTOtdrTraceFileUtil extends SMTBaseOtdrTraceFileUtil
{        
    /**
     * Directory for alarm files
     * @var string
     */
    const CUSTOM_SMARTOTU_ALARM_DIR = "/otu/smartOTU/alarms/";
    
    /**
     * SmartOTU alarm otdr file format: trace_%test Id%_%alarm discriminator%
     *
     * @var string
     */
    const SMARTOTU_OTDR_ALARM_TRACE_FILE_WO_EXTENSION = "%s_trace_%s_%s_port%s";

    /**
     * SmartOTU alarm otdr file format: trace_%test Id%_%alarm discriminator%
     * 
     * @var string
     */
    const SMARTOTU_OTDR_ALARM_TRACE_FILE = "%s_trace_%s_%s_port%s.sor";
    
    const LOCK_LAST_ACQUISITION_FILE = "/../../../tmp/lastAcquisition.lock";
    
    /**
     * Last acquisition file exclusive lock
     *
     * @var string
     */
    private static $lastAcquisitionLockFile = NULL;
    
    /**
     * Copy the measure acquisition or the last acquisition from SmartOTU directory to OTU reference detection and localisation directories.
     * Don't remove the mesure acquisition from SmartOTU directory (can be used again in case of test addition failure)
     *  
     *  NOT USED FOR HIGHSENS
     *
     * @param $portNumber string The port number where the acquisition was done.
     * @param $linkId string The link identifier.
     * @param $testId The test identifier if it exists.
     * @param $otdrShortTraceName string otdr measure trace name without path
     * @param $otdrTraceType string SMTOtdrTraceType::OTU_REFERENCE_TRACE, SMTOtdrTraceType::SMARTOTU_ACQUISITION_TRACE, SMTOtdrTraceType::LAST_TEST_ACQUISITION_TRACE
     * @param $forceCopy     Whether the copy overrides an existing trace (FALSE to update an existing test, TRUE by default for a new test)   
     * 
     * @return SMTOtdrReferenceFiles The reference traces paths copied in OTU reference detection and localisation directories
     */
    public static function copyMeasurementAsReferenceInOTU( $portNumber, $linkId, $testId, $otdrShortTraceName, $otdrTraceType, $forceCopy = TRUE )
    {
    	$otdrShortTraceName= self::replaceExtension($otdrShortTraceName, self::OTDR_FILE_SOR_EXTENSION);
    	
        //test if the new reference must be copied from SmartOTU last acquisition directory
        if ( $otdrTraceType == SMTOtdrTraceType::LAST_TEST_ACQUISITION_TRACE )
        {
        	//search on SmartOtu last acquisition directory
        	$acquisitionTrace = self::getLastMonitoringTraceNameFromSmartOtu($testId, $portNumber);
        }
        else if ( $otdrShortTraceName != NULL)
        {
            //search acquisition trace on session directory
            $acquisitionTrace = SMTSessionDirectoryUtil::getOTDRTraceFileNameFromSession( $otdrShortTraceName );
        }
        else
        {
            //search acquisition trace on session directory
            $acquisitionTrace = SMTSessionDirectoryUtil::getLastOTDRTraceFileNameFromSession( $portNumber );  
        }
        SMTLogger::getInstance()->trace("acquisition trace in session: ".$acquisitionTrace, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    
        $traceNameOnOTU = sprintf( self::REF_TRACE_NAME, $portNumber);
        $newReferenceDetTraceNameOnOTU = self::OTU_MEASURE_REF_DET_DIR.$traceNameOnOTU ;
    
        //rename reference trace name if it already exists: never replace a reference currently used
//         $count = 1;        
//         while ( file_exists( $newReferenceDetTraceNameOnOTU ) )
//         {
//             $traceWithoutDuplicateSuffix = explode( SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION , $newReferenceDetTraceNameOnOTU);
//             $traceWithoutDuplicateSuffix = explode( self::OTDR_FILE_DUPLICATE_SUFFIX , $traceWithoutDuplicateSuffix[0]);
//             $newReferenceDetTraceNameOnOTU = $traceWithoutDuplicateSuffix[0].self::OTDR_FILE_DUPLICATE_SUFFIX.$count++.SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION;
//         }
        
        if ( !$forceCopy && file_exists( $newReferenceDetTraceNameOnOTU ) )
        {
        	$traceWithoutDuplicateSuffix = explode( self::OTDR_FILE_SOR_EXTENSION , $newReferenceDetTraceNameOnOTU);
        	//si le fichier memorise existe deja, on l'ecrase
        	$newReferenceDetTraceNameOnOTU = $traceWithoutDuplicateSuffix[0].self::OTDR_FILE_MEMORIZED_SUFFIX.self::OTDR_FILE_SOR_EXTENSION;
        }
    
        SMTLogger::getInstance()->trace( sprintf("Start copying detection acquisition trace %s to %s: ", $acquisitionTrace, $newReferenceDetTraceNameOnOTU ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        if ( !copy( $acquisitionTrace, $newReferenceDetTraceNameOnOTU ) )
        {
            throw new SMTIOException( SMTIOException::COPY_FAILURE, $newReferenceDetTraceNameOnOTU );
        }
        
        //reference trace name is the same for localisation and detection (same files content).
        //$referenceTraceName = basename( $newReferenceDetTraceNameOnOTU );
            
        //$newReferenceLocTraceNameOnOTU = self::OTU_MEASURE_REF_LOC_DIR.$referenceTraceName ;
        $newReferenceLocTraceNameOnOTU = self::OTU_MEASURE_REF_LOC_DIR.$traceNameOnOTU ;
    
        if ( !$forceCopy && file_exists( $newReferenceLocTraceNameOnOTU ) )
        {
        	$traceWithoutDuplicateSuffix = explode( self::OTDR_FILE_SOR_EXTENSION , $newReferenceLocTraceNameOnOTU);
        	//si le fichier memorise existe deja, on l'ecrase
        	$newReferenceLocTraceNameOnOTU = $traceWithoutDuplicateSuffix[0].self::OTDR_FILE_MEMORIZED_SUFFIX.self::OTDR_FILE_SOR_EXTENSION;
        }
        
        SMTLogger::getInstance()->trace( sprintf("Start copying localisation acquisition trace %s to %s: ", $acquisitionTrace, $newReferenceLocTraceNameOnOTU ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        if ( !copy( $acquisitionTrace, $newReferenceLocTraceNameOnOTU) )
        {
            throw new SMTIOException( SMTIOException::COPY_FAILURE, $newReferenceLocTraceNameOnOTU );
        }    
    
        $otdrReferenceFiles = new SMTOtdrReferenceFiles();
        //$otdrReferenceFiles->setOtdrFileName( $referenceTraceName );
        $otdrReferenceFiles->setOtdrDetFilePath($newReferenceDetTraceNameOnOTU);
        $otdrReferenceFiles->setOtdrLocFilePath($newReferenceLocTraceNameOnOTU);
        $otdrReferenceFiles->setOtdrFileName( $traceNameOnOTU );
        //$otdrReferenceFiles->setOtdrDetFilePath(self::OTU_MEASURE_REF_DET_DIR.$traceNameOnOTU);
        //$otdrReferenceFiles->setOtdrLocFilePath(self::OTU_MEASURE_REF_LOC_DIR.$traceNameOnOTU);        
        
        return $otdrReferenceFiles;
    }
    
    /**
     * WARNING: that function should only be called in the case of test addition failure!
     *
     * @param SMTOtdrReferenceFiles $referenceTraceFiles
     */
    public static function removeMeasurementReferenceTracesFromOTU( SMTOtdrReferenceFiles $referenceTraceFiles )
    {
        SMTLogger::getInstance()->trace( sprintf( "Start remove OTDR acquisition trace %s or %s." , $referenceTraceFiles->getOtdrDetFilePath(), $referenceTraceFiles->getOtdrLocFilePath() ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        if ( file_exists( $referenceTraceFiles->getOtdrDetFilePath() ) )
        {
            if ( !unlink( $referenceTraceFiles->getOtdrDetFilePath()  ) )
            {
                SMTLogger::getInstance()->trace("Couldn't remove OTDR detection trace: ".$referenceTraceFiles->getOtdrDetFilePath(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            }
        }
        if ( (strlen($referenceTraceFiles->getOtdrLocFilePath()) > 0) && file_exists( $referenceTraceFiles->getOtdrLocFilePath() ) )
        {
            if ( !unlink( $referenceTraceFiles->getOtdrLocFilePath()  ) )
            {
                SMTLogger::getInstance()->trace("Couldn't remove OTDR localisation trace: ".$referenceTraceFiles->getOtdrLocFilePath(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            }
        }
    }     
    
    /**
     * Copy the alarm trace from OTU directories to SmartOtu directories.
     *
     * @param string $locTraceNameOnOTU alarm OTDR localisation trace name.
     * @param string $detTraceNameOnOTU alarm OTDR detection trace name.
     * @param int $testId The test identifier.
     * @pram int $portNumber The test optical port number
     * @param string $alarmDiscriminator The alarm discriminator. 
     *
     * @return string the OTDR alarm file name stored in SmartOTU directory.
     */
    public static function copyOtuAlarmTraceInSmartOtu( $locTraceNameOnOTU, $detTraceNameOnOTU, $shortAcqFileNameOnOTU, $testId, $portNumber, $alarmDiscriminator )
    {    
        $localisationTraceCopied = FALSE;
        
        //alarm localisation trace
        $fullTraceNameOnOTU = (($locTraceNameOnOTU != NULL) && (strlen($locTraceNameOnOTU) > 0) ) ? self::OTU_RESULT_LOC_DIR.$locTraceNameOnOTU : NULL;
    	SMTLogger::getInstance()->trace( sprintf( "Start retrieving OTDR alarm trace: %s and %s ", $locTraceNameOnOTU, $detTraceNameOnOTU ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	
    	if ( $fullTraceNameOnOTU != NULL )
    	{
	    	$fullTraceNameSmartAcqOnOTU = substr($fullTraceNameOnOTU, 0,strrpos($fullTraceNameOnOTU,".")).self::OTDR_FILE_CSOR_EXTENSION;
	    	if ( file_exists( $fullTraceNameSmartAcqOnOTU) )
	    		{
	    		$newAlarmTraceName = self::buildAlarmSmartAcqTraceName( $testId, $portNumber, $alarmDiscriminator );
	    		SMTLogger::getInstance()->trace( sprintf( "Start copying OTDR SmartAcq alarm trace %s to %s ", $fullTraceNameSmartAcqOnOTU, $newAlarmTraceName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
	
	    		if ( !copy( $fullTraceNameSmartAcqOnOTU, $newAlarmTraceName ) )
	    			{
	    			SMTLogger::getInstance()->trace( sprintf( "Failed to retrieve localisation OTDR alarm trace: %s ", $fullTraceNameSmartAcqOnOTU), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
	    			throw new SMTIOException( SMTIOException::COPY_FAILURE, $newAlarmTraceName );
	    			}
	    		else
	    			{
	    			$localisationTraceCopied = TRUE;
	    			}
	    		}
	    	else 
	    		{
	    		$fullTraceNameSmartAcqOnOTU = substr($fullTraceNameOnOTU, 0,strrpos($fullTraceNameOnOTU,".")).self::OTDR_FILE_MSOR_EXTENSION;
	    		if ( file_exists( $fullTraceNameSmartAcqOnOTU) )
	    			{
	    			$newAlarmTraceName = self::buildAlarmSmartAcqTraceName( $testId, $portNumber, $alarmDiscriminator );
	    			$newAlarmTraceName= substr($newAlarmTraceName, 0,strrpos($newAlarmTraceName,".")).self::OTDR_FILE_MSOR_EXTENSION;
	    			SMTLogger::getInstance()->trace( sprintf( "Start copying OTDR SmartAcq alarm trace %s to %s ", $fullTraceNameSmartAcqOnOTU, $newAlarmTraceName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
	    			
	    			if ( !copy( $fullTraceNameSmartAcqOnOTU, $newAlarmTraceName ) )
	    				{
	    				SMTLogger::getInstance()->trace( sprintf( "Failed to retrieve localisation OTDR alarm trace: %s ", $fullTraceNameSmartAcqOnOTU), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
	    				throw new SMTIOException( SMTIOException::COPY_FAILURE, $newAlarmTraceName );
	    				}
	    			else
	    				{
	    				$localisationTraceCopied = TRUE;
	    				}
	    			}
	    		}
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf( "locTraceNameOnOTU : NULL; detection :%s", $detTraceNameOnOTU), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    	}
    	
    	$newAlarmTraceName = self::buildAlarmTraceName( $testId, $portNumber, $alarmDiscriminator );
    	if ( $fullTraceNameOnOTU != NULL )
    	{
	    	$fullSorTraceNameOnOTU = substr($fullTraceNameOnOTU, 0,strrpos($fullTraceNameOnOTU,".")).self::OTDR_FILE_SOR_EXTENSION;
	    	if ( file_exists( $fullSorTraceNameOnOTU) )
	    	{
	    		SMTLogger::getInstance()->trace( sprintf( "Start copying OTDR alarm trace %s to %s ", $fullSorTraceNameOnOTU, $newAlarmTraceName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
	    	    	
	    	    if ( !copy( $fullSorTraceNameOnOTU, $newAlarmTraceName ) )
	        	{
	        		SMTLogger::getInstance()->trace( sprintf( "Failed to retrieve localisation OTDR alarm trace: %s ", $fullSorTraceNameOnOTU), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
	        		throw new SMTIOException( SMTIOException::COPY_FAILURE, $newAlarmTraceName );
	        	}
	        	else
	        	{
	        	    $localisationTraceCopied = TRUE;
	        	}
	    	}
    	}
    	
    	//alarm short acq localisation trace    	
    	$fullTraceNameOnOTU = (($shortAcqFileNameOnOTU != NULL) && (strlen($shortAcqFileNameOnOTU) > 0))? self::OTU_RESULT_LOC_DIR.$shortAcqFileNameOnOTU : NULL;
    	if ( $shortAcqFileNameOnOTU != NULL && $fullTraceNameOnOTU != NULL && file_exists( $fullTraceNameOnOTU ) )
    	{
    		$newAlarmTraceName = self::buildAlarmShortAcqTraceName( $testId, $portNumber, $alarmDiscriminator );
    	    SMTLogger::getInstance()->trace( sprintf( "Start copying OTDR alarm trace %s to %s ", $fullTraceNameOnOTU, $newAlarmTraceName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	    //don't throw exception for short acq
    	    if ( !copy( $fullTraceNameOnOTU, $newAlarmTraceName ) )
    	    {
    	    	SMTLogger::getInstance()->trace( sprintf( "Failed to retrieve localisation OTDR alarm trace: %s ", $fullTraceNameOnOTU ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    	    	//throw new SMTIOException( SMTIOException::COPY_FAILURE, $newAlarmTraceName );
    	    }
    	}
    	
    	//alarm detection trace
    	$fullTraceNameOnOTU = ( ($detTraceNameOnOTU != NULL) && (strlen($detTraceNameOnOTU) > 0) )? self::OTU_RESULT_DET_DIR.$detTraceNameOnOTU : NULL;
    	if ( $detTraceNameOnOTU != NULL && $fullTraceNameOnOTU != NULL && file_exists( $fullTraceNameOnOTU ) )
        {
        	$newAlarmTraceName = self::buildAlarmTraceName( $testId, $portNumber, $alarmDiscriminator, false );
            
            SMTLogger::getInstance()->trace( sprintf( "Start copying OTDR alarm trace %s to %s ", $fullTraceNameOnOTU, $newAlarmTraceName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
            
            if ( !copy( $fullTraceNameOnOTU, $newAlarmTraceName ) )
            {
            	SMTLogger::getInstance()->trace( sprintf( "Failed to retrieve detection OTDR alarm trace: %s ", $fullTraceNameOnOTU ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            	//if localisation trace was not copied in addition to the detection, throw an exception
            	if ( !$localisationTraceCopied )
            	{
            	    throw new SMTIOException( SMTIOException::COPY_FAILURE, $newAlarmTraceName );
            	}
            }            
        }
    
    	return $newAlarmTraceName;
    }    
    
    /**
     * Returns alarm OTDR full trace file name
     *
     * @param int $testId
     * @param string $portNumber
     * @param string $alarmDiscriminator the alarm discriminator
     * @param boolean $isLocalisation Whether the alarm otdr trace is the localization trace of the alarm. True by default.
     * @return string
     */
    public static function buildAlarmTraceName( $testId, $portNumber, $alarmDiscriminator, $isLocalisation = TRUE, $withExtension = TRUE )
    {
        $alarmFileDiscriminator = self::filterNonAlphaNumericCharacters( $alarmDiscriminator );
        $traceType = ($isLocalisation)? self::OTDR_TRACE_FILE_LOC : self::OTDR_TRACE_FILE_DET;
        return self::createAlarmDirectoryIfNeeded().sprintf( $withExtension? self::SMARTOTU_OTDR_ALARM_TRACE_FILE : self::SMARTOTU_OTDR_ALARM_TRACE_FILE_WO_EXTENSION, $traceType, $testId, $alarmFileDiscriminator, $portNumber);
    }
    
    /**
     * Returns alarm OTDR full trace file name
     *
     * @param int $testId
     * @param string $portNumber
     * @param string $alarmDiscriminator the alarm discriminator
     * @param boolean $isLocalisation Whether the alarm otdr trace is the localization trace of the alarm. True by default.
     * @return string
     */
    public static function buildAlarmSmartAcqTraceName( $testId, $portNumber, $alarmDiscriminator, $isLocalisation = TRUE )
    {
    	$alarmFileDiscriminator = self::filterNonAlphaNumericCharacters( $alarmDiscriminator );
    	$traceType = ($isLocalisation)? self::OTDR_TRACE_FILE_LOC : self::OTDR_TRACE_FILE_DET;
    	return self::createAlarmDirectoryIfNeeded().sprintf( self::SMARTOTU_OTDR_ALARM_TRACE_FILE_WO_EXTENSION.self::OTDR_FILE_CSOR_EXTENSION, $traceType, $testId, $alarmFileDiscriminator, $portNumber);
    }
    
    /**
     * Returns short acq alarm OTDR full trace file name
     *
     * @param int $testId
     * @param string $portNumber
     * @param string $alarmDiscriminator the alarm discriminator
     * 
     * @return string
     */    
    public static function buildAlarmShortAcqTraceName( $testId, $portNumber, $alarmDiscriminator )
    {
    	$alarmFileDiscriminator = self::filterNonAlphaNumericCharacters( $alarmDiscriminator );
    	return self::createAlarmDirectoryIfNeeded().sprintf( self::SMARTOTU_OTDR_ALARM_TRACE_FILE, self::OTDR_TRACE_FILE_SHORTACQ, $testId, $alarmFileDiscriminator, $portNumber);
    }
        
    /**
     * Remove the alarm trace from SmartOtu directory.
     *
     * @param int $testId The test identifier.
     * @param string $alarmDiscriminator The alarm discriminator.
     *
     */
    public static function removeOTUAlarmTraceFromSmartOtu( $testId, $alarmDiscriminator )
    {
    	foreach ( glob( self::buildAlarmTraceName( $testId, "*", $alarmDiscriminator, TRUE, FALSE )) as $fileToRemove )
    	{
	        SMTLogger::getInstance()->trace("Start remove localisation OTDR alarm trace: ".$fileToRemove, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
            if ( !unlink( $fileToRemove ) )
            {
                SMTLogger::getInstance()->trace("Couldn't remove OTDR localisation trace: ".$fileToRemove, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            }
        }
        
        foreach ( glob( self::buildAlarmTraceName( $testId, "*", $alarmDiscriminator, FALSE )) as $fileToRemove )
        {
	        SMTLogger::getInstance()->trace("Start remove detection OTDR alarm trace: ".$fileToRemove, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        	if ( !unlink( $fileToRemove ) )
        	{
        		SMTLogger::getInstance()->trace("Couldn't remove detection OTDR alarm trace: ".$fileToRemove, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	}
        }    

        foreach ( glob( self::buildAlarmShortAcqTraceName( $testId, "*", $alarmDiscriminator )) as $fileToRemove )
        {
            SMTLogger::getInstance()->trace("Start remove short acq OTDR alarm trace: ".$fileToRemove, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        	if ( !unlink( $fileToRemove ) )
        	{
        		SMTLogger::getInstance()->trace("Couldn't remove short acq OTDR alarm trace: ".$fileToRemove, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	}
        }        
    }    
    
    /**
     * Remove all the alarm trace from SmartOtu directories.
     *
     */
    public static function removeAllOTUAlarmTraceFromSmartOtu()
    {
        SMTLogger::getInstance()->trace("WARNING: Start remove all OTDR alarm trace in SmartOTU!", SMTLogger::INFO );
        
        $files = glob( self::createAlarmDirectoryIfNeeded().'/*'); //get all alarms files names
        foreach($files as $fileToRemove)
        {
        	if (is_file($fileToRemove))
        	{
        		SMTLogger::getInstance()->trace("Remove OTDR alarm trace: ".$fileToRemove, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        		if ( !unlink( $fileToRemove ) )
        		{
        			SMTLogger::getInstance()->trace("Couldn't remove OTDR alarm trace: ".$fileToRemove, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        		}
        	}
        }
//         foreach ( glob( self::createAlarmDirectoryIfNeeded()."/*".self::OTDR_FILE_SOR_EXTENSION ) as $fileToRemove )
//         {
//         	SMTLogger::getInstance()->trace("Start remove OTDR alarm trace: ".$fileToRemove, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
//         	if ( file_exists( $fileToRemove ) )
//         	{
//         		if ( !unlink( $fileToRemove ) )
//         		{
//         			SMTLogger::getInstance()->trace("Couldn't remove OTDR alarm trace: ".$fileToRemove, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
//         		}
//         	}
//         }
//         foreach ( glob( self::createAlarmDirectoryIfNeeded()."/*".self::OTDR_FILE_MSOR_EXTENSION ) as $fileToRemove )
//         {
//         	SMTLogger::getInstance()->trace("Start remove OTDR alarm trace: ".$fileToRemove, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        	
//         	if ( file_exists( $fileToRemove ) )
//         	{
//         		if ( !unlink( $fileToRemove ) )
//         		{
//         			SMTLogger::getInstance()->trace("Couldn't remove OTDR alarm trace: ".$fileToRemove, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
//         		}
//         	}
//         }
//         foreach ( glob( self::createAlarmDirectoryIfNeeded()."/*".self::OTDR_FILE_CSOR_EXTENSION ) as $fileToRemove )
//         {
//         	SMTLogger::getInstance()->trace("Start remove OTDR alarm trace: ".$fileToRemove, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        	
//         	if ( file_exists( $fileToRemove ) )
//         	{
//         		if ( !unlink( $fileToRemove ) )
//         		{
//         			SMTLogger::getInstance()->trace("Couldn't remove OTDR alarm trace: ".$fileToRemove, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
//         		}
//         	}
//         }
    }    
        
    /**
     * Create alarm directory if it doesn't exist.
     *
     * @return the SmartOtu alarm directory.
     */
    private static function createAlarmDirectoryIfNeeded()
    {
        $alarmDirectory = rtrim( self::CUSTOM_SMARTOTU_ALARM_DIR, '/');
		if (!is_dir( $alarmDirectory ) )
		{
			mkdir( $alarmDirectory, 0777);
		}
    	return $alarmDirectory."/";
    }    
        
    /**
     * Ensure trace file has its extension (*.sor)
     * 
     * @param string $traceName
     * 
     * @return the trace file name with an extension (*.sor)
     */
    public static function checkSorFileExtension( $traceName )
    {
        if ( !SMTUtil::endsWith( $traceName, self::OTDR_FILE_SOR_EXTENSION ) )
        {
        	$traceName = $traceName.self::OTDR_FILE_SOR_EXTENSION;
        }
        return $traceName;
    }
    
    /**
     * Filter not alpha-numeric characters of the given string
     * 
     * @param string $data The string to filter
     * @return string The string without characters that are not alpha-numeric
     */
    public static function filterNonAlphaNumericCharacters( $data )
    {
        return preg_replace("/[^0-9a-zA-Z]/i", '', $data);
    }
    
    private static function getLockLastAcquisitionFile()
    {
    	if ( self::$lastAcquisitionLockFile == NULL )
    	{
    		$handle = fopen( __DIR__.self::LOCK_LAST_ACQUISITION_FILE, "w+");
    		 
    		self::$lastAcquisitionLockFile = ( $handle != FALSE)? $handle : NULL;
    	}
    	return self::$lastAcquisitionLockFile;
    }
    
    /**
     * Acquire LastAcquisition exclusive lock
     *
     * @return boolean
     */
    private static function acquireLastAcquisitionLock()
    {
    	$retries = 0;
    	$max_retries = 6;
    
    	//SMTLogger::getInstance()->trace( "acquireLastAcquisitionLock" );
    
    	// keep trying to get a lock as long as possible ( max about 1s n(n+1)(2n+1)/6)
    	do
    	{
    		if ($retries > 0)
    		{
    			usleep( 10000 * ( $retries*$retries ) );
    		}
    		$retries += 1;
    	} while ( ( !flock(self::getLockLastAcquisitionFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
    
    	return ($retries < $max_retries);
    }
    
    /**
     * Releases LastAcquisition exclusive lock
     *
     * @return boolean
     */
    private static function releaseLastAcquisitionLock()
    {
    	//SMTLogger::getInstance()->trace( "releaseLastAcquisitionLock" );
    	return isset(self::$lastAcquisitionLockFile) && self::$lastAcquisitionLockFile != NULL? flock( self::$lastAcquisitionLockFile, LOCK_UN ) : TRUE;
    }    
}
?>