<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otau;

use app\admin\SMTOtuType;

use app\parser\SMTOtuSocket;

use app\util\SMTLogger;

use app\serviceshelper\SMTServiceHelper;

use app\http\SMTISerializable;
use app\parser\SMTOtuApi;
use app\services\setup\SMTOsxDto;
use app\services\setup\SMTOsxListDto;
use app\services\setup\SMTSwitchDto;
use app\sharedmemory\SMTMemoryManager;
use app\settings\SMTOtuStatusCacheDto;
use app\admin\SMTOtuMode;

/**
 * Retrieve switch config and detection.
 * The switch manager can be serialized in session.
 * 
 * @author Sylvain Desplat
 */
class SMTSwitchManager extends SMTServiceHelper implements SMTISerializable
{
	const SWITCH_MANAGER = "switch_manager";

	const OSX = "osx";	
	
	const OSX_POWER_ON = "ON";
	const OSX_POWER_OFF = "OFF";
	const INVALID_PORT_STRING = '-> 0';
	const INVALID_DESC_STRING = '-1';
	
    /**
     * For smartOtu, no ROTAU can be used => macros switch id = 0
     * 
     * @var number
     */
    const MACRO_SWITCH_ID = 0;
    
    /**
     * For smartOtu, switch device is limited to internal switch => device = OTU
     * 
     * @var string
     */
    const SWITCH_DEVICE = "OTU";
    
    /**
     * Dtss module position
     * 
     * @var string
     */
    const MOD3 = "MOD3";
    
    
    const OSX_MODE_ALL_DISTRIBUTOR = 0;
    const OSX_MODE_1_DISTRIBUTOR = 1;
    
    /**
     * 
     * @var String
     */
    private $switchDifference = NULL;
    
    /**
     *
     * @var boolean
     */
    private $DtssSwitch = FALSE;
    
    /**
     *
     * @var app\serviceshelper\otau\SMTMacroSwitchConf
     */
    private $switchConfig = NULL;
    
    /**
     * 
     * @var app\serviceshelper\otau\SMTMacroSwitchDet
     */
    private $switchDetect = NULL;
    
    private $switchDetectedHomogeneous = TRUE;
    
    /**
     * Retrieve switch difference, then the switch config and then the switch detected.
     * If the switch difference has already been retrieved, don't do it again. Same for detection.
     */
    function retrieve($forceDetection = TRUE)
    {
    	
//     	//in SmartOTU mode, if detection is forced, OSX alimentation must be turned on
//     	if ( $forceDetection && SMTOtuMode::isSmartOtu() )
//     	{
//     		if ( !$this->getOsxPowerOn() )
//     		{
//     			SMTLogger::getInstance()->trace("retrieve(forceDetection = TRUE) >> Force OSX power ON", SMTLogger::PINFO );
    			
//     			$this->setOsxPower(self::OSX_POWER_ON);
//     			//sleep during power on
//     			sleep(10);
//     		}
//     	}
    	
        // check difference before retrieving config (to have serial numbers updated... )
		$isSwitchDifference = $this->isDifference($forceDetection);

        //retrieve macro switch config:
        $this->switchConfig = self::retrieveConfiguredSwitch($this->getContext()->getOtuSocket());
        
        //retrieve detection if a difference exists and if detection has not already been retrieved and if a detection exists
        if ( $isSwitchDifference && ($this->switchDetect == NULL) && $this->hasDetection($forceDetection) )
        {
        	$this->switchDetect = self::retrieveDetectedSwitch($this->getContext()->getOtuSocket(), FALSE);
        }
        
        if ($forceDetection)
        {
        	SMTOtuStatusCacheDto::setSwitchDetectedOnce(TRUE);
        }
        $this->switchDetectedHomogeneous = self::retrieveDetectedSwitchHomogeneous($this->getContext()->getOtuSocket() );
        
        $this->checkDtssSwitch();
    }
    
    /**
     * Set OSX address from 1 to 30
     * 
     * @param string $osxAddress
     */
    function setOsxAddress($osxAddress)
    {
    	$this->send( sprintf(SMTOtuApi::CMD_set_osx_address, $osxAddress) );
    	
    	// force next detection
    	SMTOtuStatusCacheDto::setSwitchDetectedOnce(FALSE);
    }
    
    
    /**
     * Enable/disable sequencer during switch setup
     *
     * @param boolean $suspendSequencer TRUE to disable the sequencer
     */
    function disableSequencer($suspendSequencer)
    {
    	try
    	{
    		$this->send( SMTOtuApi::setMonitoringSequensorSuspend($suspendSequencer) );
    	}
    	catch( \Exception $e)
    	{
    		SMTLogger::getInstance()->traceException($e);
    	}
    }
    
    /**
     * Set OSX power ON/OFF
     * 
     * @param string $osxPower OSX_POWER_ON / OSX_POWER_OFF
     */
    function setOsxPower($osxPower)
    {
    	$osxPowerStatus = SMTOtuApi::RES_DISABLED;
    	if ( strcasecmp($osxPower, self::OSX_POWER_ON) == 0 )
    	{	
    		$osxPowerStatus = SMTOtuApi::RES_ENABLED;
    		
    		// force next detection
    		SMTOtuStatusCacheDto::setSwitchDetectedOnce(FALSE);
    	}
    	
     	$this->send( sprintf(SMTOtuApi::CMD_power_on_off_osx, $osxPowerStatus) );
    }
    
    /**
     * Get OSX power ON/OFF
     *
     * @return boolean
     */
    function getOsxPowerOn()
    {
    	//by default power is ON (power management not supported by OTU8kv2 old B43)
    	$powerOn = TRUE;
    	$queryPower = SMTOtuType::isCOtu() || SMTOtuType::isOTH();
    	
    	if ( SMTOtuType::isOtu8KV2() )
    	{
    		$queryPower = strcasecmp( 'TRUE', $this->sendReceive( SMTOtuApi::CMD_get_switch_molex_available) ) == 0;
    	}
    	
    	if ( $queryPower)
    	{
    		try 
    		{
    			$res = $this->sendReceive( SMTOtuApi::CMD_get_power_on_off_osx );
    			$powerOn = strcasecmp( SMTOtuApi::RES_ENABLED, $res ) == 0;
    		}
    		catch ( \Exception $e )
    		{
    			$powerOn = TRUE;
    		}
    	}
    	
    	return $powerOn;
    }
    
    /**
     * Set OSX big topology (osx on last internal switch port)
     *
     * @param string $bigTopology
     * @param SMTOtuSocket $socket
     */
    static function setBigTopology($bigTopology, SMTOtuSocket $socket)
    {
    	$socket->send( sprintf(SMTOtuApi::CMD_set_bigtopo, 0, $bigTopology) );
    	
    	// force next detection
    	SMTOtuStatusCacheDto::setSwitchDetectedOnce(FALSE);
    }
    
    /**
     * Set OSX big topology (osx on last internal switch port)
     *
     * @return whether OSX topology is a big topology
     */
    function isBigTopology()
    {
    	return strcasecmp( 'TRUE', $this->sendReceive( sprintf(SMTOtuApi::CMD_get_bigtopo, 0) ) ) == 0;
    }
    
    /**
     * Set internal switch PortsDistributorNumber (osx on last internal switch ports)
     *
     * @param string $bigTopology
     * @param SMTOtuSocket $socket
     */
    static function setPortsDistributorNumber($PortsDistributorNumber, SMTOtuSocket $socket)
    {
        $socket->send( sprintf(SMTOtuApi::CMD_set_Nb_Port_Distributeur, $PortsDistributorNumber) );
        
        // force next detection
        SMTOtuStatusCacheDto::setSwitchDetectedOnce(FALSE);
    }
    
    /**
     * Get internal switch PortsDistributorNumber (osx on last internal switch ports)
     *
     * @return int PortsDistributorNumber 
     */
    function getPortsDistributorNumber()
    {
        return intval($this->sendReceive( SMTOtuApi::CMD_get_Nb_Port_Distributeur ) );
    }
    
    /**
     * Has fixed Ports Distributor (new internal switch MOLEX on topologies C and D)
     *
     * @return int PortsDistributorNumber
     */
    function hasFixedPortsDistributor()
    {
        return strcasecmp( 'TRUE', $this->sendReceive( SMTOtuApi::CMD_Has_Fixed_Port_Distributeur) ) == 0;
    }
    
    /**
     * Full scan of osxs
     *
     * @return list of osxs full scan description
     */
    function fullScanLegacy()
    {
    	$osxList = new SMTOsxListDto();
    	
	    //scan osx with timeout of 5mn
	    $scan = $this->getContext()->getOtuSocket()->sendReceive( SMTOtuApi::CMD_scan_osu, SMTOtuSocket::DEFAULT_SLEEP, 300, TRUE );
	    $osx_address_array = explode(",",$scan);
	    
    	for ($i=0; $i<count($osx_address_array); $i++)
    	{    		
    		if( intval($osx_address_array[$i]) > 0 )
    		{
    			$osxDesc = "";
    			try
    			{
    				// si an OSX is found during scan but desc fails, don't throw exception, to avoid to loose all the scan result
    				$osxDesc = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_osuj_desc, $osx_address_array[$i]) );
    				$ports = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_osu_ports, $osx_address_array[$i]) );
    				if ( substr( $ports, -strlen( self::INVALID_PORT_STRING ) ) == self::INVALID_PORT_STRING)
    				{
    					$ports = "";
    				}
    			}
	    		//ports are retrieved from detection and can fail
	    		catch ( \Exception $e )
	    		{
	    			$ports = "";
	    			SMTLogger::getInstance()->traceException($e);
	    		}
    			$osx = new SMTOsxDto();
    			$osx->setIndex(intval($osx_address_array[$i]));
    			$osx->setDescription($osxDesc);
    			$osx->setPorts($ports);
    			$osxList->addOsx($osx);
    		}
    	}
    	
    	//retrieve big topology
    	$osxList->setIsBigTopology($this->isBigTopology());
    	
    	SMTMemoryManager::deleteAll( SMTOsxListDto::getClass() );
    	
    	// save scan in memory
    	$osxList->save();
    	
    	// force next detection
    	SMTOtuStatusCacheDto::setSwitchDetectedOnce(FALSE);
    	
    	return $osxList;
    }
    
    /**
     * Full scan of osxs/OTAM limited to *idn?
     *
     * @return list of osxs full scan description
     */
    function fullScan($scanOTAM)
    {
    	$osxList = new SMTOsxListDto();
    	$switchMissing = FALSE;
    	
    	for ($i=1; $i<= SMTMacroSwitch::OSU_DISTRIBUTOR_INDEX; $i++)
    	{
    		$osxDesc = '';
    		try
    		{
    			if ( $scanOTAM )
    			{
    				$osxDesc = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_switch_molex_idn, $i) );
    			}
    			else
    			{
    			    if ( $switchMissing )
    			    {
    				    $osxDesc = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_switch_jdsu_idn_scan, $i) );
    			    }
    			    else 
    			    {
    			        $osxDesc = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_switch_jdsu_idn, $i) );
    			    }
    			}
    			$osxDesc= preg_replace('/[[:^print:]]/', '', $osxDesc);
    			if ( $osxDesc == self::INVALID_DESC_STRING)
    			{
    				$osxDesc = '';
    			}
    			if ( strlen($osxDesc) > 0)
    			{
    			    $switchMissing = FALSE;
	    			try 
	    			{
		    			$ports = $this->sendReceive( sprintf(SMTOtuApi::CMD_get_osu_ports, $i) );
		    			if ( substr( $ports, -strlen( self::INVALID_PORT_STRING ) ) == self::INVALID_PORT_STRING)
		    			{
		    				$ports = '';
		    			}
	    			}
	    			catch(\Exception $ex )
	    			{
	    				$ports = '';
	    			}
	    			$osx = new SMTOsxDto();
	    			$osx->setIndex( $i );
	    			$osx->setDescription($osxDesc);
	    			$osx->setPorts($ports);
	    			$osxList->addOsx($osx);
    			}
    		}
    		//ports are retrieved from detection and can fail
    		catch ( \Exception $e )
    		{
    			SMTLogger::getInstance()->traceException($e);
    			$switchMissing = TRUE;
    		}
    	}
    	
    	//retrieve big topology
    	$osxList->setIsBigTopology($this->isBigTopology());
    	
    	SMTMemoryManager::deleteAll( SMTOsxListDto::getClass() );
    	
    	// save scan in memory
    	$osxList->save();
    	
    	// force next detection
    	SMTOtuStatusCacheDto::setSwitchDetectedOnce(FALSE);
    	
    	return $osxList;
    }
    /**
     * Show an OSX
     * 
     * @param unknown $address
     */
    function showOsx($osxAddress)
    {
    	$this->send( sprintf(SMTOtuApi::CMD_show_osx, $osxAddress) );
    }
    
    /**
     * Check if a DTSS module is present.
     * 
     */
    function checkDtssSwitch()
    {
        if ( SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2() || SMTOtuType::isMTS8KV2() )
        {
            try
            {
                $result = $this->sendReceive( SMTOtuApi::getModulePresentInCal( self::MOD3 ) );
                if ( strcasecmp( SMTOtuApi::RES_OK, $result) == 0)
                {
                    $this->DtssSwitch = TRUE;
                }
                else if ( $this->isDifference(FALSE) && $this->hasDetection(FALSE) )
                {
                    $result = $this->sendReceive( SMTOtuApi::getModulePresentInDet( self::MOD3 ) );
                    if ( strcasecmp( SMTOtuApi::RES_OK, $result) == 0)
                    {
                    	$this->DtssSwitch = TRUE;
                    }
                }
            }
            catch ( \Exception $e )
            {
                SMTLogger::getInstance()->traceException($e);
            }
        }
        else
        {
            $this->DtssSwitch = FALSE;
        }
    }

    /**
     * Apply given detected switch in configuration
     * 
     * @param SMTOtuSocket $socket
     * @param String $globalSwitchEncodedString
     */
    static function applySwitch(SMTOtuSocket $socket, $globalSwitchEncodedString)
    {
    	$socket->sendReceive( sprintf( SMTOtuApi::CMD_add_otau, self::SWITCH_DEVICE, $globalSwitchEncodedString, "" ), SMTOtuSocket::DEFAULT_SLEEP, 300, TRUE );
    }
    
    /**
     * Remove given detected switch from configuration
     *
     * @param SMTOtuSocket $socket
     */
    static function removeSwitch(SMTOtuSocket $socket)
    {
        $socket->send( sprintf( SMTOtuApi::CMD_remove_otau, self::MACRO_SWITCH_ID ) );
    }    

    /**
     * Apply detected switch in configuration
     *
     * @param SMTSwitchDto $detectedSwitch
     */
    static function applySwitchDetected(SMTOtuSocket $socket)
    {
    	// on ne detecte les OSXs que s'ils n'ont pas ete deja detectes
    	$detectedSwitch = self::retrieveDetectedSwitch($socket, SMTOtuStatusCacheDto::getSwitchDetectedOnce());
                
        if ( $detectedSwitch != NULL )
        {
        	//in SmartOTU mode, if OSX are used, topology must be extended
        	if ( SMTOtuMode::isSmartOtuFamily() && $detectedSwitch->hasOsx() )
        	{
        		try
        		{
        			SMTLogger::getInstance()->trace("applySwitchDetected >> setBigTopology incomplete FALSE", SMTLogger::PINFO );
        			SMTSwitchManager::setBigTopology('FALSE', $socket);
        			
        			// on force la detection avec la topologie etendue
        			$detectedSwitch = self::retrieveDetectedSwitch($socket, TRUE);
        		}
        		catch ( \Exception $e )
        		{
        			SMTLogger::getInstance()->traceException($e);
        		}
        	}
        	
    	    self::applySwitch($socket, $detectedSwitch->getGlobalSwitchEncodedString());
        }
        else
        {
            self::removeSwitch($socket);
        }
    }
    
    /**
     * Check if a DTSS module is in detection or calibration.
     *
     */
    static function isDtssSwitchInCalibOrInDet(SMTOtuSocket $socket)
    {
        $DtssSwitch = FALSE;
        if ( SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2() || SMTOtuType::isMTS8KV2() )
        {
        	try
        	{
        		$result = $socket->sendReceive( SMTOtuApi::getModulePresentInCal( self::MOD3 ) );
        		if ( strcasecmp( SMTOtuApi::RES_OK, $result) == 0)
        		{
        			$DtssSwitch = TRUE;
        		}
        		else
        		{
        			$result = $socket->sendReceive( SMTOtuApi::getModulePresentInDet( self::MOD3 ) );
        			if ( strcasecmp( SMTOtuApi::RES_OK, $result) == 0)
        			{
        				$DtssSwitch = TRUE;
        			}
        		}
        	}
        	catch ( \Exception $e )
        	{
        		SMTLogger::getInstance()->traceException($e);
        	}
        }
    	return $DtssSwitch;
    }
    
    /**
     * Is the switch from a DTSS module
     * 
     * @return boolean
     */
    function isDtssSwitch()
    {
    	return $this->DtssSwitch;
    }
    
    /**
     * Whether the OSX have the same number of outputs
     * @return boolean
     */
    function isSwitchDetectedHomogeneous()
    {
    	return $this->switchDetectedHomogeneous;
    }
    
    function hasMolexOSUDetected()
    {
    	$hasMolexSwitch = FALSE;
    	try
    	{
    		if ( $this->switchDetect!= NULL )
    		{
    			$hasMolexSwitch = $this->switchDetect->hasMolexOSU();
    		}
    		else
    		{
    			$macroSwitchDet = new SMTMacroSwitchDet($this->getContext()->getOtuSocket());
    			$hasMolexSwitch = $macroSwitchDet->hasMolexOSU();
    		}
    	}
    	catch ( \Exception $e )
    	{
    		//no log : can fail if no switch is set in the OTU
    		//SMTLogger::getInstance()->traceException($e);
    	}
    	
    	return $hasMolexSwitch;
    }
    
    function hasJdsuOSUDetected()
    {
    	$hasJdsuSwitch = FALSE;
    	try
    	{
    		if ($this->switchDetect!= NULL)
    		{
    			$hasJdsuSwitch= $this->switchDetect->hasJdsuOSU();
    		}
    		else
    		{
    			$macroSwitchDet = new SMTMacroSwitchDet($this->getContext()->getOtuSocket());
    			$hasJdsuSwitch= $macroSwitchDet->hasJdsuOSU();
    		}
    	}
    	catch ( \Exception $e )
    	{
    		//no log : can fail if no switch is set in the OTU
    		//SMTLogger::getInstance()->traceException($e);
    	}
    	return $hasJdsuSwitch;
    }
    
    function hasUOSDetected()
    {
        $hasUOSSwitch = FALSE;
        try
        {
            if ($this->switchDetect!= NULL)
            {
                $hasUOSSwitch= $this->switchDetect->hasUOS();
            }
            else
            {
                $macroSwitchDet = new SMTMacroSwitchDet($this->getContext()->getOtuSocket());
                $hasUOSSwitch= $macroSwitchDet->hasUOS();
            }
        }
        catch ( \Exception $e )
        {
            //no log : can fail if no switch is set in the OTU
            //SMTLogger::getInstance()->traceException($e);
        }
        return $hasUOSSwitch;
    }
    
    function hasMolexOSU()
    {
    	$hasMolexSwitch = FALSE;
    	try 
    	{
	    	if ( ($this->switchConfig != NULL) || ($this->switchDetect!= NULL) )
	    	{
	    		$hasMolexSwitch = ( ($this->switchConfig != NULL) && ($this->switchConfig->hasMolexOSU()) ) ||
	    		( ($this->switchDetect!= NULL) && ($this->switchDetect->hasMolexOSU()) );
	    	}
	    	else
	    	{
	    		$macroSwitchConf = new SMTMacroSwitchConf($this->getContext()->getOtuSocket());
	    		$hasMolexSwitch = $macroSwitchConf->hasMolexOSU();
	    		if ( !$hasMolexSwitch )
	    		{
	    			$macroSwitchDet = new SMTMacroSwitchDet($this->getContext()->getOtuSocket());
	    			$hasMolexSwitch = $macroSwitchDet->hasMolexOSU();
	    		}
	    	}
    	}
    	catch ( \Exception $e )
    	{
    		//no log : can fail if no switch is set in the OTU
    		//SMTLogger::getInstance()->traceException($e);
    	}
    	
        return $hasMolexSwitch;
    }
    
    function hasJdsuOSU()
    {
    	$hasJdsuSwitch = FALSE;
    	try
    	{
	    	if ( ($this->switchConfig != NULL) || ($this->switchDetect!= NULL) )
	    	{
	    		$hasJdsuSwitch= ( ($this->switchConfig != NULL) && ($this->switchConfig->hasJdsuOSU()) ) ||
	    		( ($this->switchDetect!= NULL) && ($this->switchDetect->hasJdsuOSU()) );
	    	}
	    	else
	    	{
	    		$macroSwitchConf = new SMTMacroSwitchConf($this->getContext()->getOtuSocket());
	    		$hasJdsuSwitch= $macroSwitchConf->hasJdsuOSU();
	    		if ( !$hasJdsuSwitch)
	    		{
	    			$macroSwitchDet = new SMTMacroSwitchDet($this->getContext()->getOtuSocket());
	    			$hasJdsuSwitch= $macroSwitchDet->hasJdsuOSU();
	    		}
	    	}
    	}
    	catch ( \Exception $e )
    	{
    		//no log : can fail if no switch is set in the OTU
    		//SMTLogger::getInstance()->traceException($e);
    	}
    	return $hasJdsuSwitch;
    }
    
    function hasUOS()
    {
        $hasUOSSwitch = FALSE;
        try
        {
            if ( ($this->switchConfig != NULL) || ($this->switchDetect!= NULL) )
            {
                $hasUOSSwitch= ( ($this->switchConfig != NULL) && ($this->switchConfig->hasUOS()) ) ||
                ( ($this->switchDetect!= NULL) && ($this->switchDetect->hasUOS()) );
            }
            else
            {
                $macroSwitchConf = new SMTMacroSwitchConf($this->getContext()->getOtuSocket());
                $hasUOSSwitch= $macroSwitchConf->hasUOS();
                if ( !$hasUOSSwitch)
                {
                    $macroSwitchDet = new SMTMacroSwitchDet($this->getContext()->getOtuSocket());
                    $hasUOSSwitch= $macroSwitchDet->hasUOS();
                }
            }
        }
        catch ( \Exception $e )
        {
            //no log : can fail if no switch is set in the OTU
            //SMTLogger::getInstance()->traceException($e);
        }
        return $hasUOSSwitch;
    }
    
    function isMolexSwitchPossible()
    {
    	return strcasecmp( 'TRUE', $this->sendReceive( SMTOtuApi::CMD_get_switch_molex_available) ) == 0;
    }
    
    function isUosSwitchPossible()
    {
        return strcasecmp( 'TRUE', $this->sendReceive( SMTOtuApi::CMD_get_switch_uos_available) ) == 0;
    }
    
    function isModuleCompatibleWithSwitch()
    {
    	return strcasecmp( 'TRUE', $this->sendReceive( SMTOtuApi::CMD_check_module_compatibility_with_switch ) ) == 0;
    }
    

    /**
     *
     * @return app\serviceshelper\otau\SMTMacroSwitchDet
     */
    function getDetectedSwitch()
    {
        return $this->switchDetect;
    }
    
    /**
     *
     * @return array app\services\setup\SMTSwitchDto
     */
    function getDetectedSwitchDto()
    {
    	return ($this->switchDetect != NULL)? $this->switchDetect->getSwitchesDto() : NULL;
    }
    
    /**
     *
     * @return app\serviceshelper\otau\SMTMacroSwitchConf
     */
    function getConfiguredSwitch()
    {
    	return $this->switchConfig;
    }
    
    /**
     *
     * @return array app\services\setup\SMTSwitchDto
     */
    function getConfiguredSwitchDto()
    {
    	return ($this->switchConfig != NULL)? $this->switchConfig->getSwitchesDto() : NULL;
    }
        
    /**
     * @param app\parser\SMTOtuSocket $socket
     * 
     * @return SMTMacroSwitchDet
     */
    static function retrieveDetectedSwitch(SMTOtuSocket $socket, $forceDetection = TRUE)
    {
        $dto = NULL;
        
        //search if macro switch exists
        if ($forceDetection)
            {
            	$switchDetectMacroResult = $socket->sendReceive( sprintf( SMTOtuApi::CMD_switch_detect_macro, self::MACRO_SWITCH_ID ), SMTOtuSocket::DEFAULT_SLEEP, 90);
            }
        else
            {
            	$switchDetectMacroResult = $socket->sendReceive( sprintf( SMTOtuApi::CMD_switch_detect_result, self::MACRO_SWITCH_ID ), SMTOtuSocket::DEFAULT_SLEEP, 90);
            }        
        $macroSwitchExist = strcasecmp( $switchDetectMacroResult, SMTOtuApi::RES_ON) == 0;
        
        if ( $macroSwitchExist )
        {
            $dto = SMTMacroSwitchDet::retrieve( $socket );
        }
        return $dto;
    }
    
    /**
     * Whether OSXs are homogenous
     * 
     * @param SMTOtuSocket $socket
     * 
     * @return TRUE is OSX are homogeneous
     */
    static function retrieveDetectedSwitchHomogeneous(SMTOtuSocket $socket)
    {
    	$result = $socket->sendReceive( sprintf( SMTOtuApi::CMD_switch_detect_osu_homogeneous, self::MACRO_SWITCH_ID ), SMTOtuSocket::DEFAULT_SLEEP, 90 );
    	return strcasecmp( $result, SMTOtuApi::RES_SAME) == 0;
    }
    
    /**
     *
     * @param app\parser\SMTOtuSocket $socket
     * 
     * @return SMTMacroSwitchConf
     */   
    static function retrieveConfiguredSwitch(SMTOtuSocket $socket)
    {
        $dto = NULL;
        //search if macro switch is configured
        $switchConfigMacroResult = $socket->sendReceive( sprintf( SMTOtuApi::CMD_switch_config_macro, self::MACRO_SWITCH_ID ) );
        $macroSwitchExist = strcasecmp( $switchConfigMacroResult, SMTOtuApi::RES_ON) == 0;
        
        if ( $macroSwitchExist )
        {
            $dto = SMTMacroSwitchConf::retrieve( $socket );
        }
        return $dto;
    }
    
    /**
     * Reset comparison and detection to force their retrieval.
     * 
     */
    function reset()
    {
        $this->switchDifference = NULL;
        $this->switchDetect = NULL;
    }
    
    /**
     * Compare the current macro switch in detection and calibration
     *
     * @return Whether the macro switch is identical, different in detection and calibration or whether conf is missing or det is missing
     */
    private function compareOTAUDetConfig( $forceDetection = TRUE )
    {
        if ( $forceDetection )
            {
            $this->switchDifference = $this->getContext()->getOtuSocket()->sendReceive( sprintf( SMTOtuApi::CMD_compare_switches, self::MACRO_SWITCH_ID ), SMTOtuSocket::DEFAULT_SLEEP, 120, TRUE );
//     	    $this->switchDifference = $this->sendReceive( sprintf( SMTOtuApi::CMD_compare_switches, self::MACRO_SWITCH_ID ) );
            }
        else
            {
            $this->switchDifference = $this->sendReceive( sprintf( SMTOtuApi::CMD_compare_switches_result, self::MACRO_SWITCH_ID ) );
            }    	
    }    
    
    /**
     * Whether an otau was detected.
     * 
     * @return boolean true if an otau was detected.
     */
    function hasDetection($ForceDetection = TRUE)
    {
        if ( $this->switchDifference == NULL )
        {
        	$this->compareOTAUDetConfig($ForceDetection);
        }        
        return ($this->switchDifference != NULL)? ( strcasecmp( $this->switchDifference, SMTOtuApi::RES_MISSDET) != 0 ) : FALSE;
    }
    
    /**
     * Whether a difference between detection and config was found.
     *
     * @return boolean true if a difference between detection and config was found.
     */
    function isDifference($forceDetection = TRUE)
    {
        //detect differences if they have not already been retrieved
        if ( $this->switchDifference == NULL )
        {
            $this->compareOTAUDetConfig($forceDetection);
        }
    	return ($this->switchDifference != NULL)? ( strcasecmp( $this->switchDifference, SMTOtuApi::RES_SAME) != 0 ) : FALSE;
    }    
    
    /**
     * Serialize object stored in session
     *
     * @return multitype:string
     */
    public function __sleep()
    {
    	return $this->getSerializableProperties();
    }
    
    /**
     * Returns the serializable properties of the object.
     *
     * @see \app\http\SMTISerializable::getSerializableProperties()
     */
    public function getSerializableProperties()
    {
    	return array("switchDifference","switchDetect");
    }
    
    /**
     * Deserialize object.
     */
    public function __wakeup()
    {
    }
    
    /**
     * Whether the OTU has a virtual switch
     *
     * @return Whether the OTU has a virtual switch
     */
    function hasVirtualSwitch()
    {
    	$portCount = -1;
    	try
    	{
    		$portCount = $this->sendReceive( SMTOtuApi::getVirtualSwitchPortCount() );
    		$portCount = intval($portCount);
    	}
    	catch (\Exception $e)
    	{
    		$portCount = -1;
    		//nothing to do, virtual switch not used
    	}
    	return ($portCount > 0);
    }
}
?>