<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\monitoring;

use app\util\SMTUtil;

use app\events\operations\SMTOperationEventDto;

use app\events\operations\SMTIOperationEventDto;

/**
 * Measurement event dto
 * 
 * @author Sylvain Desplat
 *
 */
class SMTMeasureOnDemandEventDto extends SMTOperationEventDto implements SMTIOperationEventDto
{          
    const MEAS_ADD_OK = "MEAS_ADD_OK";
    const MEAS_ADD = "MEAS_ADD";
    const PARAM_MISSING = "PARAM_MISSING";
    const MOD_MISSING = "MOD_MISSING";
    const LINK_UNKNOWN = "LINK_UNKNOWN";
    const SWITCH_BUFFER_SIZE = "SWITCH_BUFFER_SIZE";
    const SWITCH_BUFFER_ERROR = "SWITCH_BUFFER_ERROR";
    const REG_MEAS_FAILED = "REG_MEAS_FAILED";
    const WRONG_RANGE = "WRONG_RANGE";
    const WRONG_RES = "WRONG_RES";
    const WRONG_PULSE = "WRONG_PULSE";
    const WRONG_LASER = "WRONG_LASER";
    const MEASURE_BAD_TIMESLOT = "MEASURE_BAD_TIMESLOT";
    const MEASURE_BAD_TIME = "MEASURE_BAD_TIME";
    const WRONG_FONCTION = "WRONG_FONCTION";
    const INCOHERENT_FONCTION = "INCOHERENT_FONCTION";
    const ADD_LOCALISATION_ERROR = "ADD_LOCALISATION_ERROR";
    const REF_FILE_PARAMETER_ERROR = "REF_FILE_PARAMETER_ERROR";
    
    /**
     * Port number where the measure is performed
     * 
     * @var int
     */
    protected $portNumber;
    
    /**
     * Link identifier if the measure was launched on a link
     * @var number
     */    
    protected $linkId;
    
    /**
     * The OTDR short trace name on SmartOTu disk
     * @var string
     */
    protected $otdrTraceName;
    
    
    function getLinkId()
    {
    	return $this->linkId;
    }
    
    function setLinkId( $linkId )
    {
    	$this->linkId = SMTUtil::convertToInt( $linkId );
    }        

    public function setPortNumber( $portNumber )
    {
    	$this->portNumber = SMTUtil::convertToInt($portNumber);    
    }
    
    public function getPortNumber()
    {
    	return $this->portNumber;
    }    
    
    function getOtdrTraceName()
    {
    	return $this->otdrTraceName;
    }
    
    function setOtdrTraceName( $otdrTraceName )
    {
    	$this->otdrTraceName = $otdrTraceName;
    }    
    
    function setStatus( $status )
    {
    	if ( $this->isValidMeasureStatus( $status ) )
    	{
    		$this->status = $status;
    		$this->setLocalizedMessage( self::decodeMessage( $status ) );
    	}
    	else
    	{
    		throw new \InvalidArgumentException( MSG_ERROR_INVALID_OPERATION_STATUS );
    	}    	
    }        
    
    /**
     * Whether the status given is a measurement status
     * 
     * @param string $code mesure status
     * 
     * @return boolean
     */
    public function isValidMeasureStatus( $code )
    {                
        $valid = FALSE;
        switch( $code )
        {
        	case self::MEAS_ADD_OK:
        	case self::MEAS_ADD:
        	case self::PARAM_MISSING:
        	case self::MOD_MISSING:
        	case self::LINK_UNKNOWN:
        	case self::SWITCH_BUFFER_SIZE:
        	case self::SWITCH_BUFFER_ERROR:
        	case self::REG_MEAS_FAILED:
        	case self::WRONG_RANGE:        	
    	    case self::WRONG_RES:
	        case self::WRONG_PULSE:    	        
    	    case self::WRONG_LASER:
	        case self::MEASURE_BAD_TIMESLOT:
	        case self::MEASURE_BAD_TIME:
	        case self::WRONG_FONCTION:
            case self::INCOHERENT_FONCTION:
            case self::ADD_LOCALISATION_ERROR:
            case self::REF_FILE_PARAMETER_ERROR:        	            	         	        
        	    $valid = TRUE;
        		break;
        	default:
        		$valid = self::isOperationStatus($code);
        		break;
        }
        return $valid;        
    } 
    
    /**
     * Translates status code. If the code is not found, returns an empty string.
     * 
     * @param String $code The status code to translate
     * 
     * @return string the status message or an empty string if the status couldn't be found.
     */
    public static function decodeMessage( $code )
    {
        $message = MSG_ERROR_MEASURE_FAILED.$code;
        
        switch( $code )
        {
            case self::MEAS_ADD_OK:
            	$message = MSG_MEAS_ADD_OK;
            	break;
            case self::MEAS_ADD:
            	$message = MSG_ERROR_ONE_MEASURE_ON_DEMAND;
            	break;
            case self::PARAM_MISSING:
            	$message = MSG_ERROR_PARAM_MISSING;
            	break;
            case self::MOD_MISSING:
            	$message = MSG_ERROR_MOD_MISSING;
            	break;
            case self::LINK_UNKNOWN:
            	$message = MSG_ERROR_LINK_UNKNOWN;
            	break;
            case self::SWITCH_BUFFER_SIZE:
            	$message = MSG_ERROR_SWITCH_BUFFER_SIZE;
            	break;
            case self::SWITCH_BUFFER_ERROR:
            	$message = MSG_ERROR_SWITCH_BUFFER_ERROR;
            	break;
            case self::REG_MEAS_FAILED:
            	$message = MSG_ERROR_REG_MEAS_FAILED;
            	break;
            case self::WRONG_RANGE:
            	$message = MSG_ERROR_WRONG_RANGE;
            	break;
            case self::WRONG_RES:
            	$message = MSG_ERROR_WRONG_RES;
            	break;
            case self::WRONG_PULSE:
            	$message = MSG_ERROR_RONG_PULSE;
            	break;
            case self::WRONG_LASER:
            	$message = MSG_ERROR_WRONG_LASER;
            	break;
            case self::MEASURE_BAD_TIMESLOT:
            	$message = MSG_ERROR_MEASURE_BAD_TIMESLOT;
            	break;
        	case self::MEASURE_BAD_TIME:
        		$message = MSG_ERROR_MEASURE_BAD_TIME;
        		break;            	            	
            case self::WRONG_FONCTION:
            	$message = MSG_ERROR_WRONG_FONCTION;
            	break;
            case self::INCOHERENT_FONCTION:
            	$message = MSG_ERROR_INCOHERENT_FONCTION;
            	break;
            case self::ADD_LOCALISATION_ERROR:
            	$message = MSG_ERROR_ADD_LOCALISATION_ERROR;
            	break;
            case self::REF_FILE_PARAMETER_ERROR:
            	$message = MSG_ERROR_REF_FILE_PARAMETER_ERROR;
            	break;  			 
            default:
                $message = SMTOperationEventDto::decodeMessage( $code );
                break;
        }
        return $message;
    }    
    
    /**
     * Generate the measure on demand identifier.
     *
     * @return string the measure on demand identifier.
     */
    public static function generateMeasureOnDemandOperationId()
    {
    	return SMTOperationEventDto::MEASURE_ON_DEMAND.str_replace( " ", "_", microtime(false) );
    }
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }        
    
    /**
     * Returns the dto class name.
     *
     * @return the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }

    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\monitoring\SMTLinkTestDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    }    
}

?>