<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2014. All rights reserved.
// *********************************************************
namespace app\serviceshelper\media;

use app\serviceshelper\system\SMTOtu;

use app\services\setup\SMTSnmpDto;

use app\services\setup\SMTConfSnmpDto;
use app\serviceshelper\SMTServiceHelper;
use app\admin\SMTOtuType;
use app\parser\SMTOtuApi;
use app\util\SMTUtil;

define("MSG_SEND_SNMP_MESSAGE","TEST sent from OTU: %s");

/**
 *
 * @author Sylvain Desplat
 *
 */
class SMTSnmp extends SMTServiceHelper
{    
    /**
     * SNMP media.
     *
     * @var string
     */
    const SNMP_MEDIA = "SNMP";    
    /**
     * Snmp media
     *
     * @var string
     */
    const SNMP_MEDIA_ETH = "ETH";   
    
    const MANAGER_INDEX = 0;
    const COMMUNITY_INDEX = 1;
    const PORT_INDEX = 2;
    
    const MEDIA_INDEX = 0;
    const MIN_SNMP_INDEX = 1;
    const MAX_SNMP_INDEX = 5;
	
    const SNMP_SERVER_MAX_SIZE = 256;
    const SNMP_COMMUNITY_MAX_SIZE = 256;
    const SNMP_TEST_MESSAGE_MAX_SIZE = 30;
    const SNMP_PORT_MAX_VALUE = 65534;
    const MIBS_ARCHIVE = "jdsu-otu-mib";
    
    /**
     * Default SNMP heartbeat period in minutes.
     *
     * @var integer
     */
    const DEFAULT_SNMP_HEART_BEAT_PERIOD = 10;
    const SNMP_HEART_BEAT_MAX_PERIOD = 2880;
    const SNMP_HEART_BEAT_MIN_PERIOD = 1;
    
    /**
     * Mibs full file path on disk
     *
     */
    static function getMibsDir()
    {
        return SMTOtuType::isOTH()? '/user/harddisk/' : '/acterna/user/harddisk/';
    }
    
    /**
     * Update snmp config on OTU
     * 
     * @param SMTConfSnmpDto $snmpConfDto
     */
    function updateSnmp( SMTConfSnmpDto $snmpConfDto )
    {     	
    	$snmpIndexToRemove = self::MIN_SNMP_INDEX;
    	
    	try 
    	{
    		if ( !self::isSNMPDtoConfigValid( $snmpConfDto) )
    		{
    			$snmpConfDto->setActive( FALSE );    			
    		}
    		
    		$on = $snmpConfDto->isSnmpActive();    		
    		$this->send( SMTOtuApi::setSNMPMedia( $on ) );
    	} 
    	catch (SMTMediaException $e) 
    	{
    		$snmpConfDto->setActive( FALSE );
    		$on = $snmpConfDto->isSnmpActive();
    		$this->send( SMTOtuApi::setSNMPMedia( $on ) );
    		
    		throw $e;
    	}
    	
    	//update media period and max retry
    	if ( $on )
    	{
    		$this->send( SMTOtuApi::setMediaRetryPeriod( SMTOtuApi::MEDIA_RETRY_DELAY ) );
    		$this->send( SMTOtuApi::setSnmpMaxRetry( SMTOtuApi::MEDIA_RETRY_COUNT ) );
    		$this->setSNMPHeartbeatPeriod($snmpConfDto->getSnmpHearbeatPeriod());
    		$this->setSNMPHeartbeatOn($snmpConfDto->isSnmpHearbeatActive());
    	}
    	else
    	{
    		$this->setSNMPHeartbeatOn(FALSE);
    	}
    	    	
    	//update snmp config in OTU
    	//update emails in OTU
    	foreach ( $snmpConfDto->getSnmpArray() as $snmpDto )
    	{
    		if ( $snmpDto != NULL)
    		{    			
    			$command = SMTOtuApi::setSnmpCommand( $snmpDto->getIndex(), $snmpDto->getManager(), $snmpDto->getCommunity(), $snmpDto->getPort() );
   				$this->send( $command );
   				$snmpIndexToRemove++;
    		}
    	}
    	
    	for ( $index = $snmpIndexToRemove; $index <= self::MAX_SNMP_INDEX; $index++ )
    	{
    		$command = SMTOtuApi::setSnmpCommand( $index, "", "", 0 );
    		$this->send( $command );
    	}
    }
    
    /**
     * Test whether the given SNMP config is valid
     * 
     * @param SMTConfSnmpDto $snmpConfDto     
     * @return boolean
     * 
     * @throws SMTMediaException
     */
    static function isSNMPDtoConfigValid( SMTConfSnmpDto $snmpConfDto )
    {
        $valid = TRUE;
        
        if ( $snmpConfDto->getSnmpHearbeatPeriod() != NULL && $snmpConfDto->getSnmpHearbeatPeriod() != "" )
        {
        	if ( ( $snmpConfDto->getSnmpHearbeatPeriod() > self::SNMP_HEART_BEAT_MAX_PERIOD ) ||
        			( $snmpConfDto->getSnmpHearbeatPeriod() < self::SNMP_HEART_BEAT_MIN_PERIOD ) )
        	{
        		throw new SMTMediaException( SMTMediaException::ERROR_SNMP_INVALID_HEARTBEAT_PERIOD );
        	}
        }
        else
        {
        	$valid = FALSE;
        }
                
        foreach ( $snmpConfDto->getSnmpArray() as $snmpDto )
        {
        	if ( $snmpDto != NULL)
        	{
        		if ( $valid && ($snmpDto->getManager() != NULL) && $snmpDto->getManager() != "" )
        		{
        			if ( strlen( $snmpDto->getManager() ) >= self::SNMP_SERVER_MAX_SIZE )
        			{
        				throw new SMTMediaException( SMTMediaException::ERROR_SNMP_INVALID_SERVER_NAME );
        			}
        		}
        		else
        		{
        			$valid = FALSE;
        		}
        	
        		if ( $valid && ($snmpDto->getCommunity() != NULL) && $snmpDto->getCommunity() != "" )
        		{
        			if ( strlen( $snmpDto->getCommunity() ) >= self::SNMP_COMMUNITY_MAX_SIZE )
        			{
        				throw new SMTMediaException( SMTMediaException::ERROR_SNMP_INVALID_COMMUNITY );
        			}
        		}
        		else
        		{
        			$valid = FALSE;
        		}
        	
        		//test message can be null but cannot exceed 30 chars
        		if ( $valid && ($snmpDto->getTrapMessage() != NULL) && $snmpDto->getTrapMessage() != "" )
        		{
        			if ( strlen( $snmpDto->getTrapMessage() ) >= self::SNMP_TEST_MESSAGE_MAX_SIZE )
        			{
        				throw new SMTMediaException( SMTMediaException::ERROR_SNMP_INVALID_TEST_MESSAGE );
        			}
        		}
        	
        		if ( $valid && ($snmpDto->getPort() != NULL) && $snmpDto->getPort() != "" )
        		{
        			if ( $snmpDto->getPort() > self::SNMP_PORT_MAX_VALUE || $snmpDto->getPort() < 1 )
        			{
        				throw new SMTMediaException( SMTMediaException::ERROR_SNMP_INVALID_PORT );
        			}
        		}
        		else
        		{
        			$valid = FALSE;
        		}
        	}
        }
        
        return $valid;
    }
    
    /**
     * Fetch the snmp configuration
     *
     * @param integer $snmpIndex 
     * @param SMTConfSnmpDto $snmpConfDto
     * @param Boolean $snmpMediaOn
     *  
     * @return \app\services\setup\SMTSnmpDto
     */
    function retrieveSnmp( $snmpIndex, SMTConfSnmpDto $snmpConfDto, $snmpMediaOn )
    {
    	$snmpDto = NULL;
    	    	
	    	//look for  ,"  sequence to explode the string into an array $conf
	    	$snmpString = $this->sendReceive( SMTOtuApi::getSnmpCommand($snmpIndex), FALSE );
	    		
	    	//look for  ,"  sequence to explode the string into an array $conf
	    	//remove spaces in array values
	    	$snmpArray = SMTUtil::splitStringByComma( $snmpString );
	    	 
	    	if ( ($snmpArray[self::MANAGER_INDEX]!= SMTOtuApi::RES_UNKNOWN) && ($snmpArray[self::MANAGER_INDEX] != SMTOtuApi::RES_NONE) && ($snmpArray[self::MANAGER_INDEX] != "") && ($snmpArray[self::PORT_INDEX] != "") && ($snmpArray[self::PORT_INDEX] != "0") )
	    	{    		    		
	    		$snmpDto = new SMTSnmpDto();
	    		$snmpDto->setCommunity($snmpArray[self::COMMUNITY_INDEX]);
	    		$snmpDto->setIndex($snmpIndex);
	    		$snmpDto->setManager($snmpArray[self::MANAGER_INDEX]);
	    		$snmpDto->setPort($snmpArray[self::PORT_INDEX]);
	    	}
    
    	return $snmpDto;
    	}    
    
	/**
	 * Fetch all snmp configuration
	 * 
	 * @return \app\services\setup\SMTConfSnmpDto
	 */
	function retrieveSnmpConf()
	{
	    $snmpConfDto = SMTConfSnmpDto::buildSMTConfSnmpDto( FALSE, FALSE, self::DEFAULT_SNMP_HEART_BEAT_PERIOD);	   
	    $snmpMediaOn = $this->isSNMPMediaOn();	

	    $snmpConfDto->setActive($snmpMediaOn);
	    $snmpConfDto->setSnmpHearbeatActive($this->isSNMPHeartbeatOn());	    
		$snmpConfDto->setSnmpHearbeatPeriod($this->getSNMPHeartbeatPeriod());
	    
	    for( $index = self::MIN_SNMP_INDEX; $index<=self::MAX_SNMP_INDEX; $index++)
	    {
	    	$snmpDto = $this->retrieveSnmp( $index, $snmpConfDto, $snmpMediaOn );
		    if ( $snmpDto != NULL )
		    {
		    	$snmpConfDto->addSnmp( $snmpDto );
		    }
		    else
		    {
		    //don't fetch more snmp when an empty snmp is found
		    	break;
		    }
	    }
		
		return $snmpConfDto;
	}	
	
	/**
	 * Retrieve the Otu Mibs: create an archive and returns its name.
	 *
	 * @return the full mibs archive name; returns NULL if not found.
	 */
	static function retrieveMibs()
	{
	    $directory = self::getMibsDir();
	
		//remove previous archive
		exec ('rm -f '.$directory.self::MIBS_ARCHIVE.'*',$output);
	
		//create snapshot
		exec ('cd /usr/share/snmp/mibs; tar -hzcvf '.$directory.self::MIBS_ARCHIVE.'.tar.gz *.mib SNMPv2*.txt', $output, $success);
	
		//retrieve snapshot name
		$archiveName = NULL;
		$resultArray = glob( $directory.self::MIBS_ARCHIVE.'*' );
	
		//first mib is the mib to download; if glob fails, returns false
		if ( $success == 0 && $resultArray != FALSE && !empty( $resultArray ) )
		{
			$archiveName = $resultArray[0];
		}
	
		return $archiveName;
	}	
	
	/**
	 * Build Sms test message
	 * 
	 * @return string
	 */
	function buildTestSnmpMessage()
	{
		$testString = sprintf( MSG_SEND_SNMP_MESSAGE, "" );
		try
		{
		    $otu = new SMTOtu();
		    $otu->setContext( $this->getContext() );
		    $serialNumber = $otu->getOtuSerialNumber();
			$testString = sprintf( MSG_SEND_SNMP_MESSAGE, $serialNumber );
		}
		//catch exception: continue to sent if the OTU name retrieval failed
		catch( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
		}
		return $testString;
	}	
	
	/**
	 * Retrieve whether SNMP media is ON
	 * @return boolean Whether SNMP media is ON
	 */
	function isSNMPMediaOn()
	{
		//notification status on ALT1
		$mediaConfig = $this->sendReceive( SMTOtuApi::getSNMPMedia() );
		$mediaConfig = explode(",", $mediaConfig );
	
		$notificationStatus = ($mediaConfig[self::MEDIA_INDEX] == self::SNMP_MEDIA);

		return $notificationStatus;
	}
	
	/**
	 * Retrieve whether SNMP heartbeat is ON
	 * @return boolean Whether SNMP heartbeat is ON
	 */
	function isSNMPHeartbeatOn()
	{
		//notification status on ALT1
		$on = $this->sendReceive( SMTOtuApi::isSNMPHearbeat() );
	
		return (strcmp($on, SMTOtuApi::RES_ON) == 0);
	}	
	
	/**
	 * Set whether SNMP heartbeat is ON
	 * @param boolean $on
	 */
	function setSNMPHeartbeatOn( $on )
	{
		$this->send( SMTOtuApi::setSNMPHearbeat($on) );	
	}
	
	/**
	 * Set SNMP heartbeat period
	 * 
	 * @param integer $period
	 */
	function setSNMPHeartbeatPeriod( $period )
	{
		$this->send( SMTOtuApi::setSNMPHearbeatPeriod($period) );
	}

	/**
	 * Get SNMP heartbeat period
	 *
	 * @return integer
	 */
	function getSNMPHeartbeatPeriod()
	{
		$period = $this->sendReceive( SMTOtuApi::getSNMPHearbeatPeriod() );		
		return SMTUtil::convertToInt($period);
	}	
}
?>