<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\setup;

use app\events\SMTEventMessageManager;

use app\util\SMTLogger;

use app\util\SMTStatusDto;

use app\error\SMTAccessForbiddenRestException;

use app\settings\SMTSmartOtuSettings;

use app\view\SMTTestRestApiPage;
use app\services\SMTIService;
use app\services\SMTService;
use app\services\security\SMTAccessControl;
use app\serviceshelper\otau\SMTSwitchManager;
use app\parser\SMTOtuApi;
use app\sharedmemory\SMTMemoryManager;
use app\settings\SMTOtuStatusCacheDto;
use app\admin\SMTOtuMode;
use app\serviceshelper\otau\SMTRotauManager;
use app\admin\SMTOtuType;
use app\util\SMTInvalidValueException;

/**
 * Service to setup the OTU internal switch features and display its configuration.
 *
 * @author Sylvain Desplat
 */
class SMTSwitchService extends SMTService implements SMTIService
{
    /**
     * 
     * @var \app\serviceshelper\otau\SMTSwitchManager
     */
    private $switchManager = NULL;
    
    /**
     *
     * @var \app\serviceshelper\otau\SMTRotauManager
     */
    private $rotauManager = NULL;
        
    /**
     * Full json request data are by default stored in an array in parameter Default::$fullRequestDataName='request_Data'
     * Add or remove the switch: if switch detected is NULL, remove the switch from config
     *
     * @url POST /
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function updateSwitch( $request_data )
    {       
    	try
    	{
    		if ( $this->switchManager == NULL )
    		{
    			$this->switchManager = new SMTSwitchManager();
    			$this->switchManager->setContext( $this->getContext() );
    		}
    		//no OSX setup
    		$this->switchManager->disableSequencer(FALSE);
    		
    	    $configDetectSwitchDto = SMTConfDetSwitchDto::getInstance( $request_data );
    	    
    	    //if a difference exists,
    	    if ( $configDetectSwitchDto->getSwitchDifference() )
    	    {
        	    SMTSwitchManager::applySwitchDetected ( $this->getContext()->getOtuSocket() );
    	    }
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }    
        
    /**
     *
     * @url GET /detect/
     * 
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function fetchSwitchConfigWithDetection()
    {        
        $configDetectSwitchDto = new SMTConfDetSwitchDto();    
    	try
    	{
    		//no OSX setup
    		$this->getSwitchManager(TRUE)->disableSequencer(FALSE);
    		
    	    $configDetectSwitchDto->setConfigured( $this->getSwitchManager(TRUE)->getConfiguredSwitchDto() );
    	    $configDetectSwitchDto->setDetected( $this->getSwitchManager(TRUE)->getDetectedSwitchDto() );
    	    $configDetectSwitchDto->setSwitchDifference( $this->getSwitchManager(TRUE)->isDifference() );
    	    $configDetectSwitchDto->setDtssSwitch($this->getSwitchManager(TRUE)->isDtssSwitch());
    	    $configDetectSwitchDto->setSwitchDetectedHomogeneous($this->getSwitchManager(TRUE)->isSwitchDetectedHomogeneous());
    	    $configDetectSwitchDto->setIsOsxPowerOn($this->getSwitchManager(TRUE)->getOsxPowerOn());
    	    $configDetectSwitchDto->setOTAMPossible($this->getSwitchManager(TRUE)->isMolexSwitchPossible());
    	    $configDetectSwitchDto->setOTAMSwitch($this->getSwitchManager(TRUE)->hasMolexOSUDetected());
    	    $configDetectSwitchDto->setModuleCompatibleWithSwitch($this->getSwitchManager(TRUE)->isModuleCompatibleWithSwitch());
    	    $configDetectSwitchDto->setPortDistributorNumber($this->switchManager->getPortsDistributorNumber());
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $configDetectSwitchDto->getJsonData();
    }
    
    /**
     *
     * @url GET /
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchSwitchConfig()
    {
    	$detectOsx = FALSE;
    	$configDetectSwitchDto = new SMTConfDetSwitchDto();
    	try
    	{
    		if ( $this->switchManager == NULL )
    		{
	    		$this->switchManager = new SMTSwitchManager();
    			$this->switchManager->setContext( $this->getContext() );
    		}
    		
    		//no OSX setup
    		$this->switchManager->disableSequencer(FALSE);
    		
    		// si l'on n'est pas en mode SmartOTU, on force la detection si elle n'a jamais ete faite
    		if ( !SMTOtuMode::isSmartOtuFamily() && !SMTOtuStatusCacheDto::getSwitchDetectedOnce() )
    		{
    			$detectOsx = TRUE;
    		}
    		
    		$this->switchManager->retrieve($detectOsx);
    		
    		$configDetectSwitchDto->setConfigured( $this->switchManager->getConfiguredSwitchDto() );
    		$configDetectSwitchDto->setDetected( $this->switchManager->getDetectedSwitchDto() );
    		$configDetectSwitchDto->setSwitchDifference( $this->switchManager->isDifference() );
    		$configDetectSwitchDto->setDtssSwitch($this->switchManager->isDtssSwitch());
    		$configDetectSwitchDto->setVirtualSwitch($this->switchManager->hasVirtualSwitch());
    		$configDetectSwitchDto->setSwitchDetectedHomogeneous($this->switchManager->isSwitchDetectedHomogeneous());
    		$configDetectSwitchDto->setIsOsxPowerOn($this->switchManager->getOsxPowerOn());
    		$configDetectSwitchDto->setOTAMPossible($this->switchManager->isMolexSwitchPossible());
    		$configDetectSwitchDto->setOTAMSwitch($this->switchManager->hasMolexOSUDetected());
    		$configDetectSwitchDto->setModuleCompatibleWithSwitch($this->switchManager->isModuleCompatibleWithSwitch());
    		$configDetectSwitchDto->setPortDistributorNumber($this->switchManager->getPortsDistributorNumber());
    		if ( $configDetectSwitchDto->getPortDistributorNumber() > 0 )
    		{
    		    $configDetectSwitchDto->setFixedPortDistributorNumber($this->switchManager->hasFixedPortsDistributor());
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $configDetectSwitchDto->getJsonData();
    }
    
    /**
     *
     * @url POST /currentport
     *
     * @status 200
     *
     * Access granted to everybody (used by OTU application), but control on host name sending the request: only loopback address allowed.
     */
    function setCurrentPort($request_data)
    {
        $result = new SMTStatusDto();
        
    	try
    	{
    	    $request = $this->getContext()->getHttpRequest();
    	     
    	    if ( ($request->isLocalHTTPQuery()) || SMTSmartOtuSettings::isDebugMode() )
    	    {
    	        $this->getContext()->getLogger()->trace("receiving current port...", SMTLogger::PINFO );
    	        $this->getContext()->getLogger()->trace( print_r( $request_data, true ),SMTLogger::INFO, __FILE__,__METHOD__,__LINE__ );
    	         
    	        $currentPortOtuDto = SMTOtuCurrentPortDto::getInstance( $request_data );
    	        if ( $currentPortOtuDto != null)
    	        {    	        
    	            $this->getContext()->getLogger()->trace( "DTO current port:".print_r( $currentPortOtuDto, true ) );
    	            
    	            //save current port in memory
    	            $currentPortOtuDto->save();
    	            
    	            //notify client
    	            SMTEventMessageManager::getInstance()->createAndSendLinkTestUpdateEvent( NULL, FALSE, FALSE, $currentPortOtuDto->getCurrentPort() );
    	            
    	            //return status
    	            $result->setStatusOK();
    	        }
    	    }
    	    else
    	    {
    	        $this->getContext()->getLogger()->trace( sprintf("Remote host %s attempt to send current port: forbidden!",$request->getClientIpAddress() ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    	    	throw new SMTAccessForbiddenRestException( MSG_ERROR_REMOTE_HOST_ATTEMPT_TO_SEND_DATA, "Host: ".$request->getClientIpAddress() );
    	    }    	   
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $result->getJsonData();
    }
    
    /**
     *
     * @url POST /osx/address/{osxAddress}
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function setOsxAddress($osxAddress)
    {
    	try
    	{
    		if ( $this->switchManager == NULL )
    		{
    			$this->switchManager = new SMTSwitchManager();
    			$this->switchManager->setContext( $this->getContext() );
    		}
    		//OSX setup in progress
    		$this->switchManager->disableSequencer(TRUE);
    		
    		$this->switchManager->setOsxAddress($osxAddress);
    		
	        //sleep 2s before requesting the config
			sleep(2);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return array( SMTSwitchManager::OSX => SMTOtuApi::RES_OK );
    }
    
    /**
     *
     * @url GET /osx/scan
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function getOsxScan()
    {
    	$osxList = new SMTOsxListDto();
    	
    	try
    	{
    		if ( $this->switchManager == NULL )
    		{
    			$this->switchManager = new SMTSwitchManager();
    			$this->switchManager->setContext( $this->getContext() );
    		}
    		//start OSX setup
    		$this->switchManager->disableSequencer(TRUE);
    		
    		$osxListArray = SMTMemoryManager::fetchAll( SMTOsxListDto::getClass() );
    		if ( $osxListArray != NULL && is_array( $osxListArray) && count($osxListArray) > 0 )
    		{
    			if ( $osxListArray[0] != NULL )
    			{
    				$osxList = $osxListArray[0];
    			}
    		}
    		
    		//update big topology status
    		$osxList->setIsBigTopology($this->switchManager->isBigTopology());
    		$osxList->setIsOsxPowerOn($this->switchManager->getOsxPowerOn());
    		$osxList->setOTAMPossible($this->switchManager->isMolexSwitchPossible());
    		$osxList->setUOSPossible($this->switchManager->isUosSwitchPossible());
    		$osxList->setPortDistributorNumber($this->switchManager->getPortsDistributorNumber());
    		if ( $osxList->getPortDistributorNumber() > 0 )
    		{
    		    $osxList->setFixedPortDistributorNumber($this->switchManager->hasFixedPortsDistributor());
    		}
    		$hasMolexOsu = FALSE;
    		$hasJdsuOSU = $this->switchManager->hasJdsuOSUDetected();
    		if ( !$hasJdsuOSU )
    		{
    			$hasMolexOsu = $this->switchManager->hasMolexOSUDetected();
    		    if (!$hasMolexOsu)
    		    {
    		        $hasUos = $this->switchManager->hasUOSDetected();
    		        if ( !$hasUos )
    		        {    		        
        		    	$hasJdsuOSU = $this->switchManager->hasJdsuOSU();
        		    	if (!$hasJdsuOSU)
        		    	{
        		    		$hasMolexOsu= $this->switchManager->hasMolexOSU();
        		    		if (!$hasMolexOsu)
        		    		{
        		    		    $hasUos = $this->switchManager->hasUOS();
        		    		}
        		    	}
    		        }
    		    }
    		}
    		$osxList->computeSwitchType($hasJdsuOSU, $hasMolexOsu, $hasUos);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $osxList->getJsonData();
    }
    
    /**
     *
     * @url POST /osx/scan
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function scanOsx( $request_data )
    {
    	try
    	{
    		$otamScan = ($request_data != NULL && $request_data['otam'] != NULL)? filter_var( $request_data['otam'], FILTER_VALIDATE_BOOLEAN ) : FALSE;
    		$hasUos = FALSE;
    		if ( $this->switchManager == NULL )
    		{
    			$this->switchManager = new SMTSwitchManager();
    			$this->switchManager->setContext( $this->getContext() );
    		}
    		
    		//OSX setup in progress
    		$this->switchManager->disableSequencer(TRUE);
    		
    		$osxList= $this->switchManager->fullScan($otamScan);
    		
    		//update big topology status
    		$osxList->setIsBigTopology($this->switchManager->isBigTopology());
    		$osxList->setIsOsxPowerOn($this->switchManager->getOsxPowerOn());
    		$osxList->setOTAMPossible($this->switchManager->isMolexSwitchPossible());
    		$osxList->setUOSPossible($this->switchManager->isUosSwitchPossible());
    		$osxList->setPortDistributorNumber($this->switchManager->getPortsDistributorNumber());
    		
    		if (!$otamScan)
    		{
    		    $hasUos = $this->switchManager->hasUOSDetected();
    		}
    		$osxList->computeSwitchType(!$otamScan && !$hasUos, $otamScan, $hasUos);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $osxList->getJsonData();
    }
    
    /**
     *
     * @url POST /osx/show/{osxAddress}
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function showOsx($osxAddress)
    {
    	try
    	{    		
    		if ( $this->switchManager == NULL )
    		{
    			$this->switchManager = new SMTSwitchManager();
    			$this->switchManager->setContext( $this->getContext() );
    		}
    		
    		//OSX setup in progress
    		$this->switchManager->disableSequencer(TRUE);
    		
    		$this->switchManager->showOsx($osxAddress);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	return array( SMTSwitchManager::OSX => SMTOtuApi::RES_OK );
    }
    
    /**
     *
     * @url POST /osx/mode/{mode}
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function setOsxMode($mode)
    {
    	try
    	{    	    
    		if ( $this->switchManager == NULL )
    		{
    			$this->switchManager = new SMTSwitchManager();
    			$this->switchManager->setContext( $this->getContext() );
    		}
    		
    		//OSX setup in progress
    		$this->switchManager->disableSequencer(TRUE);
    		
    		switch( $mode)
    		{
    		    case SMTSwitchManager::OSX_MODE_ALL_DISTRIBUTOR:
    		        SMTSwitchManager::setBigTopology('TRUE', $this->getContext()->getOtuSocket() );
    		        SMTSwitchManager::setPortsDistributorNumber(0, $this->getContext()->getOtuSocket() );
    		        break;
    		    case SMTSwitchManager::OSX_MODE_1_DISTRIBUTOR:
    		        SMTSwitchManager::setBigTopology('FALSE', $this->getContext()->getOtuSocket() );
    		        SMTSwitchManager::setPortsDistributorNumber(1, $this->getContext()->getOtuSocket() );
    		        break;
    		    default:
    		        if ( intval($mode) > intval(SMTSwitchManager::OSX_MODE_1_DISTRIBUTOR) )
    		        {
    		          SMTSwitchManager::setBigTopology('FALSE', $this->getContext()->getOtuSocket() );
    		          SMTSwitchManager::setPortsDistributorNumber(intval($mode), $this->getContext()->getOtuSocket() );
    		        }
    		        else
    		        {
    		            throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "osxMode = 0,1,4,8,12", intval($mode) );
    		        }
    		        break;
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	return $this->getOsxScan();
    }
    
    /**
     *
     * @url POST /osx/power/{osxPower}
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function setOsxPower($osxPower)
    {
    	try
    	{
    		if ( $this->switchManager == NULL )
    		{
    			$this->switchManager = new SMTSwitchManager();
    			$this->switchManager->setContext( $this->getContext() );
    		}
    		
    		//OSX setup in progress
    		$this->switchManager->disableSequencer(TRUE);
    		
    		$this->switchManager->setOsxPower($osxPower);
    		
    		//sleep 10s before requesting the config if the OSX port was switched on
    		if ( strcasecmp($osxPower, SMTSwitchManager::OSX_POWER_ON) == 0 )
    		{
    			sleep(10);
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $this->getOsxScan();
    }
    
    /**
     * 
     * @return \app\serviceshelper\otau\SMTSwitchManager
     */
    private function getSwitchManager( $detectOsx = TRUE )
    {
        if ( $this->switchManager == NULL )
        {
            $this->switchManager = new SMTSwitchManager();
            $this->switchManager->setContext( $this->getContext() );
            $this->switchManager->retrieve($detectOsx);
        }
        return $this->switchManager;
    }
    
    /**
     *
     * @url GET /rotau/
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchRotauConfig()
    {
    	$configDetectRotauDto= new SMTConfDetRotauDto();
    	try
    	{
    	    if ( (SMTOtuMode::isONMSi() || SMTOtuMode::isONMS()) && ( SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2()|| SMTOtuType::isCOtu() ) )
	    	{
	    		if ( $this->rotauManager== NULL )
	    		{
	    			$this->rotauManager= new SMTRotauManager();
	    			$this->rotauManager->setContext( $this->getContext() );
	    		}
	    		
	    		$this->rotauManager->retrieve($configDetectRotauDto);	    		
	    	}
	    	else
	    	{
	    		new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "ROTAU", "not supported");
	    	}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $configDetectRotauDto->getJsonData();
    }    
    
    /**
     *
     * @url POST /rotau/{rotauAddress}
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function addRotau($rotauAddress)
    {
    	try
    	{
    	    if ( (SMTOtuMode::isONMSi() || SMTOtuMode::isONMS()) && ( SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2() || SMTOtuType::isCOtu() ) )
    		{
	    		if ( $this->rotauManager== NULL )
	    		{
	    			$this->rotauManager= new SMTRotauManager();
	    			$this->rotauManager->setContext( $this->getContext() );
	    		}
	    		$this->rotauManager->add($rotauAddress);
    		}
    		else
    		{
    			new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "ROTAU", "not supported");
    		}
    		
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $this->fetchRotauConfig();
    }
    
    /**
     *
     * @url DELETE /rotau/{$rotauId}
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function removeRotau($rotauId)
    {
    	try
    	{
    	    if ( (SMTOtuMode::isONMSi() || SMTOtuMode::isONMS()) && ( SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2() || SMTOtuType::isCOtu() ) )
    		{
	    		if ( $this->rotauManager== NULL )
	    		{
	    			$this->rotauManager= new SMTRotauManager();
	    			$this->rotauManager->setContext( $this->getContext() );
	    		}
	    		$this->rotauManager->remove($rotauId);
    		}
    		else
    		{
    			new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "ROTAU", "not supported");
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $this->fetchRotauConfig();
    }
    
    /**
     *
     * @url POST /rotau/
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function update($request_data)
    {
    	try
    	{
    	    if ( (SMTOtuMode::isONMSi() || SMTOtuMode::isONMS()) && ( SMTOtuType::isOTH() || SMTOtuType::isOtu8KV2() || SMTOtuType::isCOtu() ) )
    		{
    			$configDetectRotauDto = SMTConfDetRotauDto::getInstance( $request_data );
    			if ( $configDetectRotauDto!= null)
		    	{  
		    		$this->rotauManager= new SMTRotauManager();
		    		$this->rotauManager->setContext( $this->getContext() );
// 		    		$this->rotauManager->retrieve($configDetectRotauDto);
		    	}
		    	$index = 0;
		    	$differences = $configDetectRotauDto->getSwitchDifferences();
		    	foreach($differences as $difference)
		    	{
		    		if ($difference)
		    		{
		    			// WARNING TRAC 907: UPDATE ROTAU in OTU configuration when its detection is different from from saved configuration..
		    			// If the detection is empty, we use the method remove_rotau_ip which removes the rotau from the configuration
		    			// and removes its ip declaration in OTU
		    			if ( $configDetectRotauDto->isRotauDetected($index) )		
		    			{
		    				$this->rotauManager->update($configDetectRotauDto->getRotauDetected($index));
		    			}
		    			else
		    			{
		    				$this->rotauManager->remove($index);
		    			}
		    		}
		    		$index++;
		    	}
    		}
    		else
    		{
    			new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "ROTAU", "not supported");
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }
    
    /**
     * @url GET /test/updateswitch
     */
    function testUpdateSwitch()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$configDetectSwitchDto = new SMTConfDetSwitchDto(); 
        	$configSwitchDto = new SMTSwitchDto();
        	$detectSwitchDto = new SMTSwitchDto();
        	$configSwitchDto->setInput(1);
        	$configSwitchDto->setOutput(4);
        	$configSwitchDto->setSerialNumber("123");
        	
        	$detectSwitchDto->setInput(1);
        	$detectSwitchDto->setOutput(4);
        	$detectSwitchDto->setSerialNumber("123");
        	$detectSwitchDto->setGlobalSwitchEncodedString("SA201004");
        	
        	$configDetectSwitchDto->addConfigured( $configSwitchDto );
        	$configDetectSwitchDto->addDetected( $detectSwitchDto );
        	$configDetectSwitchDto->setSwitchDifference(TRUE);    	
        
        	$url = '/switch';
        	$json_string = json_encode( $configDetectSwitchDto->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }        
}
?>