<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI Solutions
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI Solutions
// Copyright JDSU 2017. All rights reserved.
// *********************************************************
namespace app\services\alarm;

use app\util\SMTResultSetDto;

use app\util\SMTLogger;

use app\database\SMTDatabaseException;

use app\util\SMTInvalidValueException;

/**
 * Peak optical alarm event Dto
 * 
 * @author Sylvain Desplat
 * @SMTClassTableNameAnnotation(tablename='peak_optical_alarm_event', tablekey='id') 
 */
class SMTPeakOpticalAlarmEventDto extends SMTOpticalAlarmEventDto
{
    /**
     * @var float
     */
	protected $referenceTopDistanceM;
	
	/**
	 * @var float
	 */
	protected $topDistanceM;
	
	/**
	 * @var float
	 */
	protected $bottomDistanceM;
    
    /**
     * Level in dB
     * 
     * @var float
     */
    protected $levelDb;
    
    /**
     * Whether peak level is valid
     * 
     * @var integer
     */
    protected $levelValid = SMTResultSetDto::BOOLEAN_VALUE_FALSE;
    
    /**
     * 
     * @var string
     */
    protected $peakName;

    const FETCH_CURRENT_PEAK_OPTICAL_ALARM_EVENT_OF_ALARM =
    "SELECT peak_optical_alarm_event.*, optical_alarm_event.* FROM peak_optical_alarm_event, optical_alarm_event, optical_alarm WHERE optical_alarm.id=:alarmId AND optical_alarm_event.id=optical_alarm.current_alarm_event_id and peak_optical_alarm_event.id = optical_alarm_event.id";

    const FETCH_PEAK_OPTICAL_ALARM_EVENTS_OF_ALARM =
    "SELECT * FROM peak_optical_alarm_event, optical_alarm_event WHERE optical_alarm_event.alarm_id=:alarmId AND peak_optical_alarm_event.id = optical_alarm_event.id";
    	
    /**
     * Fetch the current peak optical alarm event of the given alarm 
     * 
     * @param \SQLite3 $dbConnection
     * @param string $alarmId
     * 
     * @return SMTAttenuationOpticalAlarmEventDto
     */
    public static function fetch( \SQLite3 $dbConnection, $alarmId )
    {
        $query = self::FETCH_CURRENT_PEAK_OPTICAL_ALARM_EVENT_OF_ALARM;
    	$statement = $dbConnection->prepare( $query );
    	
    	if ( $statement === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement failed. Query: %s.", $query), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement Successfull. Query: %s.", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	}
    	    	
    	$bindParameters = ':alarmId=>'.$alarmId;
    	$statement->bindValue(':alarmId', $alarmId, SQLITE3_TEXT );
    	$resultSet = $statement->execute();
    	
    	if ( $resultSet === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch failed. Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch successfull for Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::DEBUG);
    	}    	
    	
    	$alarmEvent= parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    	
    	return $alarmEvent;
    }
    
    /**
     * Fetch the list of peak optical alarm events of an alarm
     *
     * @param \SQLite3 $dbConnection
     * @param string $alarmId
     *
     * @return array SMTPeakOpticalAlarmEventDto
     */
    public static function fetchAlarmEvents( \SQLite3 $dbConnection, $alarmId )
    {
        $query = self::FETCH_PEAK_OPTICAL_ALARM_EVENTS_OF_ALARM;
    	$statement = $dbConnection->prepare( $query );
    	
    	if ( $statement === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement failed. Query: %s.", $query), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement Successfull. Query: %s.", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	}
    	
    	$bindParameters = ':alarmId=>'.$alarmId;
    	$statement->bindValue(':alarmId', $alarmId, SQLITE3_TEXT );
    	$result = $statement->execute();
    	
    	if ( $result === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch failed. Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch successfull for Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::DEBUG);
    	}    	
    	
    	$opticalAlarmEvents = parent::getListFromResultSet($resultSet, get_class() );
    	
        return $opticalAlarmEvents;
    }       
    
    /**
     * Whether peak level is valid
     * 
     * @var boolean
     */
    public function isLevelValid()
    {
    	return SMTResultSetDto::ToBoolean( $this->levelValid );
    }    
    
    /**
     * Set whether peak level is valid
     * 
     * @var boolean $valid
     */
    public function setLevelValid( $valid )
    {
    	$this->levelValid = SMTResultSetDto::FromBoolean( $valid );
    }
    
    /**
     *
     * @param float $level
     */
    public function setLevelDb( $levelDb )
    {
    	if ( $levelDb !== NULL && !is_numeric( $levelDb ) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "levelDb", $levelDb );
    	}
    	$this->levelDb = $levelDb;
    }
    
    /**
     *
     * @return float
     */
    public function getLevelDb()
    {
    	return $this->levelDb;
    }
    
    /**
     *
     * @param float $referenceTopDistanceM
     */
    public function setReferenceTopDistanceM( $referenceTopDistanceM)
    {
    	if ( $referenceTopDistanceM!== NULL && !is_numeric( $referenceTopDistanceM) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "referenceTopDistanceM", $referenceTopDistanceM);
    	}
    	$this->referenceTopDistanceM= $referenceTopDistanceM;
    }
    
    /**
     *
     * @return float
     */
    public function getReferenceTopDistanceM()
    {
    	return $this->referenceTopDistanceM;
    }
    
    /**
     *
     * @param float $topDistanceM
     */
    public function setTopDistanceM( $topDistanceM)
    {
    	if ( $topDistanceM!== NULL && !is_numeric( $topDistanceM) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "topDistanceM", $topDistanceM);
    	}
    	$this->topDistanceM= $topDistanceM;
    }
    
    /**
     *
     * @return float
     */
    public function getTopDistanceM()
    {
    	return $this->topDistanceM;
    }
    
    /**
     *
     * @param float $bottomDistanceM
     */
    public function setBottomDistanceM( $bottomDistanceM)
    {
    	if ( $bottomDistanceM!== NULL && !is_numeric( $bottomDistanceM) )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "bottomDistanceM", $bottomDistanceM);
    	}
    	$this->bottomDistanceM= $bottomDistanceM;
    }
    
    /**
     *
     * @return float
     */
    public function getBottomDistanceM()
    {
    	return $this->bottomDistanceM;
    }
    
    /**
     *
     * @param string $peakName
     */
    public function setPeakName( $peakName )
    {
    	$this->peakName= $peakName;
    }
    
    /**
     *
     * @return string
     */
    public function getPeakName()
    {
    	return $this->peakName;
    }
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    public function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    public function getDtoClassName()
    {
    	return get_class();
    }
}

?>