<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\activity;

use app\util\SMTLogger;

use app\services\monitoring\SMTTestAcquisitionParametersDto;

use app\util\SMTUtil;

use app\util\SMTInvalidValueException;

use app\util\SMTDto;

/**
 * Otu activity dto to send to clients (created from SMTOtuActivityDto).
 * 
 * @author Sylvain Desplat
 */
class SMTActivityDto extends SMTDto
{
    /**
     * Activity type
     *  
     * @var string
     */
    protected $type = SMTActivityType::UNKNOWN;
    /**
     * Activity state
     * 
     * @var string
     */
    protected $state = SMTActivityState::UNKNOWN;
    /**
     * @var integer
     */
    protected $testId;
    /**
     * @var integer
     */
    protected $linkId;
    /**
     * @var string
     */
    protected $linkName;

    /**
     * Last monitoring timestamp of the test in UTC
     * @var number
     */
    protected $lastMonitoringTimeStamp;
    /**
     * Acquisition parameters formated as a string
     * 
     * @var SMTTestAcquisitionParametersDto
     * @SMTAttributeInfosAnnotation(classname='app\services\monitoring\SMTTestAcquisitionParametersDto')
     */
    protected $acquisitionParameters;
    /**
     * @var int
     */
    protected $portNumber;        
    
    /**
     * Set the activity type {@link SMTActivityType} values.
     *
     * @param string  $type
     */
    public function setType( $type )
    {
        if ( $type != NULL && !SMTActivityType::isValidActivityType( $type ) )
        {
            SMTLogger::getInstance()->trace("activity type unknown: ".$type, SMTLogger::ERROR );
            $this->type = SMTActivityType::UNKNOWN;
        }        
    	$this->type = $type;
    }
    /**
     * Returns the activity type.
     *
     * @return string
     */
    public function getType()
    {
    	return $this->type;
    }    

    /**
     * Set the activity state among {@link SMTActivityState} values.
     *
     * @param string  $state
     */
    public function setState( $state )
    {
        if ( $state != NULL && !SMTActivityState::isValidActivityState( $state ) )
        {
        	SMTLogger::getInstance()->trace("activity state unknown: ".$state, SMTLogger::ERROR );
        	$this->state = SMTActivityState::UNKNOWN;
        }        
    	$this->state = $state;
    }
    /**
     * Returns the activity state.
     *
     * @return string
     */
    public function getState()
    {
    	return $this->state;
    }
    
    /**
     * Set the test identifier or null if not set.
     *
     * @param integer $testId
     */
    public function setTestId( $testId )
    {
    	$this->testId = SMTUtil::convertToInt( $testId );
    }
    /**
     * Returns the test identifier or null.
     *
     * @return string
     */
    public function getTestId()
    {
    	return $this->testId;
    }    

    /**
     * Set the link identifier or null if not set.
     *
     * @param string $linkId
     */
    public function setLinkId( $linkId )
    {
    	$this->linkId = SMTUtil::convertToInt( $linkId );
    }
    /**
     * Returns the link identifier or null.
     *
     * @return string
     */
    public function getLinkId()
    {
    	return $this->linkId;
    }    
    /**
     * Set the link name or null if not set.
     *
     * @param string $linkName
     */
    public function setLinkName( $linkName )
    {
    	$this->linkName = $linkName;
    }
    /**
     * Returns the link name or null.
     *
     * @return string
     */
    public function getLinkName()
    {
    	return $this->linkName;
    }   
    
    /**
     * Set last monitoring timestamp of the test in UTC
     * 
     * @param number $lastMonitoringTimeStamp last monitoring timestamp of the test in UTC
     * @throws SMTInvalidValueException
     */
    public function setLastMonitoringTimeStamp( $lastMonitoringTimeStamp )
    {
        if ( $lastMonitoringTimeStamp !== NULL && !SMTUtil::isInt( $lastMonitoringTimeStamp ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "lastMonitoringTimeStamp", $lastMonitoringTimeStamp );
        }        
    	$this->lastMonitoringTimeStamp = $lastMonitoringTimeStamp;
    }
    
    /**
     * Get last monitoring timestamp of the test in UTC
     * @return number last monitoring timestamp of the test in UTC
     */
    public function getLastMonitoringTimeStamp()
    {
    	return $this->lastMonitoringTimeStamp;
    }    
    
    /**
     * Set the acquisition Parameter formated as a string or null if not set.
     *
     * @param SMTTestAcquisitionParametersDto $acquisitionParameter
     */
    public function setAcquisitionParameter( SMTTestAcquisitionParametersDto $acquisitionParameter )
    {
    	$this->acquisitionParameters = $acquisitionParameter;
    }
    /**
     * Returns the acquisition Parameter formated as a string or null.
     *
     * @return SMTTestAcquisitionParametersDto
     */
    public function getAcquisitionParameter()
    {
    	return $this->$acquisitionParameters;
    }
    
    /**
     * @param int $portNumber (if $portNumber is an int encoded in a string, the function try to convert it in integer)
     * @throws SMTInvalidValueException
     */
    public function setPortNumber( $portNumber )
    { 
    	$this->portNumber = SMTUtil::convertToInt( $portNumber );
    }

    public function getPortNumber()
    {
    	return $this->portNumber !== NULL ? intval($this->portNumber) : NULL;
    }        
    
    /**
     * Serialize as Json data the object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
}
?>