<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\admin;    

use app\http\SMTHttpRequest;

/**
 * 
 * @author Sylvain Desplat
 *
 */
class SMTLanguage
{        
    const LANGUAGE = "locale";
    const LANGUAGE_COOKIE = "l";
    const EN = "en";
    const FR = "fr";        
    const CN = "cn";
    const DEFAULT_LANGUAGE = self::EN;
        
    /**
     * Whether the string value is an existing language
     *
     * @param string $language
     * @return boolean
     */
    public static function isValid( $language )
    {
    	if ( $language == self::EN  || $language == self::FR || $language == self::CN)
    	{
    		return true;
    	}
    	return false;
    }
    
    /**
     * The supported languages keys.
     *
     *
     * @return string The languages key.
     */
    public static function getSupportedLanguages()
    {
        return array(self::FR,self::EN,self::CN );
    }    
    
    /**
     * Whether the current language is french
     * @return boolean Whether the current language is french
     */
    public static function isFrenchLanguage()
    {
        return self::getCurrentLanguage() == self::FR;
    }

    /**
     * Whether the current language is english
     * @return boolean Whether the current language is english
     */
    public static function isEnglishLanguage()
    {
    	return self::getCurrentLanguage() == self::EN || self::getCurrentLanguage() == self::DEFAULT_LANGUAGE;
    }
        
    /**
     * The language displayed name.
     * 
     * @param string $language
     * 
     * @return string The language name.
     */
    public static function getDisplayedLanguage( $language )
    {
        switch ( $language )
        {
        	case self::FR :
        	    return MSG_FRENCH;
        	    break;
        	case self::EN :
        		return MSG_ENGLISH;
        		break;
        	case self::CN :
        		return MSG_CHINESE;
        		break;                
        }
    }
    
    /**
     * Set language used in current session if the language is valid
     *
     * @param string $language the language code
     */
    public static function setCurrentLanguage( $language )
    {
    	if ( self::isValid( $language ) )
    	{
    		$_SESSION[self::LANGUAGE] = $language;
    	}
    }    
    
    /**
     * Return language used from current session or if not found from post request or either return default language;
     * 
     * @return string
     */
    public static function getCurrentLanguage()
    {
        //update session current language
        if ( !isset($_SESSION[self::LANGUAGE]) || isset($_POST[self::LANGUAGE] ) )
        { 
            self::setCurrentLanguage( self::retrieveLanguage() );
        }        
        
        $language = isset($_SESSION[self::LANGUAGE])?$_SESSION[self::LANGUAGE]: self::DEFAULT_LANGUAGE;
        
        if ( self::isValid( $language ) )
        {
            return $language;
        }
        
        return self::DEFAULT_LANGUAGE;    
    }
    
    /**
     * This method will check if the local defined in the bean is null. In this
     * case, this method will check if the local is stored in the cookie. If the
     * cookie does'nt contains the local, method will get the local defined in
     * the request.
     *
     */
    public static function retrieveLanguage()
    {
    	$localeFoundInCookie = false;
    	$lang = ''; 
    	
    	$request= new SMTHttpRequest();
    	
    	// Get locale from cookie
    	if ( $request->cookieExists( self::LANGUAGE_COOKIE ) )
    	{
    		$cookie = $request->cookieData( self::LANGUAGE_COOKIE  );
    		if ( isset ( $cookie ) && $cookie != '' )
    		{
    			$lang =  $cookie;
    			$localeFoundInCookie = true;
    		}
    	}
    	// If not in cookie, get locale from language header of the request
    	if ( !$localeFoundInCookie )
    	{
    		$lang = $request->getBrowserDefaultLanguage();
    	}    	
    	
    	//search language in post data, then in session and then, use the default language
    	if ( $lang == NULL )
    	{
    	    $lang = isset($_POST[self::LANGUAGE])?$_POST[self::LANGUAGE]: (isset($_SESSION[self::LANGUAGE])?$_SESSION[self::LANGUAGE] :self::DEFAULT_LANGUAGE);
    	}
    	
    	return $lang;
    }    
    
}
?>