<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app;

use Luracast\Restler\Routes;

use app\util\SMTLogger;

use Luracast\Restler\Defaults;

use app\settings\SMTSmartOtuSettings;

use app\settings\SMTSmartOtuSettingsCache;

use Luracast\Restler\Restler;
use app\http\SMTContext;

/**
 * Application Router:
 * routes REST requests to their service.
 * 
 * @author sdesplat
 */
class SMTRestlerRouter extends Restler
{
    const LOCK_CHECK_ROUTES= "/../tmp/checkRoutes.lock";
    
    /**
     * REST routes min cache size. Used to track empty arrays stored in routes cache.
     * Current size of the cache file is more than 800Ko 
     * @var integer
     */
    const MIN_CACHE_SIZE = 900000;
    
    /**
     * @var string
     */
    private static $checkRoutesLockFile = NULL;
    
     /**
      * Application context
      * 
      * @var SMTContext 
      */    
    protected $context = null;
    
    /**
     * Constructor
     *
     * @param SMTContext $context application context
     * 
     * @param boolean $productionMode
     *                              When set to false, it will run in
     *                              debug mode and parse the class files
     *                              every time to map it to the URL
     *
     * @param bool $refreshCache will update the cache when set to true
     */
    public function __construct( SMTContext $context, $productionMode = false, $refreshCache = false)
    {
        parent::__construct($productionMode, $refreshCache);
        $this->context = $context;             
    }  
    
    /**
     * 
     * @return \app\http\SMTContext
     */
    public function getContext()
    {
        return $this->context;
    }
    
    /**
     * Whether the REST route cache has already been successfully checked
     *   
     * @return boolean
     */
    private static function isRouteChecked()
    {
        $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
        return ( $settings != NULL )? $settings->isRouteCacheChecked() : FALSE;
    }
    
    /**
     * Set flag in application scope memory to indicate if the REST route cache has already been successfully checked  
     * 
     * @param boolean $isChecked
     */
    private static function updateRouteCacheCheckedStatus( $isChecked )
    {
//         SMTLogger::getInstance()->trace("updateRouteCacheCheckedStatus: ".$isChecked );
        $settings = SMTSmartOtuSettingsCache::getSettingsFromCacheMemory();
        if ( $settings == NULL )
        {
        	$settings = new SMTSmartOtuSettingsCache();
        }
        $settings->setIsRouteCacheChecked( $isChecked );
        $settings->save();
    }
    
    
    /**
     * Register services classes to Restler framework:
     * - load REST services routes from cache if it exists
     * - or regenerate the REST service routes mapping if it doesn't exist or if it is invalid 
     * 
     * @param array $service_class_names
     */
    public function loadRoutes( array &$service_class_names )
    {
        $routeChecked = self::isRouteChecked();

        if ( ($routeChecked != TRUE) || !$this->productionMode || !file_exists( Defaults::$cacheDirectory.DIRECTORY_SEPARATOR.'routes.php' ) || (filesize( Defaults::$cacheDirectory.DIRECTORY_SEPARATOR.'routes.php' ) < self::MIN_CACHE_SIZE ) )
        {            
           $this->checkRoutes( $service_class_names );
        }

//         $this->loadCache();
        
        //if loading fails, force a check of the routes
        $routes = isset($this->cache)? $this->cache->get('routes') : NULL;
        if ( $this->productionMode && (!isset($routes) || !is_array($routes) ) )//!$this->cached )
        {
            self::updateRouteCacheCheckedStatus(FALSE);
            $this->checkRoutes( $service_class_names );
            SMTLogger::getInstance()->trace( "checkRoutes $this->cached");
//             $this->loadCache();
        }
    }

    /**
     * Check that the REST services routes cached file route.php exists and has the right version
     * If the cache doesn't exist or is invalid, regenerate the REST service routes mapping.
     * 
     * @param array $service_class_names
     */
    private function checkRoutes( array &$service_class_names )
    {
        self::acquireCheckRoutesLock();
        try 
        {
            //check if the routes were not already checked when we were waiting for the lock to release
            $routeChecked = self::isRouteChecked();        
            //SMTLogger::getInstance()->trace( "route checked ".$routeChecked, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
            
            //check cache version
            $otuReleaseVersion = SMTSmartOtuSettings::getOtuVersion();
            $cacheFileVersion = Defaults::$cacheDirectory.DIRECTORY_SEPARATOR.$otuReleaseVersion;
            $validCache = file_exists( $cacheFileVersion ); 
            //SMTLogger::getInstance()->trace( "validCache ".$validCache, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
            if ( $validCache )
            {
                //check cache size
                $cacheSize = filesize( Defaults::$cacheDirectory.DIRECTORY_SEPARATOR.'routes.php' );
                if ( ($cacheSize == FALSE) || ($cacheSize < self::MIN_CACHE_SIZE) )
                {
                    $validCache = FALSE;
                }
            }
            
            if ( ( $routeChecked != TRUE ) || !$this->productionMode || !$validCache )
            {
                if ( $this->productionMode )
                {                   
                    //check if cache is valid (cache size)
                    if ( $validCache )
                    {
                       $routes = $this->cache->get('routes');
                       $validCache = isset($routes) && is_array($routes);
                    }
                    
                    //reset cache if it is invalid
                    if ( !$validCache )
                    {
                        //
                        SMTLogger::getInstance()->trace( "Invalid cache version: cleanup REST services routes", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__ );
                    	$this->cache->clear('routes');
                    	
                    	//remove everything from directory
                    	array_map('unlink', glob(Defaults::$cacheDirectory.'/*'));

                    	//if cache is invalid, reset its flag to force the reload
                    	$this->cached = NULL;
                    }
                }
                
                //register REST services
                foreach ( $service_class_names as $serviceName=>$serviceClassName )
                {
                    //reload internal routes array and register services
                	$this->addAPIClass( $serviceClassName, $serviceName );
                }

                //update cache if it's invalid
                if ($this->productionMode && !$validCache )
                {
                    SMTLogger::getInstance()->trace( "Invalid cache : recreate REST services routes", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__ );
                    
                    $this->cache->set('routes', Routes::toArray());
                    
                    //create cache file version if it doesn't exist
            		$handle = fopen( $cacheFileVersion, "w");
            		fclose($handle);            		
                }
                
                self::updateRouteCacheCheckedStatus(TRUE);
            }
            
            self::releaseCheckRoutesLock();
        }
        catch(\Exception $e)
        {
            SMTLogger::getInstance()->traceException($e);
            self::releaseCheckRoutesLock();
        }                
    }    
    
    /**
     * Lock file handling: acquire a lock
     *
     * @return boolean
     */
    private static function acquireCheckRoutesLock()
    {
        $eWouldBlock = 0;
    	$retries = 0;
    	$max_retries = 55;
    
    	SMTLogger::getInstance()->trace( "acquireCheckRoutesLock",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    
    	// keep trying to get a lock as long as possible ( max 56s) ( n(n+1)(2n+1)/6 )
    	//we use x² suite to sleep longer and longer because process should be very short except after reboot
    	do
    	{
    		if ($retries > 0)
    		{
    			usleep( 1000 * ( $retries*$retries ) );
    		}
    		$retries += 1;
    	} while ( (!flock(self::getCheckRoutesLockFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
    
    	return ($retries < $max_retries);
    }
    
    /**
     * release the lock
     *
     * @return boolean
     */
    private static function releaseCheckRoutesLock()
    {
    	SMTLogger::getInstance()->trace( "releaseCheckRoutesLock",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	return isset(self::$checkRoutesLockFile) && self::$checkRoutesLockFile != NULL? flock( self::$checkRoutesLockFile, LOCK_UN ) : TRUE;
    }

    /**
     * Lock file handle
     *
     * @return string
     */
    private static function getCheckRoutesLockFile()
    {
    	if ( self::$checkRoutesLockFile == NULL )
    	{
    		self::$checkRoutesLockFile = fopen( __DIR__.self::LOCK_CHECK_ROUTES, "w+");
    	}
    	return self::$checkRoutesLockFile;
    }
}


?>